// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Algorithm/Sort/hkMortonSort.h>
#include <Common/Base/Thread/Concurrency/hkConcurrency.h>

/// Split axis and value.

struct hkMortonSortSeperatingAxis { int m_axis; hkReal m_value; };
//
HK_INLINE hkMortonSortSeperatingAxis hkAlgorithm_Morton_computeSplitAxisAndValue(const hkAabb& aabb)
{
    // Select the largest axis (pseudo max. variance).
    hkMortonSortSeperatingAxis sav;
    sav.m_axis  = (aabb.m_max - aabb.m_min).getIndexOfMaxIComponent<3>();
    sav.m_value = (aabb.m_min(sav.m_axis) + aabb.m_max(sav.m_axis)) * 0.5f;
    return sav;
}

HK_INLINE void hkAlgorithm_Morton_splitSimple(const hkArrayView<hkVector4>& points, const hkMortonSortSeperatingAxis& sav, int minSize, hkArrayView<hkVector4>& leftOut, hkArrayView<hkVector4>& rightOut)
{
    const int axis = sav.m_axis;
    const hkReal value = sav.m_value;

    hkVector4* HK_RESTRICT  set0 = points.begin();
    hkVector4* HK_RESTRICT  set1 = points.end() - 1;

    // Consume left.
    while (set0 <= set1 && (*set0)(axis) < value) set0++;

    // Consume right.
    while (set1 > set0 && (*set1)(axis) >= value) set1--;

    // Segregate left-overs.
    if (set0 < set1)
    {
        do
        {
            // Swap one element.
            hkMath::swap(*set0, *set1);

            // Advance left.
            do set0++; while (set0 < set1 && (*set0)(axis) < value);

            // Advance right.
            do set1--; while (set1 > set0 && (*set1)(axis) >= value);
        } while (set0 < set1);
    }

    // Build sub-ranges.
    int numPoints = points.getSize();
    int leftSize = int(set0 - points.begin());

    if (leftSize < minSize || numPoints - leftSize < minSize)
    {
        // Sub-ranges too small, sort and split in half.
        hkSort(points.begin(), numPoints, [axis](const hkVector4& a, const hkVector4& b) { return a(axis) < b(axis); });
        leftSize = numPoints>>1;
    }

    leftOut  = hkArrayViewT::make( points.begin(), leftSize );
    rightOut = hkArrayViewT::make( points.begin() + leftSize, numPoints - leftSize );

    HK_ASSERT(0x615705E6, leftSize >= minSize && rightOut.getSize() >= minSize && (leftSize + rightOut.getSize()) == numPoints, "Invalid sub-ranges");
}

HK_EXPORT_COMMON void  hkAlgorithm::Morton::split(const hkArrayView<hkVector4>& points, int minNumElements, hkArrayView<hkVector4>& leftOut, hkArrayView<hkVector4>& rightOut)
{
    const int numPoints = points.getSize();

    hkAabb aabb; aabb.setEmpty(); aabb.includePoints(points.begin(), numPoints);
    const hkMortonSortSeperatingAxis sav = hkAlgorithm_Morton_computeSplitAxisAndValue(aabb);

    hkAlgorithm_Morton_splitSimple(points, sav, minNumElements, leftOut, rightOut);
}


// same as split but output the left and right aabb
HK_INLINE void hkAlgorithm_Morton_split(const hkArrayView<hkVector4>& points, const hkMortonSortSeperatingAxis& sav,
    hkArrayView<hkVector4>& leftOut,  hkAabb& leftAabbOut,
    hkArrayView<hkVector4>& rightOut, hkAabb& rightAabbOut )
{
    const int minSize = 1;
    const int axis = sav.m_axis;
    const hkReal value = sav.m_value;

    hkVector4* HK_RESTRICT  set0 = points.begin();
    hkVector4* HK_RESTRICT  set1 = points.end() - 1;
    do
    {
        // Consume left.
        while ( /*set0 <= set1 && */ (*set0)(axis) < value) leftAabbOut.includePoint(*(set0++));

        // Consume right.
        while (set1 > set0 && (*set1)(axis) >= value)
        {
            rightAabbOut.includePoint(*(set1--));
        }
        if (set0 >= set1) goto FINISHED;

        leftAabbOut. includePoint(*set1);   // reversed as we do this before the swap
        rightAabbOut.includePoint(*set0);
        hkMath::swap(*(set0++), *(set1--) );
    } while (true);
    FINISHED:

    // Build sub-ranges.
    int numPoints = points.getSize();
    int leftSize = int(set0 - points.begin());
    int rightSize = numPoints - leftSize;
#if defined(HK_DEBUG)
    for ( int i = 0; i < leftSize; i++ ) HK_ASSERT_NO_MSG( 0xf0345456, points[i]( axis ) < value );
    for ( int i = 0; i < rightSize; i++ ) HK_ASSERT_NO_MSG( 0xf0345456, points[i + leftSize]( axis ) >= value );
#endif

    if ( leftSize < minSize || rightSize < minSize )
    {
        // Sub-ranges too small, sort and split in half.
        hkSort( points.begin(), numPoints, [axis]( const hkVector4& a, const hkVector4& b ) { return a( axis ) < b( axis ); } );
        leftSize = numPoints >> 1;
    }

    leftOut = hkArrayViewT::make( points.begin(), leftSize );
    rightOut = hkArrayViewT::make( points.begin() + leftSize, numPoints - leftSize );

    HK_ASSERT( 0x615705E6, leftSize >= minSize && rightOut.getSize() >= minSize && (leftSize + rightOut.getSize()) == numPoints, "Invalid sub-ranges" );

}

void hkAlgorithm::Morton::sortDeprecated( const hkArrayView<hkVector4>& range, int mtTreshold )
{
    const int rangeSize = range.getSize();
    if ( rangeSize > 2 )
    {
        // Compute split axis and value.
        hkAabb aabb; aabb.setEmpty(); aabb.includePoints(range.begin(), rangeSize );
        const hkMortonSortSeperatingAxis sav = hkAlgorithm_Morton_computeSplitAxisAndValue(aabb);

        // Split.
        hkArrayView<hkVector4> leftRange, rightRange;
        hkAlgorithm_Morton_splitSimple(range, sav, 1, leftRange, rightRange);

        // Sort sub-ranges.
        if ( rangeSize > mtTreshold )
        {
            // Multi-threaded, process left branch asynchronously.
            hkConcurrency::AsyncHandle handle = hkConcurrency::async( [leftRange, mtTreshold]() { sortDeprecated( leftRange, mtTreshold ); } );
            sortDeprecated( rightRange, mtTreshold );
            hkConcurrency::await( handle );
        }
        else
        {
            // Single-threaded, process both branches on this thread.
            sortDeprecated( leftRange, mtTreshold );
            sortDeprecated( rightRange, mtTreshold );
        }
    }
}

void hkAlgorithm_Morton_sort(const hkArrayView<hkVector4>& range, const hkAabb& aabb, int mtThreshold )
{
    const int rangeSize = range.getSize();
    if (rangeSize > 2)
    {
        // Compute split axis and value.
        const hkMortonSortSeperatingAxis sav = hkAlgorithm_Morton_computeSplitAxisAndValue(aabb);

        // Split.
        hkArrayView<hkVector4> leftRange, rightRange;
        hkAabb leftAabb, rightAabb; leftAabb.setEmpty(); rightAabb.setEmpty();

        hkAlgorithm_Morton_split(range, sav, leftRange, leftAabb, rightRange, rightAabb);

        if ( rangeSize > mtThreshold )
        {
            // Multi-threaded, process left branch asynchronously.
            hkConcurrency::AsyncHandle handle = hkConcurrency::async( [leftRange, leftAabb, mtThreshold]() { hkAlgorithm_Morton_sort( leftRange, leftAabb, mtThreshold ); } );
            hkAlgorithm_Morton_sort( rightRange, rightAabb, mtThreshold );
            hkConcurrency::await(handle);
        }
        else
        {
            // Single-threaded, process both branches on this thread.
            hkAlgorithm_Morton_sort(leftRange, leftAabb, mtThreshold);
            hkAlgorithm_Morton_sort(rightRange, rightAabb, mtThreshold);
        }
    }
}


void hkAlgorithm::Morton::sort(const hkArrayView<hkVector4>& range, int mtThreshold )
{
    const int rangeSize = range.getSize();
    hkAabb aabb; aabb.setEmpty(); aabb.includePoints(range.begin(), rangeSize);
    hkAlgorithm_Morton_sort(range, aabb, mtThreshold);
}



HK_INLINE void hkAlgorithm_Morton_split2(
    const hkArrayView<hkVector4>& points, const hkMortonSortSeperatingAxis& sav, const hkArrayView<hkVector4>& buffer,
    hkArrayView<hkVector4>& leftOut, hkAabb& leftAabbOut,
    hkArrayView<hkVector4>& rightOut, hkAabb& rightAabbOut )
{
    const int minSize = 1;
    const int axis = sav.m_axis;
    const hkReal value = sav.m_value;

    const hkVector4* set0 = points.begin();
    const hkVector4* set1 = points.end() - 1;

    hkVector4* HK_RESTRICT  dest0 = buffer.begin();
    hkVector4* HK_RESTRICT  dest1 = buffer.end() - 1;
    HK_ASSERT_NO_MSG( 0xf04cfdaa, points.getSize() == buffer.getSize() );
    do
    {
        // Consume left.
        while ( /*set0 <= set1 && */ (*set0)(axis) < value )
        {
            hkVector4 v = *(set0++);
            leftAabbOut.includePoint( v );
            *(dest0++) = v;
        }

        // Consume right.
        while ( set1 > set0 && (*set1)(axis) >= value )
        {
            hkVector4 v = *(set1--);
            rightAabbOut.includePoint( v );
            *(dest1--) = v;
        }
        if ( set0 >= set1 ) goto FINISHED;
        {
            hkVector4 v = *(set1--);
            leftAabbOut.includePoint( v );  // reversed as we do this before the swap
            *(dest0++) = v;
        }
        {
            hkVector4 v = *(set0++);
            rightAabbOut.includePoint( v );
            *(dest1--) = v;
        }
    } while ( true );
FINISHED:

    // Build sub-ranges.
    int numPoints = points.getSize();
    int leftSize = int( dest0 - buffer.begin() );
    int rightSize = numPoints - leftSize;
#if defined(HK_DEBUG)
    for ( int i = 0; i < leftSize; i++ )  HK_ASSERT_NO_MSG( 0xf0345456, buffer[i]( axis ) < value );
    for ( int i = 0; i < rightSize; i++ ) HK_ASSERT_NO_MSG( 0xf0345456, buffer[i + leftSize]( axis ) >= value );
#endif

    if ( leftSize < minSize || rightSize < minSize )
    {
        // Sub-ranges too small, sort and split in half.
        hkSort( buffer.begin(), numPoints, [axis]( const hkVector4& a, const hkVector4& b ) { return a( axis ) < b( axis ); } );
        leftSize = numPoints >> 1;
    }

    leftOut  = hkArrayViewT::make( buffer.begin(), leftSize );
    rightOut = hkArrayViewT::make( buffer.begin() + leftSize, numPoints - leftSize );

    HK_ASSERT( 0x615705E6, leftSize >= minSize && rightOut.getSize() >= minSize && (leftSize + rightOut.getSize()) == numPoints, "Invalid sub-ranges" );
}

void hkAlgorithm_Morton_sort2( const hkArrayView<hkVector4>& range, const hkAabb& aabb, const hkArrayView<hkVector4>& buffer )
{
    const int rangeSize = range.getSize();
    if ( rangeSize > 2 )
    {
        // Compute split axis and value.
        const hkMortonSortSeperatingAxis sav = hkAlgorithm_Morton_computeSplitAxisAndValue( aabb );

        // Split.
        hkArrayView<hkVector4> leftRange, rightRange;
        hkAabb leftAabb, rightAabb; leftAabb.setEmpty(); rightAabb.setEmpty();

        hkAlgorithm_Morton_split2( range, sav, buffer, leftRange, leftAabb, rightRange, rightAabb );

        hkAlgorithm_Morton_sort2( leftRange,  leftAabb,  hkArrayView<hkVector4>( range.begin(), leftRange.getSize() ) );
        hkAlgorithm_Morton_sort2( rightRange, rightAabb, hkArrayView<hkVector4>( range.begin()+leftRange.getSize(), range.end()) );
    }
}


void hkAlgorithm::Morton::sort2( const hkArrayView<hkVector4>& range, const hkArrayView<hkVector4>& buffer )
{
// experimental, does not work
//  const int rangeSize = range.getSize();
//  hkAabb aabb; aabb.setEmpty(); aabb.includePoints( range.begin(), rangeSize );

//  hkAlgorithm_Morton_sort2( range, aabb, buffer );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
