// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Algorithm/Sort/hkSort.h>

// this: #include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>

// This is a Pseudorandom Number generator from  Num. Recip. p284:  Knuth-Lewis "quick and dirty" rand,
// otherwise known as randq1(). It's not great, but it's fast. Don't use it for "serious" work.
class hkPseudoRandomGenerator
{
public:

    HK_DECLARE_CLASS(hkPseudoRandomGenerator, New);
    HK_DETAIL_DECLARE_REFLECT_EXPORT(hkPseudoRandomGenerator, HK_EXPORT_COMMON);

    HK_EXPORT_COMMON explicit hkPseudoRandomGenerator(hkReflect::BypassCtorFlag) HK_ATTR(hk::Reflect(false));


    /// Ctor: Seed is set to 0 by default.
    HK_INLINE hkPseudoRandomGenerator(hkUint32 seed = 0){   m_seed = seed;  m_current = m_seed; }

    /// Can also reseed if desired. It wipes over  the last used, so effectively "restarts" the RNG
    HK_INLINE void setSeed(hkUint32 seed){  m_seed = seed;      m_current = m_seed; }

    /// Get seed used by generator. This may be useful to determine how the RNG was "started".
    HK_INLINE hkUint32 getSeed() const  {   return m_seed;  }

    /// Get current value (NOT equal to getRand32), but will allow you to "monitor" the RNG so you can reseed it later.
    HK_INLINE hkUint32 getCurrent() const   {   return m_current;   }

    /// Get random number as unsigned int
    HK_INLINE hkUint32 getRand32() {    m_current = 1664525U * m_current + 1013904223U;     return m_current;   }

    /// Get random integer in range [0-X) as unsigned int, with X<=256. Suitable for small values.
    HK_EXPORT_COMMON hkUint32 getRandChar(int x);

    /// Get random integer in range [0-X) as unsigned int, with X<=65536. Suitable for medium values.
    HK_EXPORT_COMMON hkUint32 getRandInt16(int x);

    // Get random real in range [0,1] using getRand32()
    HK_EXPORT_COMMON hkReal getRandReal01();

    /// Get random real in range [-1,1] using getRand32()
    hkReal getRandReal11()  {   return getRandReal01() * hkReal(2.0f) - hkReal(1.0f);   }

    /// Get random real in range [min,max] using getRand01()
    hkReal getRandRange( hkReal min, hkReal max ){      return getRandReal01() * ( max - min ) + min;   }

    /// Get a random integer in the range [start, end) (including start but not including end)
    HK_EXPORT_COMMON int  getRandRangeInt(int start, int end);

    /// Get a random hkRotation
    HK_EXPORT_COMMON void getRandomRotation(hkRotation& rotOut);

    /// Get a random quaternion
    HK_EXPORT_COMMON void getRandomRotation(hkQuaternion& rotOut);

    /// sets xyzw randomly between -1 and 1
    HK_EXPORT_COMMON void getRandomVector11(hkVector4& vecOut);

    /// sets xyzw randomly between -1 and 1
    HK_INLINE hkVector4 getRandomVector11() { hkVector4 v; getRandomVector11(v); return v; }

    /// sets xyzw randomly between -maxValue and maxValue
    HK_EXPORT_COMMON void getRandomVectorRange(hkReal maxValue, hkVector4& vecOut);

    /// sets xyzw randomly between 0 and 1
    HK_EXPORT_COMMON void getRandomVector01(hkVector4& vecOut);

    /// get a vector inside an aabb defined by min and max
    HK_EXPORT_COMMON void getRandomVectorRange(hkVector4Parameter min, hkVector4Parameter max, hkVector4& vecOut);

    /// Use barycentric. http://www.cgafaq.info/wiki/Random_Point_In_Triangle or http://www.exaflop.org/docs/cgafaq/cga6.html
    HK_EXPORT_COMMON void getRandomPointInTriangle(hkVector4Parameter A, hkVector4Parameter B, hkVector4Parameter C, hkVector4& vecOut);

    /// Randomize memory
    HK_EXPORT_COMMON void randomizeMemory(_Out_writes_bytes_all_(numBytes) void* data, int numBytes);

    /// Shuffle a piece of memory
    template<typename T>
    void shuffle(T* t, int n)
    {
        for(; n >= 65536; n--)
        {
            int k = getRand32() % n;
            hkAlgorithm::swap(t[k], t[n-1]);
        }
        for(; n >= 2; n--)
        {
            int k = getRandInt16(n) % n;
            hkAlgorithm::swap(t[k], t[n-1]);
        }
    }

protected:
    hkUint32 m_seed;
    hkUint32 m_current;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
