// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Algorithm/PseudoRandom/hkMT19937RandomGenerator.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/Io/Writer/hkStreamWriter.h>

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    // Parameters for the MT19937 generator.

    static const hkUint32 N = 624; // Degree of recurrence.
    static const hkUint32 M = 397; // Middle word.
    static const hkUint32 R = 31; // Separation point in a word.
    static const hkUint32 R_MASK = (0x1u<<R); // Mask obtained from the separation point.
    static const hkUint32 U = 11; // Additional tampering bitshift 1.
    static const hkUint32 A = 0x9908B0DF; // Coefficients of the rational normal form twist matrix.
    static const hkUint32 B = 0x9D2C5680; // Tampering bitmask 1.
    static const hkUint32 S = 7; // Tampering bitshift 1.
    static const hkUint32 C = 0xEFC60000; // Tampering bitmask 2.
    static const hkUint32 T = 15; // Tampering bitshift 2.
    static const hkUint32 L = 18; // Additional tampering bitshift 2.

    static const hkUint32 MC = 0x6c078965;
}

hkMT19937RandomGenerator::hkMT19937RandomGenerator()
:   m_state(HK_NULL)
{}

hkMT19937RandomGenerator::~hkMT19937RandomGenerator()
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    if ( m_state )
    {
        HK_MEMORY_TRACKER_REMOVE_RAW(m_state);
        hkMemHeapBlockFree(m_state, N);
        m_state = HK_NULL;
    }
}

void hkMT19937RandomGenerator::initialize(hkUint32 seed)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    if ( !m_state )
    {
        m_state = hkMemHeapBlockAlloc<hkUint32>(N);
        HK_MEMORY_TRACKER_ADD_RAW("MT19937_buffer", m_state, sizeof(hkUint32)*N);
    }

    m_state[0] = seed;
    for(hkUint32 i = 1; i < N; ++i)
    {
        // Fill the generator state.
        m_state[i] = MC * (m_state[i-1] ^ (m_state[i-1] >> 30)) + i;
    }
    m_index = 0;
}

hkUint32 hkMT19937RandomGenerator::getRand()
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    // The mersenne twister algorithm is a twisted generalized feedback shift register (TGFSR).

    HK_ASSERT_NO_MSG(0x1d29bca, m_state); // Can only do the job if it has been initialized.
    if(m_index == 0)
    {
        generateNumbers();
    }

    hkUint32 ret = m_state[m_index];
    ret = ret ^ (ret >> U);
    ret = ret ^ ((ret << S) & B);
    ret = ret ^ ((ret << T) & C);
    ret = ret ^ (ret >> L);

    m_index = (m_index+1) % N;
    return ret;
}

hkFloat32 hkMT19937RandomGenerator::getRandFloat32(hkFloat32 max)
{
    const hkFloat32 unit = 2.3283064365386962890625e-10; // 2^-32

    hkUint32 i = getRand();
    if(HK_VERY_LIKELY(i != 0))
    {
        return max*unit*hkFloat32(i);
    }
    else
    {
        /// Step exponent down by 32 and recurse
        return getRandFloat32(max*unit);
    }
}

void hkMT19937RandomGenerator::generateNumbers()
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    for(hkUint32 i = 0; i < N; ++i)
    {
        hkUint32 value =
            (m_state[i] & R_MASK) +
            (m_state[(i+1) % N] & (R_MASK-1));
        m_state[i] = m_state[(i+M) % N] ^ (value >> 1);
        if(value % 2 != 0)
        {
            // value is odd.
            m_state[i] = m_state[i] ^ A;
        }
    }
}

void hkMT19937RandomGenerator::readState(class hkStreamReader & reader)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    reader.read(&m_index, sizeof(m_index));
    reader.read(m_state, sizeof(hkUint32)*N);
}

void hkMT19937RandomGenerator::writeState(class hkStreamWriter & writer) const
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    writer.write(&m_index, sizeof(m_index));
    writer.write(m_state, sizeof(hkUint32)*N);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
