// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Algorithm/Hash/hkHash.h>
#include <Common/Base/Types/hkEndian.h>

// This implementation is based on the MurmurHash3 implementation from the
// SMHasher project available at https://code.google.com/p/smhasher/

//-----------------------------------------------------------------------------
// MurmurHash3 was written by Austin Appleby, and is placed in the public
// domain. The author hereby disclaims copyright to this source code.

//-----------------------------------------------------------------------------
// Platform-specific functions and macros

// Microsoft Visual Studio

#if defined(_MSC_VER)

#include <stdlib.h>

#define ROTL32(x,y)     _rotl(x,y)
#define ROTL64(x,y)     _rotl64(x,y)

#else   // defined(_MSC_VER)

    // Other compilers

namespace
{
    HK_ALWAYS_INLINE hkUint32 _rotl32(hkUint32 x, hkUint8 r)
    {
        return (x << r) | (x >> (32 - r));
    }

    HK_ALWAYS_INLINE hkUint64 _rotl64(hkUint64 x, hkUint8 r)
    {
        return (x << r) | (x >> (64 - r));
    }
}
#define ROTL32(x,y)     _rotl32(x,y)
#define ROTL64(x,y)     _rotl64(x,y)

#endif // !defined(_MSC_VER)

namespace
{
    //-----------------------------------------------------------------------------
    // Block read - if your platform needs to do endian-swapping or can only
    // handle aligned reads, do the conversion here

    HK_ALWAYS_INLINE hkUint32 _getBlock32(_In_reads_(i + 1) const hkUint32* p, hkLong i)
    {
#if HK_ENDIAN_BIG
        return hkEndian::swap(p[i]);
#else
        return p[i];
#endif
    }

    HK_ALWAYS_INLINE hkUint64 _getBlock64(_In_reads_(i + 1) const hkUint64* p, int i)
    {
#if HK_ENDIAN_BIG
        return hkEndian::swap(p[i]);
#else
        return p[i];
#endif
    }

    //-----------------------------------------------------------------------------
    // Finalization mix - force all bits of a hash block to avalanche

    HK_ALWAYS_INLINE hkUint32 _fmix32(hkUint32 h)
    {
        h ^= h >> 16;
        h *= 0x85ebca6b;
        h ^= h >> 13;
        h *= 0xc2b2ae35;
        h ^= h >> 16;

        return h;
    }

    HK_ALWAYS_INLINE hkUint64 _fmix64(hkUint64 k)
    {
        k ^= k >> 33;
        k *= 0xff51afd7ed558ccdull;
        k ^= k >> 33;
        k *= 0xc4ceb9fe1a85ec53ull;
        k ^= k >> 33;

        return k;
    }
}

//-----------------------------------------------------------------------------
namespace hkHash
{
    hkUint32 computeMurmurHash3(_In_reads_bytes_(dataSizeInBytes) const void* data, hkLong dataSizeInBytes, hkUint32 seed)
    {
        HK_ASSERT(0x33f63baf, (hkUlong(data) & 0x3) == 0, "Input data for hkHash::computeMurmurHash3() has to be 4-byte aligned.");

        const hkUint8* byteData = (const hkUint8*)data;
        const hkLong nblocks = dataSizeInBytes / 4;

        hkUint32 h1 = seed;

        const hkUint32 c1 = 0xcc9e2d51;
        const hkUint32 c2 = 0x1b873593;

        //----------
        // body

        const hkUint32* blocks = (const hkUint32 *)(byteData + nblocks * 4);

        for (hkLong i = -nblocks; i; i++)
        {
            hkUint32 k1 = _getBlock32(blocks, i);

            k1 *= c1;
            k1 = ROTL32(k1, 15);
            k1 *= c2;

            h1 ^= k1;
            h1 = ROTL32(h1, 13);
            h1 = h1 * 5 + 0xe6546b64;
        }

        //----------
        // tail

        const hkUint8* tail = (const hkUint8*)(byteData + nblocks * 4);

        hkUint32 k1 = 0;

        switch (dataSizeInBytes & 3)
        {
        case 3: k1 ^= tail[2] << 16;
        case 2: k1 ^= tail[1] << 8;
        case 1: k1 ^= tail[0];
            k1 *= c1; k1 = ROTL32(k1, 15); k1 *= c2; h1 ^= k1;
        };

        //----------
        // finalization

        h1 ^= dataSizeInBytes;

        h1 = _fmix32(h1);

        return h1;
    }

    void computeMurmurHash3_128(_In_reads_bytes_(dataSizeInBytes) const void* data, hkUint32 dataSizeInBytes, _Out_writes_bytes_(16) void* out, hkUint32 seed)
    {
        HK_ASSERT(0x47715ebf, (hkUlong(data) & 0x7) == 0, "Input data for hkHash::computeMurmurHash3_128() has to be 8-byte aligned.");

        const hkUint8* byteData = (const hkUint8*)data;
        const int nblocks = dataSizeInBytes / 16;

        hkUint64 h1 = seed;
        hkUint64 h2 = seed;

        const hkUint64 c1 = 0x87c37b91114253d5ull;
        const hkUint64 c2 = 0x4cf5ad432745937full;

        //----------
        // body

        const hkUint64* blocks = (const hkUint64*)(byteData);

        for (int i = 0; i < nblocks; i++)
        {
            hkUint64 k1 = _getBlock64(blocks, i * 2 + 0);
            hkUint64 k2 = _getBlock64(blocks, i * 2 + 1);

            k1 *= c1; k1 = ROTL64(k1, 31); k1 *= c2; h1 ^= k1;

            h1 = ROTL64(h1, 27); h1 += h2; h1 = h1 * 5 + 0x52dce729;

            k2 *= c2; k2 = ROTL64(k2, 33); k2 *= c1; h2 ^= k2;

            h2 = ROTL64(h2, 31); h2 += h1; h2 = h2 * 5 + 0x38495ab5;
        }

        //----------
        // tail

        const hkUint8* tail = (const hkUint8*)(byteData + nblocks * 16);

        hkUint64 k1 = 0;
        hkUint64 k2 = 0;

        switch (dataSizeInBytes & 15)
        {
        case 15: k2 ^= ((hkUint64)tail[14]) << 48;
        case 14: k2 ^= ((hkUint64)tail[13]) << 40;
        case 13: k2 ^= ((hkUint64)tail[12]) << 32;
        case 12: k2 ^= ((hkUint64)tail[11]) << 24;
        case 11: k2 ^= ((hkUint64)tail[10]) << 16;
        case 10: k2 ^= ((hkUint64)tail[9]) << 8;
        case  9: k2 ^= ((hkUint64)tail[8]) << 0;
            k2 *= c2; k2 = ROTL64(k2, 33); k2 *= c1; h2 ^= k2;

        case  8: k1 ^= ((hkUint64)tail[7]) << 56;
        case  7: k1 ^= ((hkUint64)tail[6]) << 48;
        case  6: k1 ^= ((hkUint64)tail[5]) << 40;
        case  5: k1 ^= ((hkUint64)tail[4]) << 32;
        case  4: k1 ^= ((hkUint64)tail[3]) << 24;
        case  3: k1 ^= ((hkUint64)tail[2]) << 16;
        case  2: k1 ^= ((hkUint64)tail[1]) << 8;
        case  1: k1 ^= ((hkUint64)tail[0]) << 0;
            k1 *= c1; k1 = ROTL64(k1, 31); k1 *= c2; h1 ^= k1;
        };

        //----------
        // finalization

        h1 ^= dataSizeInBytes; h2 ^= dataSizeInBytes;

        h1 += h2;
        h2 += h1;

        h1 = _fmix64(h1);
        h2 = _fmix64(h2);

        h1 += h2;
        h2 += h1;

        ((hkUint64*)out)[0] = h1;
        ((hkUint64*)out)[1] = h2;
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
