// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include<Common/Base/Types/hkEndian.h>
#include <Common/Base/Types/hkTrait.h>

namespace hkHash
{
    /// Computes the CRC32 hash of the given data with the data size (byte wise)
    HK_EXPORT_COMMON hkUint32 computeCrc32(_In_reads_bytes_(dataSize) const void* data, hkLong dataSize);

    /// Computes the CRC32 hash of a given zero terminated string
    HK_EXPORT_COMMON hkUint32 computeCrc32(_In_z_ const char* zeroTerminatedString);

    /// Extends a hash with some extra data (so used if data to hash not contiguous in mem).
    HK_EXPORT_COMMON hkUint32 appendCrc32(hkUint32 currentHash, _In_reads_bytes_(extraDataSize) const void* extraData, hkLong extraDataSize);

    /// Extends a hash with some zero terminated string
    HK_EXPORT_COMMON hkUint32 appendCrc32(hkUint32 currentHash, _In_z_ const char* zeroTerminatedString);

    /// Computes the CRC64 hash of the given data with the data size (byte wise)
    HK_EXPORT_COMMON hkUint64 computeCrc64(_In_reads_bytes_(dataSize) const void* data, hkLong dataSize);

    /// Computes the CRC64 hash of a given zero terminated string
    HK_EXPORT_COMMON hkUint64 computeCrc64(_In_z_ const char* zeroTerminatedString);

    /// Extends a hash with some extra data (so used if data to hash not contiguous in mem).
    HK_EXPORT_COMMON hkUint64 appendCrc64(hkUint64 currentHash, _In_reads_bytes_(extraDataSize)const void* extraData, hkLong extraDataSize);

    /// Extends a hash with some zero terminated string
    HK_EXPORT_COMMON hkUint64 appendCrc64(hkUint64 currentHash, _In_z_ const char* zeroTerminatedString);

    /// Computes the 32bit FNV-1 hash of the null terminated string.
    /// This asserts that the string pointer is non-null.
    HK_EXPORT_COMMON hkUint32 computeFNV1Hash32(_In_z_ const char* string);

    /// Computes the 32bit FNV-1 hash of len bytes of string.
    HK_EXPORT_COMMON hkUint32 computeFNV1Hash32(_In_reads_(len) const char* string, hkLong len);

    /// Computes the 64bit FNV-1 hash of the string.
    /// This asserts that the string pointer is non-null.
    HK_EXPORT_COMMON hkUint64 computeFNV1Hash64(_In_z_ const char* string);

    /// Extends a hash with some extra data.
    HK_EXPORT_COMMON hkUint64 appendFNV1Hash64(hkUint64 currenthash, _In_z_ const char* extraData);

    /// Computes the 32bit MurmurHash3 value of the given data with the given data size in bytes using the provided seed.
    /// Note: data has to be 4-byte aligned.
    HK_EXPORT_COMMON hkUint32 computeMurmurHash3(_In_reads_bytes_(dataSizeInBytes) const void* data, hkLong dataSizeInBytes, hkUint32 seed = 0x0);

    /// Computes the 128bit MurmurHash3 value of the given data with the given data size in bytes using the provided seed.
    /// Note: data has to be 8-byte aligned.
    /// Note: this is the x64 version, meaning that its optimized for x64 architecture, but still works on x86.
    /// It's not possible to choose x86 or x64 version at compile time, as they produce different hashes.
    HK_EXPORT_COMMON void computeMurmurHash3_128(_In_reads_bytes_(dataSizeInBytes) const void* data, hkUint32 dataSizeInBytes, _Out_writes_bytes_(16) void* out, hkUint32 seed = 0x0);

    // Hashes for common base types, as used by hkHashMap and hkHashSet, etc.

    HK_INLINE hkUint32 hkHashValue(short i);
    HK_INLINE hkUint32 hkHashValue(int i);
    HK_INLINE hkUint32 hkHashValue(long i);
    HK_INLINE hkUint32 hkHashValue(long long i);
    HK_INLINE hkUint32 hkHashValue(unsigned short i);
    HK_INLINE hkUint32 hkHashValue(unsigned i);
    HK_INLINE hkUint32 hkHashValue(unsigned long i);
    HK_INLINE hkUint32 hkHashValue(unsigned long long i);
    HK_INLINE hkUint32 hkHashValue(_In_ const void* p);
    HK_INLINE hkUint32 hkHashValue(_In_z_ const char* str);

    template<typename R>
    HK_INLINE hkUint32 hkHashValue(R(*ptr)());
    template<typename R, typename P0>
    HK_INLINE hkUint32 hkHashValue(R(*ptr)(P0));
    template<typename R, typename P0, typename P1>
    HK_INLINE hkUint32 hkHashValue(R(*ptr)(P0, P1));
    template<typename R, typename P0, typename P1, typename P2>
    HK_INLINE hkUint32 hkHashValue(R(*ptr)(P0, P1, P2));
    template<typename R, typename P0, typename P1, typename P2, typename P3>
    HK_INLINE hkUint32 hkHashValue(R(*ptr)(P0, P1, P2, P3));
    template<typename R, typename P0, typename P1, typename P2, typename P3, typename P4>
    HK_INLINE hkUint32 hkHashValue(R(*ptr)(P0, P1, P2, P3, P4));
    template<typename R, typename P0, typename P1, typename P2, typename P3, typename P4, typename P5>
    HK_INLINE hkUint32 hkHashValue(R(*ptr)(P0, P1, P2, P3, P4, P5));

    namespace Detail
    {
        /// Helper trait to find "hkUint32 T::hash()" methods (const or not)
        HK_DECLARE_HAS_METHOD_TRAIT(hkUint32, hash);

        /// Helper trait to find "hkUint32 T::getHash()" methods (const or not)
        HK_DECLARE_HAS_METHOD_TRAIT(hkUint32, getHash);
    }

    /// Generic implementation returning t.hash() if it is suitable.
    template<typename T>
    hkUint32 hkHashValue(const T& t, typename hkTrait::EnableIf<Detail::HasMethod_hash<T>::Const != 0>::Type* = 0)
    {
        return t.hash();
    }

    /// Generic implementation returning t.getHash() if it is suitable.
    template<typename T>
    hkUint32 hkHashValue(const T& t, typename hkTrait::EnableIf<Detail::HasMethod_getHash<T>::Const != 0>::Type* = 0)
    {
        return t.getHash();
    }

    /// Build a new hash from two hashes.
    HK_ALWAYS_INLINE hkUint32 combineHashValues(hkUint32 h0, hkUint32 h1);
};

#include <Common/Base/Algorithm/Hash/hkHash.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
