// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

namespace hkDirectedGraphTypes
{
    //
    //  Array-based implementation

    namespace Array
    {
        //
        //  Allocates a new element

        template <class ElementType>
        HK_INLINE typename ElementSet<ElementType>::ElementId ElementSet<ElementType>::allocate()
        {
            // Grow storage if necessary
            if ( !isValid(m_firstFreeElementId) )
            {
                grow();
            }
            HK_ASSERT_NO_MSG(0x25bfc7b5, isValid(m_firstFreeElementId));

            // Allocate the element
            const ElementId elementId   = m_firstFreeElementId;
            ElementType& element        = (*this)[elementId];
            m_firstFreeElementId        = element.getNext();

            // Add it to the list of allocated elements
            element.setNext(m_firstAllocatedElementId);
            m_firstAllocatedElementId = elementId;

            // Return it
            return elementId;
        }

        //
        //  Grows the storage

        template <class ElementType>
        void ElementSet<ElementType>::grow()
        {
            HK_ASSERT_NO_MSG(0x58ee8e13, !m_firstFreeElementId.isValid());

            // Resize. Take advantage of all allocated memory
            const int oldSize = m_elements.getSize();
            m_elements.expandBy(1);
            const int newSize = m_elements.getCapacity();
            m_elements.setSize(newSize);

            // Setup free pointers
            for (int k = oldSize; k < newSize; k++)
            {
                m_elements[k].setNext(ElementId(k + 1));
            }
            m_elements[newSize - 1].setNext(ElementId::invalid());
            m_firstFreeElementId = ElementId(oldSize);
        }

        //
        //  Frees the given element

        template <class ElementType>
        HK_INLINE void ElementSet<ElementType>::free(ElementId elementId)
        {
            // Remove it from the allocated list
            ElementId prevId = ElementId::invalid();
            for (ElementId crtId = m_firstAllocatedElementId; crtId.isValid() && (crtId != elementId); )
            {
                prevId = crtId;
                crtId = (*this)[crtId].getNext();
            }

            const ElementId nextId = (*this)[elementId].getNext();
            if ( prevId.isValid() ) {   (*this)[prevId].setNext(nextId);    }
            else                    {   m_firstAllocatedElementId = nextId; }

            // Add to free list
            (*this)[elementId].setNext(m_firstFreeElementId);
            m_firstFreeElementId = elementId;
        }
    }

    //
    //  Linked-list based implementation

    namespace List
    {
        //
        //  Allocates a new element

        template <class ElementType>
        HK_INLINE typename ElementSet<ElementType>::ElementId ElementSet<ElementType>::allocate()
        {
            ElementType* p = new ElementType();
            p->setNext(m_head);
            m_head = p;

            return p;
        }

        //
        //  Frees the given element

        template <class ElementType>
        HK_INLINE void ElementSet<ElementType>::free(ElementId elementId)
        {
            ElementType* prev = HK_NULL;
            for (ElementType* crt = m_head; crt && (crt != elementId); )
            {
                prev = crt;
                crt = crt->getNext();
            }

            ElementType* next = elementId->getNext();
            if ( prev ) {   prev->setNext(next);    }
            else        {   m_head = next;          }

            delete elementId;
        }
    }
}

//
//  Adds an edge between the two vertices

template <class VertexStorage, class EdgeStorage>
typename hkDirectedGraph<VertexStorage, EdgeStorage>::EdgeId hkDirectedGraph<VertexStorage, EdgeStorage>::addEdge(VertexId fromVertexId, VertexId toVertexId)
{
    // Allocate the edge
    const EdgeId edgeId = m_edges.allocate();
    Edge& edge          = accessEdge(edgeId);

    // Set vertices
    edge.setStartVertex(fromVertexId);
    edge.setEndVertex(toVertexId);

    // Add it to the start vertex
    Vertex& vFrom = accessVertex(fromVertexId);
    edge.setNextOutgoingEdge(vFrom.getFirstOutgoingEdge());
    vFrom.setFirstOutgoingEdge(edgeId);

    // Add it to the end vertex
    Vertex& vTo = accessVertex(toVertexId);
    edge.setNextIncomingEdge(vTo.getFirstIncomingEdge());
    vTo.setFirstIncomingEdge(edgeId);

    return edgeId;
}

//
//  Removes an edge

template <class VertexStorage, class EdgeStorage>
void hkDirectedGraph<VertexStorage, EdgeStorage>::removeEdge(EdgeId edgeId)
{
    Edge& edge = accessEdge(edgeId);

    // Remove the edge from the start vertex
    {
        Vertex& vtx     = accessVertex(edge.getStartVertex());
        EdgeId prevId   = EdgeStorage::invalidElementId();

        for (EdgeId crtId = vtx.getFirstOutgoingEdge(); EdgeStorage::isValid(crtId) && (edgeId != crtId); )
        {
            prevId = crtId;
            const Edge& e = getEdge(crtId);
            crtId = e.getNextOutgoingEdge();
        }

        if ( EdgeStorage::isValid(prevId) ) {   accessEdge(prevId).setNextOutgoingEdge(edge.getNextOutgoingEdge()); }
        else                                {   vtx.setFirstOutgoingEdge(edge.getNextOutgoingEdge());               }
    }

    // Remove the edge from the end vertex
    {
        Vertex& vtx     = accessVertex(edge.getEndVertex());
        EdgeId prevId   = EdgeStorage::invalidElementId();

        for (EdgeId crtId = vtx.getFirstIncomingEdge(); EdgeStorage::isValid(crtId) && (edgeId != crtId); )
        {
            prevId = crtId;
            const Edge& e = getEdge(crtId);
            crtId = e.getNextIncomingEdge();
        }

        if ( EdgeStorage::isValid(prevId) ) {   accessEdge(prevId).setNextIncomingEdge(edge.getNextIncomingEdge()); }
        else                                {   vtx.setFirstIncomingEdge(edge.getNextIncomingEdge());               }
    }

    edge.setStartVertex(VertexStorage::invalidElementId());
    edge.setEndVertex(VertexStorage::invalidElementId());
    m_edges.free(edgeId);
}

//
//  Adds a vertex

template <class VertexStorage, class EdgeStorage>
typename hkDirectedGraph<VertexStorage, EdgeStorage>::VertexId hkDirectedGraph<VertexStorage, EdgeStorage>::addVertex()
{
    // Allocate the vertex
    const VertexId vid  = m_vertices.allocate();

    // Set it up
    Vertex& v = accessVertex(vid);
    v.setFirstIncomingEdge(EdgeStorage::invalidElementId());
    v.setFirstOutgoingEdge(EdgeStorage::invalidElementId());

    return vid;
}

//
//  Removes a vertex

template <class VertexStorage, class EdgeStorage>
void hkDirectedGraph<VertexStorage, EdgeStorage>::removeVertex(VertexId vertexId)
{
    Vertex& vtx = accessVertex(vertexId);

    // Remove all incoming edges
    while ( EdgeStorage::isValid(vtx.getFirstIncomingEdge()) )
    {
        removeEdge(vtx.getFirstIncomingEdge());
    }

    // Remove all outgoing edges
    while ( EdgeStorage::isValid(vtx.getFirstOutgoingEdge()) )
    {
        removeEdge(vtx.getFirstOutgoingEdge());
    }

    // Free vertex
    m_vertices.free(vertexId);
}

//
//  Returns the number of incoming edges associated with the given vertex

template <class VertexStorage, class EdgeStorage>
inline int hkDirectedGraph<VertexStorage, EdgeStorage>::getNumIncomingEdges(VertexId vertexId)
{
    const Vertex& vEnd = getVertex(vertexId);

    int numEdges = 0;
    for (EdgeId incomingEdgeId = vEnd.getFirstIncomingEdge(); EdgeStorage::isValid(incomingEdgeId); numEdges++)
    {
        const Edge& incomingEdge = getEdge(incomingEdgeId);
        HK_ASSERT_NO_MSG(0x6224145c, incomingEdge.getEndVertex() == vertexId);
        incomingEdgeId = incomingEdge.getNextIncomingEdge();
    }

    return numEdges;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
