// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkHandle.h>

/// Types used by the directed graph
namespace hkDirectedGraphTypes
{
    /// Type used to wrap the user data associated with a vertex / edge
    template <class PayloadType>
    struct PayloadStorage
    {
        HK_DECLARE_CLASS(PayloadStorage, New);

        PayloadType m_payload;
    };

    template <> struct PayloadStorage<void> {};

    /// An edge in the graph
    template <class T, class VertexId, class EdgeId>
    class GenericEdge : public PayloadStorage<T>
    {
        public:

            HK_DECLARE_CLASS(GenericEdge, New);
            typedef EdgeId ElementId;

            /// Gets / sets the Id of the start vertex
            HK_INLINE VertexId getStartVertex() const       {   return m_vertexIds[0];  }
            HK_INLINE void setStartVertex(VertexId vid)     {   m_vertexIds[0] = vid;   }

            /// Gets / sets the Id of the end vertex
            HK_INLINE VertexId getEndVertex() const         {   return m_vertexIds[1];  }
            HK_INLINE void setEndVertex(VertexId vid)       {   m_vertexIds[1] = vid;   }

            /// Gets / sets the Id of the next outgoing edge from start-vertex
            HK_INLINE EdgeId getNextOutgoingEdge() const    {   return m_nextOutgoingEdgeId;    }
            HK_INLINE void setNextOutgoingEdge(EdgeId eid)  {   m_nextOutgoingEdgeId = eid;     }

            /// Gets / sets the Id of the next incoming edge into end-vertex
            HK_INLINE EdgeId getNextIncomingEdge() const    {   return m_nextIncomingEdgeId;    }
            HK_INLINE void setNextIncomingEdge(EdgeId eid)  {   m_nextIncomingEdgeId = eid;     }

            /// Gets / sets the next edge in the list of allocated / free edges
            HK_INLINE EdgeId getNext() const                {   return m_next;  }
            HK_INLINE void setNext(EdgeId eid)              {   m_next = eid;   }

        protected:

            VertexId m_vertexIds[2];        ///< The start and end vertex Ids
            EdgeId m_nextIncomingEdgeId;    ///< The Id of the next edge coming into the start vertex
            EdgeId m_nextOutgoingEdgeId;    ///< The Id of the next edge coming out of the end vertex
            EdgeId m_next;                  ///< The Id of hte next edge in the free / allocated list
    };

    /// A vertex in the graph
    template <class T, class VertexId, class EdgeId>
    class GenericVertex : public PayloadStorage<T>
    {
        public:

            HK_DECLARE_CLASS(GenericVertex, New);
            typedef VertexId ElementId;

            /// Gets / sets the Id of the first incoming edge
            HK_INLINE EdgeId getFirstIncomingEdge() const   {   return m_edgeIds[1];    }
            HK_INLINE void setFirstIncomingEdge(EdgeId eid) {   m_edgeIds[1] = eid;     }

            /// Gets / sets the Id of the first outgoing edge
            HK_INLINE EdgeId getFirstOutgoingEdge() const   {   return m_edgeIds[0];    }
            HK_INLINE void setFirstOutgoingEdge(EdgeId eid) {   m_edgeIds[0] = eid;     }

            /// Gets / sets the next allocated / free vertex
            HK_INLINE VertexId getNext() const              {   return m_next;  }
            HK_INLINE void setNext(VertexId vid)            {   m_next = vid;   }

        protected:

            EdgeId m_edgeIds[2];        ///< The Id of the first incoming / outgoing edge
            VertexId m_next;            ///< The Id of the next vertex in the (free) list
    };

    /// Types suitable for an hkArray based storage
    namespace Array
    {
        HK_DECLARE_HANDLE(VertexId, hkUint32, 0xFFFFFFFFU);
        HK_DECLARE_HANDLE(EdgeId, hkUint32, 0xFFFFFFFFU);

        /// Generic collection backed by an hkArray
        template <class T>
        class ElementSet
        {
            public:

                HK_DECLARE_CLASS(ElementSet, New);
                typedef T                       ElementType;
                typedef typename T::ElementId   ElementId;

                /// Constructor
                HK_INLINE ElementSet()
                :   m_firstFreeElementId(ElementId::invalid())
                ,   m_firstAllocatedElementId(ElementId::invalid())
                {}

                /// Returns the capacity
                HK_INLINE int getCapacity() const   {   return m_elements.getSize();    }

                /// Returns the vertex with the given Id
                HK_INLINE const ElementType& operator[](ElementId elementId) const  {   return m_elements[elementId.value()];   }
                HK_INLINE ElementType& operator[](ElementId elementId)              {   return m_elements[elementId.value()];   }

                /// Allocates a new element
                HK_INLINE ElementId allocate();

                /// Frees the given element
                HK_INLINE void free(ElementId elementId);

                /// Returns true if the given ElementId is valid
                static HK_INLINE bool HK_CALL isValid(ElementId elementId)  {   return elementId.isValid(); }

                /// Returns an invalid ElementId
                static HK_INLINE ElementId HK_CALL invalidElementId()       {   return ElementId::invalid();    }

                /// Returns the first allocated element
                HK_INLINE ElementId getFirstElement() const {   return m_firstAllocatedElementId;   }

                /// Returns the next allocated element
                HK_INLINE ElementId getNextElement(ElementId eid) const {   return (*this)[eid].getNext();  }

            protected:

                /// Grows the storage
                void grow();

            protected:

                hkArray<ElementType> m_elements;        ///< The array of elements
                ElementId m_firstFreeElementId;         ///< The Id of the first un-allocated element
                ElementId m_firstAllocatedElementId;    ///< The Id of the first allocated element
        };

        /// An hkArray backed storage for the vertices / edges in the graph
        template <class T> struct VertexSet : public ElementSet< GenericVertex  <T, VertexId, EdgeId> > {};
        template <class T> struct EdgeSet   : public ElementSet< GenericEdge    <T, VertexId, EdgeId> > {};
    }

    /// Types suitable for a linked-list implementation
    namespace List
    {
        /// A vertex / edge in the graph
        template <class TV, class TE> struct Vertex;
        template <class TV, class TE> struct Edge;
        template <class TV, class TE> struct Vertex : public GenericVertex  <TV, Vertex<TV, TE>*, Edge<TV, TE>*>    {};
        template <class TV, class TE> struct Edge   : public GenericEdge    <TE, Vertex<TV, TE>*, Edge<TV, TE>*>    {};

        /// Generic collection backed by a linked-list
        template <class T>
        class ElementSet
        {
            public:

                HK_DECLARE_CLASS(ElementSet, New);
                typedef T                       ElementType;
                typedef typename T::ElementId   ElementId;

                /// Constructor
                HK_INLINE ElementSet()
                :   m_head(HK_NULL)
                {}

                /// Returns the vertex with the given Id
                HK_INLINE const ElementType& operator[](ElementId vid) const    {   return *vid;    }
                HK_INLINE ElementType& operator[](ElementId vid)                {   return *vid;    }

                /// Allocates a new element
                HK_INLINE ElementId allocate();

                /// Frees the given element
                HK_INLINE void free(ElementId elementId);

                /// Returns true if the given ElementId is valid
                static HK_INLINE bool HK_CALL isValid(ElementId elementId)  {   return elementId != HK_NULL;    }

                /// Returns an invalid ElementId
                static HK_INLINE ElementId HK_CALL invalidElementId()       {   return HK_NULL; }

                /// Returns the first allocated element
                HK_INLINE ElementId getFirstElement() const {   return m_head;  }

                /// Returns the next allocated element
                HK_INLINE ElementId getNextElement(ElementId eid)   {   return eid->getNext();  }

            protected:

                ElementType* m_head;
        };

        /// A list backed storage for the vertices / edges in the graph
        template <class TV, class TE> struct VertexSet  : public ElementSet< Vertex<TV, TE> >   {};
        template <class TV, class TE> struct EdgeSet    : public ElementSet< Edge<TV, TE> >     {};
    }
}

// Generic directed graph
template <class VertexSet, class EdgeSet>
class hkDirectedGraph
{
    public:

        typedef typename VertexSet::ElementType Vertex;
        typedef typename VertexSet::ElementId   VertexId;
        typedef typename EdgeSet::ElementType   Edge;
        typedef typename EdgeSet::ElementId     EdgeId;

        /// Returns the vertex capacity
        HK_INLINE int getVertexCapacity() const {   return m_vertices.getCapacity();    }

    public:

        /// Returns the vertex with the given Id
        HK_INLINE const Vertex& getVertex(VertexId vid) const   {   return m_vertices[vid]; }
        HK_INLINE Vertex& accessVertex(VertexId vid)            {   return m_vertices[vid]; }

        /// Returns the edge with the given Id
        HK_INLINE const Edge& getEdge(EdgeId eid) const         {   return m_edges[eid];    }
        HK_INLINE Edge& accessEdge(EdgeId eid)                  {   return m_edges[eid];    }

        /// Adds an edge between the two vertices
        EdgeId addEdge(VertexId fromVertexId, VertexId toVertexId);

        /// Removes an edge
        void removeEdge(EdgeId edgeId);

        /// Adds a vertex
        VertexId addVertex();

        /// Removes a vertex
        void removeVertex(VertexId vertexId);

        /// Returns the number of incoming edges associated with the given vertex
        inline int getNumIncomingEdges(VertexId vertexId);

    protected:

        VertexSet m_vertices;       ///< The vertices
        EdgeSet m_edges;            ///< The edges
};

#include <Common/Base/Algorithm/Graph/hkDirectedGraph.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
