// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Algorithm/Compression/hkBufferCompression.h>
#include <Common/Base/Algorithm/Compression/hkBufferCompressionInternal.h>

namespace
{
    // memcpy(), but takes pointers as reference parameters and updates them
    HK_INLINE void copyout(hkUint8*& dst, const hkUint8*& src, int n)
    {
        while (n--) *dst++ = *src++;
    }

    template<bool enableChecks>
    HK_INLINE hkResult decompressBufferImpl(_In_reads_bytes_(inlen) const void* in_start, hk_size_t inlen, _Outref_result_bytebuffer_(outlen) void*& out_start, hk_size_t outlen)
    {
        HK_ASSERT(0x2bbf4f16, in_start != nullptr, "Invalid input buffer");
        HK_ASSERT(0x79d0c243, out_start != nullptr, "Invalid output buffer");

        const hkUint8* in = static_cast<const hkUint8*>(in_start);
        const hkUint8* const in_end = in + inlen;

        hkUint8* out = static_cast<hkUint8*>(out_start);
        const hkUint8* const out_end = out + outlen;

        while (in < in_end)
        {
            const hkUint8 control = *in;
            in++;

            // Formats:
            // {0      :3, litcount:5} followed by "litcount+1" literal bytes (1 - 32)
            // {backlen:3, offhigh :5} {offlow:8} is a reference to "backlen+2" bytes at offset "offhigh||offlow + 1"
            // {7      :3, offhigh :5} {offlow:8} {backlen:8} is a reference to "backlen+9" bytes at offset "offhigh||offlow + 1"

            hkUint32 backlen = control >> hkBufferCompression::LITCOUNT_BITS;
            const hkUint32 fieldB = (control & (hkBufferCompression::MAX_LITERAL_LEN - 1));
            if (backlen == 0)
            {
                // Literal run
                hkUint32 litlen = fieldB + 1;   // no overflow possible due to previous binary op
                if (enableChecks && (in + litlen > in_end || out + litlen > out_end))
                {
                    return HK_FAILURE;
                }
                copyout(out, in, litlen);
            }
            else // back reference
            {
                // a backreference needs at least 1 more input byte
                if (enableChecks && in >= in_end)
                {
                    return HK_FAILURE;
                }

                // Backreference
                const hkUint32 offset = (fieldB << 8) + (*in) + 1;
                in++;
                backlen += 2;
                if (backlen == 9)
                {
                    // a long backref needs one more byte (3 total)
                    if (enableChecks && in >= in_end)
                    {
                        return HK_FAILURE;
                    }

                    // long-format backreference
                    backlen += *in;
                    in++;
                }
                // Copy a range of data which is already in the out buffer
                const hkUint8* ref = out - offset;
                if (enableChecks && (out + backlen > out_end || ref < out_start))
                {
                    return HK_FAILURE;
                }
                copyout(out, ref, backlen);
            }
        }
        out_start = out;
        return HK_SUCCESS;
    }
}

hkResult hkBufferCompression::decompressBuffer(_In_reads_bytes_(inlen) const void* in_start, hk_size_t inlen, _Outref_result_bytebuffer_(outlen) void*& out_start, hk_size_t outlen)
{
    return decompressBufferImpl<true>(in_start, inlen, out_start, outlen);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
