// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Types/Geometry/Aabb16/hkAabb16.h>


#if !defined(HK_ARCH_PPC) && (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
#   define HK_1AXIS_SWEEP_USE_SIMD
#endif

/// Fast 1-axis sweep implementation
template <typename T, typename AABB>
class hk1AxisSweepT
{
    public:
        struct HK_EXPORT_COMMON KeyPair
        {
            T m_keyA;
            T m_keyB;
        };

        /// collide 2 lists, needs 4 padding elements at the end of pa && pb.
        _Ret_range_(0, maxNumPairs)
        static HK_EXPORT_COMMON int HK_CALL collide(
            _In_reads_(numA + 4) const AABB* pa, _In_range_(>=, 0) int numA,
            _In_reads_(numB + 4) const AABB* pb, _In_range_(>=, 0) int numB,
            _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkipped );

        /// collide 2 lists, needs 2 padding elements at the end of pb.
        _Ret_range_(0, maxNumPairs)
        static HK_EXPORT_COMMON int HK_CALL collidePadding2(
            _In_reads_(numA + 2) const AABB* pa, _In_range_(>=, 0) int numA,
            _In_reads_(numB + 2) const AABB* pb, _In_range_(>=, 0) int numB,
            _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkipped );


        /// collide 1 list with itself, needs 4 padding elements at the end.
        _Ret_range_(0, maxNumPairs)
        static HK_EXPORT_COMMON int HK_CALL collide(
            _In_reads_(numA + 4) const AABB* pa, _In_range_(>=, 0) int numA,
            _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkippedOut );

        /// collide 1 list with itself, needs 2 padding elements at the end.
        _Ret_range_(0, maxNumPairs)
        static HK_EXPORT_COMMON int HK_CALL collidePadding2(
            _In_reads_(numA + 2) const AABB* pa, _In_range_(>=, 0) int numA,
            _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkippedOut );

        static HK_EXPORT_COMMON void HK_CALL sortAabbs(AABB* aabbs, int size);

        // Radix sort types
        typedef typename hkRadixSort::Detector<sizeof(T)>   RadixSort;
        typedef typename RadixSort::SortData                SortData;

        static HK_EXPORT_COMMON void HK_CALL sortAabbs(_Inout_updates_(HK_NEXT_MULTIPLE_OF(4, size)) AABB* aabbs, _In_range_(>=, 0) int size, hkArray<SortData>& sortBuffer, hkArray<AABB>& tempAabbs);

        //
        //  Experimental functions
        //

        /// collide 2 lists, if HK_1AXIS_SWEEP_USE_SIMD is defined you have to pass in a buffer with size = sizeof(AABB) * (numB+5)/2.
        /// Needs 4 elements of padding at the end of pa && pb.
        _Ret_range_(0, maxNumPairs)
        static HK_EXPORT_COMMON int HK_CALL collideSIMD(
            _In_reads_(numA + 4) const AABB* pa, _In_range_(>=, 0) int numA,
            _In_reads_(numB + 4) const AABB* pb, _In_range_(>=, 0) int numB,
            _Out_writes_bytes_(bufferSizeInBytes) void* buffer, _In_range_(>=, 0) int bufferSizeInBytes,
            _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkipped );

        /// collide 1 list with itself trying to use SIMD code if SIMD code is faster, needs 2 padding elements at the end.
        _Ret_range_(0, maxNumPairs)
        static HK_EXPORT_COMMON int HK_CALL collideSIMD(
            _In_reads_(numA + 4) const AABB* pa, _In_range_(>=, 0) int numA,
            _Out_writes_bytes_(bufferSizeInBytes) void* buffer, _In_range_(>=, 0) int bufferSizeInBytes,
            _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkippedOut );

};

/// Fast 1-axis sweep implementation for hkAabb16
template <typename T>
class hk1AxisSweep16: public hk1AxisSweepT<T, hkAabb16 >
{

};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
