// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Algorithm/Collide/1AxisSweep16/hk1AxisSweep16.h>
#include <Common/Base/Algorithm/Sort/hkRadixSort.h>
#include <Common/Base/Math/Vector/hkIntVector.h>
#include <Common/Base/Types/Geometry/Aabb24_16_24/hkAabb24_16_24.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>

template<int flip, typename T, typename AABB>
HK_INLINE typename hk1AxisSweepT<T,AABB>::KeyPair* hk1AxisSweepT_appendPair(
    const AABB& aabb0, const AABB& aabb1,
    _Inout_ptrdiff_count_(end) typename hk1AxisSweepT<T,AABB>::KeyPair* HK_RESTRICT pairOut,
    _Notvalid_ const typename hk1AxisSweepT<T,AABB>::KeyPair* HK_RESTRICT end, _Out_ int& numPairsSkipped)
{
    if ( pairOut < end)
    {
        // Disable if condition always true/false warnings
        HK_DETAIL_DIAG_MSVC_SUPPRESS(25041 25042)
        if ( !flip )
        {
            aabb0.getKey(pairOut->m_keyA);
            aabb1.getKey(pairOut->m_keyB);
        }
        else
        {
            aabb1.getKey(pairOut->m_keyA);
            aabb0.getKey(pairOut->m_keyB);
        }
        pairOut++;
    }
    else
    {
        numPairsSkipped = numPairsSkipped + 1;
    }

    return pairOut;
}

template<int flipKeys, typename T, typename AABB>
HK_INLINE typename hk1AxisSweepT<T,AABB>::KeyPair* hk1AxisSweepT_scanList(
    const AABB& query, _In_reads_(_Inexpressible_()) const AABB* HK_RESTRICT sxyz,
    _Inout_ptrdiff_count_(end) typename hk1AxisSweepT<T,AABB>::KeyPair* HK_RESTRICT pairsOut,
    _Notvalid_ const typename hk1AxisSweepT<T,AABB>::KeyPair* HK_RESTRICT end, _Out_ int& numPairsSkipped )
{
    hkUint32 maxX = query.getMaxXSortKey();
    while( sxyz->getMinXSortKey() <= maxX )
    {
        hkBoolLL ov0 = AABB::yzDisjoint( query, sxyz[0] );
        hkBoolLL ov1 = AABB::yzDisjoint( query, sxyz[1] );
        hkBoolLL ov2 = AABB::yzDisjoint( query, sxyz[2] );
        hkBoolLL ov3 = AABB::yzDisjoint( query, sxyz[3] );

        if ( !((ov0&ov1) & (ov2&ov3)) )
        {
            {
                if ( !ov0 )
                {
                    pairsOut = hk1AxisSweepT_appendPair<flipKeys, T>( query, sxyz[0], pairsOut, end, numPairsSkipped  );
                }
                if (!ov1 )
                {
                    if ( sxyz[1].getMinXSortKey() <= maxX )
                    {
                        pairsOut = hk1AxisSweepT_appendPair<flipKeys, T>( query, sxyz[1], pairsOut, end, numPairsSkipped  );
                    }
                }
            }

            {
                if ( !ov2 )
                {
                    if ( sxyz[2].getMinXSortKey() <= maxX )
                    {
                        pairsOut = hk1AxisSweepT_appendPair<flipKeys, T>( query, sxyz[2], pairsOut, end, numPairsSkipped  );
                    }
                }
                if ( !ov3 )
                {
                    if ( sxyz[3].getMinXSortKey() <= maxX )
                    {
                        pairsOut = hk1AxisSweepT_appendPair<flipKeys, T>( query, sxyz[3], pairsOut, end, numPairsSkipped  );
                    }
                }
            }
        }
        sxyz+=4;
    }
    return pairsOut;
}


template<int flipKeys, typename T, typename AABB>
HK_INLINE typename hk1AxisSweepT<T,AABB>::KeyPair* hk1AxisSweepT_scanListPadding2(
    const AABB& query, _In_reads_(_Inexpressible_()) const AABB* HK_RESTRICT sxyz,
    _Inout_ptrdiff_count_(end) typename hk1AxisSweepT<T,AABB>::KeyPair* HK_RESTRICT pairsOut,
    _Notvalid_ const typename hk1AxisSweepT<T,AABB>::KeyPair* HK_RESTRICT end, _Out_ int& numPairsSkipped )
{
    hkUint32 maxX = query.getMaxXSortKey();
    while( sxyz->getMinXSortKey() <= maxX )
    {
        hkBoolLL ov0 = AABB::yzDisjoint( query, sxyz[0] );
        hkBoolLL ov1 = AABB::yzDisjoint( query, sxyz[1] );

        if ( ! (ov0&ov1)  )
        {
            if ( !ov0 )
            {
                pairsOut = hk1AxisSweepT_appendPair<flipKeys, T>( query, sxyz[0], pairsOut, end, numPairsSkipped  );
            }
            if (!ov1 )
            {
                if ( sxyz[1].getMinXSortKey() <= maxX )
                {
                    pairsOut = hk1AxisSweepT_appendPair<flipKeys, T>( query, sxyz[1], pairsOut, end, numPairsSkipped  );
                }
            }
        }
        sxyz+=2;
    }
    return pairsOut;
}

static const HK_ALIGN16( hkUint32 hk1AxisSweepT_andBitsCst[4] ) = { 0x80008000, 0x80008000,0x80008000 ,0x80008000 };

template<int flipKeys, typename T, typename AABB>
HK_INLINE typename hk1AxisSweepT<T,AABB>::KeyPair* hk1AxisSweepT_scanListSIMD(
    const AABB& query,
    _In_reads_(_Inexpressible_()) const AABB* HK_RESTRICT sxyz,
    _In_reads_(_Inexpressible_()) const hkUint16* yzMin, _In_reads_(_Inexpressible_()) const hkUint16* yzMax,
    _Inout_ptrdiff_count_(end) typename hk1AxisSweepT<T,AABB>::KeyPair* HK_RESTRICT pairsOut,
    _Notvalid_ const typename hk1AxisSweepT<T,AABB>::KeyPair* end, _Out_ int& numPairsSkipped )
{
    hkIntVector qMi;
    hkIntVector qMa;

    typedef union { AABB aabb; hkUint32 u32[4]; } aabbVals;
    const aabbVals* aabbValues = (const aabbVals*)&query;
    qMi.load<4>( aabbValues->u32 );     // load min(x,y,z,key)  : max(x,y,z,key)
    qMi.setShiftLeft128<2>(qMi);        // -> miny, minz, key, maxx,     maxy, maxz, maxw, 0
    qMa.setBroadcast<2>( qMi );
    qMi.setBroadcast<0>( qMi );

    hkIntVector andBits; andBits.load<4>(hk1AxisSweepT_andBitsCst);
    hkUint32 maxX = query.getMaxXSortKey();
    while( sxyz->getMinXSortKey() <= maxX )
    {
        hkIntVector hMi; hMi.loadNotAligned<4>( (const hkUint32*)yzMin );
        hkIntVector hMa; hMa.loadNotAligned<4>( (const hkUint32*)yzMax );
        hMa.setSubU32( hMa, qMi );
        hMi.setSubU32( qMa, hMi );
        hMa.setOr( hMa, hMi );
        hMa.setAnd( hMa, andBits );
        hkVector4Comparison overlaps = hMa.equalZeroS32();

        if ( overlaps.anyIsSet() )
        {
            hkVector4ComparisonMask::Mask mask = overlaps.getMask();
            if ( mask & hkVector4ComparisonMask::MASK_X )
            {
                pairsOut = hk1AxisSweepT_appendPair<flipKeys,T>( query, sxyz[0], pairsOut, end, numPairsSkipped  );
            }
            if ( mask & hkVector4ComparisonMask::MASK_Y )
            {
                if ( sxyz[1].getMinXSortKey() <= maxX )
                {
                    pairsOut = hk1AxisSweepT_appendPair<flipKeys,T>( query, sxyz[1], pairsOut, end, numPairsSkipped  );
                }
            }
            if ( mask & hkVector4ComparisonMask::MASK_Z )
            {
                if ( sxyz[2].getMinXSortKey() <= maxX )
                {
                    pairsOut = hk1AxisSweepT_appendPair<flipKeys,T>( query, sxyz[2], pairsOut, end, numPairsSkipped  );
                }
            }
            if ( mask & hkVector4ComparisonMask::MASK_W )
            {
                if ( sxyz[3].getMinXSortKey() <= maxX )
                {
                    pairsOut = hk1AxisSweepT_appendPair<flipKeys,T>( query, sxyz[3], pairsOut, end, numPairsSkipped  );
                }
            }
        }
        yzMin += 4*2;
        yzMax += 4*2;
        sxyz+=4;
    }

    return pairsOut;
}

template<typename AABB>
static HK_INLINE void checkPadding( _In_reads_(numA+numPadding) const AABB* pa, _In_range_(>=, 0) int numA, _In_range_(>=, 0) int numPadding )
{
#ifdef HK_DEBUG_SLOW
    HK_ASSERT( 0x5afbfde0, numA == 0 || !pa[numA-1].isEndMarker(), "numA must not include the padding elements at the end." );
    for( int i=0; i<numA-1; i++)
    {
        HK_ASSERT_NO_MSG( 0x16bac266, 0 == ( pa[i+1] < pa[i] ) );
    }
    for( int i=0; i<numPadding; i++ )
    {
        HK_ASSERT( 0x159ffbac, pa[numA+i].isEndMarker(), "max-value padding elements are required at the end." );
    }
#endif
}


// Requires 4 elements of padding at the end
template<typename T, typename AABB>
_Ret_range_(0, maxNumPairs)
int HK_CALL hk1AxisSweepT<T,AABB>::collide(
    _In_reads_(numA + 4) const AABB* pa, _In_range_(>=, 0) int numA,
    _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkippedOut )
{
    checkPadding( pa, numA, 4 );

    const KeyPair* end = pairsOut + maxNumPairs;
    KeyPair* HK_RESTRICT pairs = pairsOut;
    numPairsSkippedOut = 0;

    while ( --numA > 0 )    // this iterates numA-1
    {
        const bool dontflipKeys = false;
        pairs = hk1AxisSweepT_scanList<dontflipKeys,T>( *pa, pa+1, pairs, end, numPairsSkippedOut );
        pa++;
    }
    return int(pairs - pairsOut);
}

// Requires 2 elements of padding at the end
template<typename T, typename AABB>
_Ret_range_(0, maxNumPairs)
int HK_CALL hk1AxisSweepT<T,AABB>::collidePadding2(
    _In_reads_(numA + 2) const AABB* pa, _In_range_(>=, 0) int numA,
    _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkippedOut )
{
    checkPadding( pa, numA, 2 );

    KeyPair* end = pairsOut + maxNumPairs;
    KeyPair* HK_RESTRICT pairs = pairsOut;
    numPairsSkippedOut = 0;

    while ( --numA > 0 )    // this iterates numA-1
    {
        const bool dontflipKeys = false;
        pairs = hk1AxisSweepT_scanListPadding2<dontflipKeys,T>( *pa, pa+1, pairs, end, numPairsSkippedOut );
        pa++;
    }
    return int(pairs - pairsOut);
}

static const HK_ALIGN16(hkUint32) s_collideSIMD_maxU16[4] = { 0x7fff7fff, 0x7fff7fff, 0x7fff7fff, 0x7fff7fff };

// Requires 4 elements of padding at the end
template<typename T, typename AABB>
_Ret_range_(0, maxNumPairs)
int HK_CALL hk1AxisSweepT<T,AABB>::collideSIMD(
    _In_reads_(numA + 4) const AABB* pa, _In_range_(>=, 0) int numA,
    _Out_writes_bytes_(bufferSizeInBytes) void* buffer, _In_range_(>=, 0) int bufferSizeInBytes,
    _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkippedOut )
{
    checkPadding( pa, numA, 4 );
    HK_ASSERT_NO_MSG( 0xf0f45fd, sizeof(AABB) == 16 ); // this function assumes hkAabb16, does not work with hkAabb24

#if !defined(HK_1AXIS_SWEEP_USE_SIMD)
    return collide( pa, numA, pairsOut, maxNumPairs, numPairsSkippedOut );
#else
    hkUint16* yzMin = (hkUint16*) buffer;
    hkUint16* yzMax = yzMin + HK_NEXT_MULTIPLE_OF(4,numA)*2 + 16;
    HK_ASSERT( 0xf0ede454, hkAddByteOffset(buffer, bufferSizeInBytes) >= yzMax + HK_NEXT_MULTIPLE_OF(4,numA)*2 + 16, "Supplied buffer size is too small ");
    {
        int i=0;
        for (; i< numA; i+=4)
        {
            hkIntVector a; a.load<4>( (const hkUint32*)&pa[i] );
            hkIntVector b; b.load<4>( (const hkUint32*)&pa[i+1] );
            hkIntVector c; c.load<4>( (const hkUint32*)&pa[i+2] );
            hkIntVector d; d.load<4>( (const hkUint32*)&pa[i+3] );
            a.setShiftLeft128<2>(a);
            b.setShiftLeft128<2>(b);
            c.setShiftLeft128<2>(c);
            d.setShiftLeft128<2>(d);
            a.setPermutation<hkVectorPermutation::XZYW>( a );   // now we have min[xy] max[xy]  min[z] max[z]
            b.setPermutation<hkVectorPermutation::XZYW>( b );
            c.setPermutation<hkVectorPermutation::XZYW>( c );
            d.setPermutation<hkVectorPermutation::XZYW>( d );
            hkIntVector e; e.setMergeHead32( a, c );            // amin[xy], cmin[xy], amax[xy], cmax[xy]
            hkIntVector f; f.setMergeHead32( b, d );            // bmin[xy], dmin[xy], bmax[xy], dmax[xy]
            hkIntVector yzMi; yzMi.setMergeHead32( e, f );      // amin[xy], bmin[xy], cmin[xy], dmin[xy]
            hkIntVector yzMa; yzMa.setMergeTail32( e, f );
            yzMi.store<4>( (hkUint32*) &yzMin[i*2] );
            yzMa.store<4>( (hkUint32*) &yzMax[i*2] );
        }
        hkIntVector maxI; maxI.load<4>( s_collideSIMD_maxU16 );
        hkIntVector zero; zero.setZero();
        zero.store<4>( (hkUint32*) &yzMax[i*2] );
        maxI.store<4>( (hkUint32*) &yzMin[i*2] );
    }

    KeyPair* end = pairsOut + maxNumPairs;
    KeyPair* HK_RESTRICT pairs = pairsOut;
    numPairsSkippedOut = 0;

    while ( --numA > 0 )    // this iterates numA-1
    {
        yzMin+=2;
        yzMax+=2;

        const bool dontflipKeys = false;
        pairs = hk1AxisSweepT_scanListSIMD<dontflipKeys,T>( *pa, pa+1, yzMin, yzMax, pairs, end, numPairsSkippedOut );
        pa++;
    }
    return int(pairs - pairsOut);
#endif
}


template<typename T, typename AABB>
_Ret_range_(0, maxNumPairs)
int HK_CALL hk1AxisSweepT<T,AABB>::collide(
    _In_reads_(numA + 4) const AABB* pa, _In_range_(>=, 0) int numA,
    _In_reads_(numB + 4) const AABB* pb, _In_range_(>=, 0) int numB,
    _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkipped )
{
    checkPadding( pa, numA, 4 );
    checkPadding( pb, numB, 4 );

    KeyPair* end = pairsOut + maxNumPairs;
    KeyPair* HK_RESTRICT pairs = pairsOut;
    numPairsSkipped = 0;

    while ( true )
    {
        if ( pb < pa  )
        {
            const bool flipKeys = true;
            pairs = hk1AxisSweepT_scanList<flipKeys,T>( *pb, pa, pairs, end, numPairsSkipped );
            pb++;
            if ( --numB <= 0 ) { break; }
        }
        else
        {
            const bool dontflipKeys = false;
            pairs = hk1AxisSweepT_scanList<dontflipKeys,T>( *pa, pb, pairs, end, numPairsSkipped );
            pa++;
            if ( --numA <= 0 ) { break; }
        }
    }
    return int(pairs - pairsOut);
}


template<typename T, typename AABB>
_Ret_range_(0, maxNumPairs)
int HK_CALL hk1AxisSweepT<T,AABB>::collidePadding2(
    _In_reads_(numA + 2) const AABB* pa, _In_range_(>=, 0) int numA,
    _In_reads_(numB + 2) const AABB* pb, _In_range_(>=, 0) int numB,
    _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkipped )
{
    checkPadding( pa, numA, 2 );
    checkPadding( pb, numB, 2 );

    KeyPair* end = pairsOut + maxNumPairs;
    KeyPair* HK_RESTRICT pairs = pairsOut;
    numPairsSkipped = 0;

    while ( true )
    {
        if ( pb->getMinXSortKey() < pa->getMinXSortKey() )
        {
            const bool flipKeys = true;
            pairs = hk1AxisSweepT_scanListPadding2<flipKeys,T>( *pb, pa, pairs, end, numPairsSkipped );
            pb++;
            if ( --numB <= 0 ) { break; }
        }
        else
        {
            const bool dontflipKeys = false;
            pairs = hk1AxisSweepT_scanListPadding2<dontflipKeys,T>( *pa, pb, pairs, end, numPairsSkipped );
            pa++;
            if ( --numA <= 0 ) { break; }
        }
    }
    return int(pairs - pairsOut);
}

template<typename T, typename AABB>
_Ret_range_(0, maxNumPairs)
int HK_CALL hk1AxisSweepT<T,AABB>::collideSIMD(
    _In_reads_(numA + 4) const AABB* pa, _In_range_(>=, 0) int numA,
    _In_reads_(numB + 4) const AABB* pb, _In_range_(>=, 0) int numB,
    _Out_writes_bytes_(bufferSizeInBytes) void* buffer, _In_range_(>=, 0) int bufferSizeInBytes,
    _Inout_updates_(maxNumPairs) KeyPair* HK_RESTRICT pairsOut, _In_range_(>=, 0) int maxNumPairs, _Out_ int& numPairsSkipped )
{
    checkPadding( pa, numA, 4 );
    checkPadding( pb, numB, 4 );
    HK_ASSERT_NO_MSG( 0xf0f45fd, sizeof(AABB) == 16 ); // this function assumes hkAabb16, does not work with hkAabb26

#if !defined(HK_1AXIS_SWEEP_USE_SIMD)
    return collide( pa, numA, pb, numB, pairsOut, maxNumPairs, numPairsSkipped );
#else
    KeyPair* end = pairsOut + maxNumPairs;
    KeyPair* HK_RESTRICT pairs = pairsOut;
    numPairsSkipped = 0;

    hkUint16* yzMinA = (hkUint16*) buffer;
    hkUint16* yzMaxA = yzMinA + numB+8;
    hkUint16* yzMinB = (hkUint16*) buffer;
    hkUint16* yzMaxB = yzMinA + numB+8;
    {
        for (int i=0; i< numA; i+=4)
        {
            hkIntVector a; a.load<4>( (const hkUint32*)&pa[i] );
            hkIntVector b; b.load<4>( (const hkUint32*)&pa[i+1] );
            hkIntVector c; c.load<4>( (const hkUint32*)&pa[i+2] );
            hkIntVector d; d.load<4>( (const hkUint32*)&pa[i+3] );
            a.setShiftLeft128<2>(a);
            b.setShiftLeft128<2>(b);
            c.setShiftLeft128<2>(c);
            d.setShiftLeft128<2>(d);
            a.setPermutation<hkVectorPermutation::XZYW>( a );   // now we have min[xy] max[xy]  min[z] max[z]
            b.setPermutation<hkVectorPermutation::XZYW>( b );
            c.setPermutation<hkVectorPermutation::XZYW>( c );
            d.setPermutation<hkVectorPermutation::XZYW>( d );
            hkIntVector e; e.setMergeHead32( a, c );            // amin[xy], cmin[xy], amax[xy], cmax[xy]
            hkIntVector f; f.setMergeHead32( b, d );            // bmin[xy], dmin[xy], bmax[xy], dmax[xy]
            hkIntVector yzMi; yzMi.setMergeHead32( e, f );      // amin[xy], bmin[xy], cmin[xy], dmin[xy]
            hkIntVector yzMa; yzMa.setMergeTail32( e, f );
            yzMi.store<4>( (hkUint32*) &yzMinA[i*2] );
            yzMa.store<4>( (hkUint32*) &yzMaxA[i*2] );
        }
    }

    while ( true )
    {
        if ( pb->getMinXSortKey() < pa->getMinXSortKey() )
        {
            const bool flipKeys = true;
            pairs = hk1AxisSweepT_scanListSIMD<flipKeys,T>( *pb, pa, yzMinA, yzMaxA, pairs, end, numPairsSkipped );
            pb++;
            yzMinA += 2;
            yzMaxA += 2;
            if ( --numB <= 0 ) { break; }
        }
        else
        {
            const bool dontflipKeys = false;
            pairs = hk1AxisSweepT_scanListSIMD<dontflipKeys,T>( *pa, pb, yzMinB, yzMaxB, pairs, end, numPairsSkipped );
            pa++;
            yzMinB += 2;
            yzMaxB += 2;
            if ( --numA <= 0 ) { break; }
        }
    }
    return int(pairs - pairsOut);
#endif
}

template<typename T, typename AABB>
void HK_CALL hk1AxisSweepT<T,AABB>::sortAabbs(_Inout_updates_(HK_NEXT_MULTIPLE_OF(4, size)) AABB* aabbs, _In_range_(>=, 0) int size)
{
    int fixedSize = (size + 3) & ~3;
    hkArray<SortData> sortBuffer(fixedSize << 1);
    hkArray<AABB> tempAabbs(size);

    sortAabbs(aabbs, size, sortBuffer, tempAabbs);
}

template<typename T, typename AABB>
void HK_CALL hk1AxisSweepT<T,AABB>::sortAabbs(
    _Inout_updates_(HK_NEXT_MULTIPLE_OF(4, size)) AABB* aabbs, _In_range_(>=, 0) int size,
    hkArray<SortData>& sortBuffer, hkArray<AABB>& tempAabbs )
{
    // Make it multiple of 4
    // This is okay cos we know the aabbs array is padded with 4 extra entries
    int fixedSize = (size + 3) & ~3;
    HK_ASSERT_NO_MSG(0x48d3f24e, size < 0xffff);
    HK_ASSERT_NO_MSG(0x48d3f24e, sortBuffer.getSize() >= (fixedSize << 1));
    HK_ASSERT_NO_MSG(0x48d3f24e, tempAabbs.getSize() >= size);

    SortData* sortArray = &sortBuffer[0];
    typedef typename SortData::KeyType KeyType;
    for (int i = 0; i < fixedSize; i++)
    {
        SortData& entry = sortArray[i];

        entry.m_key = (KeyType)aabbs[i].getMinXSortKey();
        entry.m_userData = (KeyType)i;
    }
    for( int i=size; i<fixedSize; i++ )
    {
        SortData& entry = sortArray[i];
        entry.m_key = (KeyType)~0;
    }

    SortData* buffer = &sortBuffer[fixedSize];
    hkRadixSort::sort(sortArray, fixedSize, buffer);

    for (int i = 0; i < size; i++)
    {
        tempAabbs[i] = aabbs[sortArray[i].m_userData];
    }

    // Copy back
    hkString::memCpy16(aabbs, tempAabbs.begin(), size * (sizeof(AABB)/16) );
}

template class hk1AxisSweepT< hkUint16, hkAabb16 >;
template class hk1AxisSweepT< hkUint32, hkAabb16 >;
template class hk1AxisSweepT< hkUint32, hkAabb24_16_24_WithKey >;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
