// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Algorithm/Sort/hkRadixSort.h>

struct HK_EXPORT_COMMON hkKeyPair
{
    HK_DECLARE_PLACEMENT_ALLOCATOR();
    hkUint32 m_keyA;
    hkUint32 m_keyB;
};

    /// Utility to quickly find overlapping pairs of AABBs
namespace hk1AxisSweep
{
    struct AabbInt : public hkAabbUint32
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_CDINFO, hk1AxisSweep::AabbInt );

        HK_INLINE hkUint32& getKey()
        {
            // Simply returning m_min[3] here causes a compiler error when static code analysis is enabled as m_min only has 3 components.
            // minP[3] points to m_min's alignment padding, which is where the AabbInt's key is stored.
            hkUint32* minP = m_min;
            return minP[3];
        }

        HK_INLINE const hkUint32& getKey() const
        {
            const hkUint32* minP = m_min;
            return minP[3];
        }

        HK_INLINE hkUint32& getUserData()
        {
            // Simply returning m_max[3] here causes a compiler error when static code analysis is enabled as m_max only has 3 components.
            // minP[3] points to m_min's alignment padding, which is where the AabbInt's userdata is stored.
            hkUint32* maxP = m_max;
            return maxP[3];
        }

        HK_INLINE const hkUint32& getUserData() const
        {
            const hkUint32* maxP = m_max;
            return maxP[3];
        }

        HK_INLINE bool operator<(const hk1AxisSweep::AabbInt& aabb1) const
        {
            return this->m_min[0] < aabb1.m_min[0];
        }

        void operator=( const AabbInt& other )
        {
            hkString::memCpy16<sizeof(AabbInt)>( this, &other );
        }

        void setEndMarker()
        {
            m_min[0] = hkUint32(-1);
        }

            /// set from a float AABB. The float values will be converted into ordered ints
        HK_EXPORT_COMMON void set( const hkAabb& aabb, int key );

        static HK_INLINE hkUint32 yzDisjoint( const hk1AxisSweep::AabbInt& aabb, const hk1AxisSweep::AabbInt& other )
        {
            hkUint32 yab = aabb.m_max[1] - other.m_min[1];
            hkUint32 yba = other.m_max[1] - aabb.m_min[1];
            hkUint32 zab = aabb.m_max[2] - other.m_min[2];
            hkUint32 zba = other.m_max[2] - aabb.m_min[2];
            hkUint32 combined = (yab | yba) | (zab | zba);
            return combined & 0x80000000;
        }
    };


    /// Iterator for going through the 1AxisSweep results. This is equivalent to using the single-array version of hk1AxisSweep::collide.
    /// Usage is
    /// \code
    ///     hk1AxisSweep::IteratorAA iter( aabbs, numAabbs );
    ///     for (; iter.isValid(); iter.next() )
    ///     {
    ///         hkKeyPair pair;
    ///         iter.getKeyPair(pair);
    ///         // Do something with pair
    ///     }
    /// \endcode
    struct IteratorAA
    {
        public:

            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_CDINFO, hk1AxisSweep::IteratorAA );

            HK_INLINE IteratorAA(_In_reads_(numA) const hk1AxisSweep::AabbInt*, _In_range_(>, 0) int numA);

                /// Returns whether or not the iterator has more results
            HK_INLINE hkBool32 isValid() const HK_RESTRICT;

                /// Advances the iterator to the next result
            HK_INLINE void next() HK_RESTRICT;

                /// Retrieves the current result
            HK_INLINE void getKeyPair( _Out_ hkKeyPair& pair ) const;

        private:

            const hk1AxisSweep::AabbInt* HK_RESTRICT m_aabbs;
            int m_numAabbs;

                /// Index of the "outer" AABB
            int m_current;

                /// Index of the "inner" AABB
            int m_potential;
    };


    /// Iterator for going through the 1AxisSweep results. This is equivalent to using the two-array version of hk1AxisSweep::collide.
    /// Usage is
    /// \code
    ///     hk1AxisSweep::IteratorAB iter( aabbsA, numA, aabbsB, numB );
    ///     for (; iter.isValid(); iter.next() )
    ///     {
    ///         hkKeyPair pair;
    ///         iter.getKeyPair(pair);
    ///         // Do something with pair
    ///     }
    /// \endcode
    struct IteratorAB
    {
        public:
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_CDINFO, hk1AxisSweep::IteratorAB );

            HK_EXPORT_COMMON IteratorAB(_In_reads_(numA + 4) const hk1AxisSweep::AabbInt* pa, _In_range_(>, 0) int numA,
                _In_reads_(numB + 4) const hk1AxisSweep::AabbInt* pb, _In_range_(>, 0) int numB);

                /// Returns whether or not the iterator has more results
            HK_INLINE hkBool32 isValid() const HK_RESTRICT;

                /// Advances the iterator to the next result
            HK_INLINE void next()
#if !defined(HK_BOOL32_IS_STRICT)
                HK_RESTRICT
#endif
                ;

                /// Retrieves the current result
            inline void getKeyPair( _Out_ hkKeyPair& pair ) const;

        private:

                /// Which part of the "while" loop we're current in.
            hkBool32 m_aIsBigger;
            const hk1AxisSweep::AabbInt* HK_RESTRICT m_currentPtr;
            const hk1AxisSweep::AabbInt* HK_RESTRICT m_potentialPtr;

            const hk1AxisSweep::AabbInt* HK_RESTRICT m_pa;
            const hk1AxisSweep::AabbInt* HK_RESTRICT m_pb;

            int m_numA;
            int m_numB;
        protected:
            HK_DEBUG_ONLY_MEMBER(const hk1AxisSweep::AabbInt* HK_RESTRICT, m_originalA);
            HK_DEBUG_ONLY_MEMBER(const hk1AxisSweep::AabbInt* HK_RESTRICT, m_originalB);
            HK_DEBUG_ONLY_MEMBER(int, m_totalNumA);
            HK_DEBUG_ONLY_MEMBER(int, m_totalNumB);
    };

        /// This returns all overlapping AABB pairs, where one AABB is from the first list, and the other from the second list.
        /// Both lists must be appended with four AABB elements, where AABB.m_min[0] == HK_REAL_MAX.
        /// numA/numB should be equal to the actual number of elements excluding the padding AABBs.
    _Ret_range_(0, maxNumPairs)
    HK_EXPORT_COMMON int  HK_CALL collide( _In_reads_(numA + 4) const AabbInt* pa, _In_range_(>, 0) int numA,
        _In_reads_(numB + 4) const AabbInt* pb, _In_range_(>, 0) int numB,
        _Inout_count_(maxNumPairs) hkKeyPair* HK_RESTRICT pairsOut, _In_range_(>, 0) int maxNumPairs, _Out_ int& numPairsSkipped);

    HK_EXPORT_COMMON void HK_CALL collide( _In_reads_(numA) const AabbInt* pa, _In_range_(>, 0) int numA,
        _In_reads_(numB) const AabbInt* pb, _In_range_(>, 0) int numB, hkArray<hkKeyPair>& pairsOut);

        /// This returns all overlapping AABB pairs when collide pa with itself.
        /// The list must be appended with four AABB elements, where aabb.m_min[0] == HK_REAL_MAX.
        /// numA should be equal to the actual number of elements excluding the padding AABBs.
        /// The number of pairs which didn't fit into the buffer is written into numPairsSkippedInOut.
    _Ret_range_(0, maxNumPairs)
    HK_EXPORT_COMMON int  HK_CALL collide( _In_reads_(numA) const AabbInt* pa, _In_range_(>, 0) int numA,
        _Inout_count_(maxNumPairs) hkKeyPair* HK_RESTRICT pairsOut, _In_range_(>, 0) int maxNumPairs, _Out_ int& numPairsSkippedOut);

    HK_EXPORT_COMMON void HK_CALL collide(_In_reads_(numA) const hk1AxisSweep::AabbInt* pa, _In_range_(>, 0) int numA, hkArray<hkKeyPair>& pairsOut);

        /// Utility function to sort AABBs in place. This implementation is really fast, don't try to beat it
    HK_EXPORT_COMMON void HK_CALL sortAabbs(_Inout_updates_(HK_NEXT_MULTIPLE_OF(4, size)) hk1AxisSweep::AabbInt* aabbs, _In_range_(>, 0) int size);

        /// Utility function to sort AABBs in place.
        /// No extra memory is allocated during the step, even from the temp allocator.
        /// sortArray and sortedAabbs should have at least as many elements as the aabbs buffer.
    HK_EXPORT_COMMON void HK_CALL sortAabbs(_Inout_updates_(HK_NEXT_MULTIPLE_OF(4, size)) hk1AxisSweep::AabbInt* aabbs, _In_range_(>, 0) int size,
        hkArrayBase<hkRadixSort::SortData32>& sortArray, hkArrayBase<hk1AxisSweep::AabbInt>& sortedAabbs );
};

#include <Common/Base/Algorithm/Collide/1AxisSweep/hk1AxisSweep.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
