"""
Predefined workspace specifications
"""
import os


def _getWorkspaceSpecsDict():
    """
    Returns a dict containing all predefined workspace specs.

    Returned data is of the form:
    {'specType1':[spec1, spec2, ...],
     'specType2':[spec1]}
    """
    workspaceSpecsDict = {
        "HavokInternal": [{'workspaceName': "Havok",
                           'rootRelExcludePaths': []},

                          {'workspaceName': "HavokAnimationTool",
                           'outputDir': os.path.join('Tools', 'HavokBehaviorTool', 'Source'),
                           'rootRelRequiredPaths': [("Tools", "HavokBehaviorTool")],
                           'rootRelIncludePaths': [("Tools", "HavokBehaviorTool"),
                                                   ("Tools", "HavokAssembly"),
                                                   ("Tools", "Script"),
                                                   ("Demo", "Graphics"),
                                                   ("Source"),
                                                   ("Tools", "VisualDebugger")],
                            'rootRelExcludePaths': [("Source", "Engine"),
                                                    ("Source", "NewTools"),
                                                    ("Source", "Renderer")],
                           'allowedCompilers': ['vs2010', 'vs2012', 'vs2013', 'vs2015', 'vs2017'],
                           'allowedPlatforms': ['win32']},
        ],
        "PackageBuild": [{'workspaceName': "PackageBuild",
                          'rootRelExcludePaths': []},
        ],
        "HavokSdk": [{'workspaceName': "HavokSDK_RuntimeOnly",
                      'rootRelExcludePaths': [('Build',),
                                              ('Demo',),
                                              ('Source', 'Behavior', 'IKinemaBridge'),
                                              ('Source', 'ContentTools'),
                                              ('Tools',)]},

                     {'workspaceName': "HavokSDK_All",
                      'rootRelExcludePaths': [('Build',),
                                              ('Source', 'Behavior', 'IKinemaBridge'),
                                              ('Demo', 'StandaloneDemos', 'OgreCloth'),
                                              ('Demo', 'StandAloneDemos', 'Destruction', 'OgreDestruction'),
                                              ('Tools', 'PreBuild')],
                      'disabledProjectPaths': ['LegacyDX']},

                     {'workspaceName': "HavokSDK_DemosOnly",
                      'rootRelIncludePaths': [('Demo', 'Demos'),
                                              ('Demo', 'StandAloneDemos')],
                      'rootRelExcludePaths': [('Demo', 'StandaloneDemos', 'OgreCloth'),
                                              ('Demo', 'StandAloneDemos', 'Destruction', 'OgreDestruction')],
                      'disabledProjectPaths': ['LegacyDX']},

                     {'workspaceName': "HavokSDK_DemosAndRuntime",
                      'rootRelExcludePaths': [('Build',),
                                              ('Demo', 'Graphics', 'Managed'),
                                              ('Demo', 'StandaloneDemos', 'OgreCloth'),
                                              ('Demo', 'StandAloneDemos', 'Destruction', 'OgreDestruction'),
                                              ('Source', 'Behavior', 'IKinemaBridge'),
                                              ('Source', 'ContentTools'),
                                              ('Tools',)],
                      'disabledProjectPaths': ['LegacyDX']},

                     {'workspaceName': "HavokSDK_ToolsAndRuntime",
                      'rootRelExcludePaths': [('Build',),
                                              ('Demo',),
                                              ('Tools', 'HavokBehaviorTool'),
                                              ('Tools', 'PreBuild'),
                                              ('Tools', 'Script'),
                                              ('Source', 'Behavior', 'IKinemaBridge')],
                      'rootRelAlwaysIncludePaths': [("Demo", "Graphics", "Managed")],
                      'allowedPlatforms': ['win32', 'x64']},

                     {'workspaceName': "HavokSDK_OgreIntegrationDemos",
                      'rootRelIncludePaths': [('Demo', 'StandaloneDemos', 'OgreCloth'),
                                              ('Demo', 'StandAloneDemos', 'Destruction', 'OgreDestruction')]},

                     {'workspaceName': "HavokSDK_ReflectionTools",
                      'rootRelIncludePaths': [('Tools', 'FileConvert'),
                                              ('Tools', 'PreBuild')],
                      # Don't generate non-VS2012 solutions, as they'll contain FileConvert only:
                      'allowedCompilers': ['vs2012']},

                     {'workspaceName': "HavokScript",
                      'rootRelIncludePaths': [('Demo', 'Script'),
                                              ('Source', 'Script'),
                                              ('Tools', 'Script', 'Executables'),
                                              ('Tools', 'Script', 'Other')]},

        ],
    }

    return workspaceSpecsDict


class WorkspaceSpec(object):
    """
    Workspace details.
    """
    __slots__ = ('workspaceName', 'rootDir', 'outputDir', 'globPatterns', 'rootRelRequiredPaths', 'rootRelIncludePaths',
                 'rootRelExcludePaths', 'disabledProjectPaths', 'rootRelAlwaysIncludePaths', 'allowedPlatforms',
                 'allowedCompilers', 'putInTargetSubdirs',)

    def __init__(self, **kwargs):
        """
        Set all values to defaults if not specified otherwise.
        E.g. WorkspaceSpec(workspaceName="HavokSDK", allowedCompilers=['net', 'vs2010'])
        """
        self.workspaceName = ''
        self.rootDir = ''
        self.outputDir = ''
        self.globPatterns = ("*.vcproj", "*.csproj", "*.vcxproj", '*.androidproj')
        self.rootRelRequiredPaths = []
        self.rootRelIncludePaths = ["."]
        self.rootRelExcludePaths = []
        self.disabledProjectPaths = [] #  Projects will be added to worksbace but will not be built for any config.
        self.rootRelAlwaysIncludePaths = []
        self.allowedPlatforms = []
        self.allowedCompilers = []
        self.putInTargetSubdirs = False
        # paths need to be joined if they are provided as tuples
        pathKeys = ('rootRelRequiredPaths', 'rootRelIncludePaths', 'rootRelExcludePaths', 'rootRelAlwaysIncludePaths')
        # update with values provided as arguments to the constructor
        for variable, value in kwargs.iteritems():
            if variable in pathKeys:
                value = [os.path.join(*path) if (isinstance(path, tuple) or isinstance(path, list)) else path
                         for path in value]
            setattr(self, variable, value)
        assert self.workspaceName, "\n\nERROR: Found 'workspaceName' key with no value in spec:\n%s" % self

    def __str__(self):
        from pprint import pformat

        return pformat({key: getattr(self, key) for key in self.__slots__})

    def __repr__(self):
        return '<workspaceSpecs.WorkspaceSpec "%s">' % self.workspaceName


def filterModules(moduleList, workspaceSpec, platform, compiler):
    """
    Return the list of modules to include when generating the current workspace.
    """

    if workspaceSpec.allowedPlatforms and platform not in workspaceSpec.allowedPlatforms:
        return []
    if workspaceSpec.allowedCompilers and compiler not in workspaceSpec.allowedCompilers:
        return []

    # If there aren't any modules for any of the required paths return an empty list. For example, we only want to
    # generate the HavokAnimationTool solution if it's present, not because some of the libraries it uses are present.
    if workspaceSpec.rootRelRequiredPaths:
        if not all([any([os.path.normpath(m).startswith(p) for m in moduleList])
                    for p in workspaceSpec.rootRelRequiredPaths]):
            return []

    filteredModules = []
    for module in moduleList:
        excluded = False
        for excludePath in workspaceSpec.rootRelExcludePaths:
            if module.startswith(excludePath.replace('\\', '/')):
                excluded = True
                break

        if not excluded:
            if '.' in workspaceSpec.rootRelIncludePaths:
                filteredModules.append(module)
            else:
                for includePath in workspaceSpec.rootRelIncludePaths:
                    if includePath.replace('\\', '/') in module:
                        filteredModules.append(module)
                        break
    return filteredModules


def getPredefinedWorkspaceSpecTypes():
    """
    Returns a list of the predefined workspace spec types that are supported.
    """
    return sorted(_getWorkspaceSpecsDict().keys())


def getPredefinedWorkspaceSpecs(specType, rootDir):
    """
    Returns full workspace specs for the specified workspace spec type.
    """
    workspaceSpecDict = _getWorkspaceSpecsDict()
    assert specType in workspaceSpecDict, (("\n\nERROR: Unsupported predefined path grouping '%s' was specified."
                                            "\n       Supported predefined path groupings are: %s")
                                           % (specType, workspaceSpecDict.keys()))

    fullSpecsList = [WorkspaceSpec(rootDir=rootDir, **spec) for spec in workspaceSpecDict[specType]]
    return fullSpecsList
