﻿"""
    Visual Studio 2005/2008/2010 solution file generation.
"""
import os
import re
import sys
import pprint
import logging
from optparse import OptionParser
import guidGen
import project
import workspaceSpecs

workspaceTemplate = """
Microsoft Visual Studio Solution File, Format Version %(formatVersion)s
# Visual Studio %(visualStudioVersion)s
%(solutionProjects)s
%(solutionProjectDirs)s
Global
	GlobalSection(SolutionConfigurationPlatforms) = preSolution
		%(solutionConfigurationPlatforms)s
	EndGlobalSection
	GlobalSection(ProjectConfigurationPlatforms) = postSolution
		%(projectConfigurationPlatforms)s
	EndGlobalSection
	GlobalSection(SolutionProperties) = preSolution
		HideSolutionNode = FALSE
	EndGlobalSection
	%(solutionProjectNesting)s
EndGlobal"""

projListTemplate = """
Project("{%(magicGuid)s}") = "%(moduleName)s", "%(modulePath)s", "{%(guid)s}"%(dependenciesText)s
EndProject"""

dependenciesTemplate = """
	ProjectSection(ProjectDependencies) = postProject%s
	EndProjectSection"""

dependencyPairTemplate = """
		{%s} = {%s}"""

workspaceDirsTemplate = """
Project("{2150E333-8FDC-42A3-9474-1A3956D46DE8}") = "%s", "%s", "%s"
EndProject"""

guidMarker = "<GUID_MARKER_%s>"

linktypeEntryTemplate = """
		%(linktype)s|%(platform)s = %(linktype)s|%(platform)s"""

configurationTemplate = """
		{%(guid)s}.%(linktype)s|%(currentPlatform)s.ActiveCfg = %(linktype)s|%(commonPlatform)s"""

buildConfigTemplate = """
		{%(guid)s}.%(linktype)s|%(currentPlatform)s.Build.0 = %(linktype)s|%(commonPlatform)s"""

deployTemplate = """
		{%(guid)s}.%(linktype)s|%(currentPlatform)s.Deploy.0 = %(linktype)s|%(commonPlatform)s"""

projectNestingTemplate = """
		%(guid)s = %(guidMarkerWithPath)s"""

def setupLogging():
    logging.basicConfig(level=logging.DEBUG,
                        format='%(asctime)s> %(message)s',
                        datefmt='%Y/%m/%d@%H.%M.%S')


def defaultOutputFunc(message):
    """
    Default logging function.
    """
    logging.info(message)


def isSubfolderOf(folder, potentialSubFolder):
    return os.path.abspath(potentialSubFolder).lower().startswith(os.path.abspath(folder).lower())


def formatPlatform(platform):
    """
    Performs additional formatting of the platform string used in the following sections of the .sln file:
        SolutionConfigurationPlatforms
        ProjectConfigurationPlatforms
    """
    if platform == "xbox360":
        return "Xbox 360"
    elif platform.lower() == "psvita":
        return "PSVita"
    elif platform in ('nx32', 'nx64'):
        return platform.upper()
    else:
        return platform


def getSolutionConfigurationPlatforms(solution_platform, platforms, linktypes, buildIncludesCsharpProjects=False):
    """
    Build up a mapping of platform-linktype pairs. This is simple at the moment (a|a = a|a),
    it will be more complicated when we support managed code. The purpose of it is to allow
    the interconnection of dependencies between Win32 linktypes and X86 linktypes
    (i.e. non-managed - managed).
    This function works at the module level.
    """
    activePlatforms = list(platforms)
    if solution_platform != 'uwp' and buildIncludesCsharpProjects:
        activePlatforms.append('Mixed Platforms')
    solutionConfigurationPlatforms = []
    for linktype in linktypes:
        for platform in activePlatforms:
            solutionConfigurationPlatforms.append(linktypeEntryTemplate % {"linktype": linktype,
                                                                           "platform": formatPlatform(platform)})
    return ''.join(sorted(solutionConfigurationPlatforms))


def insertMatchingGuids(workspaceContents):
    """
    We find matching guidMarkers inserted earlier and replace them with a newly created random guid.
    """
    uniqueGuidMarkersList = []
    guidMarkersToReplace = re.findall(guidMarker % "\\w+", workspaceContents)

    for guidMarkerMatch in guidMarkersToReplace:
        if guidMarkerMatch not in uniqueGuidMarkersList:
            uniqueGuidMarkersList.append(guidMarkerMatch)

    for guidMarkerMatch in uniqueGuidMarkersList:
        replacementGuid = "{%s}" % guidGen.getGuidFromHash(guidMarkerMatch)
        workspaceContents = workspaceContents.replace(guidMarkerMatch, replacementGuid)

    return workspaceContents


class ProjectTreeNode(object):
    """
    A ProjectNestingTree node, represents a solution filter folder. Projects contained in each folder are stored in
    self.data list of the corresponding folder node.
    """

    def __init__(self, parent=None, name=""):
        self.parent = parent
        self.children = []
        if parent:
            parent.children.append(self)
        self.name = name
        self.data = []  # list of projects in solution folder

    def hasChild(self, name):
        """
        Check if a node with given name (str) is in self.children list.
        Returns the child node if found, otherwise returns None.
        """
        for child in self.children:
            if child.name == name:
                return child

    def addChild(self, name):
        """
        Adds a child if a given name does not exist - creates it and returns the newly created node.
        If the child with given name already exists, returns the existing child node.
        """
        child = self.hasChild(name)
        return child if child else ProjectTreeNode(self, name)

    def __str__(self):
        """
        Converts the node to a string which contains information about all parents up to the root node.
        Used for folder GUID generation (see insertMatchingGuids()).
        """
        if self.parent is None:
            return "root"
        return "%s%s%s" % (str(self.parent), "_" if str(self.parent) else "", self.name)


class ProjectNestingTree(object):
    """
    Used to keep information about project nesting within solution file.
    The nodes of the tree are or type ProjectTreeNode.
    """

    def __init__(self):
        self.root = ProjectTreeNode()

    def add(self, path, project=None):
        """
        Add a node. The node is described by a path from root, e.g. "PS3/Demos/Graphics".
        The tree nodes keep information about solution filters (folders in Solution explorer) - the projects
        are stored in ProjectTreeNode.data list (usually in the leaf node).
        This method can be called multiple times with the same path (e.g. with different project parameter).
        """
        if isinstance(path, basestring):
            path = path.split("/")
        currentNode = self.root
        for name in path:
            currentNode = currentNode.addChild(name)
        if project:
            currentNode.data.append(project)

    def traverse(self, startNode=None):
        """
        Generator used to traverse the tree nodes in depth-first order. If startNode is not provided, the tree is
        searched beginning from the root element.
        Return values from the generator are instances of ProjectTreeNode class.
        """
        if startNode is None:
            startNode = self.root
        for node in startNode.children:
            yield node
            for subnode in self.traverse(node):
                yield subnode


class WorkspaceGenerator(object):
    def __init__(self, rootDir, workspaceSpecType=None, workspaceDir=None, nestingScheme=None):
        object.__init__(self)
        # Name of the top level folder which the workspaces will be placed in
        self.havokWorkspaceDir = workspaceDir or 'Workspace'
        self.activeNestingScheme = nestingScheme or 'Condensed'

        # additional project filter nesting inside solution file (e.g. Graphics will appear within Demos folder)
        self.projectDirNesting = {"Graphics": "Demo",
                                  "Sound": "Demo",
                                  "StepByStep": "Demo"}

        self.workspaceSpecs = []
        if workspaceSpecType:
            self.workspaceSpecs = workspaceSpecs.getPredefinedWorkspaceSpecs(workspaceSpecType, rootDir)

        self.vsVersion = ''

    def getSolutionProjects(self, root, projects, relOutputDir, vsVersion):
        """
        Builds up a block of text which describes the projects in this solution.
        Each module is scanned, its project found, and the correct information
        placed into the projListTemplate, this includes any project dependencies.
        """
        projectList = []

        allGuidsInSolution = [p.guid.upper() for p in projects]

        for currentProject in projects:
            dependencies = []

            dependenciesToInclude = currentProject.dependencies + currentProject.runtimeDependencies
            currentProjectName = currentProject.name if currentProject.name != 'project_name' else currentProject.basename

            # TODO: check this fix on client side, possibly not all projects included in packages as visibility internal
            if vsVersion in ("VS2010", "VS2012", "VS2013", "VS2015", "VS2017"):
                for dependency in dependenciesToInclude:
                    if dependency.guid.upper() not in allGuidsInSolution:
                        raise Exception("Project %s depends on %s which is not in the solution" % (currentProjectName, dependency.name))
                    dependencies.append(dependencyPairTemplate % (dependency.guid, dependency.guid))

            for dependencyGuid in currentProject.getProjectReferenceGUIDs():
                if dependencyGuid not in dependencies:
                    if dependencyGuid.upper() not in allGuidsInSolution:
                        raise Exception("Project %s depends on %s which is not in the solution" % (currentProjectName, dependencyGuid))
                    dependencies.append(dependencyPairTemplate % (dependencyGuid, dependencyGuid))

            dependenciesText = ""
            if dependencies:
                dependenciesText = dependenciesTemplate % ''.join(dependencies)

            pathToProjectFromWorkspace = os.path.relpath(currentProject.path, relOutputDir)

            # store the data needed to populate the project template.
            projectList.append(('%s_%s_%s' % (currentProject.groupingPlatform,
                                              self.getModuleSolutionDir(root, currentProject),
                                              currentProject.basename),
                                {"magicGuid": currentProject.magicGuid,
                                 "moduleName": currentProject.basename,
                                 "modulePath": pathToProjectFromWorkspace,
                                 "guid": currentProject.guid,
                                 "dependenciesText": dependenciesText}))

        # The projects are sorted by platform, grouping filter and project name - this determines the order in which
        # filters are defined in project nesting section
        projectList.sort()
        projectListText = ''
        for projectDetails in sorted(projectList):
            projectListText += projListTemplate % projectDetails[1]
        return projectListText

    def keepProjectInRootDir(self):
        """
        """
        return self.activeNestingScheme == "Flat"

    def projectShouldBeNested(self, platform, projectObj):
        """
        On Xbox360 we need the Demos project to be at the top level, not nested,
        due to a bug in VS2010.
        """
        if self.keepProjectInRootDir():
            return False
        return not (self.vsVersion == "VS2010"
                    and platform.lower() == "xbox 360"
                    and projectObj.outputFile.endswith('.xex'))

    def getModuleSolutionDir(self, root, projectObj):
        """
        The solution file is nested based on project type, the returned value is the category to place module in.
        """
        assert self.activeNestingScheme in ("Condensed", "Standard", "Flat"), ("Unrecognized project nesting scheme"
                                                                               % (self.activeNestingScheme))
        if self.keepProjectInRootDir():
            return ""
        if "SOLUTION_FOLDER" in projectObj.customProperties:
            return projectObj.customProperties["SOLUTION_FOLDER"].rstrip("/")
        modulePathFromRoot = projectObj.pathFromRoot(root)
        pathComponents = [x for x in modulePathFromRoot.replace('\\', '/').split('/') if x]  # eliminate empty strings
        if self.activeNestingScheme == "Condensed":
            if 'ContentTools' in modulePathFromRoot or 'Tools' in modulePathFromRoot:
                projDir = 'Tools'
            elif 'Graphics' in modulePathFromRoot or 'Sound' in modulePathFromRoot:
                projDir = 'Graphics'
            elif 'StepByStep' in modulePathFromRoot:
                projDir = 'StepByStep'
            elif 'Demo' in modulePathFromRoot:
                projDir = 'Demos'
            else:
                projDir = 'SDK'
        elif self.activeNestingScheme == "Standard":
            if 'Max' in modulePathFromRoot:
                projDir = 'Max'
            elif 'Maya' in modulePathFromRoot:
                projDir = 'Maya'
            elif 'StepByStep' in modulePathFromRoot:
                projDir = 'StepByStep'
            else:
                projDir = pathComponents[0]
                if projDir == 'Source' or (projDir == 'Demo' and pathComponents[1] in ['Graphics', 'Sound']):
                    # use the second path component, e.g. "Demos", "Graphics", "Physics", etc.
                    projDir = pathComponents[1]
        return projDir

    def getSolutionProjectDirs(self, root, platform, projects):
        """
        Build up a description of the project nesting in the solution.
        Returns a tuple:
            - project filter definition (text) that needs to be inserted into workspace template
            - list of filter directories
        """
        projDirsText = ''
        projectNestingTree = ProjectNestingTree()

        # For each module get its appropriate parent folder
        for currentProject in projects:
            if not self.projectShouldBeNested(currentProject.platform, currentProject):
                continue
            projDir = self.getModuleSolutionDir(root, currentProject)
            if not projDir:
                continue

            projectNestingTree.add(projDir, currentProject)

        # populate workspaceDirsTemplate with project hierarchy information
        for projDir in projectNestingTree.traverse():
            projDirsText += workspaceDirsTemplate % (projDir.name,
                                                     projDir.name,
                                                     guidMarker % str(projDir))
        return projDirsText, projectNestingTree

    def getProjectConfigurationPlatforms(self, root, platforms, projects, buildIncludesCsharpProjects, actualPlatform):
        """
        Build up a mapping of platform-linktype pairs. This is simple at the moment (a|a = a|a),
        it will be more complicated when we support managed code. The purpose of it is to allow
        the interconnection of dependencies between Win32 linktypes and X86 linktypes (i.e.
        non-managed - managed).
        This function works at the project level.
        """
        projectConfigurationPlatforms = ''
        platform_mapping = {'win32': 'x86',
                            'x64': 'x64',
                            'mixed platforms': 'Mixed Platforms',
                            'any cpu': 'Mixed Platforms',
                            'x86': 'x86',
                            'arm': 'ARM'}

        solutionPlatforms = platforms
        if buildIncludesCsharpProjects:
            solutionPlatforms += ("Mixed Platforms",
                                  platform_mapping[platforms[0].lower()])
        solutionPlatforms = list(set(platforms))

        projects.sort(key=lambda x: (x.groupingPlatform,
                                     self.getModuleSolutionDir(root, x),
                                     x.name))

        for currentProject in projects:
            configs = []
            for linktype in sorted(currentProject.linktypes):
                for platform in solutionPlatforms:
                    commonPlatform = platforms[0]
                    currentPlatforms = [platform]
                    if currentProject.type in ('cs'):
                        if platform in ["win32", "x64"]:
                            currentPlatforms.append(platform_mapping[platform.lower()])
                        else:
                            platform = platforms[0]
                        commonPlatform = platform_mapping[platform.lower()]
                    for currentPlatform in sorted(currentPlatforms):
                        if actualPlatform == 'uwp':
                            commonPlatform = currentPlatform  # TODO: hack
                        currentInformation = {"guid": currentProject.guid,
                                              "linktype": linktype,
                                              "currentPlatform": formatPlatform(currentPlatform),
                                              "commonPlatform": formatPlatform(commonPlatform)}
                        configs.append(configurationTemplate % currentInformation)
                        if not currentProject.excludedFromBuild:  # If project disabled don't write in Build block
                            configs.append(buildConfigTemplate % currentInformation)
                        if currentProject.deploy:
                            configs[-1] += deployTemplate % currentInformation

            configs.sort()
            projectConfigurationPlatforms += ''.join(configs)
        return projectConfigurationPlatforms

    def getSolutionProjectNesting(self, root, platform, projects, projectNestingTree, solutionProjectDirsText):
        """
        Top level description of how to nest projects in the solution.
        """
        projectNestingText = ''

        # build solution file contents (project nesting section)
        for node in projectNestingTree.traverse():
            if str(node.parent) != "root":
                projectNestingText += projectNestingTemplate % {
                                             "guid": guidMarker % str(node),
                                             "guidMarkerWithPath": guidMarker % str(node.parent)}

            thisGuid = guidMarker % str(node)
            # If we haven't already added it
            if solutionProjectDirsText.find(thisGuid) == -1:
                solutionProjectDirsText += projListTemplate % {"magicGuid" : "2150E333-8FDC-42A3-9474-1A3956D46DE8", "moduleName": node.name, "modulePath": node.name, "guid": thisGuid, "dependenciesText": ""}
            if node.data:
                for currentProject in node.data:
                    projectNestingText += projectNestingTemplate % {
                                                 "guid": "{%s}" % currentProject.guid,
                                                 "guidMarkerWithPath": guidMarker % str(node)}

        projectNestingText = projectNestingText.strip()
        if projectNestingText:
            # Only include projectNestingSection if there is any projectNestingText
            projectNestingSection = "GlobalSection(NestedProjects) = preSolution\n\t\t%s\n\tEndGlobalSection"
            projectNestingText = projectNestingSection % projectNestingText

        return solutionProjectDirsText, projectNestingText

    def generateWorkspace(self, root, platform, projects, vsVersion, workspaceName, relOutputDir=''):
        """
        Generate a single VisualStudio solution file and write it to disc.
        Returns: solution file name.
        """
        if not relOutputDir:
            relOutputDir = self.havokWorkspaceDir

        self.vsVersion = vsVersion.upper()
        # the newer versions all use format version 12
        vsFormattedVersion = {"VS2005": "9.00",
                              "VS2008": "10.00",
                              "VS2010": "11.00",
                              "VS2012": "12.00",
                              "VS2013": "12.00",
                              "VS2015": "12.00",
                              "VS2017": "12.00"}

        buildIncludesCsharpProjects = any(p.type == "cs" or p.type == "mgd" for p in projects)
        solutionProjects = self.getSolutionProjects(root, projects, relOutputDir, self.vsVersion)
        platforms = set()
        for proj in projects:
            platforms.update(proj.platforms)
        # make sure x86 is defined for UWP (alongside Win32) even if no C# projects in sln (ensure a consistent set of platforms in every workspace)
        if platform == 'uwp' and 'x86' not in platforms:
            platforms.add('x86')
        platforms = sorted(platforms)
        linktypes = set()
        for currentProject in projects:
            linktypes.update(currentProject.linktypes)
        linktypes = list(linktypes)
        solutionConfigurationPlatforms = getSolutionConfigurationPlatforms(platform,
                                                                           platforms,
                                                                           linktypes,
                                                                           buildIncludesCsharpProjects)
        projectConfigurationPlatforms = self.getProjectConfigurationPlatforms(root,
                                                                              platforms,
                                                                              projects,
                                                                              buildIncludesCsharpProjects,
                                                                              platform)
        solutionProjectDirsText, projectNestingTree = self.getSolutionProjectDirs(root, platform, projects)
        solutionProjectDirsText, solutionNesting = self.getSolutionProjectNesting(root, platform, projects, projectNestingTree, solutionProjectDirsText)
        # vs2015 changed the format of this value from using the full year to a version number instead
        if ("VS2017" == self.vsVersion):
            visualStudioVersion = """15
VisualStudioVersion = 15.0.26228.4
MinimumVisualStudioVersion = 10.0.40219.1"""
        else:
            visualStudioVersion= "14" if "VS2015" == self.vsVersion else vsVersion[2:]

        workspace = workspaceTemplate % {
            "formatVersion": vsFormattedVersion[self.vsVersion],
            "visualStudioVersion": visualStudioVersion,
            "solutionProjects": solutionProjects.strip(),
            "solutionProjectDirs": solutionProjectDirsText.strip(),
            "solutionConfigurationPlatforms": solutionConfigurationPlatforms.strip(),
            "projectConfigurationPlatforms": projectConfigurationPlatforms.strip(),
            "solutionProjectNesting": solutionNesting.strip()}

        workspace = insertMatchingGuids(workspace).strip()

        workspaceDir = os.path.join(root, relOutputDir)
        if not os.path.exists(workspaceDir):
            os.makedirs(workspaceDir)
        workspaceFile = os.path.join(workspaceDir, workspaceName)
        try:
            from Common import utils
            utils.updateFile(workspaceFile, workspace)
        except ImportError:
            if not (os.path.isfile(workspaceFile) and open(workspaceFile).read() == workspace):
                open(workspaceFile, 'w').write(workspace)

        return workspaceFile

    def getProjectGroupsByTarget(self, workspaceSpec, sourceLevel, productsList, outputFunc):
        """
        Categorize projects
            workspaceSpec - the workspace spec to use when creating project files
            sourceLevel    - only include projects of this sourceLevel or lower
            productsList   - only include projects belonging to these products
            outputFunc    - logging function, e.g. logging.info
        """
        outputFunc("Finding and Categorising projects...")
        projectGroupsByTarget = {}

        absIncludePaths = [os.path.normpath(os.path.join(workspaceSpec.rootDir, p))
                           for p in workspaceSpec.rootRelIncludePaths]
        absExcludePaths = [os.path.normpath(os.path.join(workspaceSpec.rootDir, p))
                           for p in workspaceSpec.rootRelExcludePaths]
        absAlwaysIncludePaths = [os.path.normpath(os.path.join(workspaceSpec.rootDir, p))
                                 for p in workspaceSpec.rootRelAlwaysIncludePaths]
        absAlwaysIncludePaths = [os.path.normpath(os.path.join(workspaceSpec.rootDir, p))
                                 for p in workspaceSpec.rootRelAlwaysIncludePaths]

        for proj in project.findProjects(workspaceSpec.globPatterns,
                                         absIncludePaths,
                                         absExcludePaths,
                                         absAlwaysIncludePaths,
                                         disabledProjectPaths=workspaceSpec.disabledProjectPaths,
                                         sourceLevel=sourceLevel,
                                         productsList=productsList,
                                         outputFunc=outputFunc):
            groupingTarget = (proj.groupingPlatform, proj.compiler, proj.extraInfo)

            # Initialize the grouping target
            if groupingTarget not in projectGroupsByTarget:
                projectGroupsByTarget[groupingTarget] = {workspaceSpec.workspaceName: []}

            # Filter on allowed platforms and compilers
            def includeForFilter(filtersList, val):
                if not filtersList:
                    return True
                # Do the ".replace(' ', '')" as some platforms in the project class contain spaces (e.g. "xbox 360")
                # TODO: Clean this up properly later
                if val.replace(' ', '').lower() in filtersList:
                    return True
                return False

            projectPlatform = proj.groupingPlatform
            projectGroupingMappings = {"cafe": "wiiu",
                                       "metro_x86": "metro",
                                       "metro_arm": "metro",
                                       "apollo_x86": "apollo",
                                       "apollo_arm": "apollo"}
            if projectPlatform in projectGroupingMappings:
                projectPlatform = projectGroupingMappings[projectPlatform]
            if (includeForFilter(workspaceSpec.allowedPlatforms, projectPlatform)
            and includeForFilter(workspaceSpec.allowedCompilers, proj.compiler)):
                projectGroupsByTarget[groupingTarget][workspaceSpec.workspaceName].append(proj)

        return projectGroupsByTarget

    def getSolutionVariants(self, projects):
        """
        Returns a dictionary where keys are variant names (empty string for a non-variant solution)
        and values are lists of projects included in each solution variant.
        """
        projectsByVariant = {}
        solutionVariants = set()
        for proj in projects:
            if proj.solutionVariant not in projectsByVariant:
                projectsByVariant[proj.solutionVariant] = []
            projectsByVariant[proj.solutionVariant].append(proj)
            solutionVariants.add(proj.solutionVariant)

        # if variants are present do not generate a "generic" workspace
        if len(solutionVariants) > 1 and "" in solutionVariants:
            solutionVariants.remove("")

        # create a separate project list for each variant
        workspaces = {}
        for solutionVariant in solutionVariants:
            workspaces[solutionVariant] = []
            if "" in projectsByVariant:  # projects common to all variants
                workspaces[solutionVariant].extend(projectsByVariant[""])
            if solutionVariant:
                workspaces[solutionVariant].extend(projectsByVariant[solutionVariant])

        return workspaces

    def checkOutputFileConsistency(self, projects):
        """
        Checks if targetName of each project is unique. Returns a list of projects that have conflicting output files.
        """
        errors = []
        remainingProjects = projects[:]
        for project1 in projects:
            remainingProjects.remove(project1)
            for project2 in remainingProjects:
                if project1.groupingPlatform != project2.groupingPlatform:
                    continue  # PS3: spu and ppu projects should not be checked together
                if any(targetName in project2.targetName.values() for targetName in project1.targetName.values()):
                    errors.append([project1.path, project2.path])
        return errors

    def findProjectsAndGenerateWorkspaces(self, rootDir, workspaceSpecs=None, sourceLevel=None, productsList=None,
                                          outputFunc=defaultOutputFunc):
        """
        Generate workspaces for all targets found.
                rootDir        - the dir to scan to find projects to include
                               - generated workspaces are put in <rootDir>\Workspace
                workspaceSpecs - the workspace specs to use when generating workspace files
                                 (self.workspaceSpecs is used if not specified)
                sourceLevel    - only include projects of this sourceLevel or lower
                productsList   - only include projects belonging to these products
                outputFunc     - logging function, e.g. logging.info
        """
        outputFunc("Generating workspaces for %s" % rootDir)

        if productsList is None:
            productsList = []
        if not workspaceSpecs:
            workspaceSpecs = self.workspaceSpecs
        assert workspaceSpecs, "\n\nERROR: No workspace specs to generate workspaces for."

        generatedWorkspaces = []

        for workspaceSpec in workspaceSpecs:
            projectGroupsByTarget = self.getProjectGroupsByTarget(workspaceSpec, sourceLevel, productsList, outputFunc)

            outputFunc("Calculating Dependencies and Generating Workspaces...")
            targetsProcessed = []
            for ((platform, compiler, extraInfo), projectGroups) in projectGroupsByTarget.items():
                if platform == "anycpu":
                    # Hack, skip our internal reflection dll project.
                    continue
                for (workspaceType, projectGroup) in projectGroups.items():
                    if projectGroup:
                        vsVersions = list(set((proj.compiler for proj in projectGroup)))
                        assert len(vsVersions) == 1, ("All projects must share same compiler:\n%s"
                                                      % pprint.pformat(projectGroup))
                        allProjects = projectGroup[:]
                        for proj in projectGroup:
                            proj.setDependencies(allProjects)

                        targetsProcessed.append((platform, compiler, extraInfo))

                        platformName = platform.replace(" ", "")
                        if platform == "wiiu":
                            platformName = "WiiU"
                        elif platform == "android_arm":
                            platformName = "Android_arm"
                        elif platform == "android_x86":
                            platformName = "Android_x86"
                        elif platform == "psvita":
                            platformName = "PsVita"
                        else:
                            platformName = platform.replace(" ", "").capitalize()

                        workspaceVariants = self.getSolutionVariants(allProjects)
                        for variant, projectsToInclude in workspaceVariants.iteritems():

                            workspacePlatformName = platformName
                            if workspacePlatformName == "Cafe":
                                workspacePlatformName = "WiiU"
                            elif workspacePlatformName == "Orbis":
                                workspacePlatformName = "PS4"

                            workspaceSuffix = '%s_%s%s%s' % (workspacePlatformName,
                                                             compiler,
                                                             "_%s" % extraInfo if extraInfo else "",
                                                             "_%s" % variant if variant else "")

                            workspace = "%s_%s.sln" % (workspaceType, workspaceSuffix)

                            relOutputDir = self.havokWorkspaceDir
                            if workspaceSpec.putInTargetSubdirs:
                                relOutputDir = os.path.join(self.havokWorkspaceDir, workspaceSuffix)

                            outputFunc("\tGenerating %s..." % os.path.join(relOutputDir, workspace))

                            generatedWorkspaces.append(self.generateWorkspace(rootDir,
                                                                              platform,
                                                                              projectsToInclude,
                                                                              vsVersions[0],
                                                                              workspace,
                                                                              relOutputDir))
        return generatedWorkspaces


DEFAULT_EXCLUDES = ['ThirdParty']

COMMAND_LINE_OPTIONS = (
    (('-r', '--root'), {'action': 'store', 'dest': 'root', 'default': '',
                        'help': "the directory to search for project files [REQUIRED]"
                        }),

    (('-t', '--type'), {'action': 'append', 'dest': 'types', 'default': [],
                        'help': ("add a predefined workspace type to use - supported values are: '%s'"
                                 % ("', '".join(workspaceSpecs.getPredefinedWorkspaceSpecTypes())))
                        }),

    (('-n', '--name'), {'action': 'store', 'dest': 'workspaceName', 'default': '',
                        'help': "the workspace name to use [REQUIRED if no predefined workspace type is specified]"
                        }),

    (('-g', '--glob'), {'action': 'append', 'dest': 'globPatterns', 'default': [],
                        'help': "glob patterns to use when searching for project files, e.g. *.vcproj"
                        }),

    (('-i', '--include'), {'action': 'append', 'dest': 'rootRelIncludePaths', 'default': [],
                           'help': "root-relative paths in which to search for project files"
                           }),

    (('-x', '--exclude'), {'action': 'append', 'dest': 'rootRelExcludePaths', 'default': DEFAULT_EXCLUDES,
                           'help': (("root-relative paths to exclude when searching for project files, default values"
                                    " are: '%s' - dirs underneath excluded dirs can be re-added using -a/--always")
                                    % ("', '".join(DEFAULT_EXCLUDES)))
                           }),

    (('-a', '--always'), {'action': 'append', 'dest': 'rootRelAlwaysIncludePaths', 'default': [],
                          'help': ("always include project files in these paths even if they would "
                                   "usually be excluded by the defined exclude paths")
                          }),

    (('-d', '--disable'), {'action': 'append', 'dest': 'disabledProjectPaths', 'default': [],
                           'help': ("projects containing these substrings (e.g. 'LegacyDX') are included in "
                                    "the solution but are disabled when building")
                           }),

    (('-p', '--platform'), {'action': 'append', 'dest': 'allowedPlatforms', 'default': [],
                            'help': "if set, only generate solutions for these platforms (e.g. win32, x64)"
                            }),

    (('-c', '--compiler'), {'action': 'append', 'dest': 'allowedCompilers', 'default': [],
                            'help': "if set, only generate solutions for these compilers (e.g. vs2010, vs2012)"
                            }),

    (('-s', '--subdirs'), {'action': 'store_true', 'dest': 'targetSubdirs', 'default': False,
                           'help': "if set, put generated workspaces in target-specific subdirs"
                           }),

    (('--show-types',), {'action': 'store_true', 'dest': 'showTypes', 'default': False,
                         'help': "display the names of the pre-defined workspace spec types and exit"
                         }),

    (('--show-specs',), {'action': 'store_true', 'dest': 'showSpecs', 'default': False,
                         'help': "display the workspace generation spec(s) defined by the specified arguments and exit"
                         }),

    (('-q', '--quiet',), {'action': 'store_true', 'dest': 'quietMode', 'default': False,
                          'help': "don't print or log output, apart from generated workspace filenames and error info"
                          }),
    )


def main():
    """
    Entry point when the script is run from command line.
    """

    setupLogging()
    parser = OptionParser()
    for options in COMMAND_LINE_OPTIONS:
        parser.add_option(*options[0], **options[1])
    (options, argv) = parser.parse_args()

    supportedPredefinedTypes = workspaceSpecs.getPredefinedWorkspaceSpecTypes()

    if options.showTypes:
        print '\nPredefined workspace specs are:'
        print ' - %s' % ('\n - '.join(supportedPredefinedTypes))
        print '\nTo see spec contents re-run with --show-specs -t <SPEC_TYPE> -r <ROOT_DIR>'
        return

    if not options.root:
        print ''
        parser.print_help()
        defaultOutputFunc("\n\nERROR: A root dir must be specified using -r/--root.")
        return

    assert os.path.isdir(options.root), "\n\nERROR: Specified root is not a valid dir: %s" % options.root
    rootDir = os.path.normpath(os.path.abspath(options.root))

    assert (bool(options.types) ^ bool(options.workspaceName)), (
           "\n\nERROR: In addition to a root dir to use, a predefined workspace type or a workspace name to use "
           "must be specified (but not both). Re-run with -h/--help to see all options.")

    def nullPrint(msg):
        pass

    printFunc = nullPrint if options.quietMode else defaultOutputFunc

    workspaceSpecsList = []

    if options.types:
        for specType in options.types:
            assert specType in supportedPredefinedTypes, (
                   "\n\nERROR: Unsupported workspace type specified - use one of %s" % supportedPredefinedTypes)

            workspaceSpecsList.extend(workspaceSpecs.getPredefinedWorkspaceSpecs(specType, rootDir))

    else:
        assert options.workspaceName != "", "\n\nERROR: The workspace name must be set using -n/--name"
        spec = workspaceSpecs.WorkspaceSpec(workspaceName=options.workspaceName, rootDir=rootDir)
        workspaceSpecsList.append(spec)

    for spec in workspaceSpecsList:
        if options.globPatterns:
            spec.globPatterns = tuple(options.globPatterns)

        if options.rootRelIncludePaths:
            # If any root-rel include paths are specifed don't include the root dir itself:
            spec.rootRelIncludePaths = [p for p in spec.rootRelIncludePaths if p != '.']
            spec.rootRelIncludePaths.extend(options.rootRelIncludePaths)

        if options.rootRelExcludePaths:
            spec.rootRelExcludePaths.extend(options.rootRelExcludePaths)

        if options.rootRelAlwaysIncludePaths:
            spec.rootRelAlwaysIncludePaths.extend(options.rootRelAlwaysIncludePaths)

        if options.disabledProjectPaths:
            spec.disabledProjectPaths.extend(options.disabledProjectPaths)

        if options.allowedPlatforms:
            spec.allowedPlatforms.extend(options.allowedPlatforms)

        if options.allowedCompilers:
            spec.allowedCompilers.extend(options.allowedCompilers)

        if options.targetSubdirs:
            spec.putInTargetSubdirs = options.targetSubdirs

    if options.showSpecs:
        for spec in workspaceSpecsList:
            print '\n\n%s:' % spec.workspaceName.upper()
            print '\n%s' % spec
        return []

    if options.types:
        printFunc(("Generating workspace(s) for predefined types %s (with additional options applied "
                   "if set)...") % (', '.join(options.types)))
    else:
        printFunc("Generating workspace(s) using user-defined spec:\n%s\n" % (str(spec)))

    workspaceGenerator = WorkspaceGenerator(rootDir)
    workspaceGenerator.workspaceSpecs = workspaceSpecsList
    generatedWorkspaces = workspaceGenerator.findProjectsAndGenerateWorkspaces(rootDir, outputFunc=printFunc)

    if options.quietMode:
        if generatedWorkspaces:
            for workspaceFile in generatedWorkspaces:
                print ' - %s' % workspaceFile
        else:
            print ' - none'

    return generatedWorkspaces


if __name__ == "__main__":
    main()
    sys.exit(0)
