import os
import re
import fnmatch
from xml.etree import ElementTree as ET
from collections import defaultdict

SUPPORTED_TARGET_SPECIALIZATIONS = ['noSimd', 'win10', 'anarchy', 'double', 'noRtti', 'clang', 'neon', 'lto']


def schemaTag(tag):
    """
    Given a tag name return that name including the active msbuild schema.
    """
    return "{http://schemas.microsoft.com/developer/msbuild/2003}" + tag


def xpath(tag):
    """
    Given a tag name return a recursive xpath expression for that tag (including the necessary schema).
    """
    return './/' + schemaTag(tag)


def defaultOutputFunc(message):
    print message


class CommentParser(ET.XMLTreeBuilder):
    """
    Subclass of XMLTreeBuilder to handle recording Comments in the xml. We use
    comments to store blocks of text containing metadata so we need this.
    """
    def __init__(self):
        super(CommentParser, self).__init__()
        self._parser.CommentHandler = self.comment_tag_handler
        self._target.start("document", {})

    def close(self):
        self._target.end("document")
        return ET.XMLTreeBuilder.close(self)

    def comment_tag_handler(self, data):
        # Insert the node into the DOM as "Comment"
        self._target.start("Comment", {})
        self._target.data(data)
        self._target.end("Comment")


class Project(object):
    def __init__(self, projectPath, projectName, outputFunc=defaultOutputFunc):

        self.path = projectPath
        self.name = projectName
        self.filename = os.path.basename(self.path)
        self.basename = os.path.splitext(self.filename)[0]
        self.rawXml = self.loadXml()
        self.dom = ET.fromstring(self.rawXml, CommentParser())
        self.outputFunc = outputFunc
        self.importLibraries = set()  # for DLLs projects only
        self.listedDependencies = []
        self.runtimeDependencies = []
        self.dependencies = []
        self.platforms = []
        self.projectType = ""
        self.excludedFromBuild = False
        self.deploy = False

        self.extraInfo = []
        baseNameParts = self.basename.split('_')
        while baseNameParts and baseNameParts[-1] in SUPPORTED_TARGET_SPECIALIZATIONS:
            self.extraInfo.insert(0, baseNameParts.pop())
        self.extraInfo = '_'.join(self.extraInfo)

        self.solutionVariant = ""
        self.customProperties = {}
        self.magicGuid = "8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942"
        try:
            self.namespace = re.match('\{(.*)\}', self.firsttag("Project").tag).group(1)
        except Exception:
            self.namespace = ''
        ET.register_namespace("",self.namespace) #registers a blank prefix for the namespace when converting node to string
        self.parseProject()

    def loadXml(self):
        """Loads the project at self.path, removes any BOM, then returns as a string"""
        import codecs
        assert os.path.exists(self.path), "Project (%s) does not exist." % self.path
        rawXml = open(self.path).read()
        # Handle possible utf8 Byte Order Mark
        if rawXml.startswith(codecs.BOM_UTF8):
            rawXml = rawXml.lstrip(codecs.BOM_UTF8)
        return rawXml

    def itertags(self, tagname):
        for node in self.dom.iterfind(xpath(tagname)):
            yield node

    def firsttag(self, tagname):
        try:
            return next(self.itertags(tagname))
        except StopIteration:
            return None

    def pathFromRoot(self, root):
        """
        Path comparisons were failing because the drive letter had a different case
        in different strings (e.g. d: vs D:), seemingly only for some users.
        The drive letter is now omitted when working out the relative path.
        """
        rootNoDrive = os.path.splitdrive(root + os.path.sep)[1]
        modulePathNoDrive = os.path.splitdrive(self.path)[1]
        return modulePathNoDrive.replace(rootNoDrive, '')

    def parseProject(self):
        """
        Overridden in descendant classes.
        """
        raise NotImplementedError("Subclass me")

    def getProjectFileDependencies(self, otherProjects, havokRoot, propertyName):
        """
        Get the dependencies stored under *propertyName* in the project
        """
        otherProjectsByFilename = {}
        for p in otherProjects:
            otherProjectsByFilename[p.filename] = p
        dependenciesList = []
        for depProjFile in eval(self.customProperties.get(propertyName, "[]")):
            try:
                dependenciesList.append(otherProjectsByFilename[depProjFile])
            except KeyError:
                pass
        return dependenciesList

    def setDependencies(self, otherProjects, havokRoot=""):
        self.dependencies = []
        # only file name without extension is left - the path is ignored
        dependencyNames = []
        for dep in self.listedDependencies:
            depName = os.path.splitext(os.path.basename(dep))[0]
            dependencyNames.append(depName)
        if dependencyNames:
            for currentProject in otherProjects:
                outputFile = os.path.splitext(os.path.basename(currentProject.outputFile))[0]
                if outputFile:
                    isDep = outputFile in dependencyNames or outputFile.replace('lib', '') in dependencyNames
                    if not isDep:
                        isDep = any(os.path.splitext(lib)[0] in dependencyNames for lib in currentProject.importLibraries)
                    if isDep:
                        self.dependencies.append(currentProject)
        # Other dependency types:
        self.runtimeDependencies = self.getProjectFileDependencies(otherProjects, havokRoot, "RUNTIME_DEPENDENCIES")

    @staticmethod
    def getCustomProperties(xmlDom):
        """
        Get properties set in XML comments, e.g. solution variant is provided as <!-- VARIANT = "DX11" -->
        """

        customProperties = {}
        for comment in xmlDom.findall('.//Comment'):
            match = re.findall(r"(?P<variableName>[a-zA-Z0-9_]+)\s*=\s*"
                               r"(?P<variableValue>(\"[a-zA-Z0-9_/\-,\.\[\]\s\'+]*\")|(\'[a-zA-Z0-9_/\-,\.\[\]\s\"+]*\'))",
                               comment.text.strip())
            if match:
                for var in match:
                    variableName, variableValue = var[0:2]
                    customProperties[variableName] = variableValue[1:-1]
        return customProperties

    def getProjectReferences(self):
        """
        Get project references for Dist system.
        """
        references = {}
        namespaceString = ' xmlns="{}"'.format(self.namespace)
        for projectReferenceNode in self.itertags("ProjectReference"):
            reference = projectReferenceNode.attrib['Include']
            referenceText = ET.tostring(projectReferenceNode, encoding='utf-8').replace(namespaceString, '')
            if referenceText not in self.rawXml:
                raise Exception("\nError: project reference invalid text {}\n{}\n\n".format(reference, referenceText))
            references[reference] = referenceText
        return references

    def getProjectReferenceGUIDs(self):
        """
        Get project GUIDs for all project references.
        """
        guids = set()
        for projectReferenceNode in self.itertags("ProjectReference"):
            for projectNode in projectReferenceNode.iterfind(xpath("Project")):
                if projectNode.text is not None:
                    guids.add(self.formatGuid(projectNode.text))
        return guids

    def getProjectReferencePaths(self):
        """
        Get paths to referenced projects, relative to current project.
        """
        paths = set()
        for projectReferenceNode in self.itertags("ProjectReference"):
            path = projectReferenceNode.attrib['Include']
            if path:
                paths.add(path)
        return paths 

    def getProjectSourceLevel(self):
        """
        Return the project's source level
        """
        return self.customProperties.get("SOURCE_LEVEL", "")

    def getRequiredProducts(self):
        """
        Return the products this project belongs to
        """
        return self.customProperties.get("REQUIRED_HAVOK_PRODUCTS", "")

    @staticmethod
    def formatGuid(guidStr):
        return guidStr.lstrip('{').rstrip('}')


class Vs20052008Project(Project):
    def __init__(self, projectPath, projectName, outputFunc=defaultOutputFunc):
        super(Vs20052008Project, self).__init__(projectPath, projectName, outputFunc)
        raise NotImplementedError("Removed in the ElementTree refactor. Reinstate the code if needed.")


class Vs2010Project(Project):
    def __init__(self, projectPath, projectName, outputFunc=defaultOutputFunc):
        super(Vs2010Project, self).__init__(projectPath, projectName, outputFunc)

    def parseProject(self):
        self.customProperties = self.getCustomProperties(self.dom)
        self.solutionVariant = self.customProperties.get("VARIANT", "")
        self.platform = self.customProperties.get("PLATFORM", None)
        self.extraInfo = self.customProperties.get("EXTRA_INFO", self.extraInfo)
        self.type = "cpp"

        for node in self.itertags('CompileAsManaged'):
            if node.text is not None and node.text.strip() == "True":
                self.type = 'mgd'
                break

        if self.path.endswith('.androidproj'):
            self.magicGuid = '39E2626F-3545-4960-A6E8-258AD8476CE5'
            self.outputFile = ''
            self.deploy = True
        elif self.path.endswith('.pyproj'):
            self.magicGuid = '{888888A0-9F3D-457C-B088-3A5042F75D52}'
            self.outputFile = ''
        else:
            targetName = os.path.basename(self.firsttag('TargetName').text)
            targetExt = self.firsttag('TargetExt').text
            if targetExt is None:
                targetExt = ''
            self.outputFile = targetName + targetExt
        if self.platform and self.platform.lower() in ('nx32', 'nx64', 'nx32_win', 'nx64_win'):
            # potentially raises AttributeError
            try:
                self.projectType = self.firsttag("ConfigurationType").text  # e.g. Application
            except AttributeError:
                raise Exception('Project has no ConfigurationType setting (%s)' % self.path)
        elif self.path.endswith('.pyproj'):
            self.projectType = 'Python'
        else:
            try:
                self.projectType = self.firsttag("Keyword").text  # e.g. Application
            except AttributeError:
                self.projectType = self.firsttag("ConfigurationType").text

        def linktypeFromCondition(condition):
            return condition.split("==")[1].strip("'").split("|")[0]

        self.targetName = defaultdict(lambda: "")
        for targetNameNode in self.itertags("TargetName"):
            try:
                linktype = linktypeFromCondition(targetNameNode.attrib["Condition"])
                self.targetName[linktype] = targetNameNode.text
            except KeyError:
                self.targetName.default_factory = lambda: targetNameNode.text
        for configurationNode in self.itertags("ItemDefinitionGroup"):
            linktype = linktypeFromCondition(configurationNode.attrib["Condition"])
            importLibrary = configurationNode.findall(xpath("ImportLibrary"))
            if not importLibrary or importLibrary[0].text is None:
                continue
            importLibrary = os.path.basename(importLibrary[0].text)
            if importLibrary:
                self.importLibraries.add(importLibrary.replace("$(TargetName)", self.targetName[linktype]))

        self.listedDependencies = set()

        for elem in self.itertags("AdditionalDependencies"):
            if elem.text is not None:
                for dep in elem.text.split(';'):
                    if dep != '%(AdditionalDependencies)':
                        self.listedDependencies.add(dep)

        # WiiU does not like the libs listed in the vscproj as opposed to wiiuopt as will not have proper libpath
        # for the prelinker, so we fake them for this script only.

        for elem in self.itertags("FakeAdditionalDependencies"):
            if elem.text is not None:
                for dep in elem.text.split(';'):
                    if dep != '%(AdditionalDependencies)':
                        self.listedDependencies.add(dep)

        self.listedDependencies = list(self.listedDependencies)
        self.listedDependencies = [d.strip().replace('-l', '') for d in self.listedDependencies]

        self.guid = self.formatGuid(self.firsttag("ProjectGuid").text)

        if self.path.endswith('.pyproj'):
            self.platform = 'Any CPU'
            self.platforms = [self.platform]
        else:
            self.platforms = [elem.text for elem in self.itertags('Platform')]
            if self.platform is None:
                self.platform = self.platforms[0]
        if self.platform.lower() == "x86":
            self.groupingPlatform = u"win32"
        elif self.platform.lower() == "arm":
            if "android" in self.filename:
                self.groupingPlatform = "android_arm"
        elif self.platform.lower() == "hkandroid":
            if "android_x86" in self.basename:
                self.groupingPlatform = "android_x86"
            else:
                self.groupingPlatform = "android_arm"
        elif self.platform.lower().startswith("hk"):
            self.groupingPlatform = self.platform.lower()[2:]  # remove hk prefix, e.g. hkAndroid
        else:
            self.groupingPlatform = self.platform.lower()

        if "metro" in self.basename:
            if self.platform.lower() == "win32":
                self.groupingPlatform = "metro_x86"
            elif self.platform.lower() == "x64":
                self.groupingPlatform = "metro_x64"
            elif self.platform.lower() == "arm":
                self.groupingPlatform = "metro_arm"
        if "apollo" in self.basename:
            if self.platform.lower() == "win32":
                self.groupingPlatform = "apollo_x86"
            elif self.platform.lower() == "arm":
                self.groupingPlatform = "apollo_arm"
        if self.platform.lower() == "win32" and "wiiu" in self.basename:
            self.groupingPlatform = "wiiu"

        self.linktypes = set()
        for configNode in self.itertags('ProjectConfiguration'):
            self.linktypes.add(configNode.attrib['Include'].split('|')[0].strip())
        self.linktypes = list(self.linktypes)

        toolsVersion = self.dom.getchildren()[0].attrib.get('ToolsVersion', None)

        if "vs2010" in self.basename:
            self.compiler = "VS2010"
        elif "vs2012" in self.basename:
            self.compiler = "VS2012"
        elif "vs2013" in self.basename:
            self.compiler = "VS2013"
        elif "vs2015" in self.basename or toolsVersion == '14.0':
            self.compiler = "VS2015"
        elif "vs2017" in self.basename or toolsVersion == '15.0':
            self.compiler = "VS2017"
        elif self.path.endswith('.pyproj'):
            self.compiler = 'VS2015'  # HACK
        else:
            self.outputFunc("Error parsing version from filename: " + self.path)
            raise Exception

        if (self.platform in ('uwp', 'uwp_arm', 'uwp_x86', 'uwp_x64',
                              "metro_x86", "metro_x64", "metro_arm",
                              "apollo_x86", "apollo_arm")
        and self.projectType == "Application"):
            self.deploy = True


class CsharpProject(Project):
    def __init__(self, projectPath, projectName, outputFunc=defaultOutputFunc):
        super(CsharpProject, self).__init__(projectPath, projectName, outputFunc=defaultOutputFunc)
        self.magicGuid = "FAE04EC0-301F-11D3-BF4B-00C04F79EFBC"
        self.targetName = {}
        assemblyName = self.firsttag("AssemblyName")
        if assemblyName is not None and assemblyName.text is not None:
            self.targetName["*"] = assemblyName.text

    def parseProject(self):
        self.type = "cs"
        self.customProperties = self.getCustomProperties(self.dom)
        self.solutionVariant = self.customProperties.get("VARIANT", "")
        self.platform = self.customProperties.get("PLATFORM", None)

        ext = {"Library": ".dll",
               "WinExe": ".exe",
               "Exe": ".exe",
               'winmdobj': '.winmd',
               'AppContainerExe': '.exe'}[self.firsttag('OutputType').text]
        self.outputFile = self.firsttag('AssemblyName').text + ext
        self.listedDependencies = []
        self.guid = self.formatGuid(self.firsttag("ProjectGuid").text)
        self.platforms = set()
        self.linktypes = set()
        for pgNode in self.itertags('PropertyGroup'):
            try:
                platform = pgNode.attrib['Condition'].split('==')[1].split('|')[1].strip().replace("'", "")
                linktype = pgNode.attrib['Condition'].split('==')[1].split('|')[0].strip().replace("'", "")
                if platform:
                    self.platforms.add(platform)
                self.linktypes.add(linktype)
            except KeyError:
                pass
        self.platforms = list(self.platforms)
        self.linktypes = list(self.linktypes)
        if self.platform is None:
            self.platform = self.platforms[0]
        self.groupingPlatform = self.platform.lower()
        if self.groupingPlatform == "x86":
            self.groupingPlatform = u"win32"

        # Remove specializations from the base name to safely identify the compiler:
        if "8-0" in self.basename:
            self.compiler = "VS2005"
        elif "9-0" in self.basename:
            self.compiler = "VS2008"
        elif "vs2010" in self.basename:
            self.compiler = "VS2010"
        elif "vs2012" in self.basename:
            self.compiler = "VS2012"
        elif "vs2013" in self.basename:
            self.compiler = "VS2013"
        elif "vs2015" in self.basename:
            self.compiler = "VS2015"
        elif "vs2017" in self.basename:
            self.compiler = "VS2017"
        else:
            self.outputFunc("Error parsing version from filename: " + self.path)
            raise Exception


def getProjectObject(projectPath, outputFunc=defaultOutputFunc):
    """
    Returns a Project object instance (Vs2010Project, Vs20052008Project, CsharpProject) based on
    project file extention.
    """
    projectExtension = os.path.splitext(projectPath)[1]
    projectExtTypeMapping = {".vcproj": Vs20052008Project,
                             ".vcxproj": Vs2010Project,
                             ".cxproj": CsharpProject,
                             ".csproj": CsharpProject,
                             ".pyproj": Vs2010Project,
                             '.androidproj': Vs2010Project}

    return projectExtTypeMapping[projectExtension](projectPath, "project_name", outputFunc=outputFunc)


def includeProjectForSourceLevel(proj, sourceLevel):
    """
    Include the project if its source level is 'higher' than *productsList*
    """
    if not sourceLevel:
        return True

    orderedSourceLevels = ['PUBLIC', 'CLIENT', 'INTERNAL']

    assert sourceLevel in orderedSourceLevels, "\n\nERROR: Unrecognised source level specified: %s" % sourceLevel

    projSourceLevel = proj.getProjectSourceLevel()
    assert projSourceLevel in orderedSourceLevels, (
           "\n\nERROR: Unrecognised source level in project %s: %s" % (proj.path, sourceLevel))

    return (orderedSourceLevels.index(projSourceLevel) <= orderedSourceLevels.index(sourceLevel))


def includeProjectForProducts(proj, productsList):
    """
    Include the project if the products in use are limited to those in *productsList*
    """
    # For example:
    #    productsList = ['PHYSICS', 'DESTRUCTION', 'ANIMATION']
    #    proj.getRequiredProducts() = 'PHYSICS_2012+ANIMATION PHYSICS+ANIMATION'
    if not productsList:
        return True
    products = set(productsList)
    projProducts = [set(p.split('+')) for p in proj.getRequiredProducts().split(' ')]
    return any([p.issubset(products) for p in projProducts])


def findProjects(globPatterns, includePaths, excludePaths, alwaysIncludePaths, disabledProjectPaths,
                 sourceLevel, productsList, outputFunc=defaultOutputFunc):
    """
    Generator function that scans the filesystem for VisualStudio projects.
    The file filters are passed in globPatterns, e.g. ("*.vcproj", "*.csproj",  "*.vcxproj")
    Projects with filenames matching any disabledProjectPaths will be excluded from compilation
    (need to be manually switched on in Configuration Manager)
    """
    # Visual Studio get's very upset if solutions contain duplicate projects, so keep track of
    # project's we've found so far and don't generate any subsequent duplicates.
    foundProjects = []

    def isSubfolderOf(folder, potentialSubFolder):
        return os.path.abspath(potentialSubFolder).lower().startswith(os.path.abspath(folder).lower())

    def getProjects(includePath):
        if (not any(isSubfolderOf(excludePath, includePath) for excludePath in excludePaths)
        or any(isSubfolderOf(alwaysIncPath, includePath) for alwaysIncPath in alwaysIncludePaths)):
            for path, dirs, files in os.walk(includePath):
                for d in dirs[:]:
                    if (any(isSubfolderOf(excludePath, os.path.join(path, d)) for excludePath in excludePaths)
                    and not any(isSubfolderOf(alwaysIncPath, includePath) for alwaysIncPath in alwaysIncludePaths)):
                        dirs.remove(d)
                for f in files:
                    for globPattern in globPatterns:
                        if fnmatch.fnmatch(f, globPattern):
                            projectPath = os.path.join(path, f)
                            try:
                                proj = getProjectObject(projectPath, outputFunc)
                                if not includeProjectForSourceLevel(proj, sourceLevel):
                                    continue
                                if not includeProjectForProducts(proj, productsList):
                                    continue
                                if any(disabledPath in os.path.basename(projectPath)
                                       for disabledPath in disabledProjectPaths):
                                    proj.excludedFromBuild = True
                                yield proj
                            except Exception:
                                outputFunc("Error parsing: %s" % projectPath)

    for includePath in (includePaths + alwaysIncludePaths):
        for project in getProjects(includePath):
            # Only yield if we haven't seen this project before
            if project.path.lower() not in foundProjects:
                foundProjects.append(project.path.lower())
                yield project
