# Typically you should invoke this script as: headerToInternalStateFiles.py ../../Source
# The script will probably get the header include paths wrong if you invoke it any other way.

import os
import re
import sys
import warnings

#------------------------------------------------------
#compile regular expressions that we use over and over
#------------------------------------------------------
re_tkbms = re.compile(r"// TKBMS v1\.0(.*)TKBMS v1.0", re.DOTALL)
re_includes = re.compile(r"#include.*?\.h>")
re_internalState = re.compile( r"HKB_BEGIN_INTERNAL_STATE\([0-9]+\)*(.*)HKB_END_INTERNAL_STATE\(\)*", re.DOTALL)
#note that we remove spaces and slashes before a member to allow a "fake" member to be inside a comment
re_member = re.compile(r"\s*(?:mutable\s){0,1}/*(.*)\s+(m_.*);")
re_hasInternalState = re.compile("HKB_BEGIN_INTERNAL_STATE")
re_internalStateVersion = re.compile( r"HKB_BEGIN_INTERNAL_STATE\(([0-9]+)\)", re.DOTALL )
#given a member type name determines whether it has a finish constructor
re_hasFinish = re.compile(r"^(?:hkArray<)|(?:struct\s)|(?:class\s)")
re_isAutoGeneratedFile = re.compile("EDITS WILL BE LOST")

#delete old internal state files in case a source file has been removed
def removeIfInternalState(filename):
    #we only want to look InternalState.h and InternalState.cpp files
    if not filename.endswith("InternalState.h") and not filename.endswith("InternalState.cpp"):
        return
    
    scrubbedFileName = ""
    dir, fileWithExt = os.path.split(filename)
    name, ext = os.path.splitext(fileWithExt)
    
    # construct file name without extension and without "InternalState".  Any files that don't
    # match this do not have a matching .cpp/.h version of the file.
    idx = fileWithExt.find("InternalState")
    if( idx != -1 ):
        scrubbedFileName = fileWithExt[:idx] + ext
    
    found = False
    
    #only remove it if the source file was removed!
    for curFile in os.listdir( dir ):
        if( scrubbedFileName == curFile ):
            found = True
            break
    
    #read the input file
    if( found == False ):
        fin = open(filename, 'r')
        text = fin.read()
        fin.close()

        if re_isAutoGeneratedFile.search(text):
            os.remove(filename)

# un-camel case a class name into upper case with underbars between words
def unCamel(name):
    return re.sub('([a-z])([A-Z])', r'\1_\2', name).upper()

# This function takes the name of a file as input.  It may be relative like "../../Source/Behavior/Behavior/Generator/Clip/hkbClipGenerator.h".
# This function writes two files in the same folder as the input file <FILE>: <FILE>InternalState.h and <FILE>InternalState.cpp.
# For example, with the above input the files written would be hkbClipGeneratorInternalState.h and hkbClipGeneratorInternalState.cpp.
# The written files are cpp source files that contain a class definition for the internal state of the node and some virtual
# function definitions.
def headerToInternalState(filename, precompiledHeader, includePathLength):

    #ignore files we don't want
    if not filename.endswith('.h'):
        return
    if filename.endswith("hkbBase.h"):
        return
    if filename.endswith("InternalState.h"):
        return

    #read the input file
    fin = open(filename, 'r')
    text = fin.read()
        
    #do nothing if the file does not contain HKB_BEGIN_INTERNAL_STATE
    if not re_hasInternalState.search(text):
        return
    
    #extract the path to the file so we can put the internal state files there
    (justPath, justFilename) = os.path.split(filename)
    basePath = justPath

    #convert the base path to only use forward slashes
    basePath = basePath.replace('\\', '/')

    #convert the precompiled header path to only use forward slashes
    precompiledHeader = precompiledHeader.replace('\\', '/')

    #the relevant class names are inferred from the name of the header file
    className = justFilename[:-2]
    stateClassName = className + "InternalState"
    outFilename = justPath + '/' + stateClassName + '.cpp'
    outFilenameH = justPath + '/' + stateClassName + '.h'

    #compute the relative path by removing the first rootPathLength elements of the basePath
    relativeBasePath = basePath[includePathLength:]
    
    fileString = []
    fileStringH = []

    #copy the tkbms header from the header file to the output file, omitting SPU and SIMSPU
    m = re_tkbms.search(text)
    if m:
        tkbms_text = m.group(0);
        tkbms_text = re.sub("SIMSPU", "", tkbms_text) 
        tkbms_text = re.sub("SPU", "", tkbms_text) 
        fileString.append(tkbms_text + '\n\n')
        fileStringH.append(tkbms_text + '\n\n')
        
    #write out a warning not to edit this file
    fileString.append('// WARNING: THIS FILE IS GENERATED. EDITS WILL BE LOST.\n')
    fileString.append("// Generated from '" + relativeBasePath + '/' + justFilename + "'\n\n")
    fileStringH.append('// WARNING: THIS FILE IS GENERATED. EDITS WILL BE LOST.\n')
    fileStringH.append("// Generated from '" + relativeBasePath + '/' + justFilename + "'\n\n")

    #write out a guard to the header file
    guardName = unCamel(stateClassName) + "_H"
    fileStringH.append("#ifndef " + guardName + "\n")
    fileStringH.append("#define " + guardName + "\n\n")
    
    #include the original header in the internal state header in case there are internal types
    fileStringH.append('#include <' + relativeBasePath + '/' + justFilename + '>\n\n')

    #write the includes for the precompiled header and the header for this class
    fileString.append("#include <" + precompiledHeader + ">\n")
    fileString.append('#include <' + relativeBasePath + '/' + stateClassName + '.h>\n\n')

    #find the members
    m = re_internalState.search(text)
    memberText = m.group(1);
    members = re_member.findall(memberText)

    #classVersion = findClassVersion( filename )
    v = re_internalStateVersion.search(text)
    classVersion = v.group(1)
    
    #write out the class name and reflection macros
    fileStringH.append("class " + stateClassName + " : public hkReferencedObject\n")
    fileStringH.append("{\n")
    fileStringH.append("    //+version(" + str(classVersion) + ")\n")
    fileStringH.append("    public:\n\n")
    fileStringH.append("        HK_DECLARE_REFLECTION();\n")
    fileStringH.append("        HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BEHAVIOR );\n\n")
    fileStringH.append("        " + stateClassName + "() {}\n")

    #write out the internal state members
    for member in members:
        fileStringH.append('        ' + member[0] + ' ' + member[1] + ';\n')

    #write out the closing brace
    fileStringH.append("};\n\n")

    #write out the #endif for the header guard
    fileStringH.append("#endif\n")

    #write out the createInternalState function
    fileString.append("hkReferencedObject* " + className + "::createInternalState()\n{\n    return new " + stateClassName + "();\n}\n\n")

    #write out the getInternalState function
    fileString.append("void " + className + "::getInternalState( hkReferencedObject& internalStateOut ) const\n{\n")
    fileString.append("    " + stateClassName + "& internalState = static_cast<" + stateClassName + "&>(internalStateOut);\n\n")
    for member in members:
        fileString.append("    internalState." + member[1] + " = " + member[1] + ";\n")
    fileString.append("}\n\n")

    #write out the setInternalState function
    fileString.append("void " + className + "::setInternalState( const hkReferencedObject& internalStateIn )\n{\n")
    fileString.append("    const " + stateClassName + "& internalState = static_cast<const " + stateClassName + "&>(internalStateIn);\n\n")
    for member in members:
        fileString.append("    " + member[1] + " = internalState." + member[1] + ";\n")
    fileString.append("}\n")
    
    #join the array of strings.  this is one of the faster concatenation methods    
    endFileOut = ''.join(fileString)
    endFileOutH = ''.join(fileStringH)
    
    processed = writeFileIfDifferent( outFilename, endFileOut )
    processedH = writeFileIfDifferent( outFilenameH, endFileOutH )
    
    if( processed or processedH ):
        print( "Processing " + basePath + '/' + justFilename )


def writeFileIfDifferent( fileName, fileString ):
    shouldWriteFile = True
    
    #open output file for read (we want to check them first)
    if( os.path.isfile( fileName ) ):
        fileRead = open( fileName, 'r' )
        fileContents = fileRead.read()
        fileRead.close()
        
        if( fileContents == fileString ):
            shouldWriteFile = False
        
    # only write file if the string hasnt changed
    if( shouldWriteFile ):
        fileWrite = open( fileName, 'w' )
        fileWrite.write( fileString )
        fileWrite.close()
        return True
    
    return False

# Process all files in a directory recursively.
def processDir(where, precompiledHeader, includePathLength):
    for dirname, subdirs, files in os.walk(where):
        headers = [ os.path.join(dirname,f) for f in files ]
        for header in headers:
            removeIfInternalState(header)
        for header in headers:
            headerToInternalState(header, precompiledHeader, includePathLength)

USAGE = """%prog <WHERE>...

Recursively searches WHERE (typically ../../Source/Behavior/Behavior) for C++ header files.
Generates FILEInternalState.h and FILEInternalState.cpp for each file found."""

def main(argv):
    """
    ARGUMENTS:
        argv (list) - command line args, e.g. sys.argv[1:]. This allows other scripts to use this function directly
                      without the need to run it as a separate process.
    """
    import optparse
    parser = optparse.OptionParser(usage=USAGE)
    parser.add_option("--precompiledHeader", help="The name of the precompiled header file for your library.", default="Behavior/Behavior/hkbBehavior.h")
    parser.add_option("--includePath", help="Header includes will be expressed relative to this path. It must be a prefix of WHERE.", default="")
    options, args = parser.parse_args(argv)
    
    print( "Generating internal state for Behavior." )
    
    for arg in args:
        if os.path.isdir(arg):
            includePath = options.includePath
            if includePath == "":
                #by default we assume that all includes are relative to the input folder
                includePath = arg
            #we'll chop off this many characters from each include file path (1 extra for the slash)
            includePathLength = len(includePath)+1
            processDir(arg, options.precompiledHeader, includePathLength)
        elif os.path.isfile(arg):
            headerToInternalState(arg, options.precompiledHeader)
        else:
            warnings.warn("'%s' is not a file nor directory, skipping." % arg)
    if len(args) == 0:
        parser.error("No search path given.")

if __name__=="__main__":
    main(sys.argv[1:])
    
