"""
A Script to Configure Havok Projects and Makefiles
"""
import os
import sys
import glob
import string
import fileinput
import re
import inspect
from itertools import chain
import stat
import fnmatch
import Tkinter
import tkFont
import random
import posixpath
import shutil
import ast
import collections
import subprocess
import time

from xml.dom.minidom import parse, Comment

THIS_DIR = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
HAVOK_ROOT = os.path.normpath(os.path.join(THIS_DIR, os.pardir, os.pardir, os.pardir, os.pardir))
PACKAGE_DETAILS_DIR = os.path.join(HAVOK_ROOT, 'Docs', 'PackageDetails', 'Data')

if 'HK_LOG_DIR' in os.environ:
    sys.stdout = open(os.path.join(os.environ['HK_LOG_DIR'], "configureHavok.log", "w"))
    sys.stderr = open(os.path.join(os.environ['HK_LOG_DIR'], "configureHavokError.log"), "w")

WORKSPACEGEN_PYTHON_DIR = os.path.join(HAVOK_ROOT, "Build", "StandaloneTools", "WorkspaceGeneration", "Python")
sys.path.append(WORKSPACEGEN_PYTHON_DIR)

import project
import vsSolution

TEMPLATE_FILE_EXT = '.template'
LEVEL2_MODULE_FILE_EXT = '.level2module'
KEYCODE_FILE = os.path.join('Source', 'Common', 'Base', 'KeyCode.h')
SIMD_CONFIG_FILE = os.path.join('Source', 'Common', 'Base', 'Config', 'hkConfigSimd.h')

KEYCODE_REGEX = '#define\s+HAVOK_[A-Z0-9_]*_KEYCODE\s+".*"'

INITIAL_TEXT = "<Replace this text with your Havok product keycode(s)>"

EMPTY_KEYCODE_PRODUCTS = ['ANIMATION',
                          'CONVEX_DECOMPOSITION',
                          'FX',
                          'SCRIPT']
NO_LIB_PRODUCTS = ['SCRIPT']  # Script packages contain source but not libs - as full source is shipped.

PRODUCT_DEPENDENCIES = {  # Product dependencies excluding Common, as a check for this is already present
    'BEHAVIOR': ['ANIMATION',
                 'SCRIPT'],
    'DESTRUCTION': ['PHYSICS',
                    'CONVEX_DECOMPOSITION'],
    'DESTRUCTION_2012': ['PHYSICS_2012',
                         'CONVEX_DECOMPOSITION'],
    'FX': ['PHYSICS']}

MODULE_CONFIG_DICT = {}
LIBRARY_CONFIG_DICT = {}
SOURCE_CONFIG_DICT = {}

GENERATED_SOLUTIONS = []
__AVAILABLE_SOURCE_LEVELS = None

HK_PREBUILD_FAILURES = []

XCODE_DEV_TEAM = 'INSERT_YOUR_XCODE_DEVELOPMENT_TEAM_HERE'
HAVOK_XCODE_DEV_TEAM = 'S79S84249S'


class HkPreBuildFail():
    """
    Used to capture information about failed invocations of hkPreBuild.
    """
    def __init__(self, modulePath, command, returncode, output):
        self.modulePath = os.path.normpath(os.path.join(HAVOK_ROOT, modulePath))
        self.command = command
        self.returncode = returncode
        self.relModulePath = os.path.relpath(self.modulePath, HAVOK_ROOT)
        self.moduleName = '_'.join(self.relModulePath.split(os.path.sep))
        self.logsDirPath = os.path.join(HAVOK_ROOT, "Build", "_Logs")
        timestamp = time.strftime('%Y%m%d%H%M%S', time.localtime(time.time()))
        self.logPath = os.path.join(self.logsDirPath, "{}_{}.log".format(timestamp,
                                                                         self.moduleName))
        self.preBuildArgsPath = os.path.join(self.modulePath, 'hkPreBuild_args.txt')
        try:
            with open(self.preBuildArgsPath) as preBuildArgs:
                self.preBuildArgs = preBuildArgs.read()
        except OSError:
            self.preBuildArgs = ''
        try:
            if not os.path.isdir(self.logsDirPath):
                os.makedirs(self.logsDirPath)
            with open(self.logPath, 'w') as logFile:
                sep = '*' * 66 + '\n'
                logFile.write(sep)
                logFile.write('Command: {}\n'.format(str(command)))
                logFile.write(sep)
                logFile.write('hkPreBuild_args.txt:\n{}\n'.format(self.preBuildArgs))
                logFile.write(sep)
                logFile.write('Returncode: {}\n'.format(self.returncode))
                logFile.write(sep + '\n')
                logFile.write('Output:\n')
                logFile.write(output)
        except Exception:
            self.logPath = None


def getLineEnding(platform):
    if "linux" in platform.lower() or "macos" in platform.lower() or "ios" in platform.lower():
        return '\n'
    return "\r\n"


def getDataDir():
    """
    Populate *MODULE_CONFIG_DICT*, *LIBRARY_CONFIG_DICT* and *SOURCE_CONFIG_DICT*
    Returns True on success, False on failure.
    """
    return os.path.normpath(os.path.join(THIS_DIR, os.pardir, 'Data'))


def getPackageDetails():
    """Returns a dictionary mapping package names (minus extension) to a dictionary containing keys:
    TARGET, PRODUCT, VISIBILITY. For details on format see getPackageDataDict in
    PackageSystem/packagingUtils.py"""
    packageDetails = {}
    for f in os.listdir(PACKAGE_DETAILS_DIR):
        try:
            packageName = os.path.splitext(f)[0]
            packageDetailsPath = os.path.join(PACKAGE_DETAILS_DIR, f)
            packageDetailsStr = open(packageDetailsPath).read()
            packageDetails[packageName] = eval(packageDetailsStr)
        except Exception as e:
            print 'ERROR: Cannot parse [%s].\n[%s].' % (packageDetailsPath, e)
    return packageDetails


def getAvailableTargets():
    """Return set of targets which have been unzipped"""
    packageDetails = getPackageDetails()
    return set(d['TARGET'] for d in packageDetails.values())


def isPlatformAvailable(platform):
    """Return whether or not a given platform's packages have been unzipped"""
    return any(t.upper().startswith(platform.upper()) for t in getAvailableTargets())


def getAvailableProducts():
    """Return set of products which have been unzipped"""
    packageDetails = getPackageDetails()
    return set(d['PRODUCT'] for d in packageDetails.values())


def getAvailableSourceLevels():
    """
    Get available source levels from package details dictionaries.

    RETURNS:
        dict[str, list[str]]. Dictionary mapping product name to list of source levels present.
    """
    global __AVAILABLE_SOURCE_LEVELS
    if __AVAILABLE_SOURCE_LEVELS is None:
        __AVAILABLE_SOURCE_LEVELS = collections.defaultdict(list)
        packageDetails = getPackageDetails()
        for packageDict in packageDetails.values():
            if packageDict['TARGET'] == 'SOURCE':
                __AVAILABLE_SOURCE_LEVELS[packageDict['PRODUCT']].extend(packageDict['VISIBILITY'])
        sourceLevels = {1: 'PUBLIC', 2: 'CLIENT', 3: 'INTERNAL'}  # Map levels to strings used in project files.
        for product in __AVAILABLE_SOURCE_LEVELS:
            __AVAILABLE_SOURCE_LEVELS[product] = list(set(sourceLevels[x] for x in __AVAILABLE_SOURCE_LEVELS[product]))
    return __AVAILABLE_SOURCE_LEVELS


def loadConfigurationData():
    """
    Populate *MODULE_CONFIG_DICT*, *LIBRARY_CONFIG_DICT* and *SOURCE_CONFIG_DICT*
    Returns a list of error messages, an empty list indicates success.
    """
    errorMessages = []

    def getConfigDataFromFile(dataFile):
        """
        Read *dataFile* and return its contents as a dictionary
        """
        dataDict = {}
        absDataFile = os.path.join(getDataDir(), dataFile)
        try:
            if not os.path.isfile(absDataFile):
                errorMessages.append("Required data file does not exist: %s" % absDataFile)
            else:
                dataDict = dict(ast.literal_eval(open(absDataFile).read()))
        except Exception:
            errorMessages.append("Could not evaluate data file: %s" % absDataFile)
        return dataDict

    global MODULE_CONFIG_DICT
    global LIBRARY_CONFIG_DICT
    global SOURCE_CONFIG_DICT

    MODULE_CONFIG_DICT = getConfigDataFromFile('moduleConfig.py')
    LIBRARY_CONFIG_DICT = getConfigDataFromFile('libraryConfig.py')
    SOURCE_CONFIG_DICT = getConfigDataFromFile('sourceConfig.py')

    return errorMessages


def _idGenerator():
    id = 0L
    while 1:
        yield '%06d%08d%08d' % (long(random.random() * 1000000L), long(random.random()), id)
        id += 1
ID_GENERATOR = _idGenerator()


def notIn(lst):
    """
    Test if x is not in lst.
    """
    return lambda x: x not in lst


def writeXml(tree, filePath, utf8=False):
    """
    Write file to disk.
    """
    try:
        if os.path.isfile(filePath):
            os.remove(filePath)
        writer = open(filePath, 'w')
        if not writer:
            raise IOError
        if utf8:
            writer.write(tree.toxml("utf-8"))
        else:
            tree.writexml(writer)
        writer.close()
    except Exception:
        raise IOError


def processTag(tree, tagname, func):
    """
    Apply func to tree element tagname
    """
    map(func, tree.getElementsByTagName(tagname))


def deleteElem(elem):
    """
     Remove elem from xml tree
    """
    elem.parentNode.removeChild(elem)


def deleteTagsIf(tree, tagname, predicate):
    """
    Conditionally delete tags if predicate returns True
    """
    count = 0
    for elem in tree.getElementsByTagName(tagname):
        if predicate(elem):
            count = count + 1
            deleteElem(elem)
    return count


def replaceTagsIf(tree, tagname, predicate, alterNodeFunc):
    """
    Conditionally change tag value with alterNodeFunc if predictate returns True.
    """
    count = 0
    for elem in tree.getElementsByTagName(tagname):
        if predicate(elem):
            count = count + 1
            alterNodeFunc(elem)
    return count


def getNodeText(elem):
    """
    Retrieve contents of an element's Text-type child node.
    """
    child = elem.childNodes[0]
    return child.data


def setNodeText(elem, text):
    """
    Change data in Text element within given element. Assume this is the only element
    """
    assert (1 == len(elem.childNodes)), 'Should only be calling this on an element with a single child.'
    child = elem.childNodes[0]
    child.data = text


def isEmpty(elem):
    """
    Return true if all elem's children are text nodes.
    """
    return all(iter([n.nodeType == elem.TEXT_NODE for n in elem.childNodes]))


def doUntilDone(func):
    """
    Call func until it returns False
    """
    while func():
        pass


def getData(elem):
    """
    Return value of elem
    """
    offset = len(elem.localName) + 2
    return elem.toxml()[offset:-(offset + 1)]


def getModuleElemIncludePath(modulePath, elem):
    """
    Return a path combining modulePath and the value of elem.getAttribute('Include')
    """
    return '/'.join(modulePath.split('/') + elem.getAttribute('Include').replace('\\', '/').split('/'))


def csprojValidateLevel2SourceExists(tree):
    """
    For level2 VisualStudion 2010+ C# modules check whether all source files exist.
    """
    for elem in tree.getElementsByTagName("Compile"):
        filePath = elem.getAttribute("Include").replace("\\", os.sep)
        if filePath and not os.path.exists(filePath):
            raise InvalidLevel2ModuleError()


def configureCsproj(modulePath, filesList, libActionsDict, productList, configureDependenciesOnly, out):
    """
    No specific configuration supported for csproj, just handle stripping level2module suffix if
    appropriate
    """
    configureXmlProj(modulePath,
                     filesList,
                     libActionsDict,
                     '.csproj',
                     [csprojValidateLevel2SourceExists],
                     out,
                     sourceLevel=2)


def configureCsprojFilters(modulePath, filesList, libActionsDict, productList, configureDependenciesOnly, out):
    """
    No specific configuration supported for csproj.filters, just handle stripping level2module suffix if
    appropriate
    """
    configureXmlProj(modulePath,
                     filesList,
                     libActionsDict,
                     '.csproj.filters',
                     [csprojValidateLevel2SourceExists],
                     out,
                     sourceLevel=2)


def vcxprojValidateLevel2SourceExists(tree):
    """
    For level2 VisualStudion 2010+ modules check whether all source files exist.
    """
    for elem in tree.getElementsByTagName("ClCompile"):
        filePath = elem.getAttribute("Include").replace("\\", os.sep)
        if filePath and not os.path.exists(filePath):
            raise InvalidLevel2ModuleError()


def configureVcxproj(modulePath, filesList, libActionsDict, productList, configureDependenciesOnly, out):
    """
    Load a Visual Studio 2010 Project, and strip references to non-existing files and libs
    """
    absModulePath = os.path.join(HAVOK_ROOT, modulePath)
    if not os.path.exists(absModulePath):
        return

    isLevel2Module = False
    originalDir = os.getcwd()
    try:
        os.chdir(absModulePath)
        isLevel2Module = getTemplateFilesListInCurdir(".vcxproj", sourceLevel=2)
    finally:
        os.chdir(originalDir)

    if configureDependenciesOnly and not isLevel2Module:
        # For VS2010 the dependencies are handled by generating a solution, so do nothing
        return

    def deleteFileTag(elem):
        if elem.getAttribute('Include'):
            shouldRemoveFile = getModuleElemIncludePath(modulePath, elem) in filesList
            return shouldRemoveFile
        else:
            return False

    def fixLibs(elem):
        if not elem.parentNode.nodeName == "Link" or not elem.childNodes:
            return
        libs = elem.childNodes[0].data.split(';')
        # order of the libs needs to be preserved
        libs = collections.OrderedDict(((lib.replace('-l', ''), lib) for lib in libs))

        def __notInLibrariesList(lib):
            if lib in libActionsDict['remove']:
                return False
            if os.path.sep in lib:
                libBase = lib.split(os.path.sep)[-1]
                if libBase in libActionsDict['remove']:
                    return False
            return True
        if libs:
            libsToKeep = [libs[k] for k in filter(__notInLibrariesList, libs.keys())]
            requiredLibs = addExtraLibsIfNeeded(libsToKeep, libActionsDict)
            elem.childNodes[0].data = ';'.join(requiredLibs)

    def fixCompiledFiles(elem):
        if not elem.parentNode.nodeName == "Link":
            return
        libs = elem.childNodes[0].data.split(' ') if elem.childNodes else []
        if libs:
            libsToKeep = [l for l in libs if notIn(libActionsDict['remove'])(os.path.basename(l))]
            requiredLibs = addExtraLibsIfNeeded(libsToKeep, libActionsDict)
            elem.childNodes[0].data = ' '.join(requiredLibs)

    def fixEnabledProducts(elem):
        def inProductList(defStr):
            return defStr[len('HK_ENABLE_'):] in productList
        allDefs = [preDef for preDef in elem.childNodes[0].data.split(';')]
        enableDefs = [enableDef for enableDef in allDefs if fnmatch.fnmatch(enableDef, 'HK_ENABLE_*')]
        if enableDefs:
            otherDefs = [preDef for preDef in allDefs if preDef not in enableDefs]
            keep = filter(inProductList, enableDefs)
            otherDefs.extend(keep)
            elem.childNodes[0].data = ';'.join(otherDefs)

    jobs = [lambda vcxtree: deleteTagsIf(vcxtree, 'ClInclude', deleteFileTag),
            lambda vcxtree: deleteTagsIf(vcxtree, 'ClCompile', deleteFileTag),
            lambda vcxtree: deleteTagsIf(vcxtree, 'None', deleteFileTag),
            lambda vcxtree: processTag(vcxtree, 'AdditionalDependencies', fixLibs),
            lambda vcxtree: processTag(vcxtree, 'AdditionalOptions', fixCompiledFiles),
            lambda vcxtree: processTag(vcxtree, 'PreprocessorDefinitions', fixEnabledProducts),
            lambda vcxtree: processTag(vcxtree, 'UserPreprocessorDefinitions', fixEnabledProducts)]
    jobsLevel2 = [lambda vcxtree: vcxprojValidateLevel2SourceExists(vcxtree)]

    configureXmlProj(modulePath, filesList, libActionsDict, '.vcxproj', jobs, out)
    configureXmlProj(modulePath, filesList, libActionsDict, '.vcxproj', jobs + jobsLevel2, out, sourceLevel=2)


def configureVcxprojFilters(modulePath, filesList, libActionsDict, productList, configureDependenciesOnly, out):
    """
    Load a Visual Studio 2010 filters file, and strip references to non-existing files and folders
    """
    absModulePath = os.path.join(HAVOK_ROOT, modulePath)
    if not os.path.exists(absModulePath):
        return

    isLevel2Module = False
    originalDir = os.getcwd()
    try:
        os.chdir(absModulePath)
        isLevel2Module = getTemplateFilesListInCurdir(".vcxproj.filters", sourceLevel=2)
    finally:
        os.chdir(originalDir)

    if configureDependenciesOnly and not isLevel2Module:
        # For VS2010 the dependencies are handled by generating a solution, so do nothing
        return

    def deleteFileTag(elem):
        if elem.getAttribute('Include'):
            shouldRemoveFile = getModuleElemIncludePath(modulePath, elem) in filesList
            return shouldRemoveFile
        else:
            return False

    def deleteFilterTags(project):
        def getSuperFolders(folder):
            folders = []
            lhs = folder
            while lhs:
                folders.append(lhs)
                lhs, rhs = os.path.split(lhs)
            return folders

        tree = parse(project)
        fileTags = ["ClInclude", "ClCompile", "None", "Compile", "Image"]
        files = reduce(lambda a, b: a + b, (tree.getElementsByTagName(fileTag) for fileTag in fileTags))
        files = [e.getAttribute("Include") for e in files if e.getAttribute("Include")]
        foldersToKeep = set()
        for f in files:
            foldersToKeep.update(getSuperFolders("%s/%s" % (modulePath, os.path.dirname(f).replace('\\', '/'))))
        for elem in tree.getElementsByTagName('Filter'):
            if elem.getAttribute('Include'):
                if getModuleElemIncludePath(modulePath, elem) not in foldersToKeep:
                    deleteElem(elem)
        writeXml(tree, project)

    jobs = [lambda vcxfilterstree: deleteTagsIf(vcxfilterstree, 'ClInclude', deleteFileTag),
            lambda vcxfilterstree: deleteTagsIf(vcxfilterstree, 'ClCompile', deleteFileTag),
            lambda vcxfilterstree: deleteTagsIf(vcxfilterstree, 'None', deleteFileTag)]
    jobsLevel2 = [lambda vcxtree: vcxprojValidateLevel2SourceExists(vcxtree)]

    configureXmlProj(modulePath, filesList, libActionsDict, '.vcxproj.filters', jobs, out)
    configureXmlProj(modulePath, filesList, libActionsDict, '.vcxproj.filters', jobs + jobsLevel2, out, sourceLevel=2)

    postSteps = [(deleteFilterTags, "*.vcxproj.filters")]
    postConfigureSteps(modulePath, postSteps, out)


def getTemplateFilesListInCurdir(ext, sourceLevel=1):
    if sourceLevel == 1:
        projTemplates = glob.glob('*%s%s' % (ext, TEMPLATE_FILE_EXT))
    else:
        projTemplates = glob.glob('*%s%s*' % (ext, LEVEL2_MODULE_FILE_EXT))
        projTemplates += glob.glob('*%s%s%s*' % (ext, LEVEL2_MODULE_FILE_EXT, TEMPLATE_FILE_EXT))
    return projTemplates


class InvalidLevel2ModuleError(Exception):
    """
    Exception used to signal that level 2 module should not be processed.
    """


def configureXmlProj(modulePath, filesList, libActionsDict, ext, jobs, out, utf8=False, sourceLevel=1):
    """Loads an XML template file, parses it, performs jobs on the parse tree, then writes out the result"""
    if not os.path.exists(os.path.join(HAVOK_ROOT, modulePath)):
        return None
    availableSourceLevels = getAvailableSourceLevels()
    origdir = os.getcwd()
    os.chdir(os.path.join(HAVOK_ROOT, modulePath))
    for projTemplate in getTemplateFilesListInCurdir(ext, sourceLevel):
        outProj = projTemplate.replace(TEMPLATE_FILE_EXT, "").replace(LEVEL2_MODULE_FILE_EXT, "")
        tree = parse(projTemplate)
        customProperties = getCustomProperties(tree)
        if customProperties:
            # Determine if source levels present for any of require product sets
            # e.g. 'PHYSICS_2012+ANIMATION PHYSICS+ANIMATION'
            requiredProductSets = [set(p.split('+')) for p in customProperties['REQUIRED_HAVOK_PRODUCTS'].split(' ')]
            requiredSource = customProperties['SOURCE_LEVEL']
            requiredProductSourceLevelsPresent = any(all(p in availableSourceLevels and
                                                         requiredSource in availableSourceLevels[p]
                                                         for p in productSet)
                                                     for productSet in requiredProductSets)
            if not requiredProductSourceLevelsPresent:
                return None
        try:
            map(lambda job: job(tree), jobs)
        except InvalidLevel2ModuleError:
            # exception raised when there the module should not be processed
            break
        except Exception:
            out('\t' + outProj)
            raise
        out('\t' + outProj)
        writeXml(tree, outProj, utf8)
    os.chdir(origdir)


def getCustomProperties(xmlDom):
    """
    Get properties set in XML comments, e.g. solution variant is provided as <!-- VARIANT = "DX11" -->
    This is a previous implementation of getCustomProperties from project.py before it changed to use a different 
    xml format (see CL 366694)
    """
    def documentComments(element):
        if isinstance(element, Comment):
            yield element.data
        else:
            for subnode in element.childNodes:
                for comment in documentComments(subnode):
                    yield comment
    customProperties = {}
    for comment in documentComments(xmlDom):
        match = re.findall(r"(?P<variableName>[a-zA-Z0-9_]+)\s*=\s*"
                            r"(?P<variableValue>(\"[a-zA-Z0-9_/\-,\.\[\]\s\'+]*\")|(\'[a-zA-Z0-9_/\-,\.\[\]\s\"+]*\'))",
                            comment.strip())
        if match:
            for var in match:
                variableName, variableValue = var[0:2]
                customProperties[variableName] = variableValue[1:-1]
    return customProperties


def postConfigureSteps(modulePath, jobs, out):
    """Does jobs on files matching ext"""
    if not os.path.exists(os.path.join(HAVOK_ROOT, modulePath)):
        return None
    origdir = os.getcwd()
    os.chdir(os.path.join(HAVOK_ROOT, modulePath))
    for job in jobs:
        for proj in glob.glob(job[1]):
            job[0](proj)
    os.chdir(origdir)


def configureMakefile(modulePath, filesList, libActionsDict, productList, configureDependenciesOnly, out):
    """Load a Makefile, and strip references to non-existing files"""
    if not os.path.exists(os.path.join(HAVOK_ROOT, modulePath)):
        return None
    origdir = os.getcwd()
    os.chdir(os.path.join(HAVOK_ROOT, modulePath))

    def dotEmKayFileExistsAndIsUsable(depPath, makefile):
        """
        Returns False if either makefile or makefile template doesn't exist
        OR if not all of a module's products are present (e.g. CLOTH+ANIMATION required, only CLOTH source present).
        """
        depMakefiles = [makefile, makefile + TEMPLATE_FILE_EXT]
        if not any(os.path.exists(os.path.join(depPath, x)) for x in depMakefiles):
            return False
        depAbsPath = os.path.abspath(depPath)
        depRelPath = depAbsPath[len(HAVOK_ROOT + os.path.sep):].replace('\\', '/')
        if depRelPath not in MODULE_CONFIG_DICT:
            return False
        productMatchSatisfied = False
        for productCombo in MODULE_CONFIG_DICT[depRelPath].split(' '):
            allProductsInComboPresent = True
            for p in productCombo.split('+'):
                if p not in productList:
                    allProductsInComboPresent = False
                    break
            if allProductsInComboPresent:
                productMatchSatisfied = True
                break
        return productMatchSatisfied

    def fixMakefileDeps(line, makefile):
        start, deps = line.split(":=")
        depPaths = [d.strip() for d in deps.split(' ') if d.strip()]
        validDepPaths = [d for d in depPaths if dotEmKayFileExistsAndIsUsable(d, makefile)]
        platform = os.path.split(makefile)[1].split("_", 1)[0]
        return start + ' := ' + ' '.join(validDepPaths) + getLineEnding(platform)

    def fixMakefileLibs(line, makefile):
        start, libs = line.split(":=")
        libsToKeep = filter(notIn(libActionsDict['remove']), libs.split())
        requiredLibs = addExtraLibsIfNeeded(libsToKeep, libActionsDict)
        platform = os.path.split(makefile)[1].split("_", 1)[0]
        return start + ' := ' + ' '.join(requiredLibs) + getLineEnding(platform)

    def fixCompiledLibs(line, makefile):
        start, libs = line.split('=')
        libs = [lib for lib in libs.split() if lib != ' ']
        libs = dict(zip(map(os.path.basename, libs), libs))
        libsToKeep = [libs[k] for k in filter(notIn(libActionsDict['remove']), libs.keys())]
        requiredLibs = addExtraLibsIfNeeded(libsToKeep, libActionsDict)
        platform = os.path.split(makefile)[1].split("_", 1)[0]
        return start + ' = ' + ' '.join(requiredLibs) + getLineEnding(platform)

    lineProcessors = {
        'DEPENDENCIES': fixMakefileDeps
        }
    if not configureDependenciesOnly:
        lineProcessors.update({'LIBS': fixMakefileLibs,
                               'COMPILED_LIBS': fixCompiledLibs})

    # for level2 (Product Source) modules determine whether source files are actually available
    moduleLevel2Platforms = set()
    for filelistTemplate in getTemplateFilesListInCurdir('.filelist', sourceLevel=2):
        platform = os.path.split(filelistTemplate)[1].split("_", 1)[0]
        isLevel2Module = LEVEL2_MODULE_FILE_EXT in filelistTemplate
        infile = open(filelistTemplate, "r")
        moduleLevel2Platforms.add(platform)
        for line in infile.readlines():
            filePath = line.replace(' \\', '').strip()
            if (('.cpp' not in filePath)
            or (not isLevel2Module and filePath not in map(lambda x: x.replace('Demo/Demos/', ''), filesList))):
                continue
            elif not os.path.exists(filePath):
                # check if all non-excluded files exist
                moduleLevel2Platforms.remove(platform)
                break
        infile.close()

    for makefileTemplate in getTemplateFilesListInCurdir('.mk') + getTemplateFilesListInCurdir('.mk', sourceLevel=2):
        platform = os.path.split(makefileTemplate)[1].split("_", 1)[0]
        if LEVEL2_MODULE_FILE_EXT in makefileTemplate and platform not in moduleLevel2Platforms:
            continue
        makefile = makefileTemplate.replace(TEMPLATE_FILE_EXT, "").replace(LEVEL2_MODULE_FILE_EXT, "")
        out('\t' + posixpath.join(modulePath, makefile))
        makefileTemplateFile = open(makefileTemplate, "rb")
        outputFile = open(makefile, "wb")
        for line in makefileTemplateFile.readlines():
            for prefix in lineProcessors:
                if line.startswith(prefix):
                    outputFile.write(lineProcessors[prefix](line, makefile))
                    break
            else:
                outputFile.write(line)
        outputFile.close()
        makefileTemplateFile.close()
    for filelistTemplate in (getTemplateFilesListInCurdir('.filelist') +
                             getTemplateFilesListInCurdir('.filelist', sourceLevel=2)):
        platform = filelistTemplate.split("_", 1)[0]
        if LEVEL2_MODULE_FILE_EXT in filelistTemplate and platform not in moduleLevel2Platforms:
            continue
        filelist = filelistTemplate.replace(TEMPLATE_FILE_EXT, "").replace(LEVEL2_MODULE_FILE_EXT, "")
        out('\t' + posixpath.join(modulePath, filelist))
        outfile = open(filelist, "wb")
        infile = open(filelistTemplate, "r")
        pathsToRemove = set([p.replace('Demo/Demos/', '') for p in filesList])
        for line in infile.readlines():
            if '.cpp' not in line:
                outfile.write(line)
            else:
                strippedPath = line.replace(' \\', '').strip()
                if strippedPath not in pathsToRemove:
                    outfile.write(line)
        infile.close()
        outfile.close()
    for objlistTemplate in (getTemplateFilesListInCurdir('.obj_filelist') +
                            getTemplateFilesListInCurdir('.obj_filelist', sourceLevel=2)):
        platform = platform = os.path.split(objlistTemplate)[1].split("_", 1)[0]
        if LEVEL2_MODULE_FILE_EXT in objlistTemplate and platform not in moduleLevel2Platforms:
            continue
        objlist = objlistTemplate.replace(TEMPLATE_FILE_EXT, "").replace(LEVEL2_MODULE_FILE_EXT, "")
        filesToDelete = [os.path.basename(f).lower() for f in filesList]
        out('\t' + posixpath.join(modulePath, objlist))
        outfile = open(objlist, 'wb')
        for line in open(objlistTemplate, 'r').readlines():
            line = line.strip()
            if line.endswith('.o'):
                originalCpp = os.path.splitext(os.path.basename(line).lower())[0] + '.cpp'
                if originalCpp not in filesToDelete:
                    outfile.write(line + getLineEnding(platform))
            else:
                # assume it's a linker command and copy to output file
                outfile.write(line + getLineEnding(platform))
        outfile.close()
    for makefile in getTemplateFilesListInCurdir("Makefile", sourceLevel=2):
        makefileDest = makefile.replace(LEVEL2_MODULE_FILE_EXT, '')
        out('\t' + posixpath.join(modulePath, makefileDest))
        shutil.copy(makefile, makefileDest)
    os.chdir(origdir)


def __getXcodeProjIsas(templateLines):
    relevantLines = []
    matcher = re.compile(".*?isa\s+=\s+(\w+);")
    for line in templateLines:
        match = matcher.match(line)
        if match is not None:
            isaType = match.group(1)
            if isaType not in ('PBXProject', 'PBXShellScriptBuildPhase'):
                relevantLines.append(line)
    return relevantLines


def configureXcode(modulePath, filesList, libActionsDict, productList, configureDependenciesOnly, out):
    """Load an xcodeproj, and strip references to non-existing files and libraries"""
    if not os.path.exists(os.path.join(HAVOK_ROOT, modulePath)):
        return None

    origdir = os.getcwd()
    os.chdir(os.path.join(HAVOK_ROOT, modulePath))

    for xcodeProj in glob.glob('*.xcodeproj'):
        os.chdir(os.path.join(HAVOK_ROOT, modulePath, xcodeProj))

        # for level2 makefiles determine whether source files exist before processing
        templateFiles = getTemplateFilesListInCurdir('.pbxproj', sourceLevel=2)
        isLevel2Module = False
        isModuleTemplate = True
        if templateFiles:
            isLevel2Module = True
            # some level2 modules are not templates, they just need to be renamed
            if TEMPLATE_FILE_EXT not in templateFiles:
                isModuleTemplate = False
        else:
            templateFiles = getTemplateFilesListInCurdir('.pbxproj')

        if not templateFiles:
            continue

        try:
            templatePbxprojFilename = templateFiles[0]
            templatePbxproj = open(templatePbxprojFilename, 'r')
            if not templatePbxproj:
                raise IOError
        except Exception:
            raise IOError
        templateLines = (x.strip() for x in templatePbxproj.readlines())
        isaLines = __getXcodeProjIsas(templateLines)

        templatePbxproj.seek(0)
        templateText = templatePbxproj.read()

        # Find GUID of PBXProject.mainGroup in template. This group contains the GUID for each xcodeproj dependency.
        try:
            mainGroupGuid = re.search('mainGroup = ([0-9]+)', templateText).group(1)
        except AttributeError:
            print 'Error generating xcodeproj: \"mainGroup\" could not be found in the pbxproj template.'
            sys.exit(1)

        # Find lib directory relative to project. Any new PBXFileReferences to libs
        # will require the lib's relative path.
        try:
            libSearchPath = re.search('LIBRARY_SEARCH_PATHS =.+?\"([^\"]+)', templateText).group(1)
        except AttributeError:
            print 'Error generating xcodeproj: \"LIBRARY_SEARCH_PATHS\" could not be found in the pbxproj template.'
            sys.exit(1)
        templatePbxproj.close()

        #
        # Convert pbxproj isas to python tuples.
        #
        def __pythonize(s):
            """
            Function which takes an Xcode pbxproj isa line and turns it into eval-able python, this makes processing
            the data much simpler. We do a number of transformations:
            ';' -> ',' # this is to change the key-value seperator to match the comma python dict seperator
            '=' -> ':' # this changes the key-value relational operator to colon to match python
            re.sub('/\*(.|\r\n|\n)*?\*/', '', s) # find all block comments /* XXX */ and replace them with empty string
            This transformed data is then wrapped in () to make it a valid python tuple
            """
            def __paren(w):
                """
                Add quotes to w
                """
                return "'" + w.group(0) + "'"
            pythonStr = re.sub('[0-9A-Za-z<>./\-_\$]+',
                               __paren,
                               re.sub('/\*(.|\r\n|\n)*?\*/', '', s).replace(';', ',').replace('=', ': '))
            pythonStr = '(' + pythonStr.replace(':', ',', 1) + ')'
            pythonStr = pythonStr.replace('"\'', '"')
            pythonStr = pythonStr.replace('\'"', '"')
            return eval(pythonStr)
        isaTuples = (__pythonize(x) for x in isaLines)

        #
        # Build up dicts of source files and dependencies in pbxproj.
        #
        fileRefs = {}
        xcodeprojFileRefs = {}
        buildFiles = {}
        groups = {}
        parents = {}
        children = {}
        targetContainers = {}
        libContainers = {}
        targetDeps = {}
        refProxies = {}
        libsInTemplate = {}
        for guid, data in isaTuples:
            if data['isa'] == 'PBXFileReference':
                if '.xcodeproj' in data['path']:
                    # Keep references for xcodeprojs separate from those for source files,
                    # as 'path' has a different meaning, i.e. full path for xcodeprojs.
                    xcodeprojFileRefs[data['path']] = guid
                else:
                    fileRefs[data['path']] = guid  # We will want to search for files, so use these as key.
            if data['isa'] == 'PBXBuildFile':
                buildFiles[data['fileRef']] = guid
            if data['isa'] == 'PBXGroup':
                if 'path' in data:
                    groups[guid] = data['path']  # Store path so we can build up full path for files.
                # Store each child-parent relationship.
                for child in data['children']:
                    parents[child] = guid
                if guid not in children:
                    children[guid] = []
                    for child in data['children']:
                        children[guid].append(child)
            if data['isa'] == 'PBXContainerItemProxy':
                assert data['proxyType'] in ('1', '2')
                if data['proxyType'] == '1':
                    targetContainers[data['containerPortal']] = guid
                if data['proxyType'] == '2':
                    libContainers[data['containerPortal']] = guid
            if data['isa'] == 'PBXTargetDependency':
                targetDeps[data['targetProxy']] = guid
            if data['isa'] == 'PBXReferenceProxy':
                refProxies[data['remoteRef']] = guid
            if data['isa'] == 'XCBuildConfiguration':
                if 'OTHER_LDFLAGS' in data['buildSettings']:
                    if isinstance(data['buildSettings']['OTHER_LDFLAGS'], basestring):
                        libsInTemplate[guid] = [data['buildSettings']['OTHER_LDFLAGS']]
                    else:
                        libsInTemplate[guid] = data['buildSettings']['OTHER_LDFLAGS']

        #
        # Build full paths for references for source files using each file's parent groups.
        #
        def __getFullPath(filename):
            fileGuid = fileRefs[filename]
            path = filename
            guid = fileGuid
            while guid in parents:
                groupGuid = parents[guid]
                if groupGuid not in groups:  # Break if this group does not contain a 'path'
                    break
                path = posixpath.join(groups[groupGuid], path)
                guid = groupGuid
            return path
        fullFileRefs = {}
        for filename, guid in fileRefs.items():
            fullPath = posixpath.normpath(__getFullPath(filename))
            fullFileRefs[fullPath] = filename

        #
        # Delete projects that don't correspond to the Havok products being built.
        #
        isasToDelete = []
        xcodeprojFileRefsToRemove = []
        targetDepsToRemove = []
        projectRefsToRemove = []
        frameworkBuildFilesToRemove = []
        sourceBuildFilesToRemove = []

        def __getLibNameFromXcodeProj(xcodeprojPath):
            libName = os.path.basename(xcodeprojPath).split('.')[0]
            return libName.split('_')[0]

        def __removeProject(xcodeprojFileRefGuid):
            targetContainerGuid = targetContainers[xcodeprojFileRefGuid]
            libContainerGuid = libContainers[xcodeprojFileRefGuid]
            targetDepGuid = targetDeps[targetContainerGuid]
            refProxyGuid = refProxies[libContainerGuid]
            groupGuid = parents[refProxyGuid]
            buildFileGuid = buildFiles[refProxyGuid]

            # The xcodeproj file, and all containers, etc. should be deleted.
            isasToDelete.append(xcodeprojFileRefGuid)
            isasToDelete.append(targetContainerGuid)
            isasToDelete.append(libContainerGuid)
            isasToDelete.append(targetDepGuid)
            isasToDelete.append(refProxyGuid)
            isasToDelete.append(groupGuid)  # This group should have only 1 child, so can delete it.
            isasToDelete.append(buildFileGuid)

            # References to the xcodeproj in the project should be removed.
            # PBXProject.mainGroup contains PBXFileReference for the xcodeproj
            xcodeprojFileRefsToRemove.append(xcodeprojFileRefGuid)
            # PBXNativeTarget.dependencies contains the PBXTargetDependency
            targetDepsToRemove.append(targetDepGuid)
            # PBXProject.projectReferences contains tuple where ProjectRef is the PBXFileReference
            projectRefsToRemove.append(xcodeprojFileRefGuid)
            # PBXFrameworksBuildPhase contains the PBXBuildFile
            frameworkBuildFilesToRemove.append(buildFileGuid)
            return buildFileGuid

        deletedLibRefs = {}
        libsToRemove = {}
        for libToDelete in libActionsDict['remove']:
            for xcodeprojFileRef in xcodeprojFileRefs:
                # Unlike source files, the 'path' for xcodeprojs in the PBXFileReference is the full relative path,
                # so need to do an m*n search.
                xcodeprojModule = __getLibNameFromXcodeProj(xcodeprojFileRef)
                if libToDelete == xcodeprojModule and (xcodeprojFileRef not in deletedLibRefs):
                    xcodeprojFileRefGuid = xcodeprojFileRefs[xcodeprojFileRef]
                    __removeProject(xcodeprojFileRefGuid)
                    deletedLibRefs[xcodeprojFileRef] = 0
            for guid in libsInTemplate:
                if guid not in libsToRemove:
                    libsToRemove[guid] = set()
                for lib in libsInTemplate[guid]:
                    libName = lib.replace("-l", "")
                    if libName in libToDelete:
                        libsToRemove[guid].add(lib)
        #
        # Delete source files not for this product set.
        #
        moduleDir = os.path.join(origdir, HAVOK_ROOT, modulePath)
        # Source files ending with entries of generatedSources should not be removed
        # they are generated
        generatedSources = ['.app', '.framework', '__Resources']
        skipLevel2Module = False
        for sourceFile in fullFileRefs:
            if any([sourceFile.endswith(x) for x in generatedSources]):
                continue
            fullSourceFile = posixpath.normpath(posixpath.join(modulePath, sourceFile))
            deleteFile = fullSourceFile in filesList
            if (isLevel2Module and fullSourceFile.endswith(".cpp") and not deleteFile
            and not os.path.exists(os.path.join(moduleDir, sourceFile))):
                skipLevel2Module = True
                break
            if not deleteFile:
                deleteFile = not os.path.exists(os.path.join(moduleDir, sourceFile))
            if deleteFile and isModuleTemplate:
                filename = fullFileRefs[sourceFile]
                fileRefGuid = fileRefs[filename]
                buildFileGuid = buildFiles[fileRefGuid]
                isasToDelete.append(fileRefGuid)
                isasToDelete.append(buildFileGuid)
                sourceBuildFilesToRemove.append(buildFileGuid)  # PBXSourcesBuildPhase contains the PBXFileReference.

        if isLevel2Module and skipLevel2Module:
            continue
        if isLevel2Module and not isModuleTemplate:
            shutil.copy(templatePbxprojFilename, templatePbxprojFilename.replace(LEVEL2_MODULE_FILE_EXT, ""))
            out("\t%s" % posixpath.join(modulePath, xcodeProj, templatePbxprojFilename))
            return

        #
        # If we only have Base Source then we need to remove dependencies on any Product Source projects,
        # BUT we also need to introduce a file reference and build file for the corresponding libs for these projects.
        #
        def __getLibFilename(libName):
            return 'lib' + libName + '.a'

        def __createLibIsas(libName, fileRefGuid, buildFileGuid):
            libName = __getLibFilename(libName)
            fileRefIsa = (('%s = {isa = PBXFileReference; lastKnownFileType = archive.ar; '
                           'name = %s; path = "%s/%s"; sourceTree = SOURCE_ROOT; };')
                          % (fileRefGuid, libName, libSearchPath, libName))
            buildFileIsa = '%s = {isa = PBXBuildFile; fileRef = %s; };' % (buildFileGuid, fileRefGuid)
            return fileRefIsa, buildFileIsa

        isasToInsert = []
        xcodeprojFileRefsToAdd = []
        frameworkBuildFilesToAdd = []

        def __addFileRefDetails(lib_name, file_ref_guid, build_file_guid):
            file_ref_isa, build_file_isa = __createLibIsas(lib_name, file_ref_guid, build_file_guid)
            isasToInsert.append(file_ref_isa)
            isasToInsert.append(build_file_isa)
            # PBXProject.mainGroup should contain PBXFileReference for lib.
            xcodeprojFileRefsToAdd.append(file_ref_guid)
            # PBXFrameworksBuildPhase should contain PBXBuildFile for the lib.
            frameworkBuildFilesToAdd.append(build_file_guid)

        # Remove xcode project references and insert lib ISAs instead, wherever full source isn't present.
        for xcodeprojFileRef in xcodeprojFileRefs:
            xcodeprojGuid = xcodeprojFileRefs[xcodeprojFileRef]
            if xcodeprojGuid in xcodeprojFileRefsToRemove:
                # If this xcodeproj is for a different PRODUCT, then it won't exist,
                # but we don't want to link to a lib for it
                continue
            # We're currently in the xcodeproj dir, need to hop up one
            pathToPbxproj = os.path.join(os.pardir, xcodeprojFileRef, 'project.pbxproj')
            if os.path.exists(pathToPbxproj) or os.path.exists(pathToPbxproj + TEMPLATE_FILE_EXT):
                continue
            xcodeprojFileRefGuid = xcodeprojFileRefs[xcodeprojFileRef]
            buildFileGuid = __removeProject(xcodeprojFileRefGuid)
            libName = __getLibNameFromXcodeProj(xcodeprojFileRef)

            # We need a unique GUID for the new file and buildFile, so use the ones for the xcodeproj that
            # we're going to delete anyway.
            __addFileRefDetails(libName, xcodeprojFileRefGuid, buildFileGuid)

        # Add any lib ISAs that need to be inserted.
        for insertAction in libActionsDict['insert']:
            # e.g. {'ifPresent': 'Foo', 'addThis': 'Bar'}
            requiredLibFilename = __getLibFilename(insertAction['ifPresent'])
            if any([requiredLibFilename in l for l in isaLines]):
                libToAdd = insertAction['addThis']
                # Generate a unique GUID for the new file and buildFile references.
                __addFileRefDetails(libToAdd, file_ref_guid=ID_GENERATOR.next(), build_file_guid=ID_GENERATOR.next())

        #
        # Clean up empty groups
        #
        wasAnyGroupDeleted = True
        while wasAnyGroupDeleted:
            wasAnyGroupDeleted = False
            for group, groupsChildren in children.items():
                shouldRemoveGroup = not any(x not in isasToDelete for x in groupsChildren)
                if shouldRemoveGroup and group not in isasToDelete:
                    isasToDelete.append(group)
                    wasAnyGroupDeleted = True

        #
        # Create project file from template and write processed content.
        #
        try:
            newPbxproj = open(templateFiles[0].replace(TEMPLATE_FILE_EXT, '').replace(LEVEL2_MODULE_FILE_EXT, ''), 'w')
            templatePbxproj = open(templateFiles[0], 'r')
            if not templatePbxproj or not newPbxproj:
                raise IOError
        except Exception:
            raise IOError

        needToInsertLibs = bool(isasToInsert)
        for line in templatePbxproj.readlines():
            if HAVOK_XCODE_DEV_TEAM in line:
                line = line.replace(HAVOK_XCODE_DEV_TEAM, XCODE_DEV_TEAM)
            if needToInsertLibs and 'isa = ' in line:
                needToInsertLibs = False
                for isaText in isasToInsert:
                    newPbxproj.write(isaText + '\n')  # Insert file references and build files for libraries.
            if 'isa = ' in line:
                isaGuid = re.search('^\s*([0-9]+) = ', line).group(1)
                if isaGuid in isasToDelete:  # This ISA has been marked for delete, so don't write it.
                    continue
                if isaGuid == mainGroupGuid:
                    # Find 'children = (...)' in PBXGroup for PBXProject.mainGroup.
                    origChildrenText = re.search(r'children = \([^\)]+', line).group(0)
                    childrenText = origChildrenText
                    for xcodeprojFileRefGuid in xcodeprojFileRefsToRemove:
                        # Remove file references to xcodeprojs that aren't present.
                        childrenText = childrenText.replace(xcodeprojFileRefGuid + ',', '')
                    for xcodeprojFileRefGuid in xcodeprojFileRefsToAdd:
                        # Add file references for libs that we must use in place of xcodeprojs.
                        childrenText = childrenText + '%s, ' % xcodeprojFileRefGuid
                    line = line.replace(origChildrenText, childrenText)
                if 'isa = PBXNativeTarget' in line:
                    # Find PBXTargetDependency for xcodeproj in PBXNativeTarget.dependencies.
                    origDepsText = re.search(r'dependencies = \([^\)]+', line).group(0)
                    depsText = origDepsText
                    for targetContainerGuid in targetDepsToRemove:
                        depsText = depsText.replace(targetContainerGuid + ',', '')
                if 'isa = PBXFrameworksBuildPhase' in line:
                    origFilesText = re.search(r'files = \([^\)]+', line).group(0)
                    filesText = origFilesText
                    for buildFileGuid in frameworkBuildFilesToRemove:
                        # Remove PBXBuildFile for PBXReferenceProxy to xcodeproj from PBXFrameworksBuildPhase.
                        filesText = filesText.replace(buildFileGuid + ',', '')
                    for buildFileGuid in frameworkBuildFilesToAdd:
                        # Add PBXBuildFile for lib that we want to link against.
                        filesText = filesText + '%s, ' % buildFileGuid
                    line = line.replace(origFilesText, filesText)
                if 'isa = PBXSourcesBuildPhase' in line:
                    origFilesText = re.search(r'files = \([^\)]+', line).group(0)
                    filesText = origFilesText
                    for buildFileGuid in sourceBuildFilesToRemove:
                        # Remove PBXBuildFile for source files that are not present.
                        filesText = filesText.replace(buildFileGuid + ',', '')
                    line = line.replace(origFilesText, filesText)
                if 'isa = XCBuildConfiguration' in line:
                    for guid in libsToRemove:
                        if guid in line:
                            for lib in libsToRemove[guid]:
                                line = line.replace("{},".format(lib), '')

            else:  # PBXProject is the only isa that is written over multiple lines.
                if 'ProductGroup = ' in line and 'ProjectRef = ' in line:
                    dontWrite = False
                    for xcodeprojFileRefGuid in projectRefsToRemove:
                        if xcodeprojFileRefGuid in line:
                            dontWrite = True
                            break
                    if dontWrite:
                        # Remove (PBXGroup, PBXFileReference) tuple for xcodeproj in PBXProject.projectReferences
                        continue
            newPbxproj.write(line)
        newPbxproj.close()
    os.chdir(origdir)


def getModuleHkPreBuildArgsPath(modulePath):
    return os.path.join(HAVOK_ROOT, modulePath, 'hkPreBuild_args.txt')


def getModuleHkPreBuildArgsTemplatePath(modulePath):
    return getModuleHkPreBuildArgsPath(modulePath) + '.template'


def hasTargetProject(modulePath):
    absModulePath = os.path.join(HAVOK_ROOT, modulePath)
    moduleRootContents = os.listdir(absModulePath)
    if sys.platform == 'darwin':
        return any(f.endswith('.xcodeproj') for f in moduleRootContents)
    elif os.name == 'posix':
        return 'Makefile' in moduleRootContents
    else:
        return True


def moduleNeedsPreBuild(modulePath):
    hkPreBuildArgsTemplatePath = getModuleHkPreBuildArgsTemplatePath(modulePath)
    return os.path.isfile(hkPreBuildArgsTemplatePath) and hasTargetProject(modulePath)


def configurePreBuildArgs(modulePath, productList, out):
    """
    The list of products present must be passed to hkPreBuild so that it can exclude source files for
    products that aren't present. Internal builds on the other hand do not specify a set of products,
    and as such process all source files that it finds.
    """
    preBuildsArgsPath = getModuleHkPreBuildArgsPath(modulePath)
    preBuildsArgsTemplatePath = getModuleHkPreBuildArgsTemplatePath(modulePath)

    if moduleNeedsPreBuild(modulePath):
        with open(preBuildsArgsTemplatePath, 'r') as preBuildsArgsTemplateFile:
            contents = preBuildsArgsTemplateFile.read()
        productsStr = '\n--tkbms-products "{}"\n'.format(' '.join(productList))
        contents += productsStr
        with open(preBuildsArgsPath, 'w') as preBuildsArgsFile:
            preBuildsArgsFile.write(contents)


def runPreBuild(modulePath, filesList, libActionsDict, productList, configureDependenciesOnly, out):
    preBuildExeName = 'hkPreBuild'
    if isPlatformAvailable('linux32') and os.name == 'posix':
        preBuildBin = 'linux32'
    elif isPlatformAvailable('linux64') and os.name == 'posix':
        preBuildBin = 'linux64'
    elif sys.platform == "darwin":
        preBuildBin = 'mac'
    else:
        preBuildBin = 'windows'
        preBuildExeName = 'hkPreBuild.exe'

    preBuildExePath = os.path.join(HAVOK_ROOT, 'Tools', 'PreBuild', 'bin', preBuildBin, preBuildExeName)
    configurePreBuildArgs(modulePath, productList, out)
    absModulePath = os.path.join(HAVOK_ROOT, modulePath)
    try:
        # If we can't find a hkPreBuild_args.txt don't try to run hkPreBuild.
        if not moduleNeedsPreBuild(modulePath):
            return
        try:
            if os.name != 'nt':
                os.chmod(preBuildExePath, stat.S_IEXEC)
        except:
            pass
        command = [preBuildExePath,
                   '@hkPreBuild_args.txt',
                   '--output-path={}'.format(os.path.join(absModulePath, '_Auto')),
                   '--log-path={}'.format(os.path.join(HAVOK_ROOT, 'Build', '_Log', modulePath)),
                   '--tkbms-scan=.',
                   '-verbose=3']
        output = subprocess.check_output(args=command,
                                         cwd=absModulePath,
                                         stderr=subprocess.STDOUT)
    except subprocess.CalledProcessError as e:
        global HK_PREBUILD_FAILURES
        HK_PREBUILD_FAILURES.append(HkPreBuildFail(absModulePath,
                                                   e.cmd,
                                                   e.returncode,
                                                   e.output))
    finally:
        try:
            # Clean up some temp files that hkPreBuild creates.
            shutil.rmtree(os.path.join(absModulePath, '_Auto', 'Log'))
            os.remove(os.path.join(absModulePath, '_Auto', 'hkReflectionParser.d'))
        except Exception:
            pass


def configureModule(modulePath, filesList, libActionsDict, productList, configureDependenciesOnly, out):
    processors = [
        runPreBuild,
        configureVcxproj,
        configureVcxprojFilters,
        configureCsproj,
        configureCsprojFilters,
        configureMakefile,
        configureXcode,
        ]
    for processor in processors:
        processor(modulePath, filesList, libActionsDict, productList, configureDependenciesOnly, out)


class Configuration(Tkinter.Frame):
    def __init__(self, showGui=True, initialUserText='', master=None):
        self.gui = showGui
        self.initialUserText = initialUserText if initialUserText != '' else INITIAL_TEXT

        if self.gui:
            Tkinter.Frame.__init__(self, master)
            iconFile = os.path.join(getDataDir(), 'havok.ico')
            if os.path.isfile(iconFile) and os.name == 'nt':
                self.master.iconbitmap(default=iconFile)
            self.master.protocol("WM_DELETE_WINDOW", lambda: sys.exit(0))
            self.grid()
            self.createWidgets()
            self.master.title("Configure Havok")
            self.master.resizable(0, 0)

        self.configured = False

    def createWidgets(self):
        # Canvas
        canvas = Tkinter.Canvas(self, width=600, height=50)
        canvas.create_text(300, 25, text="Havok SDK Configuration",
                           font=tkFont.Font(family="Helvetica", size=15, weight="bold"))
        canvas.grid(row=0, column=0)

        # Frame
        keycodeTextFrame = Tkinter.Frame(self)
        keycodeTextFrame.grid(row=1, column=0)

        # Scrollbar
        keycodeScrollbar = Tkinter.Scrollbar(keycodeTextFrame, orient='vertical')
        keycodeScrollbar.grid(row=0, column=1, sticky=Tkinter.N + Tkinter.S)

        # Text
        self.keycodeText = Tkinter.Text(keycodeTextFrame, width=97, height=7, yscrollcommand=keycodeScrollbar.set,
                                        bg="white", wrap=Tkinter.WORD)
        self.keycodeText.insert(Tkinter.END, self.initialUserText)
        self.keycodeText.grid(row=0, column=0)
        keycodeScrollbar.config(command=self.keycodeText.yview)

        # Frame
        outputTextFrame = Tkinter.Frame(self)
        outputTextFrame.grid(row=3, column=0)

        # Scrollbar
        yscrollbar = Tkinter.Scrollbar(outputTextFrame)
        yscrollbar.grid(row=0, column=1, sticky=Tkinter.N + Tkinter.S)

        xscrollbar = Tkinter.Scrollbar(outputTextFrame, orient='horizontal')
        xscrollbar.grid(row=1, column=0, sticky=Tkinter.E + Tkinter.W)

        # Text
        self.outputText = Tkinter.Text(outputTextFrame, width=97,
                                       wrap=Tkinter.NONE, yscrollcommand=yscrollbar.set, xscrollcommand=xscrollbar.set,
                                       bg="#eee", takefocus=False, state=Tkinter.DISABLED)
        self.outputText.grid(row=0, column=0)
        xscrollbar.config(command=self.outputText.xview)
        yscrollbar.config(command=self.outputText.yview)

        # Button
        self.output('*******************\nHavok Configuration\n*******************')
        self.goButton = Tkinter.Button(self, text="Configure", command=self.go)
        self.goButton.grid(row=4, column=0)

    @property
    def keycodes(self):
        if self.gui:
            return self.keycodeText.get(1.0, "end")
        else:
            return self.initialUserText

    def insert_keycodes(self, keycodes, emptyKeycodes=[], keycodeFile=''):
        if not keycodeFile:
            keycodeFile = os.path.join(HAVOK_ROOT, KEYCODE_FILE)

        keycodeRegexp = re.compile(KEYCODE_REGEX)

        def strip_keycodes(keycodeFile):
            for line in fileinput.FileInput(keycodeFile, inplace=True):
                if not re.findall(keycodeRegexp, line):
                    print line,

        strip_keycodes(keycodeFile)
        for line in fileinput.FileInput(keycodeFile, inplace=True):
            if "Please enter your Havok keycode(s) here" in line:
                print line,
                for keycode in sorted(chain.from_iterable([keycodes, emptyKeycodes])):
                    if not re.findall(keycodeRegexp, keycode):
                        continue
                    if self.gui:
                        self.output('\t' + keycode)  # don't print things in a fileinput block
                    print keycode
            else:
                print line,

    def getSimdFileContents(self):
        """
        """
        return open(os.path.join(HAVOK_ROOT, SIMD_CONFIG_FILE)).read()

    def getWin32SimdRegExMatches(self):
        """
        """
        return re.findall('//\s+x86\s+#\s*\S+.*', self.getSimdFileContents())

    def getWin32LibTypes(self):
        """
        """
        libsDir = os.path.join(HAVOK_ROOT, 'Lib')
        win32Libs = glob.glob1(libsDir, 'win32*')
        win32NoSimdLibs = [x for x in win32Libs if 'noSimd' in x]
        win32SimdLibs = [x for x in win32Libs if x not in win32NoSimdLibs]
        return win32NoSimdLibs, win32SimdLibs

    def getUpdatedWin32SimdCode(self):
        """
        """
        win32SimdCode = self.getWin32SimdRegExMatches()[0]
        currentWin32SimdSetting = re.findall('#\s*(\S+.*)', win32SimdCode)[0]

        win32NoSimdLibs, win32SimdLibs = self.getWin32LibTypes()
        if win32NoSimdLibs and win32SimdLibs:
            newWin32SimdSetting = ("error HK_CONFIG_SIMD could not be set automatically for win32 as libraries "
                                   "both with and without simd optimizations appear to be present in your Havok "
                                   "distribution. Please set HK_CONFIG_SIMD manually.")
        elif win32NoSimdLibs:
            newWin32SimdSetting = "define HK_CONFIG_SIMD HK_CONFIG_SIMD_DISABLED"
        else:
            newWin32SimdSetting = "define HK_CONFIG_SIMD HK_CONFIG_SIMD_ENABLED"

        return win32SimdCode.replace(currentWin32SimdSetting, newWin32SimdSetting)

    def checkSimdFile(self):
        """
        Check if the path to the simd file exits.
        """
        simdFilePresent = os.path.isfile(os.path.join(HAVOK_ROOT, SIMD_CONFIG_FILE))
        if not simdFilePresent:
            self.output("ERROR: Couldn't find %s, which should be present." % SIMD_CONFIG_FILE)
            return False

        win32SimdRegExMatches = self.getWin32SimdRegExMatches()
        win32SimdCodeSeemsOkay = (len(win32SimdRegExMatches) == 1)
        if not win32SimdCodeSeemsOkay:
            self.output("ERROR: The contents of %s are unexpected.\n       Has it been modified?" % SIMD_CONFIG_FILE)
            return False

        simdFileNeedsModification = (self.getUpdatedWin32SimdCode() != win32SimdRegExMatches[0])
        if simdFileNeedsModification:
            simdFileIsWritable = os.stat(os.path.join(HAVOK_ROOT, SIMD_CONFIG_FILE))[0] & stat.S_IWRITE
            if not simdFileIsWritable:
                self.output(("ERROR: %s needs to be modified but is not writable.\n"
                             "       Has it been checked into source control?\n"
                             "       Please ensure that this file can be written to and try again.") %
                            SIMD_CONFIG_FILE)
                return False

        return True

    def reviewWin32SimdSetting(self):
        """
        """
        self.output('Reviewing Win32 SIMD status...')
        currentWin32SimdCode = self.getWin32SimdRegExMatches()[0]
        newWin32SimdCode = self.getUpdatedWin32SimdCode()
        simdFileNeedsModification = (newWin32SimdCode != currentWin32SimdCode)
        if simdFileNeedsModification:
            win32NoSimdLibs, win32SimdLibs = self.getWin32LibTypes()
            if win32NoSimdLibs and win32SimdLibs:
                self.output('\tFound win32 libraries both with and without simd optimizations.\n\tUpdating %s...'
                            % SIMD_CONFIG_FILE)
            elif win32NoSimdLibs:
                self.output('\tFound win32 libraries without simd optimizations.\n\tUpdating %s...' % SIMD_CONFIG_FILE)
            else:
                self.output('\tFound win32 libraries with simd optimizations.\n\tUpdating %s...' % SIMD_CONFIG_FILE)
            updatedSimdFileContents = self.getSimdFileContents().replace(currentWin32SimdCode, newWin32SimdCode)
            open(os.path.join(HAVOK_ROOT, SIMD_CONFIG_FILE), 'w').write(updatedSimdFileContents)
        self.output('\tDone')

    def getInstalledLibProducts(self):
        """
        Return a list of products with libs available in extracted packages
        This is determined by checking if any lib with an expected name exists in the Lib folder.
        """
        prefixSuffixList = [
            ['', '.lib'],
            ['', '.a'],
            ['lib', '.a'],
            ]
        libProductDict = {
            'AI': 'hkaiPathfinding',
            'ANIMATION': 'hkaAnimation',
            'BEHAVIOR': 'hkbBehavior',
            'CLOTH': 'hclCloth',
            'COMMON': 'hkBase',
            'CONVEX_DECOMPOSITION': 'hkgpConvexDecomposition',
            'DESTRUCTION': 'hkndDestruction',
            'DESTRUCTION_2012': 'hkdDestruction',
            'FX': 'hkfxFx',
            'PHYSICS': 'hknpPhysics',
            'PHYSICS_2012': 'hkpCollide',
            }

        def getLibFiles(libName):
            """
            Get list of all possible lib file names for the root libName
            """
            return map(lambda x: x[0] + libName + x[1], prefixSuffixList)
        libsRoot = os.path.join(HAVOK_ROOT, 'Lib')
        if not os.path.isdir(libsRoot):
            return []
        productsList = []
        for product in libProductDict:
            libFiles = getLibFiles(libProductDict[product])
            for root, dirs, files in os.walk(libsRoot):
                if [f for f in libFiles if f in files]:
                    productsList.append(product)
                    break
        return productsList

    def getInstalledSourceProducts(self):
        """
        Return a list of products with source available in extracted packages.
        This is determined by checking if the main header for each product is present in the expected location.
        """
        productFileDict = {
            'AI': os.path.join(HAVOK_ROOT, 'Source', 'Ai', 'Pathfinding', 'hkaiPathfinding.h'),
            'ANIMATION': os.path.join(HAVOK_ROOT, 'Source', 'Animation', 'Animation', 'hkaAnimation.h'),
            'BEHAVIOR': os.path.join(HAVOK_ROOT, 'Source', 'Behavior', 'Behavior', 'hkbBehavior.h'),
            'CLOTH': os.path.join(HAVOK_ROOT, 'Source', 'Cloth', 'Cloth', 'hclCloth.h'),
            'COMMON': os.path.join(HAVOK_ROOT, 'Source', 'Common', 'Base', 'hkBase.h'),
            'CONVEX_DECOMPOSITION': os.path.join(HAVOK_ROOT,
                                                 'Source',
                                                 'Geometry',
                                                 'ConvexDecomposition',
                                                 'hkgpConvexDecomposition.h'),
            'DESTRUCTION': os.path.join(HAVOK_ROOT, 'Source', 'Destruction', 'Destruction', 'hkndDestruction.h'),
            'DESTRUCTION_2012': os.path.join(HAVOK_ROOT, 'Source',  'Destruction2012', 'Destruction',
                                             'hkdDestruction.h'),
            'FX': os.path.join(HAVOK_ROOT, 'Source', 'Physics', 'Fx', 'hkfxBase.h'),
            'PHYSICS': os.path.join(HAVOK_ROOT, 'Source', 'Physics', 'Physics', 'hknpPhysics.h'),
            'PHYSICS_2012': os.path.join(HAVOK_ROOT, 'Source', 'Physics2012', 'Collide', 'hkpCollide.h'),

            'SCRIPT': os.path.join(HAVOK_ROOT, 'Source', 'Script', 'VM', 'lua.h'),
            }
        productsList = []
        for product in productFileDict:
            if os.path.isfile(productFileDict[product]):
                productsList.append(product)
        return productsList

    def getInstalledEmptyKeycodeProducts(self):
        """Return list non-protected products that have libs or source present."""
        return [x for x in EMPTY_KEYCODE_PRODUCTS if (x in self.getInstalledLibProducts() or x in self.getInstalledSourceProducts())]

    def getEmptyKeycodesToInsert(self):
        """Get keycodes to insert for non-protected products for which source and libs are present."""
        return ['#define HAVOK_{}_KEYCODE ""'.format(x) for x in self.getInstalledEmptyKeycodeProducts()]

    def keycodesMatchInstalledPackages(self, productList):
        """Compare the list of products for which keycodes were entered against the source and libs present."""
        self.output('Comparing keycodes to unzipped Havok packages...')
        keycodesMatchPackages = True

        libProducts = self.getInstalledLibProducts()
        libProducts.sort()

        sourceProducts = self.getInstalledSourceProducts()
        sourceProducts.sort()

        emptyKeycodeProductsPresent = self.getInstalledEmptyKeycodeProducts()
        emptyKeycodeProductsPresent.sort()

        keycodeProducts = list(productList)
        keycodeProducts.sort()

        try:
            sourceProducts.remove('COMMON')
        except Exception:
            self.output("\tERROR: Couldn't find the Havok Common source.")
            keycodesMatchPackages = False
        try:
            libProducts.remove('COMMON')
        except Exception:
            self.output("\tERROR: Couldn't find the Havok Common libraries.")
            keycodesMatchPackages = False

        keycodeProducts = map(lambda x: x.upper(), keycodeProducts)
        sourceProducts = map(lambda x: x.upper(), sourceProducts)
        libProducts = map(lambda x: x.upper(), libProducts)

        # For no-keycode products verify that either source and libs are both present, or neither are.
        # Then remove them from the list of source and lib products to match against the entered keycodes.
        for emptyKeycodeProduct in emptyKeycodeProductsPresent:
            sourcePresent = emptyKeycodeProduct in sourceProducts
            libsPresent = emptyKeycodeProduct in libProducts
            if sourcePresent or libsPresent:
                # Set error flag if no libs present and there should be libs for this product
                if not libsPresent and emptyKeycodeProduct not in NO_LIB_PRODUCTS:
                    self.output("\tERROR: Found source for %s but no corresponding libraries." % emptyKeycodeProduct)
                    keycodesMatchPackages = False
                # If no problem with libs, set error flag if no source present
                elif not sourcePresent:
                    self.output("\tERROR: Found libraries for %s but no corresponding source." % emptyKeycodeProduct)
                    keycodesMatchPackages = False
                # If no problem with either libs or source, then remove this product from lists before checking against keycodeProducts
                else:
                    sourceProducts.remove(emptyKeycodeProduct)
                    if emptyKeycodeProduct not in NO_LIB_PRODUCTS:  # Don't try to remove products that don't ship with libs, e.g. SCRIPT
                        libProducts.remove(emptyKeycodeProduct)  

        if (keycodeProducts != libProducts) or (keycodeProducts != sourceProducts):
            def formatProductList(type, products, addedProducts=None):
                if addedProducts is None:
                    addedProducts = []
                allProducts = products + ['[{}]'.format(x) for x in addedProducts]
                if not allProducts:
                    return "Didn't find {} for any products".format(type)
                return "Found {} for products: {}".format(type, ', '.join(allProducts))

            missingLibs = [x for x in keycodeProducts if x not in libProducts]
            extraLibs = [x for x in libProducts if (x not in keycodeProducts and x not in emptyKeycodeProductsPresent)]
            missingSource = [x for x in keycodeProducts if x not in sourceProducts]
            extraSource = [x for x in sourceProducts if (x not in keycodeProducts and x not in emptyKeycodeProductsPresent)]

            self.output("\tERROR: Keycodes entered don't match the unzipped source/libraries.")
            self.output("\t - " + formatProductList('keycodes', keycodeProducts))
            self.output("\t - " + formatProductList('source code', sourceProducts, emptyKeycodeProductsPresent))
            self.output("\t - " + formatProductList('libraries', libProducts, [x for x in emptyKeycodeProductsPresent if x not in NO_LIB_PRODUCTS]))
            if missingLibs:
                self.output("\t - Have keycodes but no libraries for: {}".format(', '.join(missingLibs)))
            if extraLibs:
                self.output("\t - Have libraries but no keycodes for: {}".format(', '.join(extraLibs)))
            if missingSource:
                self.output("\t - Have keycodes but no source code for: {}".format(', '.join(missingSource)))
            if extraSource:
                self.output("\t - Have source code but no keycodes for: {}".format(', '.join(extraSource)))

            keycodesMatchPackages = False

        # Once it has been verified that keycodes match source and products, verify that product dependencies are satisfied.
        productsPresent = keycodeProducts + emptyKeycodeProductsPresent
        for dependent in PRODUCT_DEPENDENCIES:
            if dependent not in keycodeProducts:
                continue
            missingDependencies = [x for x in PRODUCT_DEPENDENCIES[dependent] if x not in productsPresent]
            if missingDependencies:
                missingDepsStr = ", ".join(missingDependencies)
                self.output("\tERROR: Could not find any {} libraries or source code.\n\t - {} is required by {}.".format(
                    missingDepsStr, missingDepsStr, dependent))
                keycodesMatchPackages = False

        if keycodesMatchPackages:
            self.output("\tKeycodes match Havok distribution.")
        return keycodesMatchPackages

    def sanityCheckKeycodes(self, enteredKeycodes, productList):
        """
        Check the entered keycodes and products.
        Args:
            enteredKeycodes (List[str]): The list of keycodes the user entered.
            productsList (List[str]): The list of products extracted from the user's keycodes.
        Returns:
            bool. Are the keycodes and products consistent and correct?
        """
        badKeycodes = []
        validProductDefines = tuple(["#define HAVOK_{}_KEYCODE".format(p) for p in productList])
        for keycode in enteredKeycodes:
            if not keycode.startswith(validProductDefines):
                badKeycodes.append(keycode)
        if badKeycodes:
            badKeycodeErr = """\tERROR: Keycodes entered for unknown products.
Please remove the keycode[s] listed below and try again."""
            self.output(badKeycodeErr)
            for keycode in badKeycodes:
                self.output("\t" + keycode)
            return False

        if any(':CLIENT' in keycode.split('.')[0] for keycode in enteredKeycodes):
            self.output('ERROR:')
            self.output('\tAt least one of the keycodes entered appears to be for version 6.5 of Havok or older.')
            self.output('\tPlease check that the keycodes you have entered are correct and try again.')
            self.output('\tIf the problem persists please contact Havok support.')
            if self.gui:
                self.goButton.config(text="Done", command=self.quit)
                self.update()
            return False

        return True

    def go(self):

        errorMessages = loadConfigurationData()
        if errorMessages:
            self.output('ERROR: Could not load configuration data:')
            for errorMessage in errorMessages:
                self.output('\t%s' % errorMessage)
            self.output('\tPlease review the reported issues and try again.')
            self.output('\tIf the problem persists please contact Havok support.')
            if self.gui:
                self.goButton.config(text="Done", command=self.quit)
                self.update()
            return 1

        self.output('Checking keycodes...')
        enteredKeycodes, productList = self.extractKeycodesAndProducts(self.keycodes, self.output)

        if not self.sanityCheckKeycodes(enteredKeycodes, productList):
            return 1

        self.output('Keycodes entered for:')
        for product in productList:
            self.output('\t' + product.capitalize())

        everythingLooksOk = self.keycodesMatchInstalledPackages(productList)

        productList.extend(self.getInstalledEmptyKeycodeProducts())

        if everythingLooksOk:
            self.output('Products present:')
            for product in productList:
                self.output('\t' + product.capitalize())

        if everythingLooksOk and (not os.path.isfile(os.path.join(HAVOK_ROOT, KEYCODE_FILE))):
            self.output(r"ERROR: Couldn't find %s, so cannot add the keycodes." % KEYCODE_FILE)
            everythingLooksOk = False

        if everythingLooksOk:
            everythingLooksOk = self.checkSimdFile()

        if everythingLooksOk:
            self.output(r'Adding Keycodes to %s...' % KEYCODE_FILE)
            # Make sure that keycode.h is writable before continuing.
            keyCodeFileAtt = os.stat(os.path.join(HAVOK_ROOT, KEYCODE_FILE))[0]
            if (not keyCodeFileAtt & stat.S_IWRITE):
                self.output(("\tERROR: %s exists but is not writable. Has it been checked into source control?\n"
                             "\tPlease ensure that this file can be written to and try again.") % KEYCODE_FILE)
            else:
                # If a SCRIPT keycode is entered by the user, only insert this if SCRIPT source is present
                includeScript = 'SCRIPT' in self.getInstalledSourceProducts()
                keycodesToInsert = [k for k in enteredKeycodes if (includeScript or ('HAVOK_SCRIPT_KEYCODE' not in k))]
                self.insert_keycodes(keycodesToInsert, self.getEmptyKeycodesToInsert())

                self.reviewWin32SimdSetting()

                self.output('Getting Lib and Source Data...')
                if 'COMMON' not in productList:
                    productList.append('COMMON')
                libActionsDict = getLibInfo(productList)
                unavailableSourceDict = getSrcInfo(productList)
                self.output('\tDone')

                self.output('Configuring Makefiles and Project Files...')
                modulesConfigured = True
                try:
                    global modulesDict
                    configureModules(MODULE_CONFIG_DICT,
                                     unavailableSourceDict,
                                     libActionsDict,
                                     productList,
                                     self.output)
                except IOError:
                    modulesConfigured = False
                    self.output('Error configuring Havok modules, please check that you have permission to write '
                                'to your Havok distribution.')

                if modulesConfigured:
                    try:
                        workspaceGenerator = vsSolution.WorkspaceGenerator(HAVOK_ROOT,
                                                                           workspaceSpecType="HavokSdk",
                                                                           workspaceDir=os.path.join(HAVOK_ROOT, "Workspace"))
                        slns = workspaceGenerator.findProjectsAndGenerateWorkspaces(os.path.abspath(HAVOK_ROOT),
                                                                                    productsList=productList,
                                                                                    outputFunc=self.output)
                        global GENERATED_SOLUTIONS
                        GENERATED_SOLUTIONS += slns
                    except Exception as exc:
                        if exc.message:
                            self.output("Unable to create workspace:\n{}".format(exc.message))
                        else:
                            self.output("Unable to create workspace, please check that you have permission to "
                                        "write to your Havok distribution.")
        else:
            self.output('Please check that all the necessary packages '
                        'have been downloaded and unzipped, and try again.')
            self.output('If the problem persists please contact Havok support.')

        isAndroid = any(t.startswith('android') for t in getAvailableTargets())
        if everythingLooksOk and os.name == 'nt' and isAndroid:
            self.output('Create VSI platform directories')
            if "ProgramFiles(x86)" in os.environ:
                programFiles = os.environ["ProgramFiles(x86)"]
            else:
                programFiles = os.environ["ProgramFiles"]

            MSBuildPlatformDir = os.path.join(programFiles, "MSBuild", "Microsoft.Cpp", "v4.0", "Platforms")
            if not os.path.exists(MSBuildPlatformDir):
                self.output(("The expected MSBuild directory \"%s\" does not exist. "
                            "Please ensure MSBuild is installed") % os.path.dirname(MSBuildPlatformDir))
            else:
                AndroidPlatformDir = os.path.join(MSBuildPlatformDir, "hkAndroid")
                if not os.path.exists(AndroidPlatformDir):
                    try:
                        os.mkdir(AndroidPlatformDir)
                    except Exception:
                        self.output("Unable to create VSI folder: %s" % AndroidPlatformDir)

        if HK_PREBUILD_FAILURES:
            self.output('\n\nErrors occurred when generating reflections for some modules, you may see errors when rebuilding them.')
            for fail in HK_PREBUILD_FAILURES:
                self.output(' - {}, see log: {}'.format(fail.relModulePath, fail.logPath))
            self.output('\n\n')

        self.output('********\n Done\n********')
        self.configured = True

        if self.gui:
            self.outputText.config(state=Tkinter.DISABLED)
            self.goButton.config(text="Done", command=self.quit)
            self.update()
        return int(not everythingLooksOk)

    def output(self, message):
        if self.gui:
            self.outputText.config(state=Tkinter.NORMAL)
            self.outputText.insert(Tkinter.END, message + '\n')
            self.update()
            self.outputText.see(Tkinter.END)
            self.outputText.config(state=Tkinter.DISABLED)
        else:
            print message

    def extractKeycodesAndProducts(self, keycodeText, out):
        productList = []
        products = ['AI',
                    'BEHAVIOR',
                    'CLOTH',
                    'DESTRUCTION',
                    'DESTRUCTION_2012',
                    'PHYSICS',
                    'PHYSICS_2012',
                    'SCRIPT',
                    ]
        keycodes = []
        for keycode in keycodeText.split('\n'):
            # Ignore keycodes for non-protected products. These are generated later based on source and libs present,
            # yet it is likely that users will often enter them.
            keycode = keycode.replace(INITIAL_TEXT, '').strip()
            if keycode == '' or any('HAVOK_{}_KEYCODE'.format(x) in keycode for x in EMPTY_KEYCODE_PRODUCTS):
                continue
            keycodes.append(keycode)
            for product in products:
                if 'HAVOK_' + product + '_KEYCODE' in keycode:
                    productList.append(product)
                    break
        return keycodes, productList


def getLibInfo(productList):
    """
    Determine the library-level configuration steps that need to be performed (i.e. the libraries that either can't
    be linked or must additionally be linked, depending on what products the customer has).
    """
    libActionsDict = {'insert': [],
                      'remove': []}

    for productTag in LIBRARY_CONFIG_DICT:
        libAvailable = False
        for productElem in productTag.split():
            elemComponents = productElem.split('+')
            requiredProductsPresent = True
            for product in elemComponents:
                if product not in productList:
                    requiredProductsPresent = False
                    break
            if requiredProductsPresent:
                libAvailable = True
                break
        if (not libAvailable) and (productTag in LIBRARY_CONFIG_DICT):
            libActionsDict['remove'].extend(LIBRARY_CONFIG_DICT[productTag])

    if 'DESTRUCTION' in productList:
        if 'PHYSICS_2012' not in productList:
            libActionsDict['insert'].append({'ifPresent': 'hkndInternal',
                                             'addThis': 'hkndPhysics2012Bridge'})
        elif 'DESTRUCTION_2012' not in productList:
            libActionsDict['insert'].append({'ifPresent': 'hkndInternal',
                                             'addThis': 'hkndDestruction2012Bridge'})
    return libActionsDict


def addExtraLibsIfNeeded(libsList, libActionsDict):
    """
    Given the list of libs that are present and a libActions dict, perform any required library additions.
    """
    updatedLibsList = list(libsList)  # e.g. ['libAlpha.a', 'libBeta.a', 'libFoo.a', 'libGamma.a']

    for insertAction in libActionsDict['insert']:

        # e.g. {'ifPresent': 'Foo', 'addThis': 'Bar'}
        requiredLib = insertAction['ifPresent']
        libToAdd = insertAction['addThis']

        requiredLibFilename = None
        for libFilename in libsList:
            if requiredLib in libFilename:
                requiredLibFilename = libFilename
                break

        if requiredLibFilename:
            libFilenameToAdd = requiredLibFilename.replace(requiredLib, libToAdd)  # e.g. 'libFoo.a' => 'libBar.a'
            updatedLibsList.insert(updatedLibsList.index(requiredLibFilename), libFilenameToAdd)

    return updatedLibsList  # e.g. ['libAlpha.a', 'libBeta.a', 'libBar.a', 'libFoo.a', 'libGamma.a']


def getSrcInfo(productList):
    # Set up sourceConfigDict
    unavailableSourceDict = {}
    for module in SOURCE_CONFIG_DICT:
        unavailableSourceDict[module] = set()
        includedFiles = set()
        for productCombo in SOURCE_CONFIG_DICT[module]:
            products = productCombo.split('+')
            stripFiles = False
            for product in products:
                if product not in productList:
                    stripFiles = True
                    break
            if stripFiles:
                unavailableSourceDict[module].update(SOURCE_CONFIG_DICT[module][productCombo])
            else:
                includedFiles.update(SOURCE_CONFIG_DICT[module][productCombo])
        # in case of files common to multiple products - check if file was readded for a different product
        # and if it is the case remove that file from list of paths to remove
        unavailableSourceDict[module].difference_update(includedFiles)
    return unavailableSourceDict


def configureModules(modulesDict, unavailableSourceDict, libActionsDict, productList, out):
    """
    - Get the list of files to strip from configurable projects/makefiles (i.e. the source files that the customer
      doesn't have access to).
    - Configure modules.
    """
    for module in sorted(modulesDict.keys()):
        unavailableFilesList = []
        if module in unavailableSourceDict.keys():
            unavailableFilesList = unavailableSourceDict[module]
        configureModule(module,
                        unavailableFilesList,
                        libActionsDict,
                        productList,
                        (module not in unavailableSourceDict),
                        out)


def getInitialKeycodes(keycodeFile, keycodeProducts):
    """
    Get the keycodes in *keycodeFile* (but only those in *keycodeProducts*)
    """
    keycodes = re.findall(KEYCODE_REGEX, open(keycodeFile).read())
    if keycodeProducts:
        defines = [('HAVOK_%s_KEYCODE' % p) for p in keycodeProducts]

        missingKeycodes = [d for d in defines if d not in ''.join(keycodes)]
        assert not missingKeycodes, "\n\nERROR: No %s found in %s" % (missingKeycodes, keycodes)

        keycodes = [k for k in keycodes if any([d in k for d in defines])]

    assert keycodes, ("\n\nERROR: No keycodes %sfound in: %s" %
                      (('for %s ' % keycodeProducts if keycodeProducts else ''), keycodeFile))
    return '\n'.join(keycodes)


def main(keycodeFile=None, keycodeProducts=None, showGui=False):

    initialUserText = ''
    if keycodeFile:
        initialUserText = getInitialKeycodes(keycodeFile, keycodeProducts)

    app = Configuration(showGui, initialUserText)

    errorCode = 0
    if not showGui:
        errorCode = app.go()
    else:
        app.mainloop()

    return errorCode

COMMAND_LINE_OPTIONS = (
    (('--keycode-file',),
     {'action': 'store', 'dest': 'keycodeFile', 'default': '',
      'help': r'use the keycodes in this file'}),
    (('--keycode-products',),
     {'action': 'store', 'dest': 'keycodeProducts', 'default': '',
      'help': r"only take keycodes for these (semicolon-separated) products from keycodeFile"}),
    (('--go',),
     {'action': 'store_true', 'dest': 'go', 'default': False,
      'help': 'run without displaying the GUI'}),
    (('--xcode-dev-team',),
     {'action': 'store', 'dest': 'xcodeDevTeam', 'default': None,
      'help': 'Set the DEVELOPMENT_TEAM in Xcode projects to this value. Use "havok" for Havok team.'}),
    )

if __name__ == '__main__':
    import optparse
    parser = optparse.OptionParser()
    for options in COMMAND_LINE_OPTIONS:
        parser.add_option(*options[0], **options[1])
    options, args = parser.parse_args()

    keycodeProducts = options.keycodeProducts.split(';') if options.keycodeProducts else []

    if options.xcodeDevTeam is not None:
        XCODE_DEV_TEAM = options.xcodeDevTeam if options.xcodeDevTeam.lower() != "havok" else HAVOK_XCODE_DEV_TEAM

    sys.exit(main(options.keycodeFile, keycodeProducts, showGui=(not options.go)))
