import glob
import os
import sys
import re
import inspect

from functools import partial
from argparse import ArgumentParser

SCRIPT_DIR = os.path.abspath(os.path.dirname(inspect.getfile(inspect.currentframe())))
HAVOK_ROOT = os.path.normpath(os.path.join(SCRIPT_DIR, '..', '..', '..'))


class AssetError(Exception):
    """Raised when an error occurs processing assets"""
    pass


def getFilesByPattern(srcDir, filePattern, recurse):
    """
    Get a list of all the files that match the specified file pattern in the specified directory.
    Also, search subdirectories if the 'recurse' parameter is set to 1.
    """
    fileList = []

    def matchGlob(curDir, dirContents):
        relDir = curDir[len(srcDir)+len(os.path.sep):]
        return glob.glob1(os.path.join(srcDir, relDir), filePattern)

    def getXmlfiles(identifier, curDir, dirContents):
        fileMatches = lambda x: x.endswith(".xml") and identifier in open(os.path.join(curDir, x), "r").read(1024)
        return [x for x in dirContents if fileMatches(x)]

    # For some asset types, e.g. XML-tagfile, it is not possible to determine matching files from
    # filename alone. Thus, special patterns idenfity that custom functions must be used
    # to match assets of these types.
    SPECIAL_PATTERNS = {
        "XML_TAGFILE": partial(getXmlfiles, "<hktagfile"),
        "XML_PACKFILE": partial(getXmlfiles, "<hkpackfile"),
        }
    matchFilesFunc = SPECIAL_PATTERNS.get(filePattern, matchGlob)

    def getMatchingFiles(dummy, curDir, dirContents):
        relDir = curDir[len(srcDir)+len(os.path.sep):]
        matchingFiles = matchFilesFunc(curDir, dirContents)
        for matchingFile in matchingFiles:
            fileList.append(os.path.join(relDir, matchingFile))

    if recurse == 1:
        os.path.walk(srcDir, getMatchingFiles, None)
    else:
        getMatchingFiles(None, srcDir, [])

    return fileList


def getBinDir():
    if os.name == 'nt':
        return 'windows'
    elif sys.platform == 'darwin':
        return 'mac'
    else:
        return 'linux'


def getExe():
    if os.name == 'nt':
        return 'FileConvert.exe'
    elif sys.platform == 'darwin':
        return 'FileConvert'
    else:
        return 'FileConvert.out'


def convertAssets( assetList, srcDir, destDir, writehkt, writeXmlTagfile, excludeMetadata, fileConvertExe ):
    """
    Create copies of the specified assets using the specified binary layout rules.
    """
    upToDateAssets = 0

    # Suppress crash dialogs which appear when processing bad assets.
    if sys.platform.startswith("win"):
        # Don't display the Windows GPF dialog if the invoked program dies.
        import ctypes
        SEM_NOGPFAULTERRORBOX = 0x0002  # From MSDN
        ctypes.windll.kernel32.SetErrorMode(SEM_NOGPFAULTERRORBOX)

    def getOutfileNameForCurrentAsset(assetPath, outfileExt):
        """
        Get the output file name to use with the current asset:

        - *assetPath* = 'TestAssetPath_L8111.HKT' and *outfileExt* = '.hkt'
        """
        outfileRoot, currentExt = os.path.splitext(assetPath)
        outfileRoot = re.sub('_L[48][01]{3}$', '', outfileRoot)  # Strip any RHS '_L4101' etc.
        return '%s%s' % (outfileRoot, outfileExt)

    for asset in assetList:
        infile = os.path.join(srcDir, asset)

        if writehkt:
            outfile = getOutfileNameForCurrentAsset(asset, '.hkt')
        else:  # writeXmlTagfile
            outfile = getOutfileNameForCurrentAsset(asset, '.xml')

        outfile = os.path.join(destDir, outfile)

        if not fileConvertExe:
            fileConvertExe = os.path.join(HAVOK_ROOT, 'Tools', 'FileConvert', 'bin', getBinDir(), getExe())
			
        if not os.path.exists(fileConvertExe):
            # If the above path/file does not exist we might sit inside a HavokContentTools installation. Let's try to find the .exe locally then.
            fileConvertExe = os.path.join(SCRIPT_DIR, 'bin', getBinDir(), getExe())

        print outfile
        if not os.path.isdir(os.path.dirname(outfile)):
            os.makedirs(os.path.dirname(outfile))

        stripMetaData = '-s' if excludeMetadata else ''

        if writehkt:
            assetCommand = '""%s" %s -g "%s" "%s""' % (fileConvertExe, stripMetaData, infile, outfile)
        else:  # writeXmlTagfile
            assetCommand = '""%s" %s -x "%s" "%s""' % (fileConvertExe, stripMetaData, infile, outfile)

        # Run the command
        errorCode = os.system(assetCommand)
        if errorCode:
            print 'ERROR: FileConvert was unable to convert %s.' % infile

        else:
            upToDateAssets += 1

    if upToDateAssets > 0:
        print upToDateAssets, 'assets already exist and are up to date.'

    # Reset crash dialog suppression.
    if sys.platform.startswith("win"):
        ctypes.windll.kernel32.SetErrorMode(0)


def main(arguments):
    if (sys.version_info[0] + 0.1 * sys.version_info[1]) < 2.7:
        raise AssetError("Error: Python 2.7 or greater is needed to run this script.")

    parser = ArgumentParser()
    parser.add_argument('-s', '--srcDir', help="The directory to get files to convert from. Can be relative or absolute.", required=True)
    parser.add_argument('-d', '--destDir', help="Where to put generated files. The directory structure is mirrored if recurse is enabled.")
    parser.add_argument('-f', '--filePattern', help="Files matching this pattern are converted. *,? and [] wildcards are supported.", default='*.hkx')
    parser.add_argument('-r', '--recurse', help="Whether to recurse or not.", action='store_true', default=True)
    parser.add_argument('-m', '--excludeMetadata', help="Whether to exclude metadata when generating new files.", action='store_true', default=False)
    group = parser.add_mutually_exclusive_group(required=True)
    group.add_argument('-g', '--writehkt', help="Output cross platform binary HKT files", action='store_true', default=False)
    group.add_argument('-x', '--writeXmlTagfile', help="Output cross platform XML HKT files", action='store_true', default=False)
    parser.add_argument('-c', '--fileConvertExe', help="Path to exe to use in place of the default path", default='')
    arguments = parser.parse_args()
    arguments.destDir = arguments.destDir if arguments.destDir else arguments.srcDir

    assetsToConvert = getFilesByPattern(arguments.srcDir, arguments.filePattern, arguments.recurse)

    if len(assetsToConvert) > 0:
        convertAssets(assetsToConvert, arguments.srcDir, arguments.destDir, arguments.writehkt, arguments.writeXmlTagfile, arguments.excludeMetadata, arguments.fileConvertExe)
    else:
        print "No assets found..."

if __name__ == "__main__":
    try:
        main(sys.argv[1:])
    except AssetError as e:
        print e.message
        sys.exit(1)
