#
# Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
# prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
# Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
#
#!/usr/bin/env python

#
# Parse the output from the modified LLVM/CLANG Parser into a proper
# havokDomClass object
#
import re
import os
import util
import types
import exceptions
import reflectionDatabase

from havokDomClass import domElements

# These are used a lot in attribute parsing so pre-compile them
re_Alignment = re.compile("""(
                             __declspec\(\s*align\s*\(\s*(?P<alignmenta>\d+)\s*\)\s*\) # e.g. __declspec( align ( 16 ) )
                             |
                             HK_ALIGN(?P<alignmentb>\d+)\(                             # e.g. HK_ALIGN16(
                             |
                             HK_ALIGN\(.*?,\s*(?P<alignmentc>\d+)\s*\);                # e.g. HK_ALIGN( hkCriticalSection m_criticalSection, 16 );
                             |
                             HK_ALIGN_(?P<alignmentd>REAL)\(                            # e.g. HK_ALIGN_REAL(
                             )""", re.VERBOSE) # Alignment specifiers
re_Plus = re.compile(r"\+\w") # Plus followed by word character ( +x ) and not (x + y)
re_CommentLine = re.compile(r"^\s*//.*$") # Match empty lines, and lines containing only whitespace and comments
re_Override = re.compile("(hkArray|hkSimpleArray|hkEnum|hkFlags)\s*<\s*([^>,]+)\s*(,[^>]+)?\s*>") # Used by attribute override


# Parse errors don't need to kill the whole compilation, only the current file
# so treat them specially
class ParseErrorException(exceptions.Exception):
    def __init__(self, args = None, arglist = None):
        if arglist:
            self.filename = args
            self.msg = arglist
        else:
            self.filename = args[0]
            self.msg = args[1]    
    pass

def applyAttributes(obj, attributes):
    for key, val in attributes:
        try:
            # Try and guess the type for basic native types
            # If none of these work, just leave it as a string
            curval = getattr(obj, key)
            if isinstance(curval, types.IntType):
                try:
                    if(val.lower() == 'true'):
                        val = True
                    elif(val.lower() == 'false'):
                        val = False
                except:
                    pass
                else:
                    try:                       
                        val = int(val)
                    except:
                        assert val == "REAL" and key == "align"
                        # real alignment
            elif isinstance(curval, types.FloatType):
                val = float(val)
            # If it can't be handled numerically it is assumed to be a string
            setattr(obj, key, val)
        except AttributeError:
            # If there is an attribute with that name, set it
            # If not, dump it in the extra attribute list
            obj.attributes[key] = val

# Find the first occurence of searchStr in quotedString, excluding
# any quoted strings inside the string. Return the index of the first
# match, or -1 if not found
def findInQuoted(quotedString, searchStr):
    quoteMatch = re.search(r'(?<!\\)"(\\"|[^"])*(?<!\\)"', quotedString)
    quoteIndex = quoteMatch.start() if quoteMatch else -1
    firstSearchIndex = quotedString.find(searchStr)
    offsetIndex = 0
    while quoteIndex > 0 and quoteIndex < firstSearchIndex:
        offsetIndex += quoteMatch.end()
        quoteMatch = re.search(r'(?<!\\)"(\\"|[^"])*(?<!\\)"', quotedString[offsetIndex:])
        quoteIndex = quoteMatch.end() if quoteMatch else -1
        firstSearchIndex = quotedString[offsetIndex:].find(searchStr)
    return firstSearchIndex + offsetIndex if firstSearchIndex > -1 else -1



# Scan a line for attribute declarations
# These are of the form "// +name(val)
# TODO: Need to keep scanning until next statement
def getOldstyleAttributes(declStr):
    ret = []
    
    alignMatch = re_Alignment.search(declStr)

    if alignMatch:
        for name in ['alignmenta', 'alignmentb', 'alignmentc', 'alignmentd']:
            if alignMatch.group(name):
                ret.append(("align", alignMatch.group(name)))
        
    startIndex = declStr.find("//")
    declStr = declStr[startIndex+len("//"):]

    plusMatch = re_Plus.search(declStr)
    if plusMatch:
        startIndex = plusMatch.start()
    else:
        startIndex = -1
        
    while startIndex > -1:
        declStr = declStr[startIndex+1:]
        nameIndex = declStr.find("(")
        spaceIndex = declStr.find(" ")

        # if an opening '(' is not found or there is a space before it, check for single name attributes
        if nameIndex == -1 or (nameIndex > -1 and spaceIndex < nameIndex):
            if declStr.startswith("nosave"):
                declStr = declStr[len("nosave"):]
                startIndex = declStr.find("+")
                ret.append(("nosaveattr", True))
                continue
        
        attributeIndex = findInQuoted(declStr, ")")
        if nameIndex > -1 and attributeIndex > -1 and attributeIndex > nameIndex:
            ret.append((declStr[:nameIndex],declStr[nameIndex+1:attributeIndex]))
            declStr = declStr[attributeIndex:]
        match = re_Plus.search(declStr)
        if match:
            startIndex = match.start()
        else:
            startIndex = -1

    return ret

def applyNewAttributeList(target, attributesList):
    matches = []

    for attributes in attributesList:
        print "ATTR %s" % attributes
        in_string = False
        nested_brackets = 0
        index = 0
        lastChar = " "
        currentName = ""
        currentMatch = ""

        while index < len(attributes):
            if nested_brackets != 0 or attributes[index] != " ":
                currentMatch += attributes[index] # Skip spaces between identifiers
                
            if attributes[index] == "(":
                nested_brackets += 1
                if nested_brackets == 1:
                    currentName = currentMatch[:-1]
                    currentMatch = ""
            elif attributes[index] == ")":
                nested_brackets -= 1
                if nested_brackets == 0:
                    # Emit a value if it matches
                    print "%s," % currentName
                        # gccxml(name(value)) -> (name, value)
                    matches.append((currentName.strip(), currentMatch[:-1].strip()))
                    currentMatch = ""
            lastChar = attributes[index]
            index += 1

    for (key, val) in matches:
        if key == "nosave":
            matches.append(("nosaveattr", True))
            matches.remove((key, val))
            break

    applyAttributes(target, matches)
    return matches

# line = line number of declaration to start searching for old-style attributes
#    None -> don't search
# attributes = new-style attribute string
#    None -> don't use new attributes
def searchAttributes(target, filename, line, attributes = None, fileLines = []):
#< TODO new-style attributes
#< and more generic support than "gccxml"
    if filename.lower().endswith(".inl"):
        return
    if attributes:
        applyNewAttributeList(target, attributes)
    line -= 1  # Line no. starts at 1, fileLines indexed at 0

    originalLine = line

    while(line and line < len(fileLines)):
        memberDecl = fileLines[line]
        # Special case -- access specs may come in the middle of some attributes (particularly class version)
        # Anything else other than whitespace or comments will terminate the search
        memberDecl = memberDecl.strip()
        if line == originalLine:
            pass
        elif memberDecl == "public:" or memberDecl == "private:" or memberDecl == "protected:":
            pass
        elif memberDecl.strip() == "" or re_CommentLine.match(memberDecl):
            pass
        else:
            return

        decls = getOldstyleAttributes(memberDecl)

        # If we find a comment that is not an attribute, we're finished
        if re_CommentLine.match(memberDecl) and not decls:
            return
        
        applyAttributes(target, decls)
        
        line += 1

# Returns true if the nosave attribute should be respected
# The attribute is always respected for now as each change would require a verisoning patch
# as the member type is changed.
def checkNoSaveAttr(member, klass, location):
#    if member.type.find("*") != -1 or member.type.find("struct") != -1 or member.type.find("class") != -1 or member.type.find("Array") != -1:
#        return True
#    print "%s(%d) : warning: %s.%s (type %s) has attribute +nosave but it appears to be a simple type. Consider +serialized(false) instead" % (location['file'], location['line'], klass.name, member.name, member.type)
    return True

def recodeClass(classDef, fileLines = [], scope='::'):
    def removeFromDeclaration(decl, listToRemove):
        return "".join([x for x in decl.replace("*"," * ").replace("&"," & ").split() if x not in listToRemove])
    def templateType(type):
        if(re.match(".*<.*>", type)): # might be a template
            # These special havok types should not be treated as
            # user defined template classes.
            specialHavokTypes = ["hkArray<",
                "hkRelArray<", "class hkRelArray<", "hkInplaceArray<", 
                "hkSimpleArray<", "hkEnum<", "hkFlags<", "hkZero<", 
                "hkHomogeneousArray<", "hkRefPtr<", "class hkSmallArray<"]
            for str in specialHavokTypes:
                if(type.startswith(str)):
                    return False
            return True # template class
        return False
    ret = domElements.Class()
    ret.name = classDef['name']
    ret.location = os.path.normpath(classDef['location']['file']), classDef['location']['line']
    #< Check this
    if classDef.get("namespace_name", None):
        ret.scope = classDef['namespace_name']
        scope = classDef['namespace_name']
    else:
        ret.scope = scope
        
    ret.abstract = classDef['is_abstract']
    ret.fromheader = True

    foundRealAllocator = False
    realAllocatorAccess = "protected" # default
    foundPlacementAllocator = False

    searchAttributes(ret, classDef['location']['file'], classDef['location']['line'] + 1, None, fileLines)

    if classDef.get('is_fake_parent', False):
        ret.attributes['hk.MemoryTracker'] = "ignore=True"

#    print "NAME %s[%s,%s]" % (ret.name, classDef['bases'], len(classDef['bases']))
    if len(classDef['bases']):
        ret.parent = classDef['bases'][0]
    if len(classDef['bases']) > 1:
        for interf in classDef['bases'][1:]:
            ic = domElements.Interface()
            ic.name = interf
            ret.interface.append(ic)        

    for methodDef in classDef['methods']:
        method = domElements.Method()
        method.name = methodDef['name']
        method.visibility = methodDef['access']
        method.static = methodDef['is_static']
        #method.const = methodDef['is_const']
        method.virtual = methodDef['is_virtual']
        method.purevirtual = methodDef['is_purevirtual']
        if methodDef.get('return', None):
            method.returns = methodDef['return']

        if method.name == "operator new" and not methodDef['is_artificial']:
            # Found a real allocator
            foundRealAllocator = True
            realAllocatorAccess = method.visibility
            #< TODO: Need to get gccxml(placement) attribute or similiar

        for paramDef in methodDef['params']:
            param = domElements.Parameter()
            param.name = paramDef.get('name', '') # parameters need not be named
            param.type = paramDef.get('declared_type') # 'type' is resolved
            #param.default #< TODO
            param.description = "%s %s" % (param.type, param.name)
            method.parameter.append(param)

        # Special cases
        if method.virtual or method.purevirtual:
            ret.vtable = True
        if method.name == "staticClass":
            ret.reflected = True
        if methodDef.get("attributes", None):
            for attrString in methodDef['attributes']:
                if attrString == "placement":
                    foundPlacementAllocator = 2 # Why not just boolean??

        if not methodDef['is_artificial']:
            ret.method.append(method)

    for memberDef in classDef['members']:
        # Only non-static members here
        member = domElements.Member()

        if not memberDef['name'].startswith(ret.memberprefix):
            if ret.reflected and memberDef.get('location', None) and memberDef['location'].get('file', None):
                raise ParseErrorException((memberDef['location']['file'], "Member %s does not start with %s" % (memberDef['name'], ret.memberprefix)))
            else:
                member.has_memberprefix = False
            member.name = memberDef['name']
        else:                        
            member.name = memberDef['name'][len(ret.memberprefix):]

        member.type = memberDef['declared_type'] #< Probably need to process this

        if templateType(member.type): # template type
            member.embedtype = removeFromDeclaration(member.type, ("mutable","const","class","struct"))
            allchars = "".join([chr(x) for x in range(256)])
            member.embedtype = member.embedtype.translate(allchars, ":<>,")

        #< Ciaran: Hack until I fix up the template typedef parameters
        #if member.type.find('hkEnum') != -1 or member.type.find('hkArray') != -1 or member.type.find('hkFlags') != -1:
        member.type = re.sub(r'\bunsigned char\b', 'hkUint8', member.type)
        member.type = re.sub(r'\bsigned char\b', 'hkInt8', member.type)
        member.type = re.sub(r'\bunsigned short\b', 'hkUint16', member.type)
        member.type = re.sub(r'\bshort\b', 'hkInt16', member.type)
        member.type = re.sub(r'\bunsigned int\b', 'hkUint32', member.type)
        member.type = re.sub(r'\bunsigned long long\b', 'hkUint64', member.type)
        member.type = re.sub(r'\blong long\b', 'hkInt64', member.type)
        member.type = re.sub(r'\bunsigned long\b', 'hkUlong', member.type)
        member.type = re.sub(r'\bunsigned\b', 'hkUint32', member.type)
        member.type = re.sub(r'\bint\b', 'hkInt32', member.type)

        if ret.reflected and member.type.find("hk_size_t") > -1:
            raise ParseErrorException((memberDef['location']['file'], "hk_size_t not allowed in reflected classes (use hkUlong instead)" ))

        # This is slightly hackish as well, shouldn't be needed
        member.type = re.sub(r"hkRefPtr< ([^>]*) >", r"\1 *", member.type)
        # As is this
        member.type = re.sub(r"hkRefVariant\b", "hkReferencedObject *", member.type)
        
        # Do SimpleArray etc. here as we can apply attributes to the correct type
        if(len(ret.member) > 0):
            lastMember = ret.member[-1]
            if member.name.lower().startswith("num") and lastMember.name.lower() == member.name.lower()[3:]:
                if(len(ret.member) > 1 and ret.member[-2].name.lower().startswith(lastMember.name.lower()) and ret.member[-2].name.lower().endswith("class")):
                    secondLastMember = ret.member[-2]
                    member.name = lastMember.name
                    member.type = "hkHomogeneousArray"
                    ret.member.remove(lastMember)
                    ret.member.remove(secondLastMember)
                elif lastMember.type.endswith("*"):
                    member.type = "hkSimpleArray< %s >" % lastMember.type.rsplit("*",1)[0].rstrip()
                    member.name = lastMember.name
                    ret.member.remove(lastMember)
            elif member.name.lower().endswith("class") and lastMember.name.lower() == member.name.lower()[:-5] and member.type.lower() == "hkclass *" and lastMember.type.lower() == "void *":
                member.type = "hkVariant"
                member.name = lastMember.name
                ret.member.remove(lastMember)

        # Find any attributes (old or new) corresponding to this member
        searchAttributes(member, memberDef['location']['file'], memberDef['location']['line'], memberDef.get("attributes", None), fileLines)

##        # Fix up enums -- some non-reflected classes use C++ enums, but their underlying type
##        # (and size) is implementation-defined. For now we assume they are 4 bytes in size
##        # (MSVC default) however this is not guaranteed
##        if member.type.find("hkEnum") == -1 and member.type.find("hkFlags") == -1:
##            enumret = re.subn(r"enum\s+([\w\d_:]+)",r"int", member.type)
##            member.type = enumret[0]
##            if enumret[1] and ret.reflected:
##                print "%s(%d) : warning: Converting unsized enum %s, assuming int storage type\n" % (filename, cMember.location.line, member.name)


        # nosave and reflected(false) are treated slightly differently
        if member.attributes.has_key('nosaveattr'):
            if checkNoSaveAttr(member, ret, memberDef['location']):
                member.serialized = False
                t = member.overridetype or member.type
                if t.endswith("*"):
                    member.overridetype = "void*"
                else:
                    partial = re_Override.match(t)
                    if partial:
                        stem, c, sz = partial.groups()
                        member.overridetype = stem + ("<void*" if c.strip().endswith("*") else "<void") + (sz or "") + ">"
            del member.attributes['nosaveattr']

        member.visibility = memberDef['access']
        def HACK_member_tracked(t): # remove this when llvm extraction issues are fixed
            if t.endswith("&"): return False
            if t.startswith("hkArray<"): return True
            if t == "hkStringPtr": return True
            if t.startswith("class hkPadSpu<") and t.find("*")!=-1:
                return True
            return False
        member.memory_tracked = memberDef['is_tracked'] or HACK_member_tracked(member.type)
        member.sourcecode = "%s %s;" % (member.type, member.name)
        
        ret.member.append(member)

    for enumDef in classDef['enums']:
        #< Should we skip anonymous enums?
        if enumDef['name']:
            enum = domElements.Enum()
            enum.name = enumDef['name']
            if ret.scope != "::":
                enum.scope = ret.scope + "::" + ret.name
            else:
                enum.scope = ret.name

            searchAttributes(enum, enumDef['location']['file'], enumDef['location']['line'] + 1, None, fileLines)

            for enumVal in enumDef['decls']:
                item = domElements.EnumItem()
                item.value = enumVal['value']
                item.name = enumVal['name']
                enum.item.append(item)
            ret.enum.append(enum)
        
    #< TODO still don't handle placement allocator        
    ret.memory_declaration = (realAllocatorAccess, foundPlacementAllocator, classDef['is_referenced_object'], foundRealAllocator)
    if classDef.get('override_class', None):
        ret.override_name = re.sub(r"([\<,]\s*)unsigned long(\s*[\>,])", r"\1hkUlong\2", classDef['override_class'])
    
    for subClass in classDef['classes']:
        cppName = ""

        if ret.override_name: # get full cpp name (with <>) if available
            cppName = ret.override_name
        else:
            cppName = ret.name

        if scope != "::":
            subScope = scope + '::' + cppName
        else:
            subScope = cppName
        
        ret._class.append(recodeClass(subClass, fileLines, subScope))
            
    return ret

def parseFile(filename, LLVMoutput, project_dir = None):
    filename = reflectionDatabase.standardFileName(filename)
    ret = domElements.Document(os.path.abspath(filename), project_dir)
    txtfile = open(filename)
    if not txtfile:
        raise ParseErrorException((filename, "File not found"))
    fileLines = txtfile.readlines()
    ret.file = domElements.File()

    tkbms_dict = util.extract_tkbms_file(fileLines)
    if tkbms_dict:
        for key, value in tkbms_dict.items():
            if key in ret.file.__slots__: # workaround for e.g TAGS: !ANARCHY - see RSYS-1693
                setattr(ret.file,key,value)
    else:
        setattr(ret.file, "product", "ALL")
        setattr(ret.file, "platform", "ALL")
        setattr(ret.file, "visibility", "CUSTOMER")

    txtfile.close()

    for classDef in LLVMoutput['classList']:
        fakeParent = classDef.get('is_fake_parent', False)
        if filename == classDef['location']['file'] or fakeParent:
            try:
                # This is one of the top-level classes for that file
                if fakeParent:
                    fakeParentFile = open(classDef['location']['file'])
                    if not fakeParentFile:
                        raise ParseErrorException((classDef['location']['file'], "File not found"))
                    
                    fakeParentFileLines = fakeParentFile.readlines()                    
                    ret.file._class.append(recodeClass(classDef, fakeParentFileLines, scope="::"))
                else:
                    ret.file._class.append(recodeClass(classDef, fileLines, scope="::"))
            except ParseErrorException, e:
                print "%s: error: class %s: %s\n" % (e.filename, classDef['name'], e.msg)
                raise

    for enumDef in LLVMoutput['enumList']:
        if filename == enumDef['location']['file']:
            if enumDef['name']:
                enum = domElements.Enum()
                enum.name = enumDef['name']
                if enumDef.get("namespace_name", None):
                    enum.scope = enumDef['namespace_name']
                #enum.scope = ret.scope

                #searchAttributes(enum, enumDef['location']['line'], None, fileLines)

                for enumVal in enumDef['decls']:
                    item = domElements.EnumItem()
                    item.value = enumVal['value']
                    item.name = enumVal['name']
                    enum.item.append(item)
                ret.file.enum.append(enum)

    return ret

#
# Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
# 
# Confidential Information of Havok.  (C) Copyright 1999-2014
# Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
# Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
# rights, and intellectual property rights in the Havok software remain in
# Havok and/or its suppliers.
# 
# Use of this software for evaluation purposes is subject to and indicates
# acceptance of the End User licence Agreement for this product. A copy of
# the license is included with this software and is also available at www.havok.com/tryhavok.
# 
#