#
# Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
# prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
# Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
#
#!/usr/bin/env python

# Plugin to generate class reflection files. This should not be called directly. 
#
# A ReflectedClasses.cpp file is generated for every project containing reflected
# files. This contains the hkClass files corresponding to that project. If any unit tests
# contain serialized classes, these are placed in a seperate ReflectedClassesUnitTests.cpp
# to allow them to be stripped correctly.
#
# The output file is only written if it has changed, to reduce compilation

from havokDomClass import domToClass
import re
import util
import os
import string

_tkbms = """// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : %s
// PRODUCT    : %s
// VISIBILITY : %s
//
// ------------------------------------------------------TKBMS v1.0

//HK_REFLECTION_PARSER_EXCLUDE_FILE

// Autogenerated by generateReflections.py (reflectedClasses.py)
// Changes will not be lost unless:
// - The workspace is re-generated using build.py
// - The corresponding reflection database (reflection.db) is deleted
// - The --force-output or --force-rebuild option is added to the pre-build generateReflection.py execution

"""

_boilerPlateStart = """/* 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

//HK_REFLECTION_PARSER_EXCLUDE_FILE

// Autogenerated by generateReflections.py (reflectedClasses.py)
// Changes will not be lost unless:
// - The workspace is re-generated using build.py
// - The corresponding reflection database (reflection.db) is deleted
// - The --force-output or --force-rebuild option is added to the pre-build generateReflection.py execution

"""

_boilerPlateEnd = """/*
* Havok SDK - Level 1 file, Client Regenerated
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available from salesteam@havok.com.
* 
*/
"""

def createReflectionsDir(project_dir, options):
    """Creates the output directory for the generated reflection data.
    Reflected data will be produced in a single .cpp file for each class
    and stored in the Reflections/ subdirectory of the generateReflections.py
    output path (usually the Classes/ directory inside the project folder)."""
    
    if options.output_dir:
        dirname = options.output_dir
    else:
        dirname = os.path.join(project_dir, "Classes")
    
    # create Classes/ if it does not exist
    if not os.path.exists(dirname):
        os.mkdir(dirname)

    dirname = os.path.join(dirname,"Reflections")

    # create Classes/Reflections/ if it does not exist
    if not os.path.exists(dirname):
        os.mkdir(dirname)

    return dirname

def findClasses(classList):
    ret = classList[:]
    for c in [c for c in classList if c._class]:
        ret.extend(findClasses(c._class))
    return ret

def getAdditionalIncludesForSpecialLocation(contentsDict, product):
    curProducts = set(product.split('+'))
    header_final = ""
    # filenames are sorted to ensure consistent ordering of #includes in output files
    for (havokProductGroup, docsList) in sorted(contentsDict.items()):
        for d in docsList:
            fileProducts = set(d.file.product.upper().split('+'))
            # the current product must be a superset of the products for
            # the included file. When the included file product is ALL we always
            # include it.
            if "ALL" not in fileProducts:
                fileProducts.difference_update(curProducts)
                if len(fileProducts) != 0:
                    continue
            fullpathname = d.localfilename.replace("\\","/")
            header_final += "#include <%s>\n" % fullpathname
    return header_final

def computePlatform(platformString):
    # The platform used for the reflected file is the same as the source file,
    # but reflected file will never have platforms 'SPU', 'SIMSPU' or 'LRB'.
    platforms = platformString.upper().split(' ')
    platforms = [p for p in platforms if p not in ['SPU', 'SIMSPU', 'LRB']]
    platforms = ' '.join(platforms)
    return platforms

def rreplace(s, old, new, occurrences):
    """ Like Python standard replace, but starting from the right
    of the string. """
    li = s.rsplit(old, occurrences)
    return new.join(li)
    
def genOutputTextDictionary(location, project_dir, contentsDict, options):
    """This function returns a dictionary containing for each class the text
    to write in the corresponding reflected output file."""

    retDict = {}

    header_final = ""
    specialLocation = False

    # A location specified by the user (and different from the well known
    # special UnitTest location) may be used to express an include dependency 
    # between classes in different files, this can be handled by including 
    # all the header files for classes in the same location (different than 
    # the default UnitTest one).
    # Header file inclusion must be limited to header file which are available
    # considering the product for the current file.
    if location not in ("", "UnitTest"):
        specialLocation = True

    for (havokProductGroup, docsList) in contentsDict.items():
        for d in docsList:
            # Header and trailer to the class files
            header = ""
            trailer = ""
            # Figure out what header and trailer to use
            if not options.customer_build:
                if d.file.visibility == "CUSTOMER":
                    header += _boilerPlateStart
                    trailer += _boilerPlateEnd
                else:
                    header += _tkbms % (computePlatform(d.file.platform), d.file.product.upper(), d.file.visibility)
                    # empty trailer
        
            # Include KeyCode.h, if required.
            wrapClassesInHavokProductDefines = (not options.customer_build) and len(contentsDict.keys()) > 1
        
            fullpathname = d.localfilename.replace("\\","/")
            header += "// Generated from '%s'\n" % fullpathname
            header += "#include <%s>\n" % (d.pchfile or "Common/Base/hkBase.h")
            if not options.customer_build:
                if wrapClassesInHavokProductDefines:
                    header += "#include <Common/Base/KeyCode.h>\n"
            if wrapClassesInHavokProductDefines and havokProductGroup != '_NONE':
                header += '\n#if defined(HK_FEATURE_PRODUCT_%s)\n\n' % string.join(havokProductGroup.split('+'), ') && defined(HK_FEATURE_PRODUCT_')
            header += "#include <Common/Base/Reflection/hkClass.h>\n"
            header += "#include <Common/Base/Reflection/hkInternalClassMember.h>\n"
            header += "#include <Common/Base/Reflection/hkTypeInfo.h>\n"
            header += "#include <Common/Base/Reflection/Attributes/hkAttributes.h>\n"
            if specialLocation:
                header += getAdditionalIncludesForSpecialLocation(contentsDict, d.file.product.upper())
            else:
                fileHeader = "#include <%s>" % fullpathname
                if fileHeader not in header:
                    header += fileHeader + "\n"
            header += """
// Make VAX ignore symbols from this file
#if 1 == 0
_asm {
#endif
"""
            header += "#define True true\n"
            header += "#define False false\n"

            
            header += "" if not d.file.overridedestination else 'HK_REFLECTION_CLASSFILE_HEADER("%s");\n' % fullpathname

            if wrapClassesInHavokProductDefines and havokProductGroup != '_NONE':
                trailer = '\n#endif // HK_FEATURE_PRODUCT_%s\n' % string.join(havokProductGroup.split('+'), ' && HK_FEATURE_PRODUCT_') + trailer
            
            # Clean "Client" from location if the visibility of the document is "Client"
            # This is a quick solution to strip Client from the name to get the same reflected class file
            # name for client builds and havok builds.
            if d.file.visibility == "CLIENT":
                location = rreplace(location, "Client", "", 1)

            # Effective output file generation
            filename = d.basefilename
            filename = filename.replace(".h", "")
            filename = location + filename + "Reflection.cpp"
            retDict[filename] = header + domToClass.domToClass(d) + trailer

    return retDict

def fileListPlugin(DBObject, project_dir, options):
    """Returns the list of generated files."""
    ret = []
    if options.output_dir:
        outputDir = os.path.join(options.output_dir,"Reflections")
    else:
        outputDir = os.path.join(project_dir, "Classes", "Reflections")
    outputDict = calculateOutputFiles(DBObject, options, reflectionsOnly = True)
    for (location, contentsDict) in outputDict.items():
        for (havokProductGroup, docsList) in contentsDict.items():
            for d in docsList:
                filename = d.basefilename
                filename = filename.replace(".h", "")

                # Clean "Client" from location if the visibility of the document is "Client"
                # This is a quick solution to strip Client from the name to get the same reflected class file
                # name for client builds and havok builds.
                if d.file.visibility == "CLIENT":
                    location = rreplace(location, "Client", "", 1)

                filename = location + filename + "Reflection.cpp"
                ret.append(os.path.join(outputDir, filename))
    return ret
    
def findProductsInDocs(docs):
    productsInDocs = []
    # Imported templates should not be considered for generating the product list
    for d in [doc for doc in docs if (doc.file._class and (not all([c.override_name for c in doc.file._class])))]:
        normalisedProduct = string.join( sorted(d.file.product.split('+')), '+' ).upper()
        if normalisedProduct not in productsInDocs:
            productsInDocs.append( normalisedProduct )
    return productsInDocs

def findVisibility(docs):
    visList = [d.file.visibility for d in docs]
    if "INTERNAL" in visList:
        return "INTERNAL"
    if "CLIENT" in visList:
        return "CLIENT"
    if "CUSTOMER" in visList:
        return "CUSTOMER"
    return "PUBLIC"

def findPlatform(docs):
    # All the classes to be stored in a single file MUST have the same platform combo.
    # Assert if they don't.
    errorStr = """ERROR: Trying to put files with a mix of TKBMS platforms into a single file.
                   Force them to go in different files by adding //+hk.ReflectedFile("<location>") to the class declarations.
                   Platforms are [%s]"""
    platformsInDocs = []
    for d in docs:
        platList = [p for p in d.file.platform.upper().split(' ') if p not in ['SPU', 'SIMSPU', 'LRB']]
        normalisedPlatform = string.join( sorted(platList), ' ' )
        if normalisedPlatform not in platformsInDocs:
            platformsInDocs.append( normalisedPlatform )
    assert len(platformsInDocs) == 1, errorStr % string.join( platformsInDocs, '], [' )
    return platformsInDocs[0]

def filterFile(doc, reflectionsOnly):
    if doc.file.platform == 'NONE' or doc.file.product == 'NONE':
        return None
    
    if reflectionsOnly:
        fileText = open(doc.origfilename).read()
        if not util.hasReflectionDeclaration(fileText):
            return None
    
    return True

def calculateOutputFiles(DBObject, options, reflectionsOnly = False):
    ret = {}

    # Record which locations contain public files and which contain client.
    locationsContainingPublicFiles = []
    locationsContainingClientFiles = []

    foundHavokFile = False

    # sorted to ensure output is consistent between runs
    for doc in sorted((d for d in DBObject.getDocuments() if filterFile(d, reflectionsOnly)), key=lambda x: x.origfilename):
        # Either every file or no file should have TKBMS headers
        if doc.file.visibility != "CUSTOMER":
            foundHavokFile = True
        else:
            assert not foundHavokFile, "Project contains files with TKBMS headers, however %s does not" % (doc.origfilename)

        # Look for //+hk.ReflectedFile("<location>") in the file.
        moveList = [c.attributes.get("hk.ReflectedFile", None) for c in findClasses(doc.file._class)]
        moveSet = set([a for a in moveList if a])
        location = moveSet.pop() if len(moveSet) else ""
        assert len(moveSet) == 0, "ERROR: Found multiple hk.ReflectedFile attributes in %s." % doc.origfilename

        havokProductGroup = '_NONE' # The initial '_' is to put this group at the start when sorting later.

        # Trim quotes.
        if location.startswith('"') and location.endswith('"'):
            location = location[1:-1]

        fileIsArtificial = doc.file._class and all([c.override_name for c in doc.file._class])

        # Do some extra automated processing for Havok classes (only if hk.ReflectedFile hasn't been used to specify where the file belongs).
        if not location and not options.customer_build:

            # Classify the doc according to its Havok product.
            # If imported file is parsed first, productgroup stays at _NONE => assert
            if doc.file.product.find("ALL") == -1 and not fileIsArtificial:
                havokProductGroup = "+".join(sorted(doc.file.product.split("+")))

            # Serialize unit test classes separately.
            if re.search(r'\bUnitTest\b', doc.origfilename, re.I):
                location += "UnitTest"

            if doc.file.visibility == "INTERNAL":
                location += "Internal"
            if doc.file.visibility == "CLIENT":
                location += "Client"

        # Note what locations found public and client files are in.
        if doc.file.visibility == "PUBLIC" or doc.file.visibility == "CUSTOMER" and location not in locationsContainingPublicFiles:
            locationsContainingPublicFiles.append(location)
        elif doc.file.visibility == "CLIENT" and location not in locationsContainingClientFiles:
            locationsContainingClientFiles.append(location)

        if fileIsArtificial:
            havokProductGroup = "_ARTIFICIAL"

        # Build up the dictionary to return.
        if location not in ret:
            ret[location] = {}
        if havokProductGroup not in ret[location]:
            ret[location][havokProductGroup] = [doc]
        else:
            ret[location][havokProductGroup].append(doc)

    # Assert if the generated file will contain a mix of PUBLIC and CLIENT files.
    publicClientIntersection = [x for x in locationsContainingPublicFiles if x in locationsContainingClientFiles]
    if publicClientIntersection:
        print publicClientIntersection
        assert False, "ERROR: File contents for 'location' %s contain classes that are a mix of PUBLIC and CLIENT. This isn't allowed for packaging reasons." % publicClientIntersection

    # Do some post-processing to put generated template classes in the right place
    for loc in ret.keys():
        docList = ret[loc].get("_ARTIFICIAL", [])
        if docList:
            try:
                nonArtificialKey = [d for d in ret[loc].keys() if d != "_ARTIFICIAL"][0]
            except IndexError:
                # This will never happen for any project that is outputting files
                # but we need to handle it anyway
                nonArtificialKey = "_NONE"
                ret[loc][nonArtificialKey] = []
                
            ret[loc][nonArtificialKey].extend(docList)
            del ret[loc]["_ARTIFICIAL"]
    return ret

def outputPlugin(DBObject, project_dir, options):
    outputDict = calculateOutputFiles(DBObject, options, reflectionsOnly = True)
    if len(outputDict) != 0:
        outputDir = createReflectionsDir(project_dir, options)
        for (location, contentsDict) in outputDict.items():
            outputFiles = genOutputTextDictionary(location, project_dir, contentsDict, options)
            for (file, fileContents) in outputFiles.items():
                outputFile = os.path.join(outputDir, file)
                util.writeIfDifferent(fileContents, outputFile, options.force_rebuild, options.verbose)

#
# Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
# 
# Confidential Information of Havok.  (C) Copyright 1999-2014
# Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
# Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
# rights, and intellectual property rights in the Havok software remain in
# Havok and/or its suppliers.
# 
# Use of this software for evaluation purposes is subject to and indicates
# acceptance of the End User licence Agreement for this product. A copy of
# the license is included with this software and is also available at www.havok.com/tryhavok.
# 
#