/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

// This simple demo shows the steps required to use your custom implementation of the hkDebugDisplayHandler interface
// to display the hkDebugDisplay macros and viewer processes visual output in your own render engine

// N.B. To illustrate the simplest case we use a viewer which is not dependent on any Havok product.
// If you have licensed Havok Physics you may wish to change the #include USE_PHYSICS_EXAMPLE below, rebuild 
// and link with the Havok libraries as illustrated in the StepByStep/Physics example.
// Other products can be tested using similar code to register and set up the hkProcessContexts.

//#define USE_PHYSICS_EXAMPLE

// Base system
#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <Common/Base/Memory/Allocator/Malloc/hkMallocAllocator.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>

// Debug display, handler and process factory
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Visualize/hkDebugDisplayHandler.h>
#include <Common/Visualize/hkProcessFactory.h>
#include <Common/Visualize/hkProcess.h>
#include <Common/Visualize/hkProcessContext.h>

// Simple render system, replace with your own!
#include <StandAloneDemos/StepByStep/Visualize/RenderSystem.h>

// The simplest viewer of all which is product independent
#include <Common/Visualize/Process/hkDebugDisplayProcess.h>

// For printf
#include <Common/Base/Fwd/hkcstdio.h>

static void HK_CALL errorReport(const char* msg, void* userContext);




#ifdef USE_PHYSICS_EXAMPLE
// Physics world
#include <Physics2012/Dynamics/World/hkpWorld.h> 
#include <Physics2012/Dynamics/Entity/hkpRigidBody.h> 
#include <Physics2012/Collide/Dispatch/hkpAgentRegisterUtil.h> 
#include <Physics2012/Collide/Shape/Convex/Box/hkpBoxShape.h> 
#include <Physics2012/Utilities/Dynamics/Inertia/hkpInertiaTensorComputer.h> 

// Physics specific context and viewer processes
#include <Physics2012/Utilities/VisualDebugger/hkpPhysicsContext.h> 
#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpShapeDisplayViewer.h> 
#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpBroadphaseViewer.h> 
#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpActiveContactPointViewer.h> 
#endif



// Simple struct to hold all our generic scene representation
struct DemoWorldData
{
	void init();
	void destroy();
	void step(hkReal frameTime);

	hkTransform m_transform;

	// If using physics, make this more complex by also creating a physics world
#ifdef USE_PHYSICS_EXAMPLE
	hkpWorld* m_world;
#endif	
};


void getContexts(const DemoWorldData& DemoWorldData, hkArray<hkProcessContext*>& contexts);
void getViewerProcesses(hkArray<const char*>& processNames);



int HK_CALL main(int argc, const char** argv)
{
	// Initialize memory and base systems
	hkMemoryRouter* memoryRouter = hkMemoryInitUtil::initDefault( hkMallocAllocator::m_defaultMallocAllocator, hkMemorySystem::FrameInfo(1024 * 1024));	
	hkBaseSystem::init(memoryRouter, errorReport);	

	{
		// Initialization of our custom render system
		RenderSystem* yourRenderSystem = new RenderSystem();
		yourRenderSystem->init();
		yourRenderSystem->setupDefaultCameras(hkVector4(20, 20, 20), hkVector4(0, 0, 0), hkVector4(0, 1, 0));

		// Get display handler. Here you should produce an instance of your custom implementation of hkDebugDisplayHandler interface.
		
		hkDebugDisplayHandler* displayHandler = yourRenderSystem->getDisplayHandler();
		

		// Make a demo world (holds the Havok data for convenience)
		DemoWorldData demoWorldData;
		demoWorldData.init();
	
		hkArray<hkProcessContext*> contexts;	
		// The contexts you need depend on the Havok products you are using.
		getContexts(demoWorldData, contexts);

		// Register all processes we are interested in and pull out their names for creation below
		hkArray<const char*> processNames;
		getViewerProcesses(processNames);

		
		hkArray<hkProcess*> processes;
		for (int i = 0; i < processNames.getSize(); ++i)
		{
			hkProcess* process = hkProcessFactory::getInstance().createProcess(processNames[i], contexts);
			HK_ASSERT2(0x30675a6b, process, "Failed to create - did you register all the processes for the product(s) you are using?");
			process->m_inStream = HK_NULL;
			process->m_outStream = HK_NULL;
			process->m_processHandler = HK_NULL;
			process->m_displayHandler = displayHandler;	// Your displayhandler used here
			process->init();
			processes.pushBack(process);		
		}
		

		// Step the world
		{
			hkStopwatch frameTimer;
			frameTimer.start();

			// Update as if running at 60 frames per second.
			const hkReal frameTime = 1 / 60.0f;
			const hkReal numFrames = 600;
			for (int i = 0; i < numFrames; i++)
			{			
				demoWorldData.step(frameTime);

				// Draw some debug info through the debug display
				
				HK_DISPLAY_FRAME( demoWorldData.m_transform, 5);
				

				// Step viewer processes. Most of them do nothing here because they are driven by callbacks.
				
				for (int pi = 0; pi < processes.getSize(); ++pi)
				{
					processes[pi]->step(frameTime * 1000);
				}
				

				// Render frame
				while (frameTimer.getElapsedSeconds() < frameTime) { }
				yourRenderSystem->renderFrame();		

				frameTimer.reset();
				frameTimer.start();
			}
		}

		// Release resources in inverse order of creation
		
		for (int i = 0; i < processes.getSize(); ++i)
		{
			delete processes[i];
		}	

		// Delete contexts. Some of these may be referenced objects, in which case it
		// would be better (if you know their type) to use removeReference()
		for(int i = 0; i < contexts.getSize(); i++)
		{
			delete contexts[i];
		}
		

		demoWorldData.destroy();
		yourRenderSystem->removeReference();
	}
	hkBaseSystem::quit();
	hkMemoryInitUtil::quit();

	return 0;
}

#ifdef USE_PHYSICS_EXAMPLE

void DemoWorldData::init()
{
	m_transform.setIdentity();

	// Make a new physics world
	m_world = new hkpWorld(hkpWorldCinfo());

	// Register all collision agents	
	hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

	// Create a fixed ground body
	{		
		hkpRigidBodyCinfo bodyInfo;
		hkVector4 halfExtents; halfExtents.set(20, 0.5f, 20);
		bodyInfo.m_shape = new hkpBoxShape(halfExtents);
		bodyInfo.m_position.set(0, -0.5f, 0);
		bodyInfo.m_motionType = hkpMotion::MOTION_FIXED;
		hkpRigidBody* body = new hkpRigidBody(bodyInfo);
		bodyInfo.m_shape->removeReference();
		m_world->addEntity(body);
		body->removeReference();
	}

	// Create a dynamic falling body
	{		
		hkpRigidBodyCinfo bodyInfo;
		hkVector4 halfExtents; halfExtents.set(0.5f, 0.5f, 0.5f);
		bodyInfo.m_shape = new hkpBoxShape(halfExtents);
		hkMassProperties massProperties;
		hkpInertiaTensorComputer::computeShapeVolumeMassProperties(bodyInfo.m_shape, 1, massProperties);
		bodyInfo.setMassProperties(massProperties);
		bodyInfo.m_position.set(0, 10, 0);
		bodyInfo.m_angularVelocity.set(-2, -2, -2);
		bodyInfo.m_linearVelocity.set(5, -5, 5);
		hkpRigidBody* body = new hkpRigidBody(bodyInfo);
		bodyInfo.m_shape->removeReference();
		m_world->addEntity(body);
		body->removeReference();
	}	
}

void DemoWorldData::destroy()
{
	m_world->removeReference();
}

void DemoWorldData::step(hkReal frameTime)
{
	// Step physics world
	m_world->stepDeltaTime(frameTime);
}

void getContexts(const DemoWorldData& DemoWorldData, hkArray<hkProcessContext*>& contexts)
{
	// Create physics context and add world to it
	
	hkpPhysicsContext* physicsContext = new hkpPhysicsContext();
	physicsContext->addWorld(DemoWorldData.m_world);
	contexts.pushBack(physicsContext);	
	
}

void getViewerProcesses(hkArray<const char*>& processNames)
{
	// Push back the default display so we can see the low level hkDebugDisplay lines
	hkDebugDisplayProcess::registerProcess();
	processNames.pushBack(hkDebugDisplayProcess::getName());	

	// For a physics example, register the physics processes and push back some subset of viewer process names.
	// Note: You'll need to link with the physics libraries if you comment this code in.
	
	hkpPhysicsContext::registerAllPhysicsProcesses();	
	processNames.pushBack(hkpShapeDisplayViewer::getName());	
	processNames.pushBack(hkpBroadphaseViewer::getName());	
	processNames.pushBack(hkpActiveContactPointViewer::getName());
	
}


#else	

// Default, product independent demo: a spinning transform!

void DemoWorldData::init() 
{
	m_transform.setIdentity();
}

void DemoWorldData::destroy(){}

void DemoWorldData::step(hkReal frameTime)
{
	// Let's just spin this transform to show the scene is being stepped
	hkRotation dr;
	dr.setAxisAngle( hkVector4(0.0f, 1.0f, 0.0f), frameTime );
	hkRotation r = m_transform.getRotation();
	m_transform.getRotation().setMul( r, dr );
}

void getContexts(const DemoWorldData& DemoWorldData, hkArray<hkProcessContext*>& contexts)
{
	// These are product specific
}

void getViewerProcesses(hkArray<const char*>& processNames)
{
	hkDebugDisplayProcess::registerProcess();
	processNames.pushBack(hkDebugDisplayProcess::getName());	
}

#endif


static void HK_CALL errorReport(const char* msg, void* userContext)
{
	using namespace std;
	printf("%s", msg);
}


// Keycode
#include <Common/Base/keycode.cxx>

// Productfeatures
// We're using either no product, or only physics - we undef products even if the keycode is present so
// that we don't get the usual initialization for these products.
#undef HK_FEATURE_PRODUCT_AI
#undef HK_FEATURE_PRODUCT_ANIMATION
#undef HK_FEATURE_PRODUCT_CLOTH
#undef HK_FEATURE_PRODUCT_DESTRUCTION_2012
#undef HK_FEATURE_PRODUCT_DESTRUCTION
#undef HK_FEATURE_PRODUCT_BEHAVIOR
#ifndef USE_PHYSICS_EXAMPLE
	#undef HK_FEATURE_PRODUCT_PHYSICS_2012
#endif
#undef HK_FEATURE_PRODUCT_PHYSICS

// Also we're not using any of these.
#define HK_EXCLUDE_FEATURE_SerializeDeprecatedPre700
#define HK_EXCLUDE_FEATURE_RegisterVersionPatches
#define HK_EXCLUDE_FEATURE_RegisterReflectedClasses
#define HK_EXCLUDE_FEATURE_MemoryTracker

// We don't really need type/class info for this demo, so use this trick to make the linker happy
#include <Common/Serialize/Util/hkBuiltinTypeRegistry.h>
const hkTypeInfo* const hkBuiltinTypeRegistry::StaticLinkedTypeInfos[] = { HK_NULL };
const hkClass* const hkBuiltinTypeRegistry::StaticLinkedClasses[] = { HK_NULL };

#include <Common/Base/Config/hkProductFeatures.cxx>

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
