/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Common/Base/hkBase.h>
#include <StandAloneDemos/StepByStep/Visualize/RenderSystem.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Bridge/System/hkgSystem.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>
#include <Graphics/Common/Light/hkgLightManager.h>
#include <Graphics/Common/Texture/SkyBox/hkgSkyBox.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Common/Base/Fwd/hkcstdio.h>


hkBool RenderSystem::init()
{
	// Init graphics system
#if defined(HK_ATOM)
	hkgSystem::init("ogles2"); 
#else
	hkgSystem::init("d3d10");
#endif
	
	hkgWindow* window = hkgWindow::create();
	window->setShadowMapSize(0);
	HKG_WINDOW_CREATE_FLAG windowFlags = HKG_WINDOW_DEFAULT | HKG_WINDOW_WINDOWED;	
	bool initOk = window->initialize(windowFlags, HKG_WINDOW_BUF_COLOR | HKG_WINDOW_BUF_DEPTH32, 1280, 720, "Havok");

	// check that render is actually ok (shader drivern and platform has shaders for instance)
	bool shaderLibDriven = 
		(hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_OGLES2) || // GLSL 2
		(hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_DX11) ||   // HLSL 4.0 / 5.0
		(hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_DX9) ||   // HLSL 2.0 / 3.0
		(hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_OGLS);     // Cg   2.0

	if (shaderLibDriven)
	{
		if ( (!initOk) && ( hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_DX11) )
		{			
			HK_REPORT("HKG: DirectX not installed properly, or older OS, so trying older renderer (D3D9)\n");

			window->release();

			hkgSystem::quit();
			hkgSystem::init( "d3d9" );

			window = hkgWindow::create();
			window->setShadowMapSize(0);
			initOk = window->initialize(windowFlags, HKG_WINDOW_BUF_COLOR | HKG_WINDOW_BUF_DEPTH32, 1280, 720, "Havok");
		}

		if ( (!initOk) || (HKG_GET_RENDER_CAPS(window).m_vertexShaderMajorVersion < 2))
		{
			//printf("HKG: Have to change to a fixed function renderer as current hardware does not support shader model 2 or higher!\n");

			const char* fixedFunc = "ogles";

			window->release();

			hkgSystem::quit();
			hkgSystem::init( fixedFunc );

			window = hkgWindow::create();
			window->setShadowMapSize(0);
			initOk = window->initialize(windowFlags, HKG_WINDOW_BUF_COLOR | HKG_WINDOW_BUF_DEPTH32, 1280, 720, "Havok");
		}
	}

	if (hkgSystem::g_RendererType != hkgSystem::HKG_RENDERER_CONSOLE)
	{
#ifdef HK_DEBUG
		//printf("HKG: Using the %s renderer\n", hkgSystem::getRendererString() );
#endif
	}

	window->getContext()->lock();

	// don't allow viewport resizing
	window->setWantViewportBorders(false);
	window->setWantViewportResizeByMouse(false);	

	for(int i = 0; i < 2; ++i)
	{
		window->clearBuffers();
		window->swapBuffers();
	}

	m_window = window;
	m_displayWorld = hkgDisplayWorld::create();
	m_sceneConverter = new hkgSceneDataConverter(m_displayWorld, window->getContext());
	//env.m_reportingLevel = (hkDemoEnvironment::ReportingLevel) options.m_reportingLevel;
	m_displayHandler = new hkgDisplayHandler(m_displayWorld, m_window->getContext(), m_window);

	if( shaderLibDriven )
	{
		m_displayHandler->setShaderLib(m_sceneConverter->m_shaderLibrary);		
	}	

	setupLights();

	window->getContext()->unlock();	
	
	return true;
}


void RenderSystem::setupLights()
{
	// make some default lights
	hkgLightManager* lm = m_displayWorld->getLightManager();

	if (!lm)
	{
		lm = hkgLightManager::create(m_window->getContext());
		m_displayWorld->setLightManager( lm );
		lm->release();
		lm->lock();
	}
	else
	{
		lm->lock();
		// clear out the lights currently in the world.
		while( lm->getNumLights() > 0 )
		{
			hkgLight* l = lm->removeLight(0); // gives back reference
			l->release();
		}
	}

	// Background color
	float bg[4] = { 0.53f, 0.55f, 0.61f, 1 };
	m_window->setClearColor( bg );

	float v[4]; v[3] = 255;
	hkgLight* light;


	// the sun (direction downwards)
	{
		light = hkgLight::create();
		light->setType( HKG_LIGHT_DIRECTIONAL );
		v[0] = 255;
		v[1] = 255;
		v[2] = 255;
		v[0] /= 255; v[1] /= 255; v[2] /= 255;
		light->setDiffuse( v );
		light->setSpecular( v );
		v[0] = 0;
		v[1] = -1;
		v[2] = -0.5f;
		light->setDirection( v );
		v[0] = 0;
		v[1] = 1000;
		v[2] = 0;
		light->setPosition( v );
		light->setDesiredEnabledState( true );
		lm->addLight( light );

		// float shadowPlane[] = { 0,1,0,-0.01f };
		// light->addShadowPlane( shadowPlane );
		light->release();
	}

	hkgAabb areaOfInterest;
	areaOfInterest.m_max[0] = 10;
	areaOfInterest.m_max[1] = 10;
	areaOfInterest.m_max[2] = 10;
	areaOfInterest.m_min[0] = -10;
	areaOfInterest.m_min[1] = -10;
	areaOfInterest.m_min[2] = -10;
	setupFixedShadowFrustum(*light, areaOfInterest);

	// if se have shadow maps we only support on light by default (or else it looks dodge)
	if (m_window->getShadowMapSupport() == HKG_SHADOWMAP_NOSUPPORT)
	{
		// fill 1 - blue
		{
			light = hkgLight::create();
			light->setType( HKG_LIGHT_DIRECTIONAL );
			v[0] = 200;
			v[1] = 200;
			v[2] = 240;
			v[0] /= 255; v[1] /= 255; v[2] /= 255;
			light->setDiffuse( v );
			v[0] = 1;
			v[1] = 1;
			v[2] = 1;
			light->setDirection( v );
			v[0] = -1000;
			v[1] = -1000;
			v[2] = -1000;
			light->setPosition( v );
			light->setDesiredEnabledState( true );
			lm->addLight( light );
			light->release();
		}

		// fill 2 - yellow
		{
			light = hkgLight::create();
			light->setType( HKG_LIGHT_DIRECTIONAL );
			v[0] = 240;
			v[1] = 240;
			v[2] = 200;
			v[0] /= 255; v[1] /= 255; v[2] /= 255;
			light->setDiffuse( v );
			v[0] = -1;
			v[1] = 1;
			v[2] = -1;
			light->setDirection( v );
			v[0] = 1000;
			v[1] = -1000;
			v[2] = 1000;
			light->setPosition( v );
			light->setDesiredEnabledState( true );
			lm->addLight( light );
			light->release();
		}
	}

	lm->computeActiveSet( HKG_VEC3_ZERO );
	lm->unlock();
}


void RenderSystem::setupFixedShadowFrustum(const hkgLight& light, const hkgAabb& areaOfInterest, float extraNear, float extraFar, int numSplits, int preferedUpAxis)
{
	hkgCamera* lightCam = hkgCamera::createFixedShadowFrustumCamera( light, areaOfInterest, true, extraNear, extraFar, preferedUpAxis );

	HKG_SHADOWMAP_SUPPORT shadowSupport = m_window->getShadowMapSupport();
	if ( (numSplits > 0) && (shadowSupport == HKG_SHADOWMAP_VSM))
	{
		m_window->setShadowMapSplits(numSplits); // > 0 and you are requesting PSVSM (if the platforms supports VSM that is)
		m_window->setShadowMapMode(HKG_SHADOWMAP_MODE_PSVSM, lightCam);
	}
	else
	{
		if ((numSplits > 0) && (shadowSupport != HKG_SHADOWMAP_NOSUPPORT))
		{
			static int once = 0;
			if (once == 0)
			{
				HK_WARN_ALWAYS(0x0, "The demo is requesting PSVSM shadows, but VSM is not supported, so just reverting to normal single map, fixed projection.");
				once = 1;
			}
		}

		m_window->setShadowMapMode( HKG_SHADOWMAP_MODE_FIXED, lightCam);
	}

	lightCam->removeReference();
}


void RenderSystem::setupDefaultCameras(const hkVector4& from, const hkVector4& to, const hkVector4& up, const hkReal nearPlane, const hkReal farPlane, bool rightHanded) const
{
	hkgWindow* w = m_window;
	if ( !w )
	{
		return;
	}

	m_window->getContext()->lock();
	for(int i = 0; i < w->getNumViewports(); ++i)
	{
		hkgCamera* c = w->getViewport(i)->getCamera();

		float upN[3];
		up.store<3,HK_IO_NATIVE_ALIGNED>(&upN[0]);
		hkgVec3Normalize(upN);
		// set up camera
		c->setFrom(&from(0));
		c->setTo(&to(0));
		c->setUp(upN);
		c->setFar(float(farPlane));
		c->setNear(float(nearPlane));
		c->orthogonalize();
		c->computeModelView();
		c->computeProjection();
		c->setHandednessMode(HKG_CAMERA_HANDEDNESS_MODE( rightHanded ? HKG_CAMERA_HANDEDNESS_RIGHT : HKG_CAMERA_HANDEDNESS_LEFT) );

		w->getViewport(i)->setWorldUp( upN );
	}
	m_window->getContext()->unlock();
}


hkDebugDisplayHandler* RenderSystem::getDisplayHandler()
{
	return static_cast<hkDebugDisplayHandler*>(m_displayHandler);
}


bool RenderSystem::startFrame()
{
	HKG_TIMER_BEGIN_LIST("_Render", "UpdatePads");

	hkgWindow* window = m_window;

	HKG_TIMER_SPLIT_LIST("ClearBuffers");

	window->getContext()->lock();
	bool graphicsWindowOK = window->clearBuffers();		
	window->getContext()->unlock();

	HKG_TIMER_END_LIST();

	return graphicsWindowOK;
}


void RenderSystem::renderFrame()
{
	startFrame();
	
	HKG_TIMER_BEGIN_LIST("_Render", "render");

	hkgWindow* window = m_window;

	if (!window || (window->getWidth() == 0) || (window->getHeight() == 0) )
	{
		HKG_TIMER_BEGIN("SwapBuffers", HK_NULL);
		if (window)
		{
			window->getContext()->lock();
			window->swapBuffers();
			window->getContext()->unlock();
		}
		HKG_TIMER_END();

		HKG_TIMER_END_LIST(); // render
		return; // nothing to render too..
	}

	hkgDisplayContext* ctx = window->getContext();
	ctx->lock();

	do 
	{
		hkgViewport* masterView = window->getCurrentViewport();

		// only reason there won't be a displayWorld is if the demo is doing the rendering itself.
		hkgViewport* v = window->getViewport(0);		

		HKG_TIMER_SPLIT_LIST("SetViewport");

		v->setAsCurrent(ctx);

		sendVdbCamera(v);

		if (v->getSkyBox())
		{
			HKG_TIMER_SPLIT_LIST("SkyBox");
			v->getSkyBox()->render(ctx, v->getCamera());
		}

		hkgDisplayWorld* dw = m_displayWorld;
		/*if (dw)
		{
			dw->setFrameTime(m_frameTimer.getLastFrameTime());
		}*/
		HKG_TIMER_SPLIT_LIST("DemoPreRenderWorld");

		/*if (demo)
		{
			demo->preRenderDisplayWorld( v );
		}*/

		HKG_TIMER_SPLIT_LIST("DisplayWorld");
		if (dw)
		{
			// can't alter the world in the middle of a render pass, so it will lock itself
			dw->render(ctx, true, false); // culled with shadows (if any setup)
		}

		HKG_TIMER_SPLIT_LIST("DemoPostRenderWorld");
		/*if (demo)
		{
			demo->postRenderDisplayWorld( v );
		}*/


		HKG_TIMER_SPLIT_LIST("DrawImmediate");
		hkgDisplayHandler* dh = m_displayHandler;
		if (dh)
		{
			dh->drawImmediate();
		}	

		HKG_TIMER_SPLIT_LIST("PostEffects");

		window->applyPostEffects();

		HKG_TIMER_SPLIT_LIST("Final Pass Objects");
		{
			// only reason there won't be a displayWorld is if the demo is doing the rendering itself.				
			hkgDisplayWorld* dw2 = m_displayWorld;
			if (dw2)
			{
				window->getViewport(0)->setAsCurrent(ctx);
				dw2->finalRender(ctx, true /* frustum cull */);
			}
		}

		masterView->setAsCurrent( ctx );

		HKG_TIMER_SPLIT_LIST("DemoPostRenderWindow");
		/*if (demo)
		{
			demo->postRenderWindow(window);
		}*/
		

		// We commonly use 'swap discard' (better for SLi/CrossFire etc) as the swap effect, so 
		// that means you neeed tyo save the RT before Swap or its contents may be erased
		// A side effect of this is that the image will not have the Havok logo on the lower corner
				
		HKG_TIMER_SPLIT_LIST("SwapBuffers");

		window->swapBuffers();
		
		window->toggleCurrentStereoTargetEye();

	}  while (window->getCurrentStereoMode() == HKG_STEREO_RIGHT_EYE);

	window->waitForCompletion();	

	ctx->unlock();

	HKG_TIMER_END_LIST(); // render

	tickFrame();

	m_displayHandler->clear();
}


void RenderSystem::sendVdbCamera(hkgViewport* v)
{
	hkgCamera* c = v->getCamera();

	float floatFrom[3]; c->getFrom(floatFrom);
	float floatTo[3]; c->getTo(floatTo);
	float floatUp[3]; c->getUp(floatUp);

	hkVector4 from; from.set( floatFrom[0], floatFrom[1], floatFrom[2] );
	hkVector4 to;   to.set( floatTo[0], floatTo[1], floatTo[2] );
	hkVector4 up;   up.set( floatUp[0], floatUp[1], floatUp[2] );

	HK_UPDATE_CAMERA(from, to, up, c->getNear(), c->getFar(), c->getFOV(), "Demo Framework");	
}


void RenderSystem::tickFrame()
{
	hkgWindow* window = m_window;
	if (!window || (window->getWidth() == 0) || (window->getHeight() == 0) )
	{
		return;
	}
	HKG_TIMER_BEGIN("_Render", HK_NULL);
	HKG_TIMER_BEGIN("tick", HK_NULL);

	hkgDisplayContext* ctx = window->getContext();
	ctx->lock();

	int frameNum = ctx->advanceFrame();

	// Primary world:
	if (m_displayWorld)
	{
		m_displayWorld->advanceToFrame(frameNum, false, m_window);
		m_window->getViewport(0)->getCamera()->advanceToFrame(frameNum);
	}	

	ctx->unlock();

	HKG_TIMER_END();
	HKG_TIMER_END();
}


void RenderSystem::release()
{
	if (m_window == HK_NULL)
	{
		return;
	}

	hkgDisplayContext* ctx = m_window->getContext();
	ctx->lock();
	{
		m_window->setWindowResizeFunction( HK_NULL, HK_NULL );
		m_window->setFullscreen(false, m_window->getWidth(), m_window->getHeight());
		
		delete m_displayHandler;
		m_sceneConverter->release();
		m_displayWorld->release();		
	}
	ctx->unlock();
	m_window->release();

	// All hkg objects should be gone by now:
	hkgSystem::quit();

	m_window = HK_NULL;
}


RenderSystem::~RenderSystem()
{
	release();
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
