/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

// This simple console demo demonstrates how to load and save a Havok object using 
// the serialization system.

#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <Common/Base/Memory/Allocator/Malloc/hkMallocAllocator.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>

#include <Common/Base/Fwd/hkcstdio.h>

// Geometry
#include <Common/Base/Types/Geometry/hkGeometry.h>

// Serialize
#include <Common/Serialize/Util/hkSerializeUtil.h>

void PlatformInit();
void PlatformFileSystemInit();

static void HK_CALL errorReport(const char* msg, void* userContext)
{
	using namespace std;
	printf("%s", msg);
}

// Define a useful macro for this demo - it allow us to detect a failure, print a message, and return early
#define RETURN_FAIL_IF(COND, MSG) \
	HK_MULTILINE_MACRO_BEGIN \
		if(COND) { HK_ERROR(0x53a6a026, MSG); return 1; } \
	HK_MULTILINE_MACRO_END

#if defined(HK_PLATFORM_WINRT) || defined(HK_PLATFORM_DURANGO)
[Platform::MTAThread]
int main(Platform::Array<Platform::String^>^ args)
#elif defined( HK_PLATFORM_TIZEN )
int hkMain(int argc, const char** argv)
#else
int HK_CALL main(int argc, const char** argv)
#endif
{
	// Perfrom platform specific initialization for this demo - you should already have something similar in your own code.
	PlatformInit();

	// Need to have memory allocated for the solver. Allocate 1mb for it.
	hkMemoryRouter* memoryRouter = hkMemoryInitUtil::initDefault( hkMallocAllocator::m_defaultMallocAllocator, hkMemorySystem::FrameInfo(1024 * 1024) );
	hkBaseSystem::init( memoryRouter, errorReport );

	// Set up platform-specific file system info.
	PlatformFileSystemInit();

	{
		// Create a hkGeometry object - not usually used in the Havok runtime, but a simple enough structure
		// to illustrate the serialization functionality.
		hkGeometry* geometry;
		{
			geometry = new hkGeometry;

			// Make a 'box' geometry of dimensions 2x4x6
			hkVector4 halfExtents( 1.0f, 2.0f, 3.0f);

			hkVector4* v = geometry->m_vertices.expandBy(8);
			v[0].set( -halfExtents(0),  halfExtents(1),  halfExtents(2));
			v[1].set(  halfExtents(0),  halfExtents(1),  halfExtents(2));
			v[2].set(  halfExtents(0), -halfExtents(1),  halfExtents(2));
			v[3].set( -halfExtents(0), -halfExtents(1),  halfExtents(2));
			v[4].set( -halfExtents(0),  halfExtents(1), -halfExtents(2));
			v[5].set(  halfExtents(0),  halfExtents(1), -halfExtents(2));
			v[6].set(  halfExtents(0), -halfExtents(1), -halfExtents(2));
			v[7].set( -halfExtents(0), -halfExtents(1), -halfExtents(2));

			geometry->m_triangles.expandBy(1)->set(3, 2, 1);
			geometry->m_triangles.expandBy(1)->set(3, 1, 0);
			geometry->m_triangles.expandBy(1)->set(6, 7, 4);
			geometry->m_triangles.expandBy(1)->set(6, 4, 5);
			geometry->m_triangles.expandBy(1)->set(4, 7, 3);
			geometry->m_triangles.expandBy(1)->set(4, 3, 0);
			geometry->m_triangles.expandBy(1)->set(2, 6, 5);
			geometry->m_triangles.expandBy(1)->set(2, 5, 1);
			geometry->m_triangles.expandBy(1)->set(7, 6, 2);
			geometry->m_triangles.expandBy(1)->set(7, 2, 3);
			geometry->m_triangles.expandBy(1)->set(1, 5, 4);
			geometry->m_triangles.expandBy(1)->set(1, 4, 0);
		}
		
		// Write out an xml tagfile
		{
			hkOstream stream("geometry_xml_tagfile.xml");
			
			hkResult res = hkSerializeUtil::saveTagfile( geometry, hkGeometryClass, stream.getStreamWriter(), HK_NULL, hkSerializeUtil::SAVE_TEXT_FORMAT);
			RETURN_FAIL_IF( res != HK_SUCCESS, "Failed to save XML");
		}

		// Read back a serialized file
		{
			hkIstream stream("geometry_xml_tagfile.xml");

			hkResource* resource = hkSerializeUtil::load(stream.getStreamReader());
			hkBool32 failed = true;
			if (resource)
			{
				// Get the contained geometry
				hkGeometry* readGeometry = resource->getContents<hkGeometry>();
				
				// Check to see if the last vertex is the same, as a simple check to 
				// see if serialization has worked correctly.
				failed = ! readGeometry->m_vertices[7].equals3( readGeometry->m_vertices[7] );

				// The readGeometry is held inside the resource - so it doesn't have to be released explicitly
				// as it will be destroyed when the resource is destroyed.
				// Resource is no longer needed
				resource->removeReference();
			}
			RETURN_FAIL_IF( failed, "Failed loading XML");
		}

		// Write out an binary file
		{
			hkOstream stream("geometry.hkx");
			
			hkResult res = hkSerializeUtil::save( geometry, stream.getStreamWriter() );
			RETURN_FAIL_IF(res != HK_SUCCESS, "Failed to save binary");
		}

		// Read back a serialized file
		{
			hkIstream stream("geometry.hkx");

			hkResource* resource = hkSerializeUtil::load(stream.getStreamReader());
			hkBool32 failed = true;
			if (resource)
			{
				// Get the contained Geometry
				hkGeometry* readGeometry = resource->getContents<hkGeometry>();
				
				// Check to see if the last vertex is the same, as a simple check to 
				// see if serialization has worked correctly.
				failed = ! readGeometry->m_vertices[7].equals3( readGeometry->m_vertices[7] );

				// The readGeometry is held inside the resource - so it doesn't have to be released explicitly
				// as it will be destroyed when the resource is destroyed.
				// Resource is no longer needed
				resource->removeReference();
			}
			RETURN_FAIL_IF( failed, "Failed loading binary");
		}

		// No longer need the ref of geom as we are done
		delete geometry;
	}

	hkBaseSystem::quit();
	hkMemoryInitUtil::quit();

	return 0;
}

// [id=keycode]
#include <Common/Base/keycode.cxx>

// [id=productfeatures]
// We're not using anything product specific yet. We undef these so we don't get the usual
// product initialization for the products.
#undef HK_FEATURE_PRODUCT_AI
#undef HK_FEATURE_PRODUCT_ANIMATION
#undef HK_FEATURE_PRODUCT_CLOTH
#undef HK_FEATURE_PRODUCT_DESTRUCTION_2012
#undef HK_FEATURE_PRODUCT_DESTRUCTION
#undef HK_FEATURE_PRODUCT_BEHAVIOR
#undef HK_FEATURE_PRODUCT_PHYSICS_2012
#undef HK_FEATURE_PRODUCT_SIMULATION
#undef HK_FEATURE_PRODUCT_PHYSICS

// We are using serialization, so we need ReflectedClasses.
// The objects are being saved and then loaded immediately so we know the version of the saved data is the same 
// as the version the application is linked with. Because of this we don't need RegisterVersionPatches or SerializeDeprecatedPre700.
// If the demo was reading content saved from a previous version of the Havok content tools (common in real world Applications) 
// RegisterVersionPatches and perhaps SerializeDeprecatedPre700 are needed.
#define HK_EXCLUDE_FEATURE_SerializeDeprecatedPre700
// We can also restrict the compatibility to files created with the current version only using HK_SERIALIZE_MIN_COMPATIBLE_VERSION.
// If we wanted to have compatibility with at most version 650b1 we could have used something like:
// #define HK_SERIALIZE_MIN_COMPATIBLE_VERSION 650b1.
#define HK_SERIALIZE_MIN_COMPATIBLE_VERSION Current

#define HK_EXCLUDE_FEATURE_RegisterVersionPatches
//#define HK_EXCLUDE_FEATURE_RegisterReflectedClasses
#define HK_EXCLUDE_FEATURE_MemoryTracker

// We also need to exclude the other common libraries referenced in Source\Common\Serialize\Classlist\hkCommonClasses.h
// You may be linking these libraries in your application, in which case you will not need these #defines.
#define HK_EXCLUDE_LIBRARY_hkcdCollide
#define HK_EXCLUDE_LIBRARY_hkcdInternal
#define HK_EXCLUDE_LIBRARY_hkSceneData
#define HK_EXCLUDE_LIBRARY_hkGeometryUtilities

#include <Common/Base/Config/hkProductFeatures.cxx>

// Platform specific initialization

#include <Common/Base/System/Init/PlatformInit.cxx>

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
