/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

// Math and base include
#include <Common/Base/hkBase.h>
#include <Common/Base/System/hkBaseSystem.h>
#include <Common/Base/System/Error/hkDefaultError.h>
#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Base/Memory/Allocator/Malloc/hkMallocAllocator.h>
#include <Common/Base/System/Hardware/hkHardwareInfo.h>

// Dynamics includes
#include <Physics2012/Collide/hkpCollide.h>										
#include <Physics2012/Collide/Agent/ConvexAgent/SphereBox/hkpSphereBoxAgent.h>	
#include <Physics2012/Collide/Shape/Convex/Box/hkpBoxShape.h>					
#include <Physics2012/Collide/Shape/Convex/Sphere/hkpSphereShape.h>				
#include <Physics2012/Collide/Dispatch/hkpAgentRegisterUtil.h>					

#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastInput.h>			
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastOutput.h>			

#include <Physics2012/Dynamics/World/hkpWorld.h>								
#include <Physics2012/Dynamics/Entity/hkpRigidBody.h>							
#include <Physics2012/Utilities/Dynamics/Inertia/hkpInertiaTensorComputer.h>	

#include <Common/Base/Thread/Pool/hkCpuThreadPool.h>
#if defined(HK_PLATFORM_HAS_SPU)
#	include <Common/Base/Thread/Pool/hkSpuThreadPool.h>
#endif
#include <Common/Base/Thread/JobQueue/hkJobQueue.h>

// Visual Debugger includes
#include <Common/Visualize/hkVisualDebugger.h>
#include <Physics2012/Utilities/VisualDebugger/hkpPhysicsContext.h>				

// Keycode
#include <Common/Base/keycode.cxx>

#if !defined HK_FEATURE_PRODUCT_PHYSICS_2012
#error Physics is needed to build this demo. It is included in the common package for reference only.
#endif

// This excludes libraries that are not going to be linked
// from the project configuration, even if the keycodes are
// present

#undef HK_FEATURE_PRODUCT_AI
#undef HK_FEATURE_PRODUCT_ANIMATION
#undef HK_FEATURE_PRODUCT_CLOTH
#undef HK_FEATURE_PRODUCT_DESTRUCTION_2012
#undef HK_FEATURE_PRODUCT_BEHAVIOR
#undef HK_FEATURE_PRODUCT_PHYSICS
#define HK_FEATURE_REFLECTION_PHYSICS_2012
#define HK_CLASSES_FILE <Common/Serialize/Classlist/hkClasses.h>
#define HK_EXCLUDE_FEATURE_MemoryTracker
#define HK_EXCLUDE_FEATURE_SerializeDeprecatedPre700
#define HK_EXCLUDE_FEATURE_RegisterVersionPatches
#define HK_EXCLUDE_LIBRARY_hkGeometryUtilities
#define HK_EXCLUDE_FEATURE_hkndAssetProcessing
#define HK_EXCLUDE_FEATURE_hkndDebrisFracture_execute
#define HK_EXCLUDE_FEATURE_hkndFxDebrisFracture_execute
#include <Common/Base/Config/hkProductFeatures.cxx>

#include <stdio.h>
static void HK_CALL errorReport(const char* msg, void* userArgGivenToInit)
{
	printf("%s", msg);
}



//
// Choose the correct libs for Intel sdk
//
#include <ittnotify.h>

// Debug or not?
#if _MSC_VER < 1500 // 2005
	#ifdef _DEBUG
		// DLL ver of CRT or not?
		#ifdef _DLL
			#pragma comment(lib, "gpasdk_dd_2005.lib")
		#else
			#pragma comment(lib, "gpasdk_sd_2005.lib")
		#endif
	#else // release build
		#ifdef _DLL
			#pragma comment(lib, "gpasdk_dr_2005.lib")
		#else
			#pragma comment(lib, "gpasdk_sr_2005.lib")
		#endif
	#endif
#elif _MSC_VER < 1600 // 2008
	#ifdef _DEBUG
		// DLL ver of CRT or not?
		#ifdef _DLL
			#pragma comment(lib, "gpasdk_dd_2008.lib")
		#else
			#pragma comment(lib, "gpasdk_sd_2008.lib")
		#endif
	#else // release build
		#ifdef _DLL
			#pragma comment(lib, "gpasdk_dr_2008.lib")
		#else
			#pragma comment(lib, "gpasdk_sr_2008.lib")
		#endif
	#endif
#else // VS 2010
	#ifdef _DEBUG
		// DLL ver of CRT or not?
		#ifdef _DLL
			#pragma comment(lib, "gpasdk_dd_2010.lib")
		#else
			#pragma comment(lib, "gpasdk_sd_2010.lib")
		#endif
	#else // release build
		#ifdef _DLL
			#pragma comment(lib, "gpasdk_dr_2010.lib")
		#else
			#pragma comment(lib, "gpasdk_sr_2010.lib")
		#endif
	#endif
#endif

class GpaJobProfiler : public hkExternalJobProfiler
{
	public:

		HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, GpaJobProfiler);

		GpaJobProfiler( const char* name )
		{
			m_currentStepNumber = 0;

			m_domain = __itt_domain_createA(name);

			m_jobSubTypeVar = __itt_string_handle_createA("JobSubType");
			m_stepStr = __itt_string_handle_createA("Havok Step");

			m_jobTypeStrings.setSize(HK_JOB_TYPE_USER_0 + 1);
			m_jobTypeStrings[HK_JOB_TYPE_DYNAMICS] = __itt_string_handle_createA("Dynamics");
			m_jobTypeStrings[HK_JOB_TYPE_COLLIDE] = __itt_string_handle_createA("Collide");
			m_jobTypeStrings[HK_JOB_TYPE_COLLISION_QUERY] = __itt_string_handle_createA("CollisionQuery");
			m_jobTypeStrings[HK_JOB_TYPE_RAYCAST_QUERY] = __itt_string_handle_createA("RayCastQuery");
			m_jobTypeStrings[HK_JOB_TYPE_ANIMATION_SAMPLE_AND_COMBINE] = __itt_string_handle_createA("AnimSampleAndComb");
			m_jobTypeStrings[HK_JOB_TYPE_ANIMATION_SAMPLE_AND_BLEND] = __itt_string_handle_createA("AnimSampleAndBlend");
			m_jobTypeStrings[HK_JOB_TYPE_ANIMATION_MAPPING] = __itt_string_handle_createA("AnimMapping");
			m_jobTypeStrings[HK_JOB_TYPE_BEHAVIOR] = __itt_string_handle_createA("Behavior");
			m_jobTypeStrings[HK_JOB_TYPE_CLOTH] = __itt_string_handle_createA("Cloth");
			m_jobTypeStrings[HK_JOB_TYPE_DESTRUCTION] = __itt_string_handle_createA("Destruction");
			m_jobTypeStrings[HK_JOB_TYPE_UNIT_TEST] = __itt_string_handle_createA("UnitTest");
			m_jobTypeStrings[HK_JOB_TYPE_CHARACTER_PROXY] = __itt_string_handle_createA("CharProxy");
			m_jobTypeStrings[HK_JOB_TYPE_VEHICLE] = __itt_string_handle_createA("Vehicle");
			m_jobTypeStrings[HK_JOB_TYPE_COLLIDE_STATIC_COMPOUND] = __itt_string_handle_createA("CollideStaticCompound");
			m_jobTypeStrings[HK_JOB_TYPE_USER_0] = __itt_string_handle_createA("User");
		}

		virtual ~GpaJobProfiler(){}

		inline __itt_string_handle* getTaskString( hkJobType t )
		{
			if (t < HK_JOB_TYPE_USER_0)
				return m_jobTypeStrings[(int)t];
			else
				return m_jobTypeStrings[(int)HK_JOB_TYPE_USER_0];
		}

		virtual void onStartJob( hkJobType t, hkUint32 jobSubType )
		{
			__itt_task_begin( m_domain, __itt_null, __itt_null, getTaskString(t) );
			__itt_relation_add_to_current(m_domain, __itt_relation_is_child_of, m_currentStepID );

		}

		virtual void onEndJob( hkJobType t )
		{
			__itt_task_end(m_domain);
		}

		inline void startStep()
		{
			m_currentStepID = __itt_id_make((void*)&m_currentStepNumber, 0);
			__itt_id_create(m_domain, m_currentStepID);
			__itt_task_group(m_domain, m_currentStepID, __itt_null, m_stepStr);
		}

		inline void endStep()
		{
			__itt_id_destroy(m_domain, m_currentStepID);
			++m_currentStepNumber;
		}

		__itt_domain* m_domain;
		hkArray<__itt_string_handle*> m_jobTypeStrings;
		__itt_string_handle* m_jobSubTypeVar;

		__itt_string_handle* m_stepStr;
		hkInt32 m_currentStepNumber;
		__itt_id m_currentStepID;
};

HK_COMPILE_TIME_ASSERT(HK_JOB_TYPE_DYNAMICS == 0);
HK_COMPILE_TIME_ASSERT(HK_JOB_TYPE_HAVOK_MAX == 14); // If this fails then there are new job types to add to the above table




//
// Console app control, so we can loop and exit cleanly on quit
//
bool g_runMain = true;
HANDLE g_mainThread;
BOOL WINAPI HandlerCtrlEvent(__in DWORD dwCtrlType)
{
	if (dwCtrlType == CTRL_CLOSE_EVENT || dwCtrlType == CTRL_LOGOFF_EVENT || dwCtrlType == CTRL_SHUTDOWN_EVENT)
	{
		g_runMain = false;
		WaitForSingleObject(g_mainThread, INFINITE);
	}

	return FALSE;
}

//
// Forward declarations
//
void setupPhysics(hkpWorld* physicsWorld);
hkpRigidBody* g_ball;


int HK_CALL main(int argc, const char** argv)
{
	g_mainThread = GetCurrentThread();
	SetConsoleCtrlHandler(HandlerCtrlEvent, TRUE);


	//
	// Initialize the base system including our memory system
	//
	// Allocate 0.5MB of physics solver buffer.
	hkMemoryRouter* memoryRouter = hkMemoryInitUtil::initDefault( hkMallocAllocator::m_defaultMallocAllocator, hkMemorySystem::FrameInfo( 500* 1024 ) );
	hkBaseSystem::init( memoryRouter, errorReport );
	{
		//
		// Initialize the multi-threading classes, hkJobQueue, and hkThreadPool
		//

		// They can be used for all Havok multithreading tasks. In this exmaple we only show how to use
		// them for physics, but you can reference other multithreading demos in the demo framework
		// to see how to multithread other products. The model of usage is the same as for physics.
		// The hkThreadpool has a specified number of threads that can run Havok jobs.  These can work
		// alongside the main thread to perform any Havok multi-threadable computations.
		// The model for running Havok tasks in Spus and in auxilary threads is identical.  It is encapsulated in the
		// class hkThreadPool.  On PlayStation(R)3 we initialize the SPU version of this class, which is simply a SPURS taskset.
		// On other multi-threaded platforms we initialize the CPU version of this class, hkCpuThreadPool, which creates a pool of threads
		// that run in exactly the same way.  On the PlayStation(R)3 we could also create a hkCpuThreadPool.  However, it is only
		// necessary (and advisable) to use one Havok PPU thread for maximum efficiency. In this case we simply use this main thread
		// for this purpose, and so do not create a hkCpuThreadPool.
		hkThreadPool* threadPool;

		// We can cap the number of threads used - here we use the maximum for whatever multithreaded platform we are running on. This variable is
		// set in the following code sections.
		int totalNumThreadsUsed;

		// Get the number of physical threads available on the system
		totalNumThreadsUsed = hkHardwareInfo::getNumHardwareThreads();

		// We use one less than this for our thread pool, because we must also use this thread for our simulation
		hkCpuThreadPoolCinfo threadPoolCinfo;
		threadPoolCinfo.m_numThreads = totalNumThreadsUsed - 1;

		// This line enables timers collection, by allocating 200 Kb per thread.  If you leave this at its default (0),
		// timer collection will not be enabled.
		threadPoolCinfo.m_timerBufferPerThreadAllocation = 200000;
		threadPool = new hkCpuThreadPool( threadPoolCinfo );


		// We also need to create a Job queue. This job queue will be used by all Havok modules to run multithreaded work.
		// Here we only use it for physics.
		hkJobQueueCinfo info;
		info.m_jobQueueHwSetup.m_numCpuThreads = totalNumThreadsUsed;
		hkJobQueue* jobQueue = new hkJobQueue(info);


		// INTEL GPA:
		// Make a external job profiler for Havok
		GpaJobProfiler* gpaProfiler = new GpaJobProfiler("Havok.ProfilerExample");
		jobQueue->setExternalProfiler( gpaProfiler  );

		//
		// Enable monitors for this thread.
		//

		// Monitors have been enabled for thread pool threads already (see above comment).
		hkMonitorStream::getInstance().resize(200000);



		//
		// <PHYSICS-ONLY>: Create the physics world.
		// At this point you would initialize any other Havok modules you are using.
		//
		hkpWorld* physicsWorld;
		{
			// The world cinfo contains global simulation parameters, including gravity, solver settings etc.
			hkpWorldCinfo worldInfo;

			// Set the simulation type of the world to multi-threaded.
			worldInfo.m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_MULTITHREADED;

			// Flag objects that fall "out of the world" to be automatically removed - just necessary for this physics scene
			worldInfo.m_broadPhaseBorderBehaviour = hkpWorldCinfo::BROADPHASE_BORDER_REMOVE_ENTITY;

			physicsWorld = new hkpWorld(worldInfo);

			// Disable deactivation, so that you can view timers in the VDB. This should not be done in your game.
			physicsWorld->m_wantDeactivation = false;


			// When the simulation type is SIMULATION_TYPE_MULTITHREADED, in the debug build, the sdk performs checks
			// to make sure only one thread is modifying the world at once to prevent multithreaded bugs. Each thread
			// must call markForRead / markForWrite before it modifies the world to enable these checks.
			physicsWorld->markForWrite();


			// Register all collision agents, even though only box - box will be used in this particular example.
			// It's important to register collision agents before adding any entities to the world.
			hkpAgentRegisterUtil::registerAllAgents( physicsWorld->getCollisionDispatcher() );

			// We need to register all modules we will be running multi-threaded with the job queue
			physicsWorld->registerWithJobQueue( jobQueue );

			// Create all the physics rigid bodies
			setupPhysics( physicsWorld );
		}


		//
		// Initialize the VDB
		//
		hkArray<hkProcessContext*> contexts;


		// <PHYSICS-ONLY>: Register physics specific visual debugger processes
		// By default the VDB will show debug points and lines, however some products such as physics and cloth have additional viewers
		// that can show geometries etc and can be enabled and disabled by the VDB app.
		hkpPhysicsContext* context;
		{
			// The visual debugger so we can connect remotely to the simulation
			// The context must exist beyond the use of the VDB instance, and you can make
			// whatever contexts you like for your own viewer types.
			context = new hkpPhysicsContext();
			hkpPhysicsContext::registerAllPhysicsProcesses(); // all the physics viewers
			context->addWorld(physicsWorld); // add the physics world so the viewers can see it
			contexts.pushBack(context);

			// Now we have finished modifying the world, release our write marker.
			physicsWorld->unmarkForWrite();
		}

		hkVisualDebugger* vdb = new hkVisualDebugger(contexts);
		vdb->serve();


		//
		// Simulate the world for 1 minute.
		//


		// Take fixed time steps of 1/60th of a second.
		// This works well if your game runs solidly at 60Hz. If your game runs at 30Hz
		// you can take either 2 60Hz steps or 1 30Hz step. Note that at lower frequencies (i.e. 30 Hz)
		// more bullet through paper issues appear, and constraints will not be as stiff.
		// If you run at variable frame rate, or are likely to drop frames, you can consider
		// running your physics for a variable number of steps based on the system clock (i.e. last frame time).
		// Please refer to the user guide section on time stepping for a full treatment of this issue.

		// A stopwatch for waiting until the real time has passed
		hkStopwatch stopWatch;
		stopWatch.start();
		hkReal lastTime = stopWatch.getElapsedSeconds();

		hkReal timestep = 1.f / 60.f;
		int i = 0;
		while ( g_runMain )
		{
			// <PHYSICS-ONLY>:
			// Step the physics world. This single call steps using this thread and all threads
			// in the threadPool. For other products you add jobs, call process all jobs and wait for completion.
			// See the multithreading chapter in the user guide for details

			{
				gpaProfiler->startStep();
					physicsWorld->stepMultithreaded( jobQueue, threadPool, timestep );
				gpaProfiler->endStep();
			}

			// Step the visual debugger. We first synchronize the timer data
			context->syncTimers( threadPool );
			vdb->step();

			// Clear accumulated timer data in this thread and all slave threads
			hkMonitorStream::getInstance().reset();
			threadPool->clearTimerData();


			// <PHYSICS-ONLY>:  Display the sphereRigidBody position to the console every second
			if (++i % 60 == 0)
			{
				hkVector4 pos = g_ball->getPosition();
				printf("[%f,%f,%f]\n", pos(0), pos(1), pos(2));
				i = 0;
			}

			// Pause until the actual time has passed if you like to pretend some rendering happens here etc:
			// while (stopWatch.getElapsedSeconds() < lastTime + timestep);

			lastTime += timestep;

			// Step the graphics display (none in this demo).
		}



		//
		// Clean up physics and graphics
		//

		// <PHYSICS-ONLY>: cleanup physics
		{
			physicsWorld->markForWrite();
			physicsWorld->removeReference();
		}
		vdb->removeReference();

		// Contexts are not reference counted at the base class level by the VDB as
		// they are just interfaces really. So only delete the context after you have
		// finished using the VDB.
		context->removeReference();

		delete jobQueue;
		delete gpaProfiler;

		//
		// Clean up the thread pool
		//

		threadPool->removeReference();

	}

	hkBaseSystem::quit();
    hkMemoryInitUtil::quit();

	return 0;
}



void createBrickWall( hkpWorld* world, int height, int length, const hkVector4& position, hkReal gapWidth, hkpConvexShape* box, hkVector4Parameter halfExtents )
{
	hkVector4 posx = position;
	// do a raycast to place the wall
	{
		hkpWorldRayCastInput ray;
		ray.m_from = posx;
		ray.m_to = posx;

		ray.m_from(1) += 20.0f;
		ray.m_to(1)   -= 20.0f;

		hkpWorldRayCastOutput result;
		world->castRay( ray, result );
		posx.setInterpolate4( ray.m_from, ray.m_to, result.m_hitFraction );
	}
	// move the start point
	posx(0) -= ( gapWidth + 2.0f * halfExtents(0) ) * length * 0.5f;
	posx(1) -= halfExtents(1) + box->getRadius();

	hkArray<hkpEntity*> entitiesToAdd;

	for ( int x = 0; x < length; x ++ )		// along the ground
	{
		hkVector4 pos = posx;
		for( int ii = 0; ii < height; ii++ )
		{
			pos(1) += (halfExtents(1) + box->getRadius())* 2.0f;

			hkpRigidBodyCinfo boxInfo;
			boxInfo.m_mass = 10.0f;
			hkMassProperties massProperties;
			hkpInertiaTensorComputer::computeBoxVolumeMassProperties(halfExtents, boxInfo.m_mass, massProperties);

			boxInfo.m_mass = massProperties.m_mass;
			boxInfo.m_centerOfMass = massProperties.m_centerOfMass;
			boxInfo.m_inertiaTensor = massProperties.m_inertiaTensor;
			boxInfo.m_solverDeactivation = boxInfo.SOLVER_DEACTIVATION_MEDIUM;
			boxInfo.m_shape = box;
			//boxInfo.m_qualityType = HK_COLLIDABLE_QUALITY_DEBRIS;
			boxInfo.m_restitution = 0.0f;

			boxInfo.m_motionType = hkpMotion::MOTION_BOX_INERTIA;

			{
				boxInfo.m_position = pos;
				hkpRigidBody* boxRigidBody = new hkpRigidBody(boxInfo);
				world->addEntity( boxRigidBody );
				boxRigidBody->removeReference();
			}

			pos(1) += (halfExtents(1) + box->getRadius())* 2.0f;
			pos(0) += halfExtents(0) * 0.6f;
			{
				boxInfo.m_position = pos;
				hkpRigidBody* boxRigidBody = new hkpRigidBody(boxInfo);
				entitiesToAdd.pushBack(boxRigidBody);
			}
			pos(0) -= halfExtents(0) * 0.6f;
		}
		posx(0) += halfExtents(0)* 2.0f + gapWidth;
	}
	world->addEntityBatch( entitiesToAdd.begin(), entitiesToAdd.getSize());

	for (int i=0; i < entitiesToAdd.getSize(); i++){ entitiesToAdd[i]->removeReference(); }
}

void setupPhysics(hkpWorld* physicsWorld)
{
	//
	//  Create the ground box
	//
	{
		hkVector4 groundRadii( 70.0f, 2.0f, 140.0f );
		hkpConvexShape* shape = new hkpBoxShape( groundRadii , 0 );

		hkpRigidBodyCinfo ci;

		ci.m_shape = shape;
		ci.m_motionType = hkpMotion::MOTION_FIXED;
		ci.m_position = hkVector4( 0.0f, -2.0f, 0.0f );
		ci.m_qualityType = HK_COLLIDABLE_QUALITY_FIXED;

		physicsWorld->addEntity( new hkpRigidBody( ci ) )->removeReference();
		shape->removeReference();
	}

	hkVector4 groundPos( 0.0f, 0.0f, 0.0f );
	hkVector4 posy = groundPos;

	//
	// Create the walls
	//

	int wallHeight = 8;
	int wallWidth  = 8;
	int numWalls = 6;
	hkVector4 boxSize( 1.0f, 0.5f, 0.5f);
	hkpBoxShape* box = new hkpBoxShape( boxSize , 0 );
	box->setRadius( 0.0f );

	hkReal deltaZ = 25.0f;
	posy(2) = -deltaZ * numWalls * 0.5f;

	for ( int y = 0; y < numWalls; y ++ )			// first wall
	{
		createBrickWall( physicsWorld, wallHeight, wallWidth, posy, 0.2f, box, boxSize );
		posy(2) += deltaZ;
	}
	box->removeReference();

	//
	// Create a ball moving towards the walls
	//

	const hkReal radius = 1.5f;
	const hkReal sphereMass = 150.0f;

	hkVector4 relPos( 0.0f,radius + 0.0f, 50.0f );

	hkpRigidBodyCinfo info;
	hkMassProperties massProperties;
	hkpInertiaTensorComputer::computeSphereVolumeMassProperties(radius, sphereMass, massProperties);

	info.m_mass = massProperties.m_mass;
	info.m_centerOfMass  = massProperties.m_centerOfMass;
	info.m_inertiaTensor = massProperties.m_inertiaTensor;
	info.m_shape = new hkpSphereShape( radius );
	info.m_position.setAdd4(posy, relPos );
	info.m_motionType  = hkpMotion::MOTION_BOX_INERTIA;

	info.m_qualityType = HK_COLLIDABLE_QUALITY_BULLET;


	hkpRigidBody* sphereRigidBody = new hkpRigidBody( info );
	g_ball = sphereRigidBody;

	physicsWorld->addEntity( sphereRigidBody );
	sphereRigidBody->removeReference();
	info.m_shape->removeReference();

	hkVector4 vel(  0.0f,4.9f, -100.0f );
	sphereRigidBody->setLinearVelocity( vel );


}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
