/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

// Math and base include
#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <Common/Base/System/Error/hkDefaultError.h>
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Base/Memory/Allocator/Malloc/hkMallocAllocator.h>

#include <Common/Base/System/Io/Socket/hkSocket.h>

#include <Common/Base/System/Io/FileSystem/hkServerFileSystem.h>
#include <Common/Base/System/Io/FileSystem/hkFileServer.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Base/System/Io/OptionParser/hkOptionParser.h>

#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/System/Io/OStream/hkOStream.h>

#include <Common/Base/Fwd/hkcstdarg.h>

#include <stdio.h>


static void HK_CALL log(const char* msg, ...)
{
	va_list argptr = NULL;
	va_start(argptr, msg);
	vprintf(msg, argptr);
	fflush(stdout);
	va_end(argptr);  
}

static bool s_verbose = false;


static void HK_CALL errorReport(const char* msg, void* userArgGivenToInit)
{
	if(s_verbose)
		log("%s\n", msg);
}


static void HK_CALL updateStatusFile(const char* fileServRoot, const char* msg)
{
	hkStringBuf pathOut;
	pathOut.printf("%s/fileServStatus.txt", fileServRoot);
	hkOstream fout(pathOut);
	if (fout.isOk())
	{
		fout << msg;
	}
}

struct FileServerOptions
{
	FileServerOptions()
	{
		hkString::memSet(this, 0, sizeof(*this));
	}

	const char* m_ipAddress;
	const char* m_rootDir;
	int m_port;
	int m_timeoutMinutes;
};


int HK_CALL main(int argc, const char** argv)
{
#if defined(HK_COMPILER_HAS_INTRINSICS_IA32) && HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED	
	_MM_SET_FLUSH_ZERO_MODE(_MM_FLUSH_ZERO_ON);
#endif

	FileServerOptions options; 
	hkOptionParser parser("FileServer", "Havok simple network fileserver" );

	hkOptionParser::Option fsoptions[] = 
	{
		hkOptionParser::Option("a", "address", "Host IP address", &options.m_ipAddress, "localhost"),
		hkOptionParser::Option("p", "port", "Host port", &options.m_port, hkServerFileSystem::HK_SERVER_FILE_SYSTEM_DEFAULT_PORT),
		hkOptionParser::Option("r", "root", "File serving root", &options.m_rootDir ),
		hkOptionParser::Option("t", "timeout", "Timeout (in minutes)", &options.m_timeoutMinutes, 2 ),
		hkOptionParser::Option("v", "verbose", "Enable verbose output", &s_verbose, false ),
	};

	parser.setOptions(fsoptions, HK_COUNT_OF(fsoptions));

	if( parser.parse( argc, argv ) != hkOptionParser::PARSE_SUCCESS )
	{
		return -1;
	}


	//TODO error handling on the input args
		
		// Initialize the base system including our memory system
		// Allocate 0.5MB of Physics solver buffer.
	hkMemorySystem::FrameInfo finfo(500000);
	hkMemoryRouter* memoryRouter = hkMemoryInitUtil::initDefault(hkMallocAllocator::m_defaultMallocAllocator, finfo);
	hkBaseSystem::init( memoryRouter, errorReport );
	
	log("FILE SERVER: Havok simple network fileserver started\n");
	hkBool reconnecting = false;
	
	while (1)
	{
		if(reconnecting)
		{
			updateStatusFile(options.m_rootDir, "RECONNECTING");
		}
		else
		{
			updateStatusFile(options.m_rootDir, "SEARCHING");
		}
		const char* ipAddr = options.m_ipAddress;
		const int port = options.m_port;
		const int timeout = options.m_timeoutMinutes;

		hkSocket* socket = hkSocket::create();
		hkStopwatch t;
		t.start();
		hkResult res;
		log("FILE SERVER: Trying to connect to %s:%d..\n", ipAddr, port);
		while( (res = socket->connect(ipAddr, port)) != HK_SUCCESS )
		{
			if (t.getElapsedSeconds() > 60*timeout) 
			{
				break;
			}
			// xx Sleep
		}

		if (res == HK_SUCCESS)
		{
			// Connect
			log("FILE SERVER: Connected, serving from %s.\n", options.m_rootDir ? options.m_rootDir : "local dir");
			updateStatusFile(options.m_rootDir, "CONNECTED");
			hkFileServer::ThreadData td;
			hkCriticalSection dataLock(1000);
			td.m_socket = socket;
			td.m_dataLock = &dataLock;

			hkStringBuf rootDir = "";
			if(options.m_rootDir)
			{
				rootDir.setJoin(options.m_rootDir, "/");
			}
			td.m_rootDir = rootDir;

			hkFileServer::ReturnCode serverReturn = hkFileServer::serve( &td );
			if(serverReturn == hkFileServer::RETURN_SOCKET_CLOSED)
			{
				log("FILE SERVER: Client finished.\n");
				break; // done
			}
			else if (serverReturn == hkFileServer::RETURN_SOCKET_ERROR)
			{
				log("FILE SERVER: hkFileServer::serve() received an error (RETURN_SOCKET_ERROR). Restarting...\n");
				reconnecting = true;
			}
			else
			{
				log("FILE SERVER: hkFileServer::serve() was told to exit (RETURN_SERVER_SHUTDOWN). Restarting...\n");
				reconnecting = true;
			}

		}
		else
		{			
			log("FILE SERVER: Could not connect after %d mins, aborting.\n", timeout);
			break; // done
		}
	}
	updateStatusFile(options.m_rootDir, "DONE");
	log("FILE SERVER: Exiting..\n");
	hkBaseSystem::quit();
	hkMemoryInitUtil::quit();

	return 0;
}



// Keycode
#include <Common/Base/keycode.cxx>

// This excludes libraries that are not going to be linked
// from the project configuration, even if the keycodes are
// present
#undef HK_FEATURE_PRODUCT_AI
#undef HK_FEATURE_PRODUCT_ANIMATION
#undef HK_FEATURE_PRODUCT_CLOTH
#undef HK_FEATURE_PRODUCT_DESTRUCTION_2012
#undef HK_FEATURE_PRODUCT_DESTRUCTION
#undef HK_FEATURE_PRODUCT_BEHAVIOR
#undef HK_FEATURE_PRODUCT_PHYSICS_2012
#undef HK_FEATURE_PRODUCT_PHYSICS

#define HK_EXCLUDE_FEATURE_RegisterVersionPatches
#define HK_EXCLUDE_FEATURE_MemoryTracker
#define HK_EXCLUDE_FEATURE_SerializeDeprecatedPre700
#define HK_EXCLUDE_LIBRARY_hkCompat
#define HK_EXCLUDE_FEATURE_RegisterReflectedClasses


//#include "hkProductFeaturesBasic.h"
#include <Common/Base/Config/hkProductFeatures.cxx>

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
