/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */
#ifndef HK_IMGUI_WIDGET_H
#define HK_IMGUI_WIDGET_H

#include <Graphics/ImGui/ImGuiLayout.h>
#include <Graphics/ImGui/ImGuiContext.h>
#include <Common/Base/Types/Color/hkColor.h>

namespace ImGui
{
namespace Widget
{
		/// Convenience class for layouts to automatically (un)register themselves as the active layout.
	template < typename LAYOUT >
	struct ScopedLayout : public LAYOUT
	{
		ScopedLayout( Context& ctx )
			: LAYOUT(ctx.layout().remaining())
			, m_ctx(ctx)
		{
			m_ctx.pushLayout( this );
		}
		ScopedLayout( Context& ctx, const Rectangle& r )
			: LAYOUT(r)
			, m_ctx(ctx)
		{
			m_ctx.pushLayout( this );
		}

		template< typename P0 >
		ScopedLayout( Context& ctx, P0 p0 )
			: LAYOUT(ctx.layout().remaining(), p0)
			, m_ctx(ctx)
		{
			m_ctx.pushLayout( this );
		}
		template< typename P0 >
		ScopedLayout( Context& ctx, const Rectangle& r, P0 p0 )
			: LAYOUT(r, p0)
			, m_ctx(ctx)
		{
			m_ctx.pushLayout( this );
		}
		
		~ScopedLayout()
		{
			Pair usedp = this->used();
			if( m_ctx.popLayout(this) )
			{
				m_ctx.layout().alloc(usedp);
			}
		}
		Context& m_ctx;
	};

		/// Helper class to assert if the widget was used incorrectly.	
		/// Widgets such as Button need to have wasClicked or wasToggled called to work correctly.
	class StateWasQueried
	{
		public:
			StateWasQueried() : m_used(false) {}
			~StateWasQueried() { HK_ASSERT2(0x7fe8218b, m_used, "You need to query the enclosing object to work correctly (i.e. call isOpen, wasClicked etc.");}
			void notify() const { m_used = true; }
		private:
			mutable hkBool m_used;
	};

		/// Simple button.
	class Button
	{
		public:

			Button(ImGui::Context& im);
			Button(ImGui::Context& im, const char* label);

			~Button();

			Button& label(const char*);
			Button& image(hkUint32* rgb, int w, int h, ImGui::Argb color);

				///
			hkBool wasClicked() const;
				/// Two-state button.
			hkBool wasToggled(hkBool& active) const;
			hkBool isActive(hkBool& active) const;

			Button& accelerator(char a, char b=0, char c=0);

		protected:

			bool _button(hkBool active) const;

			ImGui::Context& m_gui;
			PaddedLayout m_layout;
			char m_accelerator[4];
			StateWasQueried m_operatorBoolCalled;
	};

		/// Container for a single row of rectangles.
	typedef ScopedLayout<ImGui::HorizontalLayout> HorizontalBox;

		/// Container for a single column of rectangles.
	typedef ScopedLayout<ImGui::VerticalLayout> VerticalBox;

		/// Similar to HorizontalBox, except it wraps round to the next line.
	typedef ScopedLayout<ImGui::FlowLayout> FlowBox;

		/// Pads a single child widget.
	typedef ScopedLayout<ImGui::PaddedLayout> PaddedArea;

		/// Widget which contains a child widget which may be clipped to the scroll bounds.
	struct ScrolledArea
	{
		ScrolledArea(Context& ctx, ScrollState state[2]);
		~ScrolledArea();

		int _getAxisTranslation(int axis);

		Context& m_ctx;
		ScrollState* m_state;
		Rectangle m_available;
		hkBool m_scrollbarVisible[2];
		ManualLayout m_manual;
	};


		/// Numeric control with increment/decrement.
	struct Spinner
	{
		Spinner(ImGui::Context& im, hkReal& value, hkReal increment) : m_gui(im), m_fmt("%g"), m_value(value), m_increment(increment) { }
		~Spinner();
		Spinner& format(const char* fmt) { m_fmt = fmt; return *this; }

		ImGui::Context& m_gui;
		const char* m_fmt;
		hkReal& m_value;
		hkReal m_increment;
		// todo - integer type
		// todo - range
		// todo - log/linear etc scale
		// todo - keyboard editing
	};

		/// 4 spinners for an rgb value
	struct ColorSpinner
	{
		ColorSpinner(ImGui::Context& im, ImGui::Argb& color);
	};

		/// Simple label.
	struct Label
	{
			/// Label with optional color.
		Label( ImGui::Context& m_gui, const char* m_text, ImGui::Argb m_color=hkColor::BLACK );
	};

	struct Image
	{
			/// Draw an image.
		Image( ImGui::Context& m_gui, const hkUint32* argb, int w, int h, ImGui::Argb bgcolor );
	};

		/// Single radio button.
	struct RadioButton
	{
			/// 
		RadioButton( ImGui::Context& gui, int thisTag, const char* label, int* activeTag );
		RadioButton( ImGui::Context& gui, int thisTag, int* activeTag );
		~RadioButton();
		ImGui::Context& m_gui;
		ManualLayout m_layout;
		int m_thisTag;
		int* m_activeTag;
	};

		/// Space with an optional fill color.
	struct Spacer
	{
			/// Space is factor of font size
		Spacer( ImGui::Context& m_gui, hkReal xspace, hkReal yspace, ImGui::Argb color = 0 );
	};

		/// Collapsible group of widgets.
	struct Expander
	{
		Expander( ImGui::Context& im, const char* label, hkBool& expanded );
		~Expander();

		hkBool isOpen() const { m_operatorBoolCalled.notify(); return m_expanded; }

		ImGui::Context& m_gui;
		const char* m_label;
		hkBool& m_expanded;
		Rectangle m_exRect;
		ManualLayout m_manual;
		StateWasQueried m_operatorBoolCalled;
	};

		/// Floating group of widgets. TODO: maintain z order of windows
	struct Window
	{
		Window( ImGui::Context& gui, Rectangle& rect, const char* title = HK_NULL );
		~Window();

		Rectangle available() const { return m_manual.remaining(); }
		ImGui::Context& m_gui;
		ManualLayout m_manual;
		Rectangle& m_rect;
	};

		/// Text edit box.
	struct TextEdit
	{
		TextEdit(ImGui::Context& gui, hkStringPtr& text, TextEditState& state);
	};

		/// Progress bar with optional label.
	struct Progress
	{
		Progress(ImGui::Context& gui, hkReal val) : m_gui(gui), m_val(val), m_text(HK_NULL) { }
		~Progress();
		Progress& text(const char* str);

		ImGui::Context& m_gui;
		hkReal m_val;
		const char* m_text;
	};

		/// Tree widget.
	struct Tree
	{
		enum IconType
        {
			ICON_NONE,
			ICON_ADD,
			ICON_DEL
        };
		Tree(ImGui::Context& gui, const char* label, int level, hkBool expandable, hkBool& open)
			: m_gui(gui), m_label(label), m_level(level), m_expandable(expandable), m_open(&open), m_iconType(ICON_NONE)
		{
		}
		Tree(ImGui::Context& gui, const char* label, int level, IconType iconType=ICON_NONE)
			: m_gui(gui), m_label(label), m_level(level), m_expandable(false), m_open(HK_NULL), m_iconType(iconType)
		{
		}
			/// Returns whether the tree node is open or not.
		hkBool isOpen() const;

		ImGui::Context& m_gui;
		const char* m_label;
		int m_level;
		hkBool m_expandable;
		hkBool* m_open;
		IconType m_iconType;
	};

		/// Multicolumned widget.
	struct Table
	{
		Table( ImGui::Context& gui, int* colwidths, int ncol )
			: m_gui(gui), m_table(m_gui.layout().remaining(), colwidths, ncol)
		{
			m_gui.pushLayout( &m_table );
		}
		~Table()
		{
			Pair p = m_gui.layout().used();
			if( m_gui.popLayout(&m_table) )
			{
				m_gui.layout().alloc(p);
			}
		}

		ImGui::Context& m_gui;
		TableLayout m_table;
	};

	struct SizeGroup
	{
		SizeGroup( int* colwidths, int ncol );
		~SizeGroup();

		struct Row : public Layout
		{
			Row(ImGui::Context& gui, SizeGroup& g);
			~Row();
			virtual Rectangle alloc2(int sx, int sy);
			virtual Rectangle remaining() const;
			virtual Pair used() const;

			ImGui::Context& m_gui;
			SizeGroup& m_group;
			Rectangle m_usable;
			Rectangle m_allocated;
			int m_curCol;
		};

		int* m_externColWidths;
		hkArray<int> m_cols;
	};

		/// Tabbed widget which can contain a single child per tab.
	struct Notebook
	{
		Notebook( ImGui::Context& gui, NotebookState& state);
		Notebook( ImGui::Context& gui, const Rectangle& r, NotebookState& state);
		~Notebook();

			/// Returns true if the label with the name specified is active.
			/// If it is active the ImGui client code needs to display the panels interface by, 
			/// doing ImGui element calls.
		bool isActivePage( const char* label );

	protected:

		void _init();

		ImGui::Context& m_gui;
		Rectangle m_available; // whole area including bar + contents
		Pair m_barSize; // height of the tab bar
		int m_pageCount;
		int m_tabOffset;
		int m_tabSize;
		NotebookState& m_state;
		ManualLayout m_child;
	};

		/// Group of widgets with an optional label
	struct Frame
	{
		Frame(Context& ctx, const char* label)
			: m_ctx(ctx)
		{
			Spacer(ctx, 0, 0.5f);
			if( label )
			{
				Label(ctx, label);
			}
			Spacer(ctx, 0, 0.25f);
		}
		~Frame()
		{
			Spacer(m_ctx, 0, 0.25f);
		}
		Context& m_ctx;
	};
}
}

#endif // HK_IMGUI_WIDGET_H

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
