/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */
#ifndef HK_IMGUI_LAYOUT_H
#define HK_IMGUI_LAYOUT_H

namespace ImGui
{
		/// Layouts are used by widgets to give them rectangles to render themselves into.
		/// The basic idea is that a layout is initially given a rectangle which it chops
		/// into pieces via the alloc2 call. The initial rectangle is only a maximum, it
		/// would be unusual to use it entirely. After all allocations have been made, we can
		/// query how much was used.
		///
		/// Because the layout has to return rectangles piecemeal, before it has seen all requests,
		/// it is impossible to layout correctly. E.g. consider a 2x2 table where the columns of the
		/// second row are wider than the first row. When allocating the widths in the first row we
		/// can't know in advance the sizes of the second row.
		///
		/// One approach to fixing this "psychic" problem is to run multiple passes
		/// of the GUI e.g. a "measure" pass without handling any input to discover all widgets and a second
		/// pass which processes input normally. The other approach (and the one taken here) is to accept
		/// a one frame delay and to remember the widths from the previous frame. This has the advantage
		/// of allowing the caller to manipulate the columns widths with no extra API, but relies on
		/// the GUI remaining relatively stable from frame to frame. In the example case of the table, the user
		/// must supply storage for the column widths from the previous frame.
		///
		/// One other simplification which the automatic layout uses is that the widget gravity is
		/// constant. That is, we always start with the top-left-most widget. In the case of hierarchical
		/// widgets, we maintain a stack of layouts. Each layout maintains a "used" amount and when it
		/// is finished (popped off the stack), that amount is allocated from its parent. See
		/// ImGui::Widget::ScopedLayout which illustrates this process.
		///
		/// We keep the *Layout classes (which do not know about the hierarchy) separate from the widget
		/// container classes (which explicitly use the hierarchy) so that widgets may use the layout
		/// algorithms internally without pushing them on the stack. (Currently many layouts are done manually
		/// but both Widget and Painter could use them)
	class Layout
	{
		public:

			virtual ~Layout() {}

				/// Request an area, using a Pair to describe the size
			inline Rectangle alloc(Pair p) { return alloc2(p[0], p[1]); }

				/// Request an area of sx*sy pixels.
			virtual Rectangle alloc2(int sx, int sy) = 0;

				/// Query the maximum area remaining.
			virtual Rectangle remaining() const = 0;

				/// Query the amount currently allocated.
				/// Used in particular by the scrolling code and hierarchical layout.
			virtual Pair used() const = 0;
	};

		/// Layout which holds a single child with a border of padding.
	class PaddedLayout : public Layout
	{
		public:

			PaddedLayout(const Rectangle& area);

			inline void setPad( int left, int top, int right, int bottom ) { m_left=left; m_top=top; m_right=right, m_bottom=bottom; }
			inline void getPad( int& left, int& top, int& right, int& bottom ) const { left=m_left; top=m_top; right=m_right, bottom=m_bottom; }

			// Implement Layout
			virtual Rectangle alloc2(int sx, int sy);
			virtual Rectangle remaining() const;
			virtual Pair used() const;
			Rectangle areaUnpadded() const { return Rectangle(m_usable.pos, m_used); }

		protected:

			Rectangle m_usable;
			Pair m_used;
			int m_left;
			int m_top;
			int m_right;
			int m_bottom;
	};

		/// Base class for horizontal and vertical layouts
	class LinearLayout : public Layout
	{
		public:

			LinearLayout(const Rectangle& area, hkBool horizontal) : m_usable(area), m_used(0,0), m_isHorizontal(horizontal), m_centered(false) { }

				/// Make sure we use at lease 'size' pixels perpendicular to the allocation direction.
			LinearLayout& reserve(int size) { int& x = m_used[m_isHorizontal?1:0]; x = hkMath::max2(x,size); return *this; }
			LinearLayout& setCentered(hkBool c) { m_centered = c; return *this; }
				// Implement Layout
			virtual Rectangle alloc2(int sx, int sy);
			virtual Rectangle remaining() const;
			virtual Pair used() const;
			virtual Rectangle allocated() const;

		protected:

			Rectangle m_usable;
			Pair m_used;
			int m_isHorizontal;
			hkBool m_centered; 
	};

		/// Layout for a single row of rectangles.
	class HorizontalLayout : public LinearLayout
	{
		public:

			HorizontalLayout(const Rectangle& area) : LinearLayout(area,true) { }
	};
	
		/// Layout for a single column of rectangles.
	class VerticalLayout : public LinearLayout
	{
		public:

			VerticalLayout(const Rectangle& area) : LinearLayout(area,false) { }
	};

		/// Layout which allows manual placement of children.
	class ManualLayout : public Layout
	{
		public:

			ManualLayout( const Rectangle& rect ) : m_usable(rect), m_offset(0,0), m_used(0,0) {}
		
			// Implement Layout
			virtual Rectangle alloc2(int sx, int sy);
			virtual Rectangle remaining() const;
			virtual Pair used() const;

		//protected: // todo - scrolling needs this public - make a clean interface

			Rectangle m_usable;
			Pair m_offset;
			Pair m_used;
	};

		/// Similar to HorizontalLayout, except it wraps round to the next line.
	class FlowLayout : public Layout
	{
		public:

			FlowLayout(const Rectangle& rect, FlowLayoutState* state);

			// Implement Layout
			virtual Rectangle alloc2(int sx, int sy);
			virtual Rectangle remaining() const;
			virtual Pair used() const;

		protected:

			Rectangle m_usable; ///< Area usuable by children
			int m_curRowHeight; ///< Current row height so far
			int m_maxRowWidth; ///< Max width of all rows so far
			Pair m_curOffset; ///< Current cursor witin the usable area.
			int m_wrapPadding; ///< When we wrap, the number of pixels which were left blank on the preceding line.
			hkUint64 m_wrapBit; ///< The bitmask for the current child (1<<numAllocs)
			FlowLayoutState* m_state;  ///< Persistent state (bitmask of children which need wrapping)
	};

		/// Layout with several columns.
		/// Each alloc2 call advances to the next column.
	class TableLayout : public Layout
	{
		public:

				/// The colwidths are persisted from the previous frame, they are used as a guess for the current
				/// widths. On output they are set to the minimum needed widths, they may be changed arbitrarily
				/// between invocations.
			TableLayout( const Rectangle& rect, int* colwidths, int ncol );
			~TableLayout();

			// Implement Layout
			virtual Rectangle alloc2(int sx, int sy);
			virtual Rectangle remaining() const;
			virtual Pair used() const;

		protected:

			Rectangle m_available;
			hkArray<int> m_colWidths;
			int* m_colWidthsExtern;
			int m_numCols;
			int m_curCol;
			int m_curRowHeight;
			Pair m_used;
	};
}

#endif // HK_IMGUI_LAYOUT_H

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
