/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */
#ifndef HK_IMGUI_CONTEXT_H
#define HK_IMGUI_CONTEXT_H

#include <Common/Base/System/Stopwatch/hkStopwatch.h>

namespace ImGui
{
	class Layout;

		/// This contains all the persistent state of the ImGui which the user need not manage.
		/// Remember that most of the user visible (i.e. widget) state is stored externally by
		/// the user. The things we manage which are:
		///	* bookkeeping of input state and time
		///	* handling common operations based on the state (e.g. button presses & drags)
		/// * storing some transient state such as the currently active layout and the painter.
		///
		/// For methods that require state it is the client responsibility to maintain the state 
		/// between calls to the UI update. Thus using a local variable of type TextEditState 
		/// (ie a type that holds gui state), that goes in and out of scope through each gui update, 
		/// is probably incorrect.
		/// 
		/// In order to use the context it is most convenient to use the classes defined in the 
		/// ImGuiWidget.h.
		/// These classes will call the appropriate methods on the context, but will also handling 
		/// the appropriate drawing of the components. The helper classes should be used as local variables
		/// within the ImGui code - so that the correct calling sequence, to make input and drawing 
		/// work correctly.
	struct Context
	{
		public:

			EXT_SIMPLE_ALLOCATOR();

			enum ActionBits
			{
				ACTION_NONE = 0,
				ACTION_MOUSE_DOWN,
				ACTION_MOUSE_DRAG
			};
			typedef hkEnum<ActionBits, int> ActionState;

				/// Ctor
			Context();
				/// Dtor
			~Context();

				/// Set the painter - which will be used for drawing the context state
			void setPainter(Painter* painter);

				/// Update the context state. Should be called after the immediate mode Gui code has been executed
				/// and before it is rendered. 
			void tick();

				/// Informs context of mouse state. ButtonPressed and buttonReleased, act as booleans as currently only supports 
				/// a single button. X and y specifies the current coordinates of the mouse.
			void mouse(int buttonPressed, int buttonReleased, int x, int y);
				/// Informs the context of a keyboard change of state. Called to indicate a keyboard key has been depressed. 
				/// Most keys are indicated by their legend (ie a comes in as A). 
				/// TODO: Currently keys such as shift come in as hkg framework codes. To make this work independently from hkg
				/// the key codes need to be independent.
			void keyboardDown( int buttonDown );
			void keyboardUp( int buttonDown );
				/// Informs the context that a button has been depressed on the gamepad. The value passed in is the button 
				/// index on the game pad.
			void gamepad( int buttonDownIndex );

				// Behaviors. TODO: these should all output the WidgetFlags so the painter can draw them appropriately

				/// Do a button. Returns the button state (true if it has been clicked). Rectangle defines the rectangle of the button
				/// The widgetFlags passed in may be modified.
			hkBool doButton( const Rectangle& rect, WidgetFlags& flags );
				/// Returns true if the keys pressed (in accel) are all down. If any is not down it will return false.
			hkBool doAccelerator( const char* accel );

				/// Do a scroll bar. Axis = 0 for horizontal scroll bar, and 1 for vertical. Returns true if the scroll bar has been moved.
				/// ScrollState holds the scroll state of the scroll bar, and may be modified by this call.
			hkBool32 doScrollBar( int axis, const Rectangle& scrollbar, ScrollState& state, WidgetFlags& flags );
				/// Used for scrolly areas with persistent state (think: scrolling with smartphones)
			hkBool32 doScrollDrag( int axis, const Rectangle& visibleClient, ScrollState& state, WidgetFlags& flags, int dragThreshold=10 );
				/// Used for scrolly areas with no inertia or persistent state (think: moving windows)
			int doScrollDrag( int axis, const Rectangle& visible, WidgetFlags& flags, int dragThreshold=10 );

				/// Do a 'spinner' - a box that displays a value, that can be change by clicking and dragging. 
				/// Returns true if the spinner value has changed.
			hkBool doSpinner( int axis, const Rectangle& rect, hkReal& value, hkReal increment, WidgetFlags& flags);
				/// Do a text entry box. Returns true if 'text' is changed. 
				/// 'state' is used to store the current text entry state. 
				/// Thre Rescaler is used to calculate a character position from the pixel x position in the text. 
			hkBool doTextEntry( const Rectangle& rect, hkStringPtr& text, TextEditState& state, Rescaler& indexFromPos);

				/// For keyboard/gamepad navigation, all the widgets in the given rectangle should be considered a group.
				/// When navigating we try to match within the group first and only consider other groups if that fails.
			void addFocusGroup( const Rectangle& rect, const char* name=HK_NULL, int id=0 );

				/// Get the current action state
			ActionState getActionState() { return m_action; }
				/// Set the current action state
			void setActionState(ActionState a) { m_action = a; }

				/// Push layout to be used
			void pushLayout(Layout* l);
				/// Pop layout on the top of the layout stack. Checks that the layout passed in is the layout on the top of the stack.
				/// Returns true if layouts remain on the stack
			hkBool popLayout(Layout* l);
				/// Get the current layout on the top of the stack
			Layout& layout(int up=0) const;

				/// Get the current painter
			Painter& painter() const { return *m_painter; }

				/// Return a counter which increments every \a jiffiesPerSecond seconds
			int jiffies(int jiffiesPerSecond = 1);

			const Pair& getMousePress() const { return m_mousePress; }
			const Pair& getMouseCur() const { return m_mouseCur; }

		protected:

			struct FocusGroup
			{
				Rectangle rect;
				Pair lastPos;
				const char* name;
				int id;
			};

			hkArray<int> m_keysDown;
			int m_padState;
			Pair m_mouseCur;
			Pair m_mousePrev;
			int m_mouseButton; 
			Pair m_mousePress; 
			int m_padButton; 
			Pair m_padPress; 
			Pair m_focusPos; 
			hkStopwatch m_time;
			Painter* m_painter;
			hkArray<Layout*> m_layout;
			hkEnum<ActionBits, int> m_action;
			hkArray<Rectangle> m_focusRects;
			hkArray<FocusGroup> m_focusGroups;
			int m_focusLayer;
			Pair m_keyboardPos;
	};
}

#endif // HK_IMGUI_H

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
