/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#ifndef HK_GRAPHICS_PICK_DATA
#define HK_GRAPHICS_PICK_DATA

#include <Graphics/Common/hkgObject.h>

// forward decl.
class hkgDisplayWorld;

/// hkgFaceSetPickData, returned by hkgFaceSet::intersect().
/// Note: We only return the closest triangle ATM for ray intersect, and first encountered triangle for frustum intersect.
///		  Returning data for every triangle in a frustum query may be a huge amount of data and it's currently not needed.
struct hkgFaceSetPickData
{
		HKG_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( hkgFaceSetPickData );

		hkgFaceSetPickData() : m_dist(-1.0f) {}

		//////////////////////////////////////////////////////////////////////////
		// Primary data filled out by hkgFaceSet::intersect().
		//////////////////////////////////////////////////////////////////////////

		/// The distance along ray of triangle hit point (only filled out by ray intersect). 
		/// Use  (dir*m_dist + from)  for world point
		float m_dist;

		/// The index of the primitive in the hkgFaceSet in which triangle presides that was hit.
		int m_primitiveIndex; 
		
		/// The index in the primitive of the triangle (i-th triangle, not i-th connectivity value)
		int m_primitiveTriIndex; 

		/// The 3 indices that make up the triangle (so that you don't have to walk the above information
		/// if you don't want to). Index into the vertex set with these values to get the vertex positions.
		int m_verts[3]; 

		/// The normal to the triangle that was hit
		float m_localNormal[3];

		//////////////////////////////////////////////////////////////////////////
		// Extra data filled out by the caller.
		//////////////////////////////////////////////////////////////////////////

		/// The index of the material face set in the hkgGeometry holding this hkgFaceSet.
		int		m_matFaceSetIndex;

		/// The index in the material face set of this hkgFaceSet, used in conjunction with the above index.
		int		m_matFaceSetSubIndex;

		/// Determine if the pick info is valid
		inline bool isValid()
		{
			return (m_dist > 0);
		}
};

/// hkgGeometryPickData, returned by hkgGeometry::intersect().
struct hkgGeometryPickData
{
	HKG_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( hkgGeometryPickData );

	/// The hkgFaceSetPickDatas that came from successfull hkgFaceSet::intersect calls.
	extArray<hkgFaceSetPickData>		m_facePickDatas;

	//////////////////////////////////////////////////////////////////////////
	// Extra data filled out by the caller.
	//////////////////////////////////////////////////////////////////////////

	/// Local position of the hit on this geometry in the parent hkgDisplayObject space.
	float	m_localPos[3];

	/// The index of this geometry in the hkgDisplayObject of this geometry.
	int		m_geomIndex;

	//////////////////////////////////////////////////////////////////////////
	// Internal helper functions for building up the results
	//////////////////////////////////////////////////////////////////////////

	/// The distance to the closest sub-item (only valid for intersect ray results)
	inline float getClosestDistance() const
	{
		return m_facePickDatas[0].m_dist;
	}

	/// Determine if the pick info is valid
	inline bool isValid()
	{
		for ( int i = 0; i < m_facePickDatas.getSize(); i++ )
		{
			if (!m_facePickDatas[i].isValid())
			{
				return false;
			}
		}

		return (m_facePickDatas.getSize() > 0);
	}
};


/// hkgDisplayObjectPickData, returned by hkgDisplayObject::intersect().
struct hkgDisplayObjectPickData
{
	HKG_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( hkgDisplayObjectPickData );
	
	/// The hkgGeometryPickDatas that came from successful hkgGeometry::intersect calls.
	extArray<hkgGeometryPickData>	m_geomPickDatas;

	//////////////////////////////////////////////////////////////////////////
	// Extra data filled out by the caller.
	//////////////////////////////////////////////////////////////////////////

	/// The index into the hkgDisplayWorld of this display object.
	int							m_objectIndex;	

	/// The world normal of the hit on the object. This is the normal from the hkgFaceSet transformed
	/// by the hkgDisplayObject.m_trans if one. It is not valid for non-ray queries.
	float						m_worldNormal[3];

	/// The world position of the hit on the object. This is the hit point from the hkgFaceSet transformed
	/// by the hkgDisplayObject.m_trans if one.
	float						m_worldPos[3];

	//////////////////////////////////////////////////////////////////////////
	// Internal helper functions for building up the results
	//////////////////////////////////////////////////////////////////////////

	/// The distance to the closest sub-item (only valid for intersect ray results)
	inline float getClosestDistance() const
	{
		return m_geomPickDatas[0].getClosestDistance();
	}

	/// Determine if the pick info is valid
	inline bool isValid()
	{
		for ( int i = 0; i < m_geomPickDatas.getSize(); i++ )
		{
			if (!m_geomPickDatas[i].isValid())
			{
				return false;
			}
		}

		return (m_geomPickDatas.getSize() > 0);
	}
};

// hkgViewportPickData, returned from hkgViewport::pick().
struct hkgViewportPickData
{
	HKG_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( hkgViewportPickData );

	/// The hkgDisplayObjectPickDatas that came from calls to hkgDisplayObject::intersect()
	extArray<hkgDisplayObjectPickData>	m_objectPickDatas;

	//////////////////////////////////////////////////////////////////////////
	// Extra data filled out by caller
	//////////////////////////////////////////////////////////////////////////

	/// The pointer to the world that was used to query for intersection. Stored here for
	/// convenience.
	hkgDisplayWorld*			m_world;

	/// The total number of triangles that where successfully hit during the test.
	int							m_trianglesHit;

	//////////////////////////////////////////////////////////////////////////
	// Helper functions for accessing "closest" point results (only valid for intersect ray results).
	// These should only be called for valid results.
	//////////////////////////////////////////////////////////////////////////

	/// The closest pick data
	inline hkgDisplayObjectPickData& getClosestObjectPickData()
	{
		return m_objectPickDatas[0];
	}

	/// The distance to the closest sub-item (only valid for intersect ray results)
	inline float getClosestDistance() const
	{
		return m_objectPickDatas[0].getClosestDistance();
	}

	/// The object index of the closest sub-item (only valid for intersect ray results)
	inline int getClosestObjectIndex() const
	{
		return m_objectPickDatas[0].m_objectIndex;
	}

	/// The world pos of the closest sub-item (only valid for intersect ray results)
	inline const float* getClosestObjectWorldPos() const
	{
		return m_objectPickDatas[0].m_worldPos;
	}

	/// The world pos of the closest sub-item (only valid for intersect ray results)
	inline const float* getClosestObjectWorldNormal() const
	{
		return m_objectPickDatas[0].m_worldNormal;
	}

	/// The local pos of the closest sub-item (only valid for intersect ray results)
	inline const float* getClosestGeometryLocalPos() const
	{
		return m_objectPickDatas[0].m_geomPickDatas[0].m_localPos;
	}

	/// The geometry index of the closest sub-item (only valid for intersect ray results)
	inline int getClosestGeometryIndex() const
	{
		return m_objectPickDatas[0].m_geomPickDatas[0].m_geomIndex;
	}

	//////////////////////////////////////////////////////////////////////////
	// Internal helper functions for building up the results
	//////////////////////////////////////////////////////////////////////////
	
	/// Determine if the pick info is valid
	inline bool isValid()
	{
		for ( int i = 0; i < m_objectPickDatas.getSize(); i++ )
		{
			if (!m_objectPickDatas[i].isValid())
			{
				return false;
			}
		}

		return (m_objectPickDatas.getSize() > 0);
	}
};

#endif // HK_GRAPHICS_PICK_DATA

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
