/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

inline hkgVertexSet::hkgVertexSet(hkgDisplayContext* context)
: m_context(context), m_numVerts(0), m_bufferAccess(HKG_VERTEX_BUFFER_ACCESS_DEFAULT), m_vertexFormat(HKG_VERTEX_FORMAT_NONE), m_currentLockType(HKG_LOCK_NONE), m_lockedBuf( HK_NULL )
{
	for (int i=0; i < HKG_VERTEX_COMPONENT_NUM; ++i)
	{
		m_storageOffsets[i] = 0;
		m_storageFormats[i] = HKG_VERTEX_STORAGE_NONE;
		m_storageStrides[i] = 0;
	}

	HKG_CHECK_CONTEXT_LOCK_STATUS(context);
	m_vertexMapping = HK_NULL;
}

inline hkgVertexSet::~hkgVertexSet()
{
	if( m_vertexMapping )
	{
		delete m_vertexMapping;
	}
}

inline int hkgVertexSet::getNumVerts() const
{
	return m_numVerts;
}

inline HKG_VERTEX_FORMAT hkgVertexSet::getVertexFormat() const
{
	return m_vertexFormat;
}

inline bool hkgVertexSet::isDynamic() const
{
	return m_bufferAccess & HKG_VERTEX_BUFFER_ACCESS_DYNAMIC;
}

inline bool hkgVertexSet::isWriteOnly() const
{
	return (m_bufferAccess & HKG_VERTEX_BUFFER_ACCESS_READ )== 0;
}

inline HKG_VERTEX_STORAGE hkgVertexSet::getVertexComponentFormat(HKG_VERTEX_COMPONENT component) const
{
	return m_storageFormats[component];
}

inline hkLong hkgVertexSet::getVertexComponentOffset(HKG_VERTEX_FORMAT component) const
{
	return m_storageOffsets[component];
}

inline void* hkgVertexSet::getVertexComponentData( HKG_VERTEX_COMPONENT component, int index, hkUint8* data )
{
	if (!m_lockedBuf)
	{
		HK_WARN_ALWAYS(0x101, "You must lock the vertex buffer before being able to query or set data in it." );
		return HK_NULL;
	}

	HK_ASSERT2(0x32435ef, m_currentLockType & HKG_LOCK_READMASK, "You have not locked the buffer in the correct mode. You are reading from a write only buffer." );
	HK_ASSERT2(0x32435ee, (m_vertexFormat & (1<<component)) != 0, "You are trying to set data that does not exist in this vertex format." );

	unsigned char* v = (m_lockedBuf + m_storageOffsets[component]) + ( m_storageStrides[component] * index ); // always make m_storageOffsets relative to lockedBuf
	
	return v;
}

HK_FORCE_INLINE void hkgVertexSet::setVertexComponentData(HKG_VERTEX_COMPONENT component, int index, const void* data)
{
	if (!m_lockedBuf)
	{
		HK_WARN_ALWAYS(0x102, "You must lock the vertex buffer before being able to query or set data in it." );
		return;
	}

	HK_ASSERT2(0x102, index >= 0 && index < m_numVerts, "Index out of range for vertex component.");
	HK_ASSERT2(0x32435ef, m_currentLockType & HKG_LOCK_WRITEMASK, "You have not locked the buffer in the correct mode. You are writing to a read only buffer." );
	HK_ASSERT2(0x32435ee, (m_vertexFormat & (1<<component)) != 0, "You are trying to set data that does not exist in this vertex format." );
	{
		unsigned char* HK_RESTRICT v = (m_lockedBuf + m_storageOffsets[component]) + ( m_storageStrides[component] * index ); // always make m_storageOffsets relative to lockedBuf

		HKG_VERTEX_STORAGE storage = m_storageFormats[ component ]; 
		const unsigned char* HK_RESTRICT cd = (const unsigned char*)data;
		int numBytes = g_storageSizes[storage];

		for (int i = 0; i < numBytes; i++)
		{
			v[i] = cd[i];
		}
	}
}

HK_FORCE_INLINE void hkgVertexSet::setVertexComponentData(HKG_VERTEX_COMPONENT component, int index, const double* data)
{
	HKG_VERTEX_STORAGE storage = m_storageFormats[ component ]; 
	int numBytes = g_storageSizes[storage];
	int numFloats = numBytes/sizeof(float);
	float buffer[16];
	for (int i =0; i < numFloats; i++)
	{
		buffer[i] = float(data[i]);
	}
	setVertexComponentData(component, index, (void*)buffer);
}


inline const void* hkgVertexSet::getVertexComponentData( HKG_VERTEX_COMPONENT component, int index, hkUint8* data ) const
{
	return const_cast<hkgVertexSet*>(this)->getVertexComponentData(component, index, data);
}

inline int hkgVertexSet::getComponentSize( HKG_VERTEX_COMPONENT component ) const
{
	return g_storageSizes[ m_storageFormats[component] ];
}

inline hkgDisplayContext* hkgVertexSet::getContext() const
{
	return m_context;
}

inline HKG_MATERIAL_VERTEX_HINT hkgVertexSet::getMaterialHints() const
{
	HKG_MATERIAL_VERTEX_HINT h = HKG_MATERIAL_VERTEX_HINT_NONE;

	if (m_vertexFormat & HKG_VERTEX_FORMAT_TANGENT)
	{
		h |= HKG_MATERIAL_VERTEX_HINT_TANGENTS;
	}

	if (m_vertexFormat & HKG_VERTEX_FORMAT_BLENDWEIGHTS)
	{
		h |= HKG_MATERIAL_VERTEX_HINT_BLENDING;
	}

	bool usingInstancedColors = m_context->getCurrentInstanceObject() && m_context->getCurrentInstanceObject()->getSupportsColorBuffer();
	if ( (m_vertexFormat & HKG_VERTEX_FORMAT_COLOR) || (usingInstancedColors))
	{
		h |= HKG_MATERIAL_VERTEX_HINT_VCOLOR;
	}

	if (m_context->getRenderPassInfo() & HKG_RENDER_PASS_INSTANCED)
	{
		h |= HKG_MATERIAL_VERTEX_HINT_INSTANCED;
	}

	if (m_bufferAccess & HKG_VERTEX_BUFFER_ACCESS_DYNAMIC)
	{
		h |= HKG_MATERIAL_VERTEX_HINT_DYNAMICVB;
	}

	return h;
}

inline int hkgVertexSet::getNumTextureCoordsUsed() const
{
	HKG_VERTEX_FORMAT tf = HKG_VERTEX_FORMAT_TEX0;
	int numT = 0;
	while ( (m_vertexFormat & tf) && (numT < HKG_VERTEX_COMPONENT_MAX_TCOORDS) )
	{
		tf = (HKG_VERTEX_FORMAT)( tf << 1 ); // shift to next tex coord define.
		numT++;
	}

	if (m_vertexFormat & HKG_VERTEX_FORMAT_TANGENT)
	{
		numT++;
	}

	if (m_vertexFormat & HKG_VERTEX_FORMAT_BITANGENT)
	{
		numT++;
	}

	return numT;
}

bool hkgVertexSet::isContiguous() const
{
    return true;
}

inline void* hkgVertexSet::getVertexComponentStartPointer( HKG_VERTEX_COMPONENT component )
{
	HK_ASSERT2(0x101, m_lockedBuf, "You must lock the vertex buffer before being able to query or set data in it." );
	return ( m_lockedBuf + m_storageOffsets[component] ); // offset may be to different buffer
}

inline const void* hkgVertexSet::getVertexComponentStartPointer( HKG_VERTEX_COMPONENT component ) const
{
	HK_ASSERT2(0x101, m_lockedBuf, "You must lock the vertex buffer before being able to query or set data in it." );
	return ( m_lockedBuf + m_storageOffsets[component] ); // offset may be to different buffer
}

inline int hkgVertexSet::getVertexComponentStride( HKG_VERTEX_COMPONENT component ) const
{
	return m_storageStrides[ component ];
}

inline bool hkgVertexSet::isVertexComponentAligned( HKG_VERTEX_COMPONENT component, int alignment ) const
{
	HK_ASSERT2(0x101, m_lockedBuf, "You must lock the vertex buffer before being able to query or set data in it." );

	int stride = getVertexComponentStride( component );
	HK_ASSERT(0x0, (stride > 0));
	const hkUlong v0 = (hkUlong)( m_lockedBuf + m_storageOffsets[component] ); // offset may be to different buffer
	const hkUlong v1 = (hkUlong)( m_lockedBuf + m_storageOffsets[component] + stride );
	const hkUlong nmv0 = HK_NEXT_MULTIPLE_OF(alignment, v0);
	const hkUlong nmv1 = HK_NEXT_MULTIPLE_OF(alignment, v1);
	return (nmv0==v0) && (nmv1==v1);
}

inline HKG_VERTEX_COMPONENT hkgVertexSet::componentFromFormat( HKG_VERTEX_FORMAT f )
{
	HKG_VERTEX_COMPONENT c = 0;
	if ( f > HKG_VERTEX_FORMAT_NONE)
	{
		while ( (c < HKG_VERTEX_COMPONENT_NUM) && ((f & 0x01) == 0) )
		{
			f = f >> 1;
			++c;
		}
	}
	return c;
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
