/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#ifndef HK_GRAPHICS_FONT_H
#define HK_GRAPHICS_FONT_H

#include <Common/Base/Types/hkBaseTypes.h>
#include <Graphics/Common/Texture/hkgTexture.h>
#include <Graphics/Common/Font/hkgFontDefines.h>

struct FT_FaceRec_;


/// A simple 2D texture based font that assumes an orthographic projection (but can be used with any projection).
/// Can handle any texture of a given format, as outlined in the setTexture() method description.
class hkgFont  : public hkgReferencedObject
{
	public:

			/// Callback interface for batch rendering of texts
		class TextCallback
		{
		public:

			struct TextItem
			{
				const char* m_text;
				float m_x;
				float m_y;
				float m_color[4];
			};

			virtual ~TextCallback() {}

				/// Text provider callback method.
				/// Returns a TestItem object pointer which is allocated and maintained by the callback object.
			virtual const TextItem* getText() = 0;

		};

		/// Create a platform specific font object.
		static hkgFont* (HK_CALL *create)( );

		// Simple Ascii based texture 
		void loadFromBuiltin(hkgDisplayContext* context, bool thick = false);
		
		// Unicode / OpenType font support
		bool loadFontFromFile(hkgDisplayContext* context, const char* filename);

		// Unicode / OpenType font support
		//bool loadFontFromMemory(hkgDisplayContext* context, hkUint8* memPtr, hkUint32 memSize);
	
			/// Get the pointer to the current texture for the font. It may be HK_NULL.
		inline const hkgTexture* getTexture(int i = 0) const;
		inline hkgTexture* getTexture(int i = 0);
		inline int getNumTextures() const;

			/// Setup the draw state for the font. This will adjust the render state of the
			/// context to match that expected of the font, such as alpha and depth modes.
			/// Note that this is a static call and can be called anytime, but render() calls
			/// for text should follow this call or else the render state may be altered.
		static void		HK_CALL setDrawState( hkgDisplayContext* context );

			/// Render the given text to the current context's viewport.
			/// It will assume that the camera projection is in an orthographic 
			/// projection already (but of course can be any projection) and that 
			/// the world matrix is the identity. It will always enable textures even 
			/// if the current viewport in the context  requests otherwise. Assumes 
			/// that the draw state is correct (will not check for speed reasons so you can fool it if you wish)
			/// The color is RGBA as 4 floats of 0-->1 range.
		virtual void	render( hkgDisplayContext* context, const char* text, const float x, const float y, const float color[4]) = 0;
		
			/// Render text in batch provided by the callback interface to the current context's viewport.
			/// It will assume that the camera projection is in an orthographic 
			/// projection already (but of course can be any projection) and that 
			/// the world matrix is the identity. It will always enable textures even 
			/// if the current viewport in the context requests otherwise. Assumes 
			/// that the draw state is correct (will not check for speed reasons so you can fool it if you wish)
		virtual void	renderBatch( hkgDisplayContext* context, TextCallback* callback );

			/// Resolve the upper right and lower left texture coordinates for a given 
			/// character. You can then use these coordinates to texture any polygon if
			/// you wish to use the texture outside of this class.
			/// relX and relY are 0..1 wrt to the rest of the characters
		void		resolve( hkUlong charCode, float& u0, float& v0, float& u1, float& v1, 
			float& startX, float& startY, float& relX, float &relY, float& advance);
	
			/// A quicker resolve, used to work out how long a string will be in pixels say
		void		preResolve( hkUlong charCode, float& advance );

			/// Set the character width, in pixels on the screen. The default is correct for characters
			/// of 8 pixels high and with the default texture size. Works out as about 10 pixels wide
			/// to avoid distortion.
		inline void		setCharWidth(float w);

			/// Set the character height, in pixels on the screen. The default is 8 pixels high 
		inline void		setCharHeight(float h);

			/// Get the width of the characters on the screen, in pixels.
		inline float	getCharWidth() const;
			
			/// Get the height of the characters on the screen, in pixels.
		inline float	getCharHeight() const;
	
			/// Set the width of a tab stop, in pixels.
		inline void	setTabStopSize(float pixels);

			/// Get the width of a tab stop, in pixels.
		inline float	getTabStopSize() const;

			/// Get the maximum ascent as a fraction of charHeight.
		inline float getMaxAscent() const;

			/// Get the maximum descent as a negative fraction of charHeight.
		inline float getMaxDescent() const;

	protected:

		hkgFont();
		virtual ~hkgFont();

		void renderGlyph(int textureIndex, hkUint8* destData, hkUlong c, int xTexel, int yTexel);
		void renderGlyphRect(int textureIndex, hkUint8* destDataRect, hkUlong c, int xTexel, int yTexel);
		int numPrintableChars( const char* utf8 );

		struct CharBitmap
		{
			int textureIndex;
			int age;
			float xLL;
			float yLL;
			float xUR;
			float yUR;
			float startOffsetX; // may be negative
			float startOffsetY; // may be negative
			float sizeX;
			float sizeY;
			float advance;
		};

		extArray<CharBitmap> m_bitmaps;
		extPointerMap<hkUlong, int> m_characterToBitmap;
		extArray<unsigned char> m_fontData;

		struct TextureInfo
		{
			struct GridRef
			{
				int xTexels;
				int yTexels;
			};
			extArray<GridRef> m_freeSlots;
			hkgTexture* m_texture;
		};
		extArray<TextureInfo> m_textures;

		hkgFontType		m_fontType;
		
		float			m_charWidth;
		float			m_charHeight;
		float			m_tabStopWidth;
		float			m_maxAscent;
		float			m_maxDescent;

		FT_FaceRec_*	m_face;
};

#include <Graphics/Common/Font/hkgFont.inl>

#endif

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
