/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#ifndef HKG_DISPLAY_KEYFRAME_H
#define HKG_DISPLAY_KEYFRAME_H

#include <Common/Base/Types/hkBaseTypes.h>
#include <Graphics/Common/Math/hkgMath.h>
#include <Graphics/Common/hkgObject.h>

class hkgDisplayObject;

// Graphic key frame for replay system
// A keyframe stores the minimum necessary to set the state of the display world and obtain the same render output
class hkgDisplayKeyFrame
{
	public:

		/// Base class for a keyframe
		class KeyFrame : public hkReferencedObject
		{
			public:

				HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO);

			public:

				/// Constructor
				HK_FORCE_INLINE KeyFrame(const hkReferencedObject* owner)
				:	hkReferencedObject()
				,	m_owner(owner)
				{}

			protected:

				/// Copy constructor
				HK_FORCE_INLINE KeyFrame(hkgDisplayKeyFrame& dstKfBuffer, const hkgDisplayKeyFrame& srcKfBuffer, const KeyFrame& other)
				:	hkReferencedObject()
				,	m_owner(other.m_owner)
				{}

			public:

				/// Clones this keyframe
				virtual KeyFrame* clone(hkgDisplayKeyFrame& dstKfBuffer, const hkgDisplayKeyFrame& srcKfBuffer) const = 0;

				/// Sets this frame as a linear interpolation of the given frames
				virtual void setInterpolate(const hkgDisplayKeyFrame& kfBufferStart, const KeyFrame* kfStart,
											const hkgDisplayKeyFrame& kfBufferEnd, const KeyFrame* kfEnd,
											hkgDisplayKeyFrame& kfBufferInterp, hkReal interpFactor) = 0;

				/// Returns the owning display object
				HK_FORCE_INLINE const hkReferencedObject* getOwner() const {  return m_owner;	}

			protected:

				hkRefPtr<const hkReferencedObject> m_owner;	///< The display object that produced this keyframe
		};

		// Transforms for each display object
		struct ObjectTransform : public KeyFrame
		{
			public:
				
				/// Constants
				enum
				{
					MAT_SIZE = 16,
				};

			public:

				/// Constructor
				ObjectTransform(const hkReferencedObject* owner);

			protected:

				/// Copy constructor
				ObjectTransform(hkgDisplayKeyFrame& dstKfBuffer, const hkgDisplayKeyFrame& srcKfBuffer, const ObjectTransform& other);

			public:

				/// Allocates key frame transform memory
				void allocateTransforms(hkgDisplayKeyFrame& kfBuffer, int numInstances, int numBonesPerInstance);

				/// Returns the world from instance transform
				HK_FORCE_INLINE float* accessWorldFromInstanceTm(hkgDisplayKeyFrame& kfBuffer, int instanceIdx)
				{
					return kfBuffer.accessMemory<float>(m_worldFromInstanceOffset + sizeof(float) * instanceIdx * MAT_SIZE);
				}

				HK_FORCE_INLINE const float* getWorldFromInstanceTm(const hkgDisplayKeyFrame& kfBuffer, int instanceIdx) const
				{
					return kfBuffer.getMemory<float>(m_worldFromInstanceOffset + sizeof(float) * instanceIdx * MAT_SIZE);
				}

				/// Returns the instance from bone transform
				HK_FORCE_INLINE float* accessInstanceFromBoneTm(hkgDisplayKeyFrame& kfBuffer, int instanceIdx, int boneIdx)
				{
					return kfBuffer.accessMemory<float>(m_instanceFromBoneOffset + sizeof(float) * (instanceIdx * m_numBonesPerInstance + boneIdx) * MAT_SIZE);
				}
				HK_FORCE_INLINE const float* getInstanceFromBoneTm(const hkgDisplayKeyFrame& kfBuffer, int instanceIdx, int boneIdx) const
				{
					return kfBuffer.getMemory<float>(m_instanceFromBoneOffset + sizeof(float) * (instanceIdx * m_numBonesPerInstance + boneIdx) * MAT_SIZE);
				}

				/// Gets / sets the per-bone instance identifier that will be used to match transforms during interpolation
				HK_FORCE_INLINE int getBoneInstanceId(const hkgDisplayKeyFrame& kfBuffer, int instanceIdx, int boneIdx) const
				{
					const float* ptrBoneIds = getInstanceFromBoneTm(kfBuffer, m_numInstances, 0);
					const float* ptrBoneId	= &ptrBoneIds[instanceIdx * m_numBonesPerInstance + boneIdx];

					return *reinterpret_cast<const int*>(ptrBoneId);
				}
				HK_FORCE_INLINE void setBoneInstanceId(hkgDisplayKeyFrame& kfBuffer, int instanceIdx, int boneIdx, int id)
				{
					float* ptrBoneIds	= accessInstanceFromBoneTm(kfBuffer, m_numInstances, 0);
					float* ptrBoneId	= &ptrBoneIds[instanceIdx * m_numBonesPerInstance + boneIdx];

					*reinterpret_cast<int*>(ptrBoneId) = id;
				}

				/// Returns the number of instances / bones per instance
				HK_FORCE_INLINE int getNumInstances() const			{	return m_numInstances;			}
				HK_FORCE_INLINE int getNumBonesPerInstance() const	{	return m_numBonesPerInstance;	}

				/// Clones this keyframe
				virtual KeyFrame* clone(hkgDisplayKeyFrame& dstKfBuffer, const hkgDisplayKeyFrame& srcKfBuffer) const HK_OVERRIDE;

				/// Sets this frame as a linear interpolation of the given frames
				virtual void setInterpolate(const hkgDisplayKeyFrame& kfBufferStart, const KeyFrame* kfStart,
											const hkgDisplayKeyFrame& kfBufferEnd, const KeyFrame* kfEnd,
											hkgDisplayKeyFrame& kfBufferInterp, hkReal interpFactor) HK_OVERRIDE;

		protected:

			hkUint32 m_worldFromInstanceOffset;		///< Transforms from instance local space to world
			hkUint32 m_instanceFromBoneOffset;		///< Transforms from bone local space to instance local space
			int m_numInstances;
			int m_numBonesPerInstance;
	};

	public:

		/// Return a frame which is the result of the interpolation of the two given input key frames
		static void getInterpolatedFrame(const hkgDisplayKeyFrame& startKey, const hkgDisplayKeyFrame& endKey, hkReal s, hkgDisplayKeyFrame& frameOut);

		/// Allocates memory
		HK_FORCE_INLINE hkUint32 allocMemory(hkUint32 size)
		{
			const hkUint32 alignedSize	= HK_NEXT_MULTIPLE_OF(16, size);
			const hkUint32 offset		= HK_NEXT_MULTIPLE_OF(16, m_memory.getSize());
			const hkUint32 newSize		= offset + alignedSize;

			m_memory.setSize(newSize);
			return offset;
		}

		/// Clears the frame
		HK_FORCE_INLINE void clear()
		{
			m_keyFrames.clear();
			m_memory.clear();
		}

		/// Returns a pointer to the memory
		template <typename T> const T* getMemory(hkUint32 offset) const		{	return reinterpret_cast<const T*>(&m_memory[offset]);	}
		template <typename T> T* accessMemory(hkUint32 offset)				{	return reinterpret_cast<T*>(&m_memory[offset]);			}

	public:

		hkArray< hkRefPtr<KeyFrame> > m_keyFrames;
		hkArray<hkUint8> m_memory;

		int m_numSolidObjects;
		int m_numAlphaObjects;
		int m_numMixedObjects;
};

#endif //HKG_DISPLAY_KEYFRAME_H

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
