/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#ifndef HK_GRAPHICS_DISPLAY_CONTEXT_H
#define HK_GRAPHICS_DISPLAY_CONTEXT_H

#include <Common/Base/Types/hkBaseTypes.h>
#include <Common/Base/Types/Color/hkColor.h>
#include <Graphics/Common/Shader/hkgShader.h>
#include <Graphics/Common/hkgObject.h>
#include <Graphics/Common/DisplayContext/hkgDisplayContextDefines.h>
#include <Graphics/Common/DisplayContext/hkgDisplayStatistics.h>

// We include the window header so that we can use it for 
// m_owner inlined (must be after the above defs)

class hkgViewport;
class hkgTexture;
class hkgVertexSet;
class hkgShaderContext;
class hkgShaderEffect;
class hkgShaderEffectCollection;
class hkgShaderLib;
class hkgMaterial;
class hkgWindow;
class hkgBlendMatrixSet;
class hkgFontLibrary;
class hkgInstancedDisplayObject;

#define HKG_CHECK_CONTEXT_LOCK_STATUS(ctx) HK_ASSERT2(0x4354260, ctx->isLocked(), "You are making graphics calls without locking the display context") 

// Real-time per-frame gpu statistics can be turned by this define ( DX11 only for now )
//#define HK_GPU_STATS

#ifdef HK_GPU_STATS
#	define HK_ON_GPU_STATS(CODE)		CODE
#else
#	define HK_ON_GPU_STATS(CODE)
#endif

/// The display context. One of the core classes in the graphics library. It represents the rendering state of
/// graphics hardware / underlying API, so that the library can work with any API even if the API stores no state 
/// itself, like the pure DirectX H/W abstraction layer on the Xbox. The display context not only stores the state 
/// but is a virtual abstraction so that direct state changes can be made on all platforms and so we can do 
/// immediate mode rendering through this class as well.
/// It exposes a limited number of render state options, representing the lowest common denominator required to
/// produce sufficient graphical apps.
/// You will see that the user never creates a context (there is no public constructor), as there is only one per 
/// piece of hardware / API context and it is created as needed.

struct hkgRenderCaps
{
	// General
	hkUint32 m_maxDedicatedVideoMem; //in MB
	hkBool m_supportsMRT;
	hkBool m_supportsOIT;
	hkBool m_supportsPostEffects;
	hkBool m_supportsDisplacementMapping;


	// Textures
	hkUint16 m_maxAnisotropy;
	hkUint16 m_maxNumTextures;
	hkUint16 m_maxTexture1DDimension;
	hkUint16 m_maxTexture2DDimension;
	hkUint16 m_maxTexture3DDimension;
	hkUint16 m_maxTextureCubeDimension;
	hkBool m_supportsDXT;
	hkBool m_supportsNamedSamplers;
	hkBool m_supportsHDR;

	// Shaders
	hkBool m_shaderCompilationSupported;
	hkUint8 m_vertexShaderMinorVersion;
	hkUint8 m_vertexShaderMajorVersion;
	hkUint8 m_geometryShaderMinorVersion;
	hkUint8 m_geometryShaderMajorVersion;
	hkUint8 m_hullAndDomainShaderMinorVersion;
	hkUint8 m_hullAndDomainShaderMajorVersion;
	hkUint8 m_pixelShaderMinorVersion;
	hkUint8 m_pixelShaderMajorVersion; 
	extArray<hkgShader::ShaderDefine> m_platformShaderDefines;
	

	// Lighting
	hkUint32 m_maxNumLights;

	// Deform
	HKG_MESHBLENDING_SUPPORT m_automaticSkinSupport;
	hkUint16 m_maxSkinBlendMatrices;
	hkBool m_favorHardwareSkinning; // normally if vshader >= 2 we do it, but some platforms don't have a good enough VS to make it worth while
	
	void init();
	inline void copyShaderDefines( extArray<hkgShader::ShaderDefine>& out ) const; 

};

class hkgDisplayContext : public hkgReferencedObject
{
		friend class hkgWindow;
	public:

		// Debug statistics
#ifdef HK_GPU_STATS
		hkgDisplayStatistics     m_statistics;
		hkgDisplayStatistics m_prevStatistics;
#endif

	public:

			/// This allows you to find out the limitations of the current renderer
			/// Use it to limit your demo setup etc
		inline const hkgRenderCaps& getRenderCaps() const;

		
			// Returns new frame id.
		virtual int advanceFrame();
		inline int getCurrentFrameId() const; 

			/// Get the current vertex stream options. Most platforms have a concept of a current
			/// source for each vertex component or set of components, and sometimes it is desirable
			/// to disable all of them, or to selectively disable parts.
		inline HKG_VERTEX_OPTIONS getVertexOptions() const;

			/// Set the current vertex stream options. Most platforms have a concept of a current
			/// source for each vertex component or set of components, and sometimes it is desirable
			/// to disable all of them, or to selectively disable parts.
		inline void setVertexOptions( HKG_VERTEX_OPTIONS o );

			/// Set the current viewport. The viewport is hook though which the context can figure
			/// out what portion of the framebuffer it is using and what state is desired for that view
			/// This method just sets the internal current viewport to point to v. The viewport is not 
			/// referenced by the context (does not inc ref count).
		inline void				setCurrentViewport(hkgViewport*	v);

			/// Get the current viewport. The viewport is hook though which the context can figure
			/// out what portion of the framebuffer it is using and what state is desired for that view
			/// The viewport is not referenced by the context (does not inc ref count).
		inline hkgViewport*		getCurrentViewport() const;
		
			/// Get the current active 2D texture. The texture is referenced by the context (incs ref count).
			/// This texture will be used when texturing is enabled.
		inline hkgTexture*		getCurrentTextureByOrder(int bindOrder) const;

		inline hkgMaterial*		getCurrentMaterial() const;
		inline HKG_MATERIAL_VERTEX_HINT getCurrentMaterialHints() const;

			/// Get the context from which shaders will draw their context information (input 'constants')
		inline hkgShaderContext* getShaderContext() const;

			/// Get the current active set of shaders.
		inline hkgShaderEffect*		getCurrentShaderEffect() const;
			
		virtual hkgShaderEffectCollection* getCurrentGlobalShaderEffectCollection( ) const;

			/// Get the current active set of blend matrices.
		inline hkgBlendMatrixSet* getCurrentBlendMatrices() const;

			/// Get the current active set of vertices.
		inline hkgVertexSet* getCurrentVertexSet() const;

		virtual void 			setCurrentSoleTexture(hkgTexture* t, HKG_TEXTURE_MODE mode); // will bind to it aswell, and unbind all other stages (if any)
		void					setCurrentTextures( hkgTexture*const* t, const HKG_TEXTURE_MODE* modes, int numT ); // will 0 all others,. and will assume stage 0 start
	
		void					refreshNamedTextures(); 
		void					clearCurrentTextures();

			/// Set the current material. Can be null (will unbind current if so)
			/// Will no bind if is already the current material
		virtual void		    setCurrentMaterial(class hkgMaterial* m, const hkgVertexSet* vertexSetHint ); // will bind to it aswell

			/// Set the current active shader. 
		virtual void		    setCurrentShaderEffect(hkgShaderEffect* s, const hkgVertexSet* vertexSetHint ); // will bind to it aswell
		
		void					setCurrentGlobalShaderEffectCollection( hkgShaderEffectCollection* globalShaders );

			/// Set the current active blend matrix set
		virtual void		    setCurrentBlendMatrices(hkgBlendMatrixSet* bm); // will bind to it aswell

			/// Set the current active blend matrix set
		virtual void		    setCurrentVertexSet(hkgVertexSet* vs); // will bind (setStreams) to it aswell

			/// A context has a window that owns it. Each window has a context at the moment, so there
			/// is a one to one mapping from window to context.
		inline const hkgWindow*	getOwner() const;

			/// A context has a window that owns it. Each window has a context at the moment, so there
			/// is a one to one mapping from window to context.
		inline hkgWindow*		getOwner();

			/// Get the current render state.The returned value is a bitfield of enabled options.
		inline HKG_ENABLED_STATE  getEnabledState() const;	
		
			/// Enable/Disable per vertex lighting..
		virtual void setLightingState(bool on) = 0;

			/// Enable/Disable 2D texture mapping, using the current texture as the map. 1D & 3D textures 
			/// not supported yet.
		virtual void setTexture2DState(bool on) = 0;

			/// Enable/Disable alpha blending. If blending is enabled, a (src alpha, 1-src alpha) style 
			/// blend is enabled, which is a common transparent surface rendering mode. Modes such as 
			/// additive blending are not yet supported through this interface.
		virtual void setBlendState(bool on) = 0;

			/// Enable/Disable whether we read the Z buffer when doing the pixel updates. If z read is 
			/// turned off, we bypass the z buffer test. If z read is on, the z test is update on 
			/// 'less than'. Typically for say transparent objects, we render solids first with z read 
			/// and write on, then render the transparents with z read on, but z write off.
		virtual void setDepthReadState(bool on) = 0;

			/// Enable/Disable whether we write to the Z buffer when doing the pixel updates. If z write
			/// is turned off, even if the pixel passed the z test during the z read, we bypass the z buffer
			/// update of the z value and update the color only. Typically for say transparent objects, we 
			/// render solids first with z read and write on, then render the transparents with z read on, 
			/// but z write off.
		virtual void setDepthWriteState(bool on) = 0;

			/// Set the depth bias for the Z buffer when doing the pixel updates. This will offset the
			/// depth by the biased amount while rendering, allowing you to render the same geometry twice
			/// but over the preceding one without turning depth read/write off. This is useful for shadows
			/// or when drawing a wireframe over the actual mesh.
		virtual void setDepthBias(float val) = 0;

			/// Turn on or off backface culling. Although it may seem obvious the backface culling 
			/// would improve speed, it is really system dependant as it may cause blocks in the 
			/// triangle pipeline, especially on PlayStation(R)2 where the culling is done in the VU and has to break up
			/// the triangle strips for instance if culling is enabled. 	
		virtual void setCullFaceState(bool on ) = 0;

			/// Turn on or off rendering in a wireframe mode. On GameCube this is done a linestrips so 
			/// is slower compared to other platforms that have native support for this mode.
		virtual void setWireframeState(bool on) = 0;
		
			/// Enable or disable the given light number. It assumes that if you are enabling a given light
			/// that you have set them up using the hkgLight class. This is merely a state change.
		virtual void setLightState(int light, bool on) = 0;

		virtual void setTwoSidedLightingState(bool twoSided) { /* most platforms don't suppport two sided by default */ }

			/// Enable or disable 2d scissoring.
		virtual void setScissorState(bool on);

			/// Enable/Disable (per pixel) fog
		virtual void setFogState(bool on);
			
			/// Set the fog parameters. Fnear and ffar are from 0..1, with respect to camera near/far plane.
		virtual void setLinearFogParameters(float fnear, float ffar );
	
			/// Only makes sense if mode is exp or exp2, with associated density (1/(e^d*density) etc)
		virtual void setExpFogParameters(HKG_FOG_MODE fogMode, float density  );

			/// Get fog parameters.
		virtual void getFogParams( HKG_FOG_MODE& fogMode, float& fnear, float& ffar, float& density);

			/// Set the scissor rectangle.
		virtual void setScissorRect(int ll_x, int ll_y, int ur_x, int ur_y );

		virtual void setLineWidth(float v) { } // not all platforms support this concept for wireframe etc. < 0 == default

			/// Alter if needed the current state to match the given desired state. This is used to 
			/// make sure that the context matches your desired rendering view, without having to query 
			/// each state in turn. Default args are dangerous on this func as modfes will change in a 
			/// non obvious way
		void matchState( HKG_ENABLED_STATE desiredState, 
						HKG_CULLFACE_MODE desiredCullfaceMode /*= HKG_CULLFACE_CW*/,
						HKG_BLEND_MODE desiredBlendMode /*= HKG_BLEND_MODULATE*/,
						HKG_ALPHA_SAMPLE_MODE desiredAlphaSampleMode /*= HKG_ALPHA_SAMPLE_COVERAGE*/ );
		
			/// This method makes the underlying hardware state match that assumed by this context. This
			/// Is required at startup to get the hardware in a set state, and also after the state of
			/// the hardware becomes undetermined (for instance after D3D resets once it loses context).
			/// The method simply calls 'set state' on each controlled state with the context idea of which
			/// is enabled.
		void syncState(); 
		
			/// Two modes, CW and CCW. Specify false as second arg if you want CCW (in a right hand system)
			/// culling.
		inline HKG_CULLFACE_MODE getCullfaceMode() const;

			/// Two modes, CW and CCW. Specify false as second arg if you want CCW (in a right hand system)
			/// culling. Will set the mode and call setCullFaceState() to reflect the change if culling is on.
		inline void setCullfaceMode( HKG_CULLFACE_MODE m );

			/// Additive or Modulated blend mode.
		inline HKG_BLEND_MODE getBlendMode() const;

			/// Additive or Modulated blend mode.
		inline void setBlendMode( HKG_BLEND_MODE m );

			/// AlphaToCoverage support etc
		inline HKG_ALPHA_SAMPLE_MODE getAlphaSampleMode() const;

			/// AlphaToCoverage support etc
		inline void setAlphaSampleMode( HKG_ALPHA_SAMPLE_MODE m );

			/// How color is set
		inline HKG_COLOR_MODE getColorMode() const;

			/// How color is set
		inline void setColorMode( HKG_COLOR_MODE c );

			/// What type of debugging we would like to do
		inline HKG_DEBUG_FLAGS getDebugFlags() const;

		inline void setDebugFlags( HKG_DEBUG_FLAGS c );

			/// What type of rendering are we doing
		inline HKG_RENDER_PASS_INFO getRenderPassInfo() const;

		inline void setRenderPassInfo( HKG_RENDER_PASS_INFO p );

			/// What, if any, stage is locked externally (with a shadow map for instance)
		inline HKG_TEXTURE_STAGE_LOCK_MODE getTextureStageLockMode() const;
		inline int getTextureStageLockCount() const;

		inline void setTextureStageLockMode( HKG_TEXTURE_STAGE_LOCK_MODE l );
		inline void setTextureStageLockCount( int c );

		// Immediate mode rendering

			/// Pop matrix will take the current modelview matrix off the stack and restore
			/// the previous matrix as the current matrix.
		virtual void popMatrix() = 0;

			/// Push matrix will place the current modelview matrix on the stack. It does not need
			/// to alter the current matrix.
		virtual void pushMatrix() = 0;

			/// Multiply the current modelview matrix by m. Matrices in this graphics lib are OpenGL style,
			/// so are column major, so item(row, col) is m[row + col*4].
		virtual void multMatrix( const float* m) = 0;

			/// Begin a set of immediate mode rendering primitives. Do not alter state while inside a group.
			/// This is quite similar to glBegin( ), where HKG_IMM_GROUP defines how the items will be interpreted.
			/// Call endGroup() once all vertices have been defined for the group.
		virtual void beginGroup(HKG_IMM_GROUP type) = 0;

			/// Set the current color, RGB as floats 0.0-->1.0. Alpha will be set as 1.0
		virtual void  setCurrentColor3( const float* c) = 0;

			/// Set the current color, RGBA as floats 0.0-->1.0
		virtual void  setCurrentColor4( const float* c) = 0;
		
			/// Set the current color in packed form: [ARGB] == 0xAARRGGBB
		virtual void  setCurrentColorPacked( hkColor::Argb c ) = 0; 

			/// Set the current texture coordinate uv[0] = u, uv[1] = v.
			/// These can be <0 or >1, and the clamp mode is 'repeat' at the moment.
			/// Be careful on some platforms very high u or v values can cause artifacts
			/// such as on PlayStation(R)2 with u or v in the 100s.
		virtual void  setCurrentTextureCoord(const float* uv) = 0;

			/// Set the current normal to be used in per vertex lighting. If lighting is disabled
			/// then there is not much point in specifying a normal, but a default normal will be
			/// assumed.
		virtual void  setCurrentNormal(const float* n) = 0;

			/// Set the vertex position. This call will define the vertex, so all 'current' attributes 
			/// for the vertex will be used as of this point, so all color,normal,etc. must be set before 
			/// this call to affect this vertex.
		virtual void  setCurrentPosition(const float* p) = 0;

		void  setCurrentPosition(const double* p){ float f[3]; f[0] = float(p[0]); f[1] = float(p[1]); f[2] = float(p[2]); setCurrentPosition(f); }


			/// End the group. This will batch all the vertices together that where specified after the last
			/// beginGroup(). There is no guarantee that the vertices will appear in the buffer straight away
			/// until flush is called. Make sure you specify enough vertices to satisfy the type of primitive
			/// you are rendering.
		virtual void endGroup(bool useCurrentShaders = false) = 0;

			/// Flush the pipeline of commands to the hardware. After this call it can be assumed that all
			/// outstanding rendering has been completed.
		virtual void flush() = 0;

			// Handy debug routines. Optionally set the state to match (and restore) a default state (no lighting etc).
		void drawUnitAxis(bool setState = true);

			/// Feed the shader constants. If you use packed normals etc in the vertex set, you may need to 
			/// pass a representative vertex set here for the shader to see the unpack scales etc
		void feedShaders(bool onlyFeedDirty, const hkgVertexSet* vertexHints);

			// Owner will not be reference() counted against this object
		static hkgDisplayContext* (*create)(hkgWindow* owner);

		virtual void lock() const;
		bool isLocked() const;
		virtual void unlock() const;

		// For dynamic textures and vertex sets, normally vid reset will cause invalidation in dx renderers etc
		inline void registerTexture(hkgTexture* t) const;
		inline bool isRegisteredTexture(hkgTexture* t) const;
		inline void unregisterTexture(hkgTexture* t) const;
		
		inline void registerVertexSet(hkgVertexSet* vs) const;
		inline bool isRegisteredVertexSet(hkgVertexSet* vs) const;
		inline void unregisterVertexSet(hkgVertexSet* vs) const;


		inline int getNumCurrentTextures() const;

		void setGamma( float g ); // eg 1..3. Default 2.2 
		inline float getGamma() const; 

		void setExposure( float e ); // eg -10..10. Default 0
		inline float getExposure() const ; 

		virtual void setHdrEnabled(bool on);
		inline bool getHdrEnabled() const;

		virtual void setRigidSkinningEnabled(bool on);
		inline bool getRigidSkinningEnabled() const;

		inline void getColorRangeParams(float* f ) const; // { color range correction enabled,  1/gamma,  pow(2,exposure) }

			// For a given scene, bias is the amount (default 0.0001) by which to offset the test by (in the depths when using VSM
			// are from 0..1 where 1 is the furthest, so 0.0001 is 0.01% of the depth of the shadow region, so for big scenes you can make this smaller.
			// The epsilon is controls the variance, so that it does not go to 0, so is just a small number, but can be used to alter the
			// influence of the variance (has to be positive though).
		virtual void setShadowVsmBias( float bias, float epsilon );
		virtual void getShadowVsmBias( float& bias, float& epsilon );

		void setAlphaScale(float s);
		float getAlphaScale() const;

		inline hkgFontLibrary* getFontLibrary() const;
		void setFontLibrary(hkgFontLibrary* l);

		inline hkgMaterial* getDefaultMaterial(); // alter as you please

		virtual hkgShaderEffectCollection* getDefaultShaderEffectCollection() const { return HK_NULL; } // may be null, what is used if the material haas no shaders on platforms that require them
		void rebindTextures();

		void setCurrentInstanceObject(const hkgInstancedDisplayObject* inst);
		inline const hkgInstancedDisplayObject* getCurrentInstanceObject() const { return m_currentInstance; }


			// Debug statistics
#ifdef HK_GPU_STATS
		inline void advanceStatistics( void )
		{
			m_prevStatistics = m_statistics;

			m_statistics.clearAll();
		}
#else
		inline void advanceStatistics( void ) {}
#endif

		struct TextureUseStats
		{
			hkgTexture* m_tex;
			int m_allocatedSize;
			int m_numMips;
			bool m_compressed;
		};
		hkArray<TextureUseStats> m_allocatedTextures;
		TextureUseStats& findOrMakeTextureStats(hkgTexture* t);
		void removeTextureStats(hkgTexture* t);

		void reportTextureStats();

		/// Sets the override material
		void setMaterialOverride(hkgMaterial* mtlOverride);
		HK_FORCE_INLINE const hkgMaterial* getMaterialOverride() const	{	return m_mtlOverride;	}
		HK_FORCE_INLINE hkgMaterial* accessMaterialOverride()			{	return m_mtlOverride;	}

		/// Returns the default material override
		hkgMaterial* getDefaultMaterialOverride();

		/// Gets / sets the shader lib
		HK_FORCE_INLINE void setShaderLib(hkgShaderLib* shaderLib)		{	m_shaderLib = shaderLib;	}
		HK_FORCE_INLINE const hkgShaderLib* getShaderLib() const		{	return m_shaderLib;	}
		HK_FORCE_INLINE hkgShaderLib* accessShaderLib()					{	return m_shaderLib;	}

	public:

		struct BoundTexture
		{
			BoundTexture() { }
			BoundTexture(hkgTexture* t, int s, int bs) : m_tex(t), m_bindIndex(s), m_bindStage(bs) { }
			hkgTexture* m_tex;
			int m_bindIndex;  // out 0 based index texture stage. Can be -1 for named texture matches
			int m_bindStage;  // actual h/w stage
		};

	
	protected:

		hkgDisplayContext(hkgWindow* owner); 
		virtual void cleanup();
		virtual ~hkgDisplayContext();
		
		virtual void computeRenderCaps(hkgRenderCaps& c) const = 0;

		virtual void setCurrentTextureByOrder(int orderIndex, hkgTexture* t, HKG_TEXTURE_MODE mode); // will bind to it aswell
		virtual void setCurrentTextureDirect(int actualBindStage, hkgTexture* t, HKG_TEXTURE_MODE mode); // will bind to it aswell

		void computeThickLineOffset( float s[3], float e[3], int pixels, float offset_s[3], float offset_e[3] ) const; 

		mutable hkgRenderCaps m_currentRenderCaps;
		mutable bool		  m_currentCapsDirty; // Turning on msaa etc may change cur caps
	
		HKG_ENABLED_STATE   m_currentState;
		HKG_VERTEX_OPTIONS  m_vertexOptions;	
		HKG_CULLFACE_MODE   m_cullfaceMode;
		HKG_BLEND_MODE	    m_blendMode;
		HKG_ALPHA_SAMPLE_MODE	m_alphaSampleMode;
		HKG_COLOR_MODE		m_colorMode;
		HKG_FOG_MODE		m_fogMode;
		float				m_fogNear;
		float				m_fogFar;
		float				m_fogDensity;
		HKG_DEBUG_FLAGS	    m_debugFlags;
		HKG_RENDER_PASS_INFO m_renderPass;
		HKG_TEXTURE_STAGE_LOCK_MODE m_textureStageLock;
		int					m_textureStageLockCount;
		int					m_frameId;
		hkgWindow*		    m_owner;
		hkgViewport*	    m_currentViewport;

		float				m_exposure;
		float				m_powTwoExposure;
		float				m_oneOverGamma; 
		bool				m_hdrEnabled;
		bool				m_rigidSkinningEnabled;
	

		

		inline int findBoundTextureIndexByOrder(int textureOrder, int startIndexHint = 0) const;
		extArray<BoundTexture>  m_currentBoundTextureStages; // ordered to make search faster
		bool			m_currentBoundTextureAreNamed;

		//if all object share a material in a group there is no need to rebind to it
		hkgMaterial*		m_currentBoundMaterial;
		HKG_MATERIAL_VERTEX_HINT m_currentBoundMaterialHints;

			// Shader support
		mutable hkgShaderContext*   m_shaderContext;
		hkgShaderEffect*			m_currentBoundShaderEffect;
		hkgShaderEffectCollection*	m_globalShaderEffectCollection;

		hkgBlendMatrixSet*  m_currentBlendMatrices;
		hkgVertexSet*		m_currentVertexSet;
		hkgShaderEffect*    m_currentVertexSetShaderEffectUsed;
		const hkgInstancedDisplayObject* m_currentInstance;

		unsigned int		m_maxVerts; // usually (2^16 - 1) == 65,535 verts. The context will split to this 
										// boundary so the user does not need to worry about it.

		float	m_shadowVsmBias;
		float   m_shadowVsmEpsilon;
		float	m_alphaScale;

		hkgFontLibrary* m_fontLib;
		hkgShaderLib* m_shaderLib;
		hkgMaterial* m_defaultMaterial;
		hkgMaterial* m_defaultMtlOverride;	/// A default material that can be used to override any other material
		hkgMaterial* m_mtlOverride;

		mutable extArray<hkgTexture*> m_texturePalette;
		mutable extArray<hkgVertexSet*> m_vertexSets;

		mutable class extCriticalSection m_contextLock;
		static HK_THREAD_LOCAL(int) m_threadLocalLockCount;

};

class hkgDisplayContextAutoLock
{
	public:
			/// Ctor
		hkgDisplayContextAutoLock(hkgDisplayContext* context)
			:	m_context(context)
		{
			context->lock();
		}
			/// Dtor
		~hkgDisplayContextAutoLock()
		{
			m_context->unlock();
		}
	private:
		hkgDisplayContext* m_context;
};

#include <Graphics/Common/DisplayContext/hkgDisplayContext.inl>

#endif //HK_GRAPHICS_DISPLAY_CONTEXT_H

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
