/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#ifndef HKG_SERIES_GRAPH_H_
#define HKG_SERIES_GRAPH_H_

#include <Graphics/Common/hkgObject.h>
#include <Common/Base/Types/Color/hkColor.h>

class hkgDisplayContext;
class hkgFont;


/// This class acts as the base class for graphs in the demo framework. It stores an arbitrary number of real-valued
/// series of positive numbers coming from a fixed number of channels (e.g from different threads).
/// When the render function is called it will draw a graph of all the series for each channel.
class hkgSeriesGraph : public  hkgReferencedObject
{
	public:

		enum Format 
		{
			INT_VALUES,	 // no decimal part
			SI_PREFIX,	 // thousands are converted to k,  millions to M, etc
			FLOAT_VALUES // 3 digits
		};

	public:

		HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DISPLAY);

		hkgSeriesGraph();
		virtual ~hkgSeriesGraph();

		//
		// Sampling
		//

		/// Set the number of channels available to all series.
		void setNumChannels( int numChannels );

		/// Adds a new data series. It won't create the same series twice.
		void addSeries( const char* name, int color = hkColor::BLACK, bool  thick = false);

		/// Removes the given series
		void removeSeries( const char* name );

		/// Adds a data point in the given series from the given channel.
		void addData( const char* name, int channelIndex, hkReal value );

		/// Adds a data point in the given series from the given channel.
		void addData( int seriesIndex, int channelIndex, hkReal value );

		/// Update scale to avoid spikes etc. Should be called after all the data has been added, and before render()
		void postProcess();

		//
		// Rendering
		//

		/// Set the render target. Can be called multiple times (e.g. for resize).
		bool setDisplayContext( hkgDisplayContext* context, int width , int height ); 

		/// Set the label format.
		void setFormat( Format format, const char* unit );

		/// Render the graphs.
		void render( int xPos, int yPos, bool useCurrentViewport = false );

	protected:

		/// A series of data coming from several channels.
		struct Series
		{
			HKG_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR ( hkgSeriesGraph::Series );

			/// All data from the same channel is stored in a ring buffer of values.
			struct Channel
			{
				HKG_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( hkgSeriesGraph::Series::Channel );

				extArray< hkReal > m_samples; ///< Ring buffer of samples
				hkReal m_lastValue;			 ///< The last added value
				int m_head;					 ///< Index where to store the next value
				hkReal m_max;				 ///< Maximum current value
				hkReal m_peak;				 ///< Historical peak value
			};

			/// Initialize the series. It is possible to reinitialize it when the size of the graph changes.
			void reset( int numChannels, int numSamples );

			hkStringPtr m_name;				///< Name of the series
			hkColor::Argb m_color;			///< Color of the graph
			hkBool32 m_thick;				///< Whether to draw thick lines or not
			extArray<Channel> m_channels;	///< Data for each channel
		};

	protected:

		// Calculate legend width, samples per channel, etc based on the render dimensions.
		void refit();

		// Render all the charts and the tick labels
		void renderCharts( int xPos, int yPos, int height );

		// Render the given chart
		void renderChart( int xPos, int yPos, int width, int height, int channelIdx, hkReal scaleY, hkReal tickIncrement );

		// Render the legend.
		void renderLegend( int xPos, int yPos );

		// The series of data.
		extArray< Series* > m_series;

		// Display variables.
		hkRefPtr<hkgDisplayContext> m_context;
		hkUint32 m_width;
		hkUint32 m_height;
		hkUint32 m_padding;
		hkUint32 m_labelWidth;
		hkUint32 m_legendWidth;
		hkgFont* m_font;
		Format m_format;
		const char* m_unit;

		int m_numChannels; 
		hkUint32 m_numSamplesPerChannel;	// == the pixel width of each chart

		// Scale 
		hkReal m_lastScaleY;
};


#endif // HKGSERIESGRAPH_H_

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
