/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

//
// Havok 2.0 Visual Debug Display Manager
//

#ifndef HKG_DISPLAY_HANDLER
#define HKG_DISPLAY_HANDLER

#include <Common/Base/hkBase.h>
#include <Common/Base/Ext/hkBaseExt.h>
#include <Graphics/Common/hkgObject.h>

#include <Common/Visualize/hkDebugDisplayHandler.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>

static const int HKG_DISPLAY_HANDLER_DEFAULT_GRID_SPAN = 200;
static const int HKG_DISPLAY_HANDLER_DEFAULT_GRID_BASE = 10;
static const hkReal HKG_DISPLAY_HANDLER_DEFAULT_LINE_GAP = 5; // in pixels

class hkDisplayGeometry;
class hkgDisplayWorld;
class hkgDisplayContext;
class hkgDisplayWindow;
class hkgDisplayObject;
class hkgMeshSystem;
class hkgMeshBody;
class hkgCamera;
class hkgShaderLib;

class hkgKeyboard;

class hkgDisplayHandler : public hkgReferencedObject, public hkDebugDisplayHandler
{
	public:

		HKG_DECLARE_CLASS_ALLOCATOR();
		hkgDisplayHandler(hkgDisplayWorld* displayWorld, hkgDisplayContext* displayContext, class hkgWindow* window);
		virtual ~hkgDisplayHandler();

		//
		// Display handler functions
		//
		using hkDebugDisplayHandler::addGeometry;
		virtual hkResult addGeometry(const hkArrayBase<hkDisplayGeometry*>& geometries, const hkTransform& transform, hkUlong id, int tag, hkUlong shapeIdHint, hkGeometry::GeometryType geometryType = hkGeometry::GEOMETRY_STATIC) HK_OVERRIDE;
		virtual hkResult addGeometryInstance(hkUlong originalInstanceId, const hkTransform& transform, hkUlong id, int tag, hkUlong shapeIdHint) HK_OVERRIDE;
		virtual hkResult setGeometryPickable(hkBool isPickable, hkUlong id, int tag) HK_OVERRIDE;
		virtual hkResult setGeometryVisibility(int geometryIndex, bool isVisible, hkUlong id, int tag) HK_OVERRIDE;
		virtual hkResult setGeometryColor( hkColor::Argb color, hkUlong id, int tag) HK_OVERRIDE;
		virtual hkResult setGeometryColor(hkColor::Argb color, hkgDisplayObject* displayObject);
		virtual hkResult setGeometryTransparency(float alpha, hkUlong id, int tag) HK_OVERRIDE;
		virtual hkResult updateGeometry(const hkTransform& transform, hkUlong id, int tag) HK_OVERRIDE;
		virtual hkResult updateGeometry( const hkMatrix4& transform, hkUlong id, int tag ) HK_OVERRIDE;
		virtual hkResult updateGeometries(const hkTransform* transforms, const hkUlong* ids, int numIds, int tag) HK_OVERRIDE;
		virtual hkResult skinGeometry(hkUlong* ids, int numIds, const hkMatrix4* poseModel, int numPoseModel, const hkMatrix4& worldFromModel, int tag ) HK_OVERRIDE;
		virtual void addTextureSearchPath(const char* path) HK_OVERRIDE;
		virtual void clearTextureSearchPaths() HK_OVERRIDE;
		virtual hkResult removeGeometry(hkUlong id, int tag, hkUlong shapeIdHint) HK_OVERRIDE;
		virtual hkResult updateGeometryVerts( hkUlong geomID, const hkArray<hkVector4>& newVerts, int tag ) HK_OVERRIDE;

		virtual hkResult updateCamera(const hkVector4& from, const hkVector4& to, const hkVector4& up, hkReal nearPlane, hkReal farPlane, hkReal fov, const char* name) HK_OVERRIDE;

		virtual hkResult displayPoint(const hkVector4& position, hkColor::Argb color, int id, int tag) HK_OVERRIDE;
		virtual hkResult displayLine(const hkVector4& start, const hkVector4& end, hkColor::Argb color, int id, int tag) HK_OVERRIDE;
		virtual hkResult displayTriangle(const hkVector4& a, const hkVector4& b, const hkVector4& c, hkColor::Argb color, int id, int tag) HK_OVERRIDE;
		virtual hkResult displayText(const char* text, hkColor::Argb color, int id, int tag) HK_OVERRIDE;
		virtual hkResult display3dText(const char* text, const hkVector4& pos, hkColor::Argb color, int id, int tag) HK_OVERRIDE;

		// The following displayXXX2d functions add render elements to lists of 2d display primitives.
		// Position has to be supplied in normalized window coordinates (z and w components ignored)
		// Providing 0x00FFFFFF as color is regarded as a dummy element for determining the order of rendering;
		// for all elements of a 2d primitive after the x-th dummy element in the primitive's list of render elements, 
		// all elements of all 2d primitives before the x-th dummy element in their respective lists will be rendered earlier.
		// Rendering of the set of primitives between the x-th and (x+1)-th dummy elements corresponds in the order of
		// the functions listed below.
		virtual hkResult displayTriangle2d(const hkVector4& a, const hkVector4& b, const hkVector4& c, hkColor::Argb color, int id, int tag) HK_OVERRIDE;
		virtual hkResult displayPoint2d(const hkVector4& position, hkColor::Argb color, int id, int tag) HK_OVERRIDE;
		virtual hkResult displayLine2d(const hkVector4& start, const hkVector4& end, hkColor::Argb color, int id, int tag) HK_OVERRIDE;
		virtual hkResult displayText2d(const char* text, const hkVector4& pos, hkReal sizeScale, hkColor::Argb color, int id, int tag) HK_OVERRIDE;

		virtual hkResult displayAnnotation(const char* text, int id, int tag) HK_OVERRIDE;

		virtual hkResult displayGeometry(const hkArrayBase<hkDisplayGeometry*>& geometries, const hkTransform& transform, hkColor::Argb color, int id, int tag) HK_OVERRIDE;
		virtual hkResult displayGeometry(const hkArrayBase<hkDisplayGeometry*>& geometries, hkColor::Argb color, int id, int tag) HK_OVERRIDE;

		virtual hkResult sendMemStatsDump(const char* data, int length) HK_OVERRIDE;

		virtual hkResult holdImmediate() HK_OVERRIDE; // does nothing as single buffered
		virtual hkResult step(hkReal frameTimeInMs);

		virtual void lockForUpdate() HK_OVERRIDE;
		virtual void unlockForUpdate() HK_OVERRIDE;

		void drawImmediate(); // draw
		void clear(); // clear the lists

		void clearAndDeallocate();

		void setImmediateModeEnabled(bool on); // starts of enabled, but sometimes it is ahndy to just chop ouit 
		bool getImmediateModeEnabled() const { return m_allowImmediateMode; }

		//
		// Functions specific to the hkg display handler
		//

		//
		// Mapping: "id/type" to "display object"
		//
		hkUlong getDisplayObjectId(const hkgDisplayObject* displayObject);

		hkResult setPickable(hkUlong id, hkBool isPickable);

			// will not hold ref to obj
			// Note that the id at the moment is the addr of the collidable for the obj
		void addPrecreatedDisplayObject( hkUlong id, hkgDisplayObject* obj );
		hkgDisplayObject* getPrecreatedDisplayObject( hkUlong id );
		void removePrecreatedDisplayObject( hkUlong id );

		hkgDisplayObject* findDisplayObject( hkUlong id );
		const extPointerMap<hkUlong, hkgDisplayObject*>& getObjectMap() { return m_objectMap; }

		void clearCache(); // will just remove all cached display objs
		void clearDisplay(); // will remove all added geoms etc, including the cached objs

			/// By default if there are no precreated display objects registered
			/// for a body that comes in through addGeometry, this handler will
			/// create an ugly but usefull display version for it.
			/// You can turn this off with this flag.
		void setAutoDisplayCreationState( bool on );
		bool getAutoDisplayCreationState();

		void setAutoInstancingSizeHint( int numObjectsPerGeom );
		int getAutoInstancingSizeHint() const;

		void setDisplayBodyCachingEnabled( bool on );
		bool getDisplayBodyCachingEnabled() const;
		
		void setAllShadowCasters(bool on) { m_allShadowCasters = on; }

		void setAllowColorChangeOnPrecreated( bool on );
		void setTextureMapScale(float s);
		void setTextureMapLocalMode(bool local); // local or world verts used for texture vert creation
		void setFixedTextureMapMode(bool on); // if on, will give each tri the tcoords { 0,0  1,0  1,1  }

		void setDefaultMaterial(class hkgMaterial* mat);
		hkgMaterial* getDefaultMaterial() const;

		void setEnableHardwareSkinning( bool enable );

		void setShaderLib( hkgShaderLib* shaderLib );
		hkgShaderLib* getShaderLib() const;

		hkgDisplayContext* getContext() { return m_displayContext; }
		const hkgDisplayContext* getContext() const { return m_displayContext; }

		hkgDisplayWorld* getDisplayWorld() { return m_displayWorld; }
		const hkgDisplayWorld* getDisplayWorld() const { return m_displayWorld; }

		hkgMeshSystem* getMeshSystem() { return m_meshSystem; }
		const hkgMeshSystem* getMeshSystem() const { return m_meshSystem; }

		void setDepthTest( hkBool depthRead,  hkBool depthWrite ) { m_useDepthRead = depthRead; m_useDepthWrite = depthWrite; } // If enabled, triangles and lines will be drawn depth-tested
		void setDepthTestText( hkBool depthRead ) { m_useDepthReadForText = depthRead;} // If enabled, text will be drawn depth-tested


		// Immediate mode data
		struct DisplayPoint
		{
			hkVector4 m_position;
			hkColor::Argb m_color;
		};

		struct DisplayLine
		{
			hkVector4 m_start;
			hkVector4 m_end;
			hkColor::Argb m_color;
		};

		struct DisplayTriangle
		{
			hkVector4 m_a;
			hkVector4 m_b;
			hkVector4 m_c;
			hkColor::Argb m_color;
		};

		struct DisplayText
		{
			char m_buffer[256];
			hkVector4 m_pos;
			hkColor::Argb m_color;
		};

		struct DisplayTextScaled
		{
			char m_buffer[256];
			hkVector4 m_pos;
			hkColor::Argb m_color;
			hkReal m_sizeScale;
		};

		extArray<DisplayPoint>& getDisplayPoints();
		extArray<DisplayLine>& getDisplayLines();
		extArray<DisplayTriangle>& getDisplayTriangles();

		extArray<DisplayPoint>& getDisplayPoints2d();
		extArray<DisplayLine>& getDisplayLines2d();
		extArray<DisplayTriangle>& getDisplayTriangles2d();

	protected:
		
		hkResult addMesh(hkMeshBody* mesh, hkUlong id, int tag);
		hkResult removeMesh(hkUlong id, int tag);

	protected:

		class hkgDisplayContext*	m_displayContext;
		class hkgWindow*			m_window;
		class hkgDisplayWorld*	m_displayWorld;
		class hkgMaterial*     m_defaultMaterial;

		class hkgFont*			m_font;
		class hkgCamera*			m_fontCam;

		float m_textureMapScale;
		bool m_textureMapLocal;
		bool m_fixedMapMode;
		bool m_allowImmediateMode;

		hkUint32 m_instanceSizeHint;

		// Mapping: id (collidable) to hkg display object
		bool m_autoCreateEnabled;
		bool m_allowColorChangeOnPrecreated;
		bool m_displayCachingEnabled;
		bool m_allShadowCasters;

		extPointerMap<hkUlong, hkgDisplayObject*> m_objectMap;
		extPointerMap<const hkgDisplayObject*, hkUlong> m_reverseObjectMap;
		extPointerMap<hkUlong, hkgDisplayObject*> m_precreatedObjectMap;
		extPointerMap<hkUlong, hkgMeshBody*> m_meshesMap;
		
		hkgMeshSystem* m_meshSystem;
		hkgShaderLib* m_shaderLib;
		bool m_shaderLibSupportsHWSkinning;
		
		struct GeomCache
		{
			HKG_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(GeomCache);
			extArray< class hkgGeometry* > m_geoms;
		};
		extPointerMap<hkUlong, GeomCache*> m_shapeObjectCache;

		extArray<class hkgInstancedDisplayObject*> m_instances;

		// single buffered. What you get is what you just asked for so no longer a frame behind..
		extArray<DisplayPoint> m_displayPoints;
		extArray<DisplayPoint> m_displayPoints2d;
		extArray<DisplayLine> m_displayLines;
		extArray<DisplayLine> m_displayLines2d;
		extArray<DisplayTriangle> m_displayTriangles;
		extArray<DisplayTriangle> m_displayTriangles2d;
		extArray<DisplayText> m_displayTexts;
		extArray<DisplayTextScaled> m_displayTexts2d;

		mutable class extCriticalSection m_dataLock;

		hkPseudoRandomGenerator m_rand;

		hkBool m_useDepthRead;
		hkBool m_useDepthWrite;
		hkBool m_useDepthReadForText;

};

#endif // HKG_DISPLAY_HANDLER

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
