/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/UseCase/WeldingIssues/WeldingIssues.h>

#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexTranslate/hkpConvexTranslateShape.h>

#include <Physics2012/Dynamics/World/hkpPhysicsSystem.h>

#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/Geometry/VertexSet/hkgVertexSet.h>
#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>

#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppUtility.h>
#include <Physics2012/Collide/Shape/Compound/Collection/StorageExtendedMesh/hkpStorageExtendedMeshShape.h>
#include <Physics/Constraint/Data/BallAndSocket/hkpBallAndSocketConstraintData.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics2012/Collide/Util/Welding/hkpMeshWeldingUtility.h>
#include <Physics2012/Collide/Agent3/Machine/Nn/hkpAgentNnTrack.h>
#include <Physics2012/Dynamics/Collide/hkpSimpleConstraintContactMgr.h>
#include <Physics2012/Collide/Shape/Deprecated/Mesh/hkpMeshShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/SimpleMesh/hkpSimpleMeshShape.h>
#include <Physics2012/Collide/Shape/Deprecated/StorageMesh/hkpStorageMeshShape.h>

#include <Common/Base/Reflection/hkClass.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Base/Reflection/Registry/hkTypeInfoRegistry.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>

// Common
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>
#include <Demos/DemoCommon/Utilities/Character/CharacterUtils.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>

// Scene Data
#include <Common/SceneData/Scene/hkxScene.h>
// Camera handling
#include <Common/Visualize/hkDebugDisplay.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>

#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpActiveContactPointViewer.h>
#include <Physics2012/Utilities/CharacterControl/CharacterRigidBody/hkpCharacterRigidBodyListener.h>

//Character controller
#include <Physics2012/Utilities/CharacterControl/CharacterRigidBody/hkpCharacterRigidBody.h>
#include <Physics2012/Utilities/CharacterControl/CharacterRigidBody/hkpCharacterRigidBodyCinfo.h>

#define WELD_OPEN_EDGES (hkBool)( false )

DEMO_OPTIONS_DEFINE(WeldingIssues,Options);

WeldingIssues::WeldingIssues( hkDemoEnvironment* env )
: hkDefaultPhysics2012Demo( env )
, m_time( 0.0f )
, m_loader( HK_NULL )
{

	//
	// Setup the camera
	//
	{
		hkVector4 from( 5.f, 6.f,  25.f );
		hkVector4 to( 5.f,  2.0f, 10.f );
		hkVector4 up( 0.0f, 1.0f, 0.0f );

		setupDefaultCameras( env, from, to, up );
	}


	//
	// Create the world
	//
	{
		hkpWorldCinfo info;

		info.m_gravity.set( 0.0f, -10.0f, 0.0f );
		info.m_contactPointGeneration = hkpWorldCinfo::CONTACT_POINT_ACCEPT_ALWAYS;
		info.m_enableDeactivation = false;
		m_world = new hkpWorld( info );
		m_world->lock();

		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

		m_debugViewerNames.pushBack( hkpActiveContactPointViewer::getName() );

		setupGraphics();
		disableBackFaceCulling();
	}


	createWorld();


	
	hkArray< hkpMeshWeldingUtility::ShapeInfo > shapes;

	hkpPhysicsSystem* system = m_world->getWorldAsOneSystem();
	const hkArray< hkpRigidBody* >& rigidBodies = system->getRigidBodies();
	

	// Collect all shapes
	for ( int i = 0; i < rigidBodies.getSize(); ++i )
	{
		const hkTransform& transform = rigidBodies[ i ]->getTransform();
		const hkpShape* shape = rigidBodies[ i ]->getCollidable()->getShape();

		if ( shape->getType() == hkcdShapeType::MOPP )
		{
			const hkpMoppBvTreeShape* mopp  = static_cast< const hkpMoppBvTreeShape* >( shape );

			hkpMeshWeldingUtility::ShapeInfo info;
			info.m_transform = transform;
			info.m_shape = mopp;

			shapes.pushBack( info );
		}
	}

	// Create welding info for mesh considering all other meshes
	for ( int i = 0; i < rigidBodies.getSize(); ++i )
	{
		const hkpShape* shape = rigidBodies[ i ]->getCollidable()->getShape();
		if ( shape->getType() == hkcdShapeType::MOPP )
		{
			const hkpMoppBvTreeShape* mopp  = static_cast< const hkpMoppBvTreeShape* >( shape );
			hkpShapeCollection* collection = const_cast<hkpShapeCollection*>(mopp->getShapeCollection());

			const hkTransform& transform = rigidBodies[ i ]->getTransform();				

			hkpMeshWeldingUtility::computeWeldingInfoMultiShape( transform, collection, hkpWeldingUtility::WELDING_TYPE_ANTICLOCKWISE, shapes, WELD_OPEN_EDGES );
		}
	}

	system->removeReference();


	m_world->unlock();
}


WeldingIssues::~WeldingIssues( void )
{
	m_world->markForWrite();
	m_world->removeReference();
	m_world = HK_NULL;
	
	
	for(int i = 0 ; i < m_character.getSize() ; i ++)
	{
		m_character[ i ]->removeReference();
	}

	m_character.clearAndDeallocate();

	delete m_loader;
}


hkpRigidBody* WeldingIssues::createSphere( hkVector4Parameter position, const hkEnum< hkpMotion::MotionType, hkInt8 >& motionType, const hkEnum< hkpCollidableQualityType, hkInt8 >& qualityType )
{
	hkpSphereShape* sphere = new hkpSphereShape( 0.25f );

	hkpRigidBodyCinfo ci;
	ci.m_shape = sphere;
	ci.m_motionType = motionType;
	ci.m_position = position;
	ci.m_qualityType = qualityType;
	ci.m_restitution = 0.0f;
	ci.m_friction = 1.0f;


	hkpRigidBody* rb = new hkpRigidBody( ci );
	if (motionType == hkpMotion::MOTION_DYNAMIC)
	{
		hkMatrix3 zeroInvInertia; zeroInvInertia.setZero();
		rb->setInertiaInvLocal(zeroInvInertia);
	}

	m_world->addEntity( rb )->removeReference();

	sphere->removeReference();

	if ( qualityType == HK_COLLIDABLE_QUALITY_BULLET )
	{
		HK_SET_OBJECT_COLOR( hkUlong(rb->getCollidable()), hkColor::semiTransparent(hkColor::YELLOW));
	} 
	else 
	{
		HK_SET_OBJECT_COLOR( hkUlong(rb->getCollidable()), hkColor::semiTransparent(hkColor::ORANGE)  );
	}

	if (motionType == hkpMotion::MOTION_DYNAMIC)
	{
		HK_SET_OBJECT_COLOR( hkUlong(rb->getCollidable()), hkColor::semiTransparent(hkColor::RED) );
	}

	return rb;
}


hkpCharacterRigidBody* WeldingIssues::createCharacter( hkVector4Parameter position, const hkEnum< hkpMotion::MotionType, hkInt8 >& motionType, const hkEnum< hkpCollidableQualityType, hkInt8 >& qualityType )
{
	hkpCapsuleShape* capsule = new hkpCapsuleShape( hkVector4( 0.0f, 0.0f, 0.0f ), hkVector4( 0.0f,  0.5f, 0.0f ), .25f );


	hkpCharacterRigidBodyCinfo info;
	{		
		info.m_unweldingHeightOffsetFactor = m_options.m_unweldingHeightOffsetFactor;
		info.m_mass = 100.0f;
		info.m_shape = capsule;
		info.m_maxForce = 1000.0f;
		info.m_up = hkVector4( 0 , 1 , 0 );
		info.m_position.setAdd4( hkVector4(0.f,.25f,0.f), position );
		info.m_friction = 1.0f;		
	}

	/////////////////////////////////////////////////////////////


	hkpCharacterRigidBody* rb = new hkpCharacterRigidBody( info );
	{			
		hkpCharacterRigidBodyListener* listener = new hkpCharacterRigidBodyListener();
		rb->setListener( listener );
		listener->removeReference();
		rb->getRigidBody()->setMotionType(motionType);
		rb->getRigidBody()->setQualityType(qualityType);
	}

	m_world->addEntity( rb->getRigidBody() );
	capsule->removeReference();

	if (motionType == hkpMotion::MOTION_DYNAMIC)
	{
		hkMatrix3 zeroInvInertia; zeroInvInertia.setZero();
		rb->getRigidBody()->setInertiaInvLocal(zeroInvInertia);
	}

	if ( qualityType == HK_COLLIDABLE_QUALITY_BULLET)
	{
		HK_SET_OBJECT_COLOR( hkUlong(rb->getRigidBody()->getCollidable()), hkColor::YELLOW  & 0xf0ffffff );
	} 
	else 
	{
		HK_SET_OBJECT_COLOR( hkUlong(rb->getRigidBody()->getCollidable()), hkColor::ORANGE & 0xf0ffffff );
	}

	if (motionType == hkpMotion::MOTION_DYNAMIC)
	{
		HK_SET_OBJECT_COLOR( hkUlong(rb->getRigidBody()->getCollidable()), hkColor::RED & 0xf0ffffff );
	}

	return rb;
}

hkpRigidBody* WeldingIssues::createMopp( const hkVector4* verts, hkUint16 numVerts, const hkUint16* indices, hkUint16 numTriangles, hkVector4Parameter pos, hkpWeldingUtility::WeldingType weldingType )
{
	hkpShapeCollection* mesh = createMesh(verts, numVerts, indices, numTriangles, pos, hkpShapeCollection::COLLECTION_EXTENDED_MESH, weldingType );

	hkpMoppCompilerInput mci;
	hkpMoppCode* code = hkpMoppUtility::buildCode( mesh, mci );
	hkpMoppBvTreeShape* mopp = new hkpMoppBvTreeShape( mesh, code );

	mesh->removeReference();
	code->removeReference();

	// Add it to the world
	hkpRigidBodyCinfo info;
	info.m_shape = mopp;
	info.m_position = pos;
	info.m_motionType = hkpMotion::MOTION_FIXED;
	info.m_restitution = 0.0f;
	info.m_friction = 1.0f;

	hkpRigidBody* rb = new hkpRigidBody( info );
	mopp->removeReference();

	m_world->addEntity(rb)->removeReference();

	return rb;
}

hkpShapeCollection* WeldingIssues::createMesh( const hkVector4* verts, hkUint16 numVerts, const hkUint16* indices, hkUint16 numTriangles, hkVector4Parameter pos, hkpShapeCollection::CollectionType collectionType, hkpWeldingUtility::WeldingType weldingType )
{
	switch( collectionType )
	{
	case hkpShapeCollection::COLLECTION_EXTENDED_MESH:
		{
			hkpStorageExtendedMeshShape* mesh = new hkpStorageExtendedMeshShape;

			hkpExtendedMeshShape::TrianglesSubpart subPart;
			subPart.m_vertexBase = &verts[0](0);
			subPart.m_vertexStriding = sizeof( hkVector4 );
			subPart.m_numVertices = numVerts;
			subPart.m_indexBase = indices;
			subPart.m_stridingType = hkpExtendedMeshShape::INDICES_INT16;
			subPart.m_indexStriding = sizeof( hkUint16 ) * 3;

			subPart.m_numTriangleShapes = numTriangles;
			mesh->addTrianglesSubpart(subPart);	
			
			return mesh;
		}		
		break;
	case hkpShapeCollection::COLLECTION_SIMPLE_MESH:
		{
			hkpSimpleMeshShape* mesh = new hkpSimpleMeshShape;

			for( hkInt32 i = 0; i < numVerts; ++i )
			{
				mesh->m_vertices.pushBack(verts[i]);
			}

			for(hkInt32 i = 0; i < numTriangles; i++)
			{
				hkpSimpleMeshShape::Triangle& triangle = mesh->m_triangles.expandOne();
				triangle.m_a = indices[i*3 + 0];
				triangle.m_b = indices[i*3 + 1];
				triangle.m_c = indices[i*3 + 2];
			}

			return mesh;
		}
		break;
	case hkpShapeCollection::COLLECTION_MESH_SHAPE:
		{
			hkpStorageMeshShape* mesh = new hkpStorageMeshShape;

			hkpMeshShape::Subpart subPart;
			subPart.m_vertexBase = &verts[0](0);
			subPart.m_vertexStriding = sizeof( hkVector4 );
			subPart.m_numVertices = numVerts;
			subPart.m_indexBase = indices;
			subPart.m_stridingType = hkpMeshShape::INDICES_INT16;
			subPart.m_indexStriding = sizeof( hkUint16 ) * 3;
			subPart.m_numTriangles = numTriangles;

			mesh->addSubpart(subPart);	
			return mesh;
		}
		break;
	default:
		HK_ASSERT2(0, false, "Unsupported collection type.");
		return HK_NULL;
	}
}

void WeldingIssues::createWorld( void )
{
	hkReal hb = 0.5f;		
	hkReal hh = 1.5f;		

	m_character.setSize( 14 );


	//
	// Create the ground base 
	//
	{
		hkVector4 halfExtents( 50.0f, 0.3f, 50.0f );
		
		hkVector4 verts[ 8 ];
		verts[0].set( -1.0f,  1.0f, -1.0f );
		verts[1].set( -1.0f,  1.0f,  1.0f );
		verts[2].set(  1.0f,  1.0f,  1.0f );
		verts[3].set(  1.0f,  1.0f, -1.0f );
		verts[4].set( -1.0f, -1.0f, -1.0f );
		verts[5].set( -1.0f, -1.0f,  1.0f );
		verts[6].set(  1.0f, -1.0f,  1.0f );
		verts[7].set(  1.0f, -1.0f, -1.0f );

		for( int i = 0; i < 8; ++i )
		{
			verts[ i ].mul4( halfExtents );
		}

		hkUint16 indices[] =
		{
			0,1,2,
			0,2,3,
			1,5,6,
			1,6,2,
			2,6,7,
			2,7,3,
			0,3,7,
			0,7,4,
			1,0,4,
			1,4,5,
			4,7,6,
			4,6,5
		};

		createMopp( verts, 8, indices, 12, hkVector4( 0.0f, -halfExtents( 1 ), 0.0f ) );
	}	


	//
	// MOPP prism with hinged box above it and open top and closed bottom
	//
	{
		hkVector4 verts[6];
		verts[0].set(  -hb,    hh, -hb );
		verts[1].set( 0.0f,    hh,  hb );
		verts[2].set(   hb,    hh, -hb );
		verts[3].set(  -hb, -0.1f, -hb );
		verts[4].set( 0.0f, -0.1f,  hb );
		verts[5].set(   hb, -0.1f,- hb );

		hkUint16 indices[] = 
		{
			0,3,4,
			0,4,1,
			0,5,3,
			0,2,5,
			1,4,5,
			2,1,5,
			3,5,4
		};

		const hkpCollidable* body = createMopp( verts, 6, indices, 7, hkVector4::getZero() )->getCollidable();
		HK_SET_OBJECT_COLOR( hkUlong(body),  hkColor::semiTransparent(hkColor::ORANGE) );


		//
		// Create a box dangling over the MOPP shape
		//
		{
			hkpBoxShape* shape = new hkpBoxShape( hkVector4( 0.5f, 0.5f, 0.5f ), 0.0f );

			hkVector4 axis; 
			axis.set( 1.0f, 1.0f, 1.0f );
			axis.normalize3();
			
			hkpRigidBodyCinfo ci;
			ci.m_shape = shape;
			ci.m_motionType = hkpMotion::MOTION_DYNAMIC;
			ci.m_position.set( 0.0f, 2.1f, 0.0f );
			ci.m_rotation.setAxisAngle( axis, 45.0f * HK_REAL_PI / 180.f );
			ci.m_numShapeKeysInContactPointProperties = 1;

			hkpRigidBody* rb = new hkpRigidBody( ci );
			m_world->addEntity( rb )->removeReference();

			//
			// Create ball and socket constraint.
			// 
			{
				// Set the pivot
				hkVector4 up;
				up.set( 0.0f, 1.0f, 0.0f );

				hkVector4 pivot = ci.m_position;
				pivot.addMul4( 0.707f, up );

				// Create the constraint
				hkpBallAndSocketConstraintData* constraintData = new hkpBallAndSocketConstraintData(); 
				constraintData->setInWorldSpace( rb->getTransform(), hkTransform::getIdentity(), pivot );
				
				hkpConstraintInstance* constraint = new hkpConstraintInstance( rb, HK_NULL, constraintData );
				m_world->addConstraint( constraint ); 	
				
				constraintData->removeReference();
				constraint->removeReference();
			}
			shape->removeReference();
		}

		//
		// Create another prism for kicks
		//
		createMopp( verts, 6, indices, 7, hkVector4( 5.0f, 0.0f, 0.0f ) );
	}

	//
	// Ramps
	//
	{
		hb = 1.5f;
		hh = 1.0f;

		hkVector4 verts[6];
		verts[0].set( -hb, hh, -hb );
		verts[1].set( -hb,  0,  hb );
		verts[2].set(  hb,  0,  hb );
		verts[3].set(  hb, hh, -hb );
		verts[4].set(  hb,  0, -hb );
		verts[5].set( -hb,  0, -hb );

		//
		// Ramp with sides missing
		//
		{
			hkUint16 indices[] = 
			{
				0,1,2,
				3,0,2,
				5,4,1,
				4,2,1,
				0,4,5,
				3,4,0
			};

			const hkpCollidable* body = createMopp( verts, 6, indices, 6, hkVector4( 10.0f, 0.0f, 0.0f ) )->getCollidable();
			HK_SET_OBJECT_COLOR( hkUlong( body ),  hkColor::semiTransparent(hkColor::PURPLE) );
		}

		//
		// Create capsules
		//
		m_character[0] = createCharacter( hkVector4( 12.0f, 0.0f, 0.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_BULLET );
		m_character[1] = createCharacter( hkVector4(  8.0f, 0.0f, 0.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_DEBRIS );

		//
		// Free capsule
		//
		m_character[13] = createCharacter( hkVector4( 15.0f, 0.0f, 5.0f ), hkpMotion::MOTION_DYNAMIC, HK_COLLIDABLE_QUALITY_BULLET );

		//
		// Put a free box in the world
		//
		{
			hkpBoxShape* shape = new hkpBoxShape( hkVector4( 0.25f, 0.25f, 0.25f ) , 0.0f );

			hkpRigidBodyCinfo ci;
			ci.m_shape = shape;

			ci.m_motionType = hkpMotion::MOTION_DYNAMIC;
			ci.m_position.set( 12.0f, 1.0f, 5.0f );

			hkpRigidBody* rb = new hkpRigidBody( ci );
			m_world->addEntity( rb )->removeReference();
			shape->removeReference();
		}

		//
		// Ramp with sides
		//
		{
			hkUint16 indices[] = 
			{
				0,1,2,
				3,0,2,
				5,4,1,
				4,2,1,
				0,4,5,
				3,4,0,
				0,5,1,
				3,2,4
			};

			createMopp( verts, 6, indices, 8, hkVector4( 20.0f, 0.0f, 0.0f ) );

			m_character[2] = createCharacter( hkVector4( 22.0f, 0.0f, 0.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_BULLET );
			m_character[3] = createCharacter( hkVector4( 18.0f, 0.0f, 0.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_DEBRIS );
		}

		//
		// ramp with sides using 2 MOPPs
		//
		{
			//
			// ramp no sides
			//
			{
				hkUint16 indices[] = 
				{
					0,1,2,
					3,0,2,
					5,4,1,
					4,2,1,
					0,4,5,
					3,4,0
				};

				const hkpCollidable* body = createMopp( verts, 6, indices, 6, hkVector4( 30.0f, 0.0f, 0.0f ) )->getCollidable();
				HK_SET_OBJECT_COLOR( hkUlong (body ),  hkColor::semiTransparent(hkColor::GREEN) );
			}

			//
			// just the sides
			//
			{
				hkUint16 indices[] = 
				{
					0,5,1,
					3,2,4
				};

				const hkpCollidable* body = createMopp( verts, 6, indices, 2, hkVector4( 30.0f, 0.0f, 0.0f ) )->getCollidable();
				HK_SET_OBJECT_COLOR( hkUlong( body ), hkColor::semiTransparent(hkColor::RED) );
			}

			m_character[4] = createCharacter( hkVector4( 32.0f, 0.0f, 0.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_BULLET );
			m_character[5] = createCharacter( hkVector4( 28.0f, 0.0f, 0.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_DEBRIS );
		}
	}

	//
	// Half walls
	//
	{
		const hkReal hw = 1.0f;
		const hkReal hl = 0.5f;
		const hkReal h  = 0.5f;
		
		hkVector4 verts[8];
		verts[0].set( -hw,  h, -hl );
		verts[1].set( -hw,  h,  hl );
		verts[2].set(  hw,  h,  hl );
		verts[3].set(  hw,  h, -hl );
		verts[4].set( -hw, -h, -hl );
		verts[5].set( -hw, -h,  hl );
		verts[6].set(  hw, -h,  hl );
		verts[7].set(  hw, -h, -hl );

		
		hkUint16 indices[] = 
		{
			0,1,2,
			3,0,2,
			1,5,6,
			2,1,6,
			3,2,6,
			3,6,7,
			0,3,7,
			0,7,4,
			5,4,6,
			4,7,6,
			0,4,5,
			1,0,5
		};

		//
		// Create one MOPP with welding
		//
		hkpRigidBody* halfWall1 = createMopp( verts, 8, indices, 12, hkVector4( 10.0f, 0.0f, 10.0f ) );
		HK_SET_OBJECT_COLOR( hkUlong ( halfWall1->getCollidable() ), hkColor::semiTransparent(hkColor::PURPLE) );

		m_character[6] = createCharacter( hkVector4( 12.0f, 0.0f, 10.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_BULLET );
		m_character[7] = createCharacter( hkVector4( 12.0f, 0.0f, 12.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_BULLET );
		m_character[8] = createCharacter( hkVector4(  8.0f, 0.0f, 12.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_BULLET );

//		m_world->addWorldPostSimulationListener( this );

		//
		// Create another MOPP without welding
		// 
		hkpRigidBody* halfWall2 = createMopp( verts, 8, indices, 12, hkVector4( 20.0f, 0.0f, 10.0f ), hkpWeldingUtility::WELDING_TYPE_NONE );
		HK_SET_OBJECT_COLOR( hkUlong( halfWall2->getCollidable() ), hkColor::semiTransparent(hkColor::RED) );

		m_character[9]  = createCharacter( hkVector4( 22.0f, 0.0f, 10.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_BULLET );
		m_character[10] = createCharacter( hkVector4( 22.0f, 0.0f, 12.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_BULLET );
		m_character[11] = createCharacter( hkVector4( 18.0f, 0.0f, 12.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_BULLET );
	}

	//
	// Platform with triangle between corners
	//
	{
		hkReal hWidth = 1.0f;
		hkReal hLength = 0.5f;
		hkReal hHeight = 0.5f;
		
		hkVector4 verts[8];
		verts[0].set( -hWidth,  hHeight, -hLength );
		verts[1].set( -hWidth,  hHeight,  hLength );
		verts[2].set(  hWidth,  hHeight,  hLength );
		verts[3].set(  hWidth,  hHeight, -hLength );
		verts[4].set( -hWidth, -hHeight, -hLength );
		verts[5].set( -hWidth, -hHeight,  hLength );
		verts[6].set(  hWidth, -hHeight,  hLength );
		verts[7].set(  hWidth, -hHeight, -hLength );


		hkUint16 indices[] = 
		{
			0,1,2,
			3,0,2,
			1,5,6,
			2,1,6,
			0,3,7,
			0,7,4,
			0,4,5,
			1,0,5,
			3,2,6,
			3,6,7,
			2,0,4,
			4,6,2
		};

		const hkpCollidable* bump = createMopp( verts, 8, indices, 12, hkVector4( 30.0f, 0.0f, 10.0f ) )->getCollidable();
		HK_SET_OBJECT_COLOR( hkUlong( bump ), hkColor::semiTransparent(hkColor::YELLOWGREEN)  );

		m_character[12] = createCharacter( hkVector4( 30.0f, 0.5f, 10.0f ), hkpMotion::MOTION_KEYFRAMED, HK_COLLIDABLE_QUALITY_BULLET );
	}


	//
	// Pyramid
	//
	{
		hkReal radius = 0.1f;
		hkReal height = 2.0f;
		
		hkVector4 vertices[ 5 ];
		vertices[ 0 ].set( -radius,   0.0f, -radius );
		vertices[ 1 ].set( -radius,   0.0f,  radius );
		vertices[ 2 ].set(  radius,   0.0f,  radius );
		vertices[ 3 ].set(  radius,   0.0f, -radius );
		vertices[ 4 ].set(    0.0f, height,    0.0f );

		hkUint16 indices[] = 
		{
			2,1,0,
			0,3,2,
			1,2,4,
			2,3,4,
			4,3,0,
			0,1,4
		};

		const hkpCollidable* bump = createMopp( vertices, 5, indices, 6, hkVector4( 0.0f, 0.0f, 10.0f ) )->getCollidable();
		HK_SET_OBJECT_COLOR( hkUlong( bump ), hkColor::semiTransparent(hkColor::YELLOW));

		createSphere( hkVector4( 0.0f, 8.0f, 10.f ), hkpMotion::MOTION_DYNAMIC, HK_COLLIDABLE_QUALITY_BULLET );
	}
}


hkDemo::Result WeldingIssues::stepDemo( void )
{
	// Advance time
	m_time += m_timestep;

	m_world->lock();

	//Identify keyframe capsules by numbers and update unweldingHeightOffsetFactor
	for(int i = 0 ; i < m_character.getSize()  ; i++)
	{
		hkVector4 pos = m_character[i]->getPosition();
		pos.add4( hkVector4( 0 , 0.75f , 0 ) );
		hkStringBuf label;
		label.printf( "%d" , i );	
		HK_DISPLAY_3D_TEXT( label.cString(), pos , hkColor::BLACK);

		m_character[i]->m_unweldingHeightOffsetFactor = m_options.m_unweldingHeightOffsetFactor;
	}

	// Animate the keyframed capsules
	for ( int c = 0; c < 6; ++c )
	{
		hkReal t = hkMath::sin( 0.25f * m_time * HK_REAL_PI );
		hkReal s = hkMath::sin( m_time * HK_REAL_PI );

		if ( c < 3 )
		{
			const int i = 2 * c;
			const int j = i + 1;

			hkVector4 rampCenter; 
			rampCenter.set( 10.0f * ( c + 1 ), 0.0f, 0.0f );

			hkReal capsuleHeight = 0.65f;
			hkReal h = 1.1f;
			hkReal w = 3.0f;

			hkVector4 position1;
			position1.setAdd4( hkVector4( w/2.f+.1f + s*.1f, h*t/2.f + capsuleHeight + s*.1f, -w*t/2.f ), rampCenter );
			m_character[i]->getRigidBody()->setPosition( position1 );

			hkVector4 position2;
			hkReal x = -w/2.f-.1f - s*.1f;
			hkReal y = h*t/2.f + capsuleHeight + s*.1f;
			hkReal z = -w*t/2.f;
			position2.setAdd4( hkVector4( x, y, z ), rampCenter );
			m_character[j]->getRigidBody()->setPosition( position2 );
			
		} 
		else if ( c < 5 ) 		{
			const int i = 6 + 3 * ( c - 3 );
			const int j = i + 1;
			const int k = j + 1;

			hkVector4 wallCenter; 
			wallCenter.set( 10.f * ( c - 2 ), 0.0f, 10.0f );

			hkReal capsuleHeight = 0.3f;
			hkReal capsuleRadius = 0.25f;


			// Put capsule at corner
			hkVector4 position1;
			position1.setAdd4( hkVector4( -1.0f + 0.1f * t - capsuleRadius, capsuleHeight, -0.5f - 0.1f * t - capsuleRadius ), wallCenter );
			m_character[ i ]->getRigidBody()->setPosition( position1 );			

			// Put capsule on edge
			hkVector4 position2;
			position2.setAdd4( hkVector4( 0.25f * t, capsuleHeight, -0.6f - capsuleRadius ), wallCenter );
			m_character[ j ]->getRigidBody()->setPosition( position2 );

			// Put capsule on other corner
			hkVector4 position3;
			position3.setAdd4( hkVector4( 1.0f + 0.1f * t + capsuleRadius, capsuleHeight, 0.5f - 0.1f * t + capsuleRadius ), wallCenter );
			m_character[ k ]->getRigidBody()->setPosition( position3 );
		}
		else 
		{
			hkVector4 position; 
			position.set( 30.0f, 0.75f, 10.0f );

			// Put capsule at corner
			position.add4( hkVector4( 0.2f * t, 0.0f, -0.2f * t ) );
			m_character[ 12 ]->getRigidBody()->setPosition( position );
		}
	}

	m_world->unlock();

	return hkDefaultPhysics2012Demo::stepDemo();
}


static const char helpString[] = \
"A demo which shows the problems with welding.\n"
"\n"
"The code which overrides welding information for character rigid bodies can be applied to all the capsules by tweaking "
"the unweldingHeightOffsetFactor parameter.";

HK_DECLARE_DEMO(WeldingIssues, HK_DEMO_TYPE_PHYSICS_2012 | HK_DEMO_TYPE_CRITICAL, "Demonstrates problems with welding", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
