/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

// Demo framework
#include <Demos/demos.h>

// Shapes and Physics
#include <Physics2012/Collide/Shape/Misc/Bv/hkpBvShape.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics2012/Collide/Query/Collector/BodyPairCollector/hkpAllCdBodyPairCollector.h>
#include <Physics2012/Utilities/Collide/hkpShapeGenerator.h>

// Data
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Container/BitField/hkBitField.h>

// Character control
#include <Physics2012/Utilities/CharacterControl/CharacterProxy/hkpCharacterProxy.h>
#include <Physics2012/Utilities/CharacterControl/CharacterRigidBody/hkpCharacterRigidBody.h>
#include <Physics2012/Utilities/CharacterControl/StateMachine/hkpDefaultCharacterStates.h>

// Display includes
#include <Common/Visualize/Process/hkDebugDisplayProcess.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpSweptTransformDisplayViewer.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpShapeDisplayViewer.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpBroadphaseViewer.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpPhantomDisplayViewer.h>

// Header
#include <Demos/Physics2012/UseCase/TriggerVolume/TriggerVolumeComparison/TriggerVolumeComparisonDemo.h>

// Constructor
TriggerVolumeComparisonDemo::TriggerVolumeComparisonDemo( hkDemoEnvironment* env )
:	hkDefaultPhysics2012Demo( env, DEMO_FLAGS_NO_SERIALIZE )
{
	//
	// Setup the camera
	//
	if ( m_variantId == VARIANT_DETECTION_COMPARISON )	// Broadphase v narrowphase etc: camera from front
	{
		hkVector4 pos ( -10.0f, 50.0f, 200.0f );
		hkVector4 to  ( -10.0f, 50.0f, 0.0f );
		hkVector4 up  ( 0.0f, 1.0f, 0.0f );
		setupDefaultCameras( env, pos, to, up );
	}
	if ( m_variantId == VARIANT_CONTINUOUS ) // Continuous physics: camera from above
	{
		hkVector4 pos ( 0.0f, 100.0f, 0.0f );
		hkVector4 to  ( 0.0f, 0.0f, 0.0f );
		hkVector4 up  ( 0.0f, 0.0f, -1.0f );
		setupDefaultCameras( env, pos, to, up );
	}

	//
	// Create the world
	//
	{
		hkpWorldCinfo info;

#if defined(HK_PLATFORM_MULTI_THREAD) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
		info.m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_MULTITHREADED;
#else
		info.m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_CONTINUOUS;
#endif

		info.setupSolverInfo( hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM );
		info.setBroadPhaseWorldSize( 1000.f );
		info.m_enableDeactivation = false;
		info.m_collisionTolerance = 0.05f;
		info.m_contactPointGeneration = info.CONTACT_POINT_ACCEPT_ALWAYS;

		m_world = new hkpWorld( info );
		m_world->lock();

		setupGraphics();
	}

	//
	// Register collision agents
	//
	{
		hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );
	}

	//
	// Setup some variables used by all the shapes
	//

	// Sphere info
	hkReal radius = 1.0f;
	m_sphereShape = new hkpSphereShape( radius );
	m_sphereInfo.m_shape = m_sphereShape;
	m_sphereInfo.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;

	// Character controller info
	hkVector4 vertexA( 0.0f, (2.0f - radius), 0.0f );
	hkVector4 vertexB( 0.0f, (radius - 2.0f), 0.0f );		
	m_charControlShape = new hkpCapsuleShape(vertexA, vertexB, radius);

	// Common grid setup
	hkBool includeFixedRb  = false;
	hkBool includeFixedTV  = true;
	hkBool includeFixedPCS = true;
	hkBool includeFixedSP  = true;
	hkBool includeFixedAP  = true;
	m_gridHorizontalOffset	= hkVector4( 20.0f, 0.0f, 0.0f );
	m_numCols = includeFixedRb + includeFixedTV + includeFixedPCS + includeFixedSP + includeFixedAP;

	// Create fixed volumes - one of each
	setupFixedVolumes( includeFixedRb, includeFixedTV, includeFixedPCS, includeFixedSP, includeFixedAP );

	if ( m_variantId == VARIANT_DETECTION_COMPARISON )	// Broadphase v narrowphase, character controllers, interactions with each other
	{
		//
		// Create shape grid
		//

		// Variant-0 grid setup
		m_gridVerticalOffset	= hkVector4( 0.0f, 15.0f, 0.0f );
		m_gridVelocity			= hkVector4( 0, -10, 0 );

		m_numRows = 7;

		m_gridOrigin = hkVector4 ( 0, 10, 0 );
		m_gridOrigin.addMul4( (1 - (hkReal)m_numCols)/2, m_gridHorizontalOffset );

		int rowIndex = 0;
		for ( int colIndex = 0; colIndex < m_numCols; colIndex++ )
		{
			// Create moving rigid body
			addGridRigidBody( rowIndex, colIndex );
			rowIndex++;

			// Create moving trigger volume
			addGridTriggerVolume( rowIndex, colIndex );
			rowIndex++;

			// Create moving phantom callback shape
			addGridPhantomCallbackShape( rowIndex, colIndex );
			rowIndex++;

			// Create moving shape phantom
			addGridShapePhantom( rowIndex, colIndex );
			rowIndex++;

			// Create moving AABB phantom
			addGridAabbPhantom( rowIndex, colIndex );
			rowIndex++;

			// Create moving character rigid body
			addGridCharacterRigidBody( rowIndex, colIndex );
			rowIndex++;

			// Create moving character rigid body
			addGridCharacterProxy( rowIndex, colIndex );
			rowIndex = 0;
		}
	}
	else if ( m_variantId == VARIANT_CONTINUOUS ) // Continuous physics
	{
		//
		// TOI variant
		//

		m_sphereInfo.m_maxLinearVelocity = HK_REAL_MAX;
		m_sphereInfo.m_qualityType = HK_COLLIDABLE_QUALITY_BULLET;
		m_sphereInfo.m_mass = 10.100f;

		m_gridVerticalOffset	= hkVector4( 0.0f, 0.0f, 0.0f );
		m_gridVelocity			= hkVector4( 0, 0, 500 );

		m_numRows = 1;

		m_gridOrigin = hkVector4 ( 0, 0, -20 );
		m_gridOrigin.addMul4( (1 - (hkReal)m_numCols)/2, m_gridHorizontalOffset );

		for ( int colIndex = 0; colIndex < m_numCols; colIndex++ )
		{
			addGridRigidBody(0, colIndex);
		}
	}
	m_world->unlock();
}


// Destructor
TriggerVolumeComparisonDemo::~TriggerVolumeComparisonDemo()
{
	m_world->lock();

	// fixed volumes
	m_fixedTriggerVolume->removeReference();
	m_fixedPhantomCallbackShape->removeReference();
	m_fixedShapePhantom->removeReference();
	m_fixedAabbPhantom->removeReference();

	// Character controllers
	for ( int i = 0; i < m_characterProxies.getSize(); i++ ) // there should never be a different number of proxies and rigid bodies
	{
		m_characterRigidBodies[i]->removeReference();
		m_characterProxies[i]->removeReference();
	}

	// Common shapes
	m_sphereShape->removeReference();
	m_charControlShape->removeReference();

	m_world->unlock();

}

// Inherited methods
void TriggerVolumeComparisonDemo::setupContexts(hkArray<hkProcessContext*>& contexts)
{
	hkDefaultPhysics2012Demo::setupContexts(contexts);
	{

		// Both variants: display shapes, phantoms
		m_debugViewerNames.clear();
		m_debugViewerNames.pushBack( hkDebugDisplayProcess::getName() );
		m_debugViewerNames.pushBack( hkpPhantomDisplayViewer::getName() );
		m_debugViewerNames.pushBack( hkpShapeDisplayViewer::getName() );

		if ( m_variantId == VARIANT_CONTINUOUS )
		{
			// Display the toi clock
			enableDisplayingToiInformation( true );

			// Add interpolated motion and broadphase viewers
			m_debugViewerNames.pushBack( hkpBroadphaseViewer::getName() );
			m_debugViewerNames.pushBack( hkpSweptTransformDisplayViewer::getName() );
		}
	}
}

TriggerVolumeComparisonDemo::Result TriggerVolumeComparisonDemo::stepDemo( void )
{
	// The aabb phantoms and shape phantoms must be moved manually.
	// Also, narrowphase overlaps with shape phantoms must be checked here

	// The phantoms will only be moving in demo variant 0, so I only need to implement that movement scheme
	// All other phantoms will have the motion state hkpMotion::MOTION_FIXED, so they can be sorted out that way

	if ( m_variantId == VARIANT_DETECTION_COMPARISON )
	{
		HK_TIMER_BEGIN("StepDemo", HK_NULL);

		m_world->lock();

		// Manually move all the entities that the custom action won't work on
		moveShapePhantoms();
		moveAabbPhantoms();
		moveCharacterRigidBodies();
		moveCharacterProxies();

		// Detect narrowphase overlaps with the shape phantoms
		collideShapePhantoms();

		// UI help
		displayFixedObjectLabels();
		displayResultsText();

		m_world->unlock();

		HK_TIMER_NAMED_END("StepDemo");

		return hkDefaultPhysics2012Demo::stepDemo();
	}

	if ( m_variantId == VARIANT_CONTINUOUS )
	{
		m_world->lock();

		// setup asynchronous timestep
		hkReal slowMotionFactor = 100.0f;
		hkReal physicsDeltaTime = 0.1f;
		hkReal frameDeltaTime = physicsDeltaTime / slowMotionFactor;

		// (No moving phantoms to move)

		// Detect narrowphase overlaps with the shape phantom
		collideShapePhantoms();

		// UI help
		displayFixedObjectLabels();
		displayResultsText();

		m_world->unlock();

		hkDefaultPhysics2012Demo::stepAsynchronously(m_world, frameDeltaTime, physicsDeltaTime);
		return DEMO_OK;
	}
	else
	{
		return DEMO_ERROR;
	}
}

// functions for moving the shape and AABB phantoms, and controlling the character controllers
void TriggerVolumeComparisonDemo::collideShapePhantoms()
{
	for ( int i = 0; i < m_shapePhantoms.getSize(); i++ )
	{
		//
		// Test for narrowphase overlaps, and colour the shape as appropriate
		//
		DemoShapePhantom* currShPh = m_shapePhantoms[i];
		hkpAllCdBodyPairCollector coll;
		currShPh->getPenetrations( coll );

		// If there are entities overlapping with the hkpShapePhantom:
		if ( !coll.getHits().isEmpty() )
		{
			currShPh->setResultBits( coll );
			HK_SET_OBJECT_COLOR((hkUlong)currShPh->getCollidable(), currShPh->m_colourHit);
		}
		// Otherwise if there are no overlaps
		else
		{
			HK_SET_OBJECT_COLOR((hkUlong)currShPh->getCollidable(), currShPh->m_colourNoHit);
		}
	}

}

void TriggerVolumeComparisonDemo::moveShapePhantoms()
{
	for ( int i = 0; i < m_shapePhantoms.getSize(); i++ )
	{
		DemoShapePhantom* currShPh = m_shapePhantoms[i];
		//
		// Update position
		//
		if ( !currShPh->m_isFixed ) // should always be fixed for grid shape phantoms
		{
			currShPh->move( m_world->m_dynamicsStepInfo.m_stepInfo.m_deltaTime );
		}
	}

}

void TriggerVolumeComparisonDemo::moveAabbPhantoms()
{
	for ( int i = 0; i < m_aabbPhantoms.getSize(); i++ )
	{
		DemoAabbPhantom* currAabbPh = m_aabbPhantoms[i];
		//
		// Update position
		//
		if ( !currAabbPh->m_isFixed )
		{
			currAabbPh->move( m_world->m_dynamicsStepInfo.m_stepInfo.m_deltaTime );
		}
	}

}

void TriggerVolumeComparisonDemo::moveCharacterRigidBodies()
{
	int numRBs = m_characterProxies.getSize();

	//Apply the character controllers
	{
		HK_TIMER_BEGIN( "Simulate character rigid bodies", HK_NULL );

		for (int i=0; i < numRBs; i++)
		{
			// calculate the fulcrum point so the character controller knows when to switch direction
			hkReal characterRbRowNumber = 5.0f;
			hkVector4 pos = m_characterRigidBodies[i]->getPosition();
			hkVector4 vel = m_characterRigidBodies[i]->getLinearVelocity();
			hkVector4 origPos; origPos.setAddMul4(m_gridOrigin, m_gridVerticalOffset, characterRbRowNumber);
			origPos.addMul4((hkReal)i, m_gridHorizontalOffset);

			hkVector4 fulcrum; fulcrum.setMul4( characterRbRowNumber - m_numRows/2, m_gridVerticalOffset );
			hkVector4 toFulcrum; toFulcrum.setSub4(fulcrum,pos);
			hkReal amplitude = origPos.distanceTo3( fulcrum );

			// If it's exceeded its boundary and going the wrong way, flip velocity
			if ( toFulcrum.length3().getReal() > amplitude && vel.dot3(toFulcrum).getReal() < 0 )
			{
				vel.setNeg3( vel );

				// Update the velocity of the character proxy and integrate
				m_characterRigidBodies[i]->setLinearVelocity(vel, m_timestep);
			}
		}
		HK_TIMER_END();
	}
}
void TriggerVolumeComparisonDemo::moveCharacterProxies()
{
	int numProxies = m_characterProxies.getSize();

	//Apply the character controllers
	{
		HK_TIMER_BEGIN( "Simulate character proxies", HK_NULL );

		for (int i=0; i < numProxies; i++)
		{
			// calculate the fulcrum point so the character controller knows when to switch direction
			hkReal proxyRowNumber = 6.0f;
			hkVector4 pos = m_characterProxies[i]->getPosition();
			hkVector4 vel = m_characterProxies[i]->m_velocity;
			hkVector4 origPos; origPos.setAddMul4(m_gridOrigin, m_gridVerticalOffset, proxyRowNumber);
			origPos.addMul4((hkReal)i, m_gridHorizontalOffset);

			hkVector4 fulcrum; fulcrum.setMul4( proxyRowNumber - m_numRows/2, m_gridVerticalOffset );
			hkVector4 toFulcrum; toFulcrum.setSub4(fulcrum,pos);
			hkReal amplitude = origPos.distanceTo3( fulcrum );

			// If it's exceeded its boundary and going the wrong way, flip velocity
			if ( toFulcrum.length3().getReal() > amplitude && vel.dot3(toFulcrum).getReal() < 0 )
			{
				vel.setNeg3( vel );
			}

			hkStepInfo csi;
			csi.m_deltaTime = m_timestep;
			csi.m_invDeltaTime = 1.0f / m_timestep;

			// Update the velocity of the character proxy and integrate
			m_characterProxies[i]->setLinearVelocity(vel);
			m_characterProxies[i]->integrate( csi, m_world->getGravity() ); //m_world->getGravity()
		}
		HK_TIMER_END();
	}
}


// functions for adding shapes to the shape grid, down here for readability
void TriggerVolumeComparisonDemo::addGridRigidBody( int row, int col )
{
	// Set sphere position and shape
	hkVector4 currentPosition = m_gridOrigin;
	currentPosition.addMul4( (hkReal)row, m_gridVerticalOffset );
	currentPosition.addMul4( (hkReal)col, m_gridHorizontalOffset );
	m_sphereInfo.m_position = currentPosition;
	m_sphereInfo.m_shape = m_sphereShape;

	// create rigid body
	hkpRigidBody* sphereRB = new hkpRigidBody( m_sphereInfo );

	// Put the entity type in the user data, so the triggers know what they've hit
	sphereRB->setUserData( ENTITY_TYPE_RIGIDBODY );

	// Add the rigid body to the world
	m_world->addEntity( sphereRB );

	// Add our custom action to sphere
	addPingpongAction( sphereRB, row, currentPosition );

	sphereRB->removeReference();
}

void TriggerVolumeComparisonDemo::addGridTriggerVolume( int row, int col )
{
	// Calculate position offset from grid origin
	hkVector4 currentPosition = m_gridOrigin;
	currentPosition.addMul4( (hkReal)row, m_gridVerticalOffset );
	currentPosition.addMul4( (hkReal)col, m_gridHorizontalOffset );

	// Create rigid body
	m_sphereInfo.m_position = currentPosition;
	m_sphereInfo.m_shape = m_sphereShape;
	hkpRigidBody* movingTriggerVolRb = new hkpRigidBody ( m_sphereInfo );

	// Create a trigger volume from the rigid body
	DemoTriggerVolume* movingTriggerVol = new DemoTriggerVolume( movingTriggerVolRb );

	// Put the entity type in the user data, so the triggers know what they've hit
	movingTriggerVolRb->setUserData( ENTITY_TYPE_TRIGGERVOLUME );

	m_world->addEntity( movingTriggerVolRb );

	HK_SET_OBJECT_COLOR((hkUlong)movingTriggerVolRb->getCollidable(), movingTriggerVol->m_colourNoHit);

	// Add our custom action to trigger volume
	addPingpongAction( movingTriggerVolRb, row, currentPosition );

	// Reference counting
	movingTriggerVolRb->removeReference();
	movingTriggerVol->removeReference();
}

void TriggerVolumeComparisonDemo::addGridPhantomCallbackShape( int row, int col )
{
	// Calculate position offset from grid origin
	hkVector4 currentPosition = m_gridOrigin;
	currentPosition.addMul4( (hkReal)row, m_gridVerticalOffset );
	currentPosition.addMul4( (hkReal)col, m_gridHorizontalOffset );

	// Create the phantom callback shape
	DemoPhantomCallbackShape* movingPhantomCallbackShape = new DemoPhantomCallbackShape( );
	hkpBvShape* movingPCSbvShape = new hkpBvShape( m_sphereShape, movingPhantomCallbackShape );

	movingPhantomCallbackShape->setUserData( reinterpret_cast<hkUlong>(m_world) );
	m_sphereInfo.m_shape = movingPCSbvShape;
	m_sphereInfo.m_position = currentPosition;

	hkpRigidBody* movingPhantomCallbackShapeRB = new hkpRigidBody( m_sphereInfo );

	// Put the entity type in the user data, so the triggers know what they've hit
	movingPhantomCallbackShapeRB->setUserData( ENTITY_TYPE_PHANTOMCALLBACKSHAPE );

	m_world->addEntity( movingPhantomCallbackShapeRB );

	HK_SET_OBJECT_COLOR((hkUlong)movingPhantomCallbackShapeRB->getCollidable(), movingPhantomCallbackShape->m_colourNoHit);

	// Add our custom action to phantom callback shape
	addPingpongAction( movingPhantomCallbackShapeRB, row, currentPosition );

	// Reference counting
	movingPCSbvShape->removeReference();
	movingPhantomCallbackShape->removeReference();
	movingPhantomCallbackShapeRB->removeReference();
}

void TriggerVolumeComparisonDemo::addGridShapePhantom( int row, int col )
{
	// Calculate position offset from grid origin
	hkVector4 currentPosition = m_gridOrigin;
	currentPosition.addMul4( (hkReal)row, m_gridVerticalOffset );
	currentPosition.addMul4( (hkReal)col, m_gridHorizontalOffset );

	// demoShapePhantom needs a transformation...
	hkTransform transf = hkTransform::getIdentity();
	transf.setTranslation( currentPosition );

	DemoShapePhantom* movingShapePhantom = new DemoShapePhantom( m_sphereShape, transf );

	// Motion details - we can't move our phantom with the custom action, so we'll have to move it in stepDemo()
	movingShapePhantom->m_isFixed = false;
	hkVector4 pivot; pivot.setMul4( (hkReal)row - m_numRows/2, m_gridVerticalOffset );
	movingShapePhantom->setupMotion( m_gridVelocity, pivot );

	// Put the entity type in the user data, so the triggers know what they've hit
	movingShapePhantom->setUserData( ENTITY_TYPE_SHAPEPHANTOM );

	m_world->addPhantom( movingShapePhantom );
	m_shapePhantoms.pushBack( movingShapePhantom );
	HK_SET_OBJECT_COLOR((hkUlong)movingShapePhantom->getCollidable(), hkColor::semiTransparent( hkColor::GREEN ));

	// Reference counting
	movingShapePhantom->removeReference();
}

void TriggerVolumeComparisonDemo::addGridAabbPhantom( int row, int col )
{
	// Calculate position offset from grid origin
	hkVector4 currentPosition = m_gridOrigin;
	currentPosition.addMul4( (hkReal)row, m_gridVerticalOffset );
	currentPosition.addMul4( (hkReal)col, m_gridHorizontalOffset );

	// Create the AABB
	hkAabb movingAabb;
	hkReal radius = m_sphereShape->getRadius();
	hkVector4 sphereHalfSize( radius, radius, radius );
	movingAabb.m_min.setSub4( currentPosition, sphereHalfSize );
	movingAabb.m_max.setAdd4( currentPosition, sphereHalfSize );

	// Create the AABB Phantom and add it to the world
	DemoAabbPhantom* movingAabbPhantom = new DemoAabbPhantom( movingAabb );

	// Motion details
	movingAabbPhantom->m_isFixed = false;
	hkVector4 pivot; pivot.setMul4( (hkReal)row - m_numRows/2, m_gridVerticalOffset );
	movingAabbPhantom->setupMotion( m_gridVelocity, pivot );

	// Put the entity type in the user data, so the triggers know what they've hit
	movingAabbPhantom->setUserData( ENTITY_TYPE_AABBPHANTOM );

	// Add the phantoms to the  world and phantom list
	m_world->addPhantom( movingAabbPhantom );
	m_aabbPhantoms.pushBack( movingAabbPhantom );

	// Reference counting
	movingAabbPhantom->removeReference();
}

void TriggerVolumeComparisonDemo::addGridCharacterRigidBody( int row, int col )
{
	// Calculate position offset from grid origin
	hkVector4 currentPosition = m_gridOrigin;
	currentPosition.addMul4( (hkReal)row, m_gridVerticalOffset );
	currentPosition.addMul4( (hkReal)col, m_gridHorizontalOffset );

	// Construct a character rigid body
	hkpCharacterRigidBodyCinfo crbCinfo;
	crbCinfo.m_shape = m_charControlShape;
	crbCinfo.m_position = currentPosition;
	crbCinfo.m_maxForce = 1000.0f;
	crbCinfo.m_mass = 100.0f;
	crbCinfo.m_up.setNeg4( m_world->getGravity() );
	crbCinfo.m_up.normalize3();

	hkpCharacterRigidBody* movingCRB = new hkpCharacterRigidBody( crbCinfo );
	movingCRB->setLinearVelocity(m_gridVelocity, m_timestep);

	// Put the entity type in the user data, so the triggers know what they've hit
	movingCRB->getRigidBody()->setUserData( ENTITY_TYPE_CHAR_RIGIDBODY );

	m_world->addEntity( movingCRB->getRigidBody() );
	m_characterRigidBodies.pushBack( movingCRB );
}

void TriggerVolumeComparisonDemo::addGridCharacterProxy( int row, int col )
{
	// Calculate position offset from grid origin
	hkVector4 currentPosition = m_gridOrigin;
	currentPosition.addMul4( (hkReal)row, m_gridVerticalOffset );
	currentPosition.addMul4( (hkReal)col, m_gridHorizontalOffset );
	hkTransform transf = hkTransform::getIdentity();
	transf.setTranslation( currentPosition );

	hkpShapePhantom* proxyShapePhantom = new hkpSimpleShapePhantom( m_charControlShape, transf );

	// Put the entity type in the user data, so the triggers know what they've hit
	proxyShapePhantom->setUserData( ENTITY_TYPE_CHAR_PROXY );

	m_world->addPhantom( proxyShapePhantom );

	// Reference counting
	proxyShapePhantom->removeReference();

	// Construct a character proxy
	hkpCharacterProxyCinfo cpci;
	cpci.m_position = currentPosition;
	cpci.m_up.setNeg4( m_world->getGravity() );
	cpci.m_up.normalize3();	
	cpci.m_velocity = m_gridVelocity;

	cpci.m_shapePhantom = proxyShapePhantom;
	hkpCharacterProxy* characterProxy = new hkpCharacterProxy( cpci );
	m_characterProxies.pushBack( characterProxy );
}


void TriggerVolumeComparisonDemo::setupFixedVolumes( hkBool includeRB, hkBool includeTV, hkBool includePCS, hkBool includeSP, hkBool includeAP )
{

	//
	// Create the fixed volumes for all variants
	//

	// Common box geometry
	hkVector4 firstBoxPos; firstBoxPos.setMul4( -(m_numCols - 1.0f)/2 - 1, m_gridHorizontalOffset );
	hkReal fixedShapeRotation = HK_REAL_PI / 4.0f;
	hkVector4 fixedShapeNarrowPhaseSize ( 5.0f, 2.0f, 1.0f );
	hkReal aabbX = cos(fixedShapeRotation)*fixedShapeNarrowPhaseSize(0) + sin(fixedShapeRotation)*fixedShapeNarrowPhaseSize(1);
	hkReal aabbY = sin(fixedShapeRotation)*fixedShapeNarrowPhaseSize(0) + cos(fixedShapeRotation)*fixedShapeNarrowPhaseSize(1);
	hkVector4 fixedAabbSize ( aabbX, aabbY, fixedShapeNarrowPhaseSize(2) );

	// Common box info
	hkpRigidBodyCinfo boxRbInfo;
	hkpShape* boxShape = new hkpBoxShape( fixedShapeNarrowPhaseSize, 0 );
	hkpShape* convexVerticesShape = hkpShapeGenerator::createConvexVerticesBox( fixedShapeNarrowPhaseSize, 0 );
	boxRbInfo.m_rotation.setAxisAngle( hkVector4( 0.0f, 0.0f, 1.0f ), fixedShapeRotation);
	boxRbInfo.m_position = firstBoxPos;
	boxRbInfo.m_motionType = hkpMotion::MOTION_FIXED;
	boxRbInfo.m_qualityType = HK_COLLIDABLE_QUALITY_FIXED;

	// Rigid Body
	if ( includeRB )
	{
		// Set position and shape
		boxRbInfo.m_position.add4( m_gridHorizontalOffset );
		boxRbInfo.m_shape = boxShape;

		// Create rigid body
		hkpRigidBody* fixedRB = new hkpRigidBody( boxRbInfo );

		// Set entity type for the triggers' data collection
		fixedRB->setUserData( ENTITY_TYPE_RIGIDBODY );

		// Add to world, reference count
		m_world->addEntity( fixedRB );
		fixedRB->removeReference();
	}

	//
	// Create fixed trigger volume
	//
	if ( includeTV )
	{
		// Set position and shape
		boxRbInfo.m_position.add4( m_gridHorizontalOffset );
		// Create a box-like convex vertices shape for the triggerVolume to avoid using the box-box agent
		// with deep-collisions.
		boxRbInfo.m_shape = convexVerticesShape;

		// Create rigid body
		hkpRigidBody* fixedTriggerVolumeRb = new hkpRigidBody( boxRbInfo );

		// Set entity type for the triggers' data collection
		fixedTriggerVolumeRb->setUserData( ENTITY_TYPE_TRIGGERVOLUME );

		// Make the rigid body a trigger volume
		m_fixedTriggerVolume = new DemoTriggerVolume( fixedTriggerVolumeRb );

		// Add rigid body to world
		m_world->addEntity( fixedTriggerVolumeRb );

		// Set colour
		HK_SET_OBJECT_COLOR((hkUlong)fixedTriggerVolumeRb->getCollidable(), m_fixedTriggerVolume->m_colourNoHit);

		// Reference count
		fixedTriggerVolumeRb->removeReference();
	}

	//
	// Create fixed phantom callback shape
	//
	if ( includePCS )
	{
		// Create phantom callback shape, link with box in bounding volume shape
		m_fixedPhantomCallbackShape = new DemoPhantomCallbackShape( );
		hkpBvShape* fixedPCSbvShape = new hkpBvShape( boxShape, m_fixedPhantomCallbackShape );

		// Set the shape user data to be a pointer to the world, for timestamp collection
		m_fixedPhantomCallbackShape->setUserData( reinterpret_cast<hkUlong>(m_world) );

		// Set position and shape
		boxRbInfo.m_position.add4( m_gridHorizontalOffset );
		boxRbInfo.m_shape = fixedPCSbvShape;

		// Create rigid body
		hkpRigidBody* fixedPhantomCallbackShapeRB = new hkpRigidBody( boxRbInfo );

		// Set entity type for the triggers' data collection
		fixedPhantomCallbackShapeRB->setUserData( ENTITY_TYPE_PHANTOMCALLBACKSHAPE );

		// Add rigid body to world
		m_world->addEntity( fixedPhantomCallbackShapeRB );

		// Set colour
		HK_SET_OBJECT_COLOR((hkUlong)fixedPhantomCallbackShapeRB->getCollidable(), m_fixedPhantomCallbackShape->m_colourNoHit);

		// Reference counting
		fixedPCSbvShape->removeReference();
		fixedPhantomCallbackShapeRB->removeReference();
	}

	//
	// Create fixed shape phantom
	//
	if ( includeSP )
	{
		// demoShapePhantom needs a transformation...
		boxRbInfo.m_position.add4( m_gridHorizontalOffset );
		hkVector4 pos = boxRbInfo.m_position;
		hkQuaternion rot; rot.setAxisAngle( hkVector4( 0, 0, 1 ), fixedShapeRotation );
		hkTransform transf; transf.setRotation( rot );
		transf.setTranslation( pos );

		// Create shape phantom
		m_fixedShapePhantom = new DemoShapePhantom( boxShape, transf );
		m_fixedShapePhantom->m_isFixed = true;

		// Set entity type for the triggers' data collection
		m_fixedShapePhantom->setUserData( ENTITY_TYPE_SHAPEPHANTOM );

		// Add phantom to world
		m_world->addPhantom( m_fixedShapePhantom );

		// Set colour
		HK_SET_OBJECT_COLOR((hkUlong)m_fixedShapePhantom->getCollidable(), hkColor::semiTransparent( hkColor::GREEN ));

		// Store in array for access by the movement and custom collision detection functions in stepDemo
		m_shapePhantoms.pushBack(m_fixedShapePhantom);
	}

	//
	// Create fixed AABB phantom
	//
	if ( includeAP )
	{
		// Create the AABB
		hkAabb fixedAabb;
		boxRbInfo.m_position.add4( m_gridHorizontalOffset );
		hkVector4 pos = boxRbInfo.m_position;
		fixedAabb.m_min.setSub4( pos, fixedAabbSize );
		fixedAabb.m_max.setAdd4( pos, fixedAabbSize );

		// Create the phantom
		m_fixedAabbPhantom = new DemoAabbPhantom( fixedAabb );
		m_fixedAabbPhantom->m_isFixed = true;

		// Set entity type for the triggers' data collection
		m_fixedAabbPhantom->setUserData( ENTITY_TYPE_AABBPHANTOM );

		// Add the phantom to the world
		m_world->addPhantom( m_fixedAabbPhantom );

		// Store in array for access by the movement functions in stepDemo
		m_aabbPhantoms.pushBack( m_fixedAabbPhantom );
	}
	convexVerticesShape->removeReference();
	boxShape->removeReference();
}

// Function to add a PingpongAction to a rigid body
void TriggerVolumeComparisonDemo::addPingpongAction( hkpRigidBody* Rb, const int& rowIndex, const hkVector4& position )
{
	hkVector4 pivot; pivot.setMul4( (hkReal)rowIndex - m_numRows/2, m_gridVerticalOffset );
	PingpongAction* fly = new PingpongAction( Rb, m_gridVelocity, position, pivot );
	m_world->addAction(fly);
	fly->removeReference();
}

// Display labels on the fixed volumes
void TriggerVolumeComparisonDemo::displayFixedObjectLabels()
{
	hkVector4 textPos, centreOffsetForBelowText;
	if ( m_variantId == VARIANT_DETECTION_COMPARISON )
	{
		centreOffsetForBelowText = hkVector4( 0.0f, -10.0f, 0.0f );
	}
	else
	{
		centreOffsetForBelowText = hkVector4( 0.0f, 0.0f, 5.0f );
	}

	textPos = m_fixedTriggerVolume->getTriggerBody()->getTransform().getTranslation();
	textPos.add4( centreOffsetForBelowText );
	HK_DISPLAY_3D_TEXT( "TV", textPos, hkColor::BLACK );

	textPos.setAdd4( textPos, m_gridHorizontalOffset );
	HK_DISPLAY_3D_TEXT( "PCS", textPos, hkColor::BLACK );

	textPos = m_fixedShapePhantom->getTransform().getTranslation();
	textPos.add4( centreOffsetForBelowText );
	HK_DISPLAY_3D_TEXT( "SP", textPos, hkColor::BLACK );

	m_fixedAabbPhantom->getAabb().getCenter( textPos );
	textPos.add4( centreOffsetForBelowText );
	HK_DISPLAY_3D_TEXT( "AP", textPos, hkColor::BLACK );
}

// Functions for displaying the results in a text box
void TriggerVolumeComparisonDemo::displayResultsText()
{
	hkStringBuf resultString, timerString, outStr;
	outStr.printf( "" );

	// Heading - different for each variant
	if ( m_variantId == VARIANT_DETECTION_COMPARISON )
	{
		outStr.appendPrintf( "Current time:\t\t\t\t\t\t%.5f\n\n", m_world->getCurrentTime() );
		outStr.appendPrintf( "\t\tRB\tTV\tPCS\tSP\tAP\tCRB\tCP\tLast Enter \tLast Leave\n" );
	}
	else
	{
		outStr.appendPrintf( "Current time:\t%.5f\n", m_world->getCurrentTime() );
		outStr.appendPrintf( "\t\tRB\tLast Enter \tLast Leave\n" );
	}

	// Trigger Volume
	formatResultString( m_fixedTriggerVolume->m_resultBits, resultString );
	formatTimerString( m_fixedTriggerVolume->m_lastEnterEvent, m_fixedTriggerVolume->m_lastLeaveEvent, timerString );
	outStr.appendPrintf( "TV:\t%s \t", resultString.cString() );
	outStr.appendPrintf( "%s\n", timerString.cString() );

	// Phantom Callback Shape
	formatResultString( m_fixedPhantomCallbackShape->m_resultBits, resultString );
	formatTimerString( m_fixedPhantomCallbackShape->m_lastEnterEvent, m_fixedPhantomCallbackShape->m_lastLeaveEvent, timerString );
	outStr.appendPrintf( "PCS:\t%s \t", resultString.cString() );
	outStr.appendPrintf( "%s\n", timerString.cString() );

	// Shape Phantom
	formatResultString( m_fixedShapePhantom->m_resultBits, resultString );
	formatTimerString( m_fixedShapePhantom->m_lastEnterEvent, m_fixedShapePhantom->m_lastLeaveEvent, timerString );
	outStr.appendPrintf( "SP:\t%s \t", resultString.cString() );
	outStr.appendPrintf( "%s\n", timerString.cString() );

	// AABB Phantom
	formatResultString( m_fixedAabbPhantom->m_resultBits, resultString );
	formatTimerString( m_fixedAabbPhantom->m_lastEnterEvent, m_fixedAabbPhantom->m_lastLeaveEvent, timerString );
	outStr.appendPrintf( "AP:\t%s \t", resultString.cString() );
	outStr.appendPrintf( "%s\n", timerString.cString() );

	// Key - different for each variant
	outStr.appendPrintf( "\nB: detects broadphase" );
	outStr.appendPrintf( "\nN: detects narrowphase" );
	outStr.appendPrintf( "\n" );
	if ( m_variantId == VARIANT_DETECTION_COMPARISON )
	{
		outStr.appendPrintf( "\nRB:\tRigid Body (Grey ball)" );
		outStr.appendPrintf( "\nTV:\tTrigger Volume (Blue ball and box)" );
		outStr.appendPrintf( "\nPCS:\tPhantom Callback Shape (Green ball and box)" );
		outStr.appendPrintf( "\nSP:\tShape Phantom (Yellow ball and box with AABB)" );
		outStr.appendPrintf( "\nAP:\tAABB Phantom (Yellow AABB)" );
		outStr.appendPrintf( "\nCRB:\tCharacter Rigid Body (Red capsule)" );
		outStr.appendPrintf( "\nCP\tCharacter Proxy (Yellow capsule)" );
	}
	else
	{
		outStr.appendPrintf( "\nRB:\tRigid Body (Grey ball)" );
		outStr.appendPrintf( "\nTV:\tTrigger Volume (Blue box)" );
		outStr.appendPrintf( "\nPCS:\tPhantom Callback Shape (Green box)" );
		outStr.appendPrintf( "\nSP:\tShape Phantom (Yellow box with AABB)" );
		outStr.appendPrintf( "\nAP:\tAABB Phantom (Yellow AABB)" );
	}
	m_env->m_textDisplay->outputText( outStr, 20, 100, 0xffffffff );
}

void TriggerVolumeComparisonDemo::formatResultString( const hkBitField* bf, hkStringBuf& outStr )
{
	// Bits flip permanently on detection of the appropriate hit type
	// Bitfield is two bits each for rigid body, trigger volume, phantom callback shape, shape phantom, AABB phantom,
	// character rigid body and character proxy. The first bit is for broadphase collision, the second for narrowphase.
	// For no hit, output char is --
	// For broadphase hit, first - is changed to B
	// For narrowphase hit, second - is changed to N

	outStr.clear();
	if ( m_variantId == VARIANT_DETECTION_COMPARISON )
	{
		for ( int i = 0; i < bf->getSize()/2; i++ )
		{
			if ( bf->get(2*i) == 0 )
			{
				outStr.append("-");
			}
			else
			{
				outStr.append("B");
			}
			if ( bf->get(2*i + 1) == 0 )
			{
				outStr.append("-\t");
			}
			else
			{
				outStr.append("N\t");
			}
		}
		outStr.chompEnd(1);
	}
	if ( m_variantId == VARIANT_CONTINUOUS )
	{
		if ( bf->get(0) == 0 )
		{
			outStr.append( "-" );
		}
		else
		{
			outStr.append( "B" );
		}
		if ( bf->get(1) == 0 )
		{
			outStr.append( "-" );
		}
		else
		{
			outStr.append( "N" );
		}
	}
}

void TriggerVolumeComparisonDemo::formatTimerString( const hkTime& enterTime, const hkTime& leaveTime, hkStringBuf& outStr )
{
	outStr.clear();
	if ( enterTime == 0 )
	{
		outStr.appendPrintf( "-- \t\t\t" );
	}
	else
	{
		outStr.appendPrintf( "%f \t", enterTime );
	}
	if ( leaveTime == 0 )
	{
		outStr.appendPrintf( " --" );
	}
	else
	{
		outStr.appendPrintf( " %f", leaveTime );
	}
}

void TriggerVolumeComparisonDemo::addResultToBitField( const EntityType volTypeID, const CollisionPhase phaseID, hkBitField* bf )
{
	// Bits flip permanently on detection of the appropriate hit type
	// Bitfield is two bits each for rigid body, trigger volume, phantom callback shape, shape phantom, AABB phantom,
	// character rigid body and character proxy. The first bit is for broadphase collision, the second for narrowphase.
	if ( phaseID == COLLISION_PHASE_BROADPHASE )
	{
		HitType hitTypes[] = {
			INVALID_HIT_TYPE,
			BROADPHASE_RIGIDBODY,
			BROADPHASE_TRIGGERVOLUME,
			BROADPHASE_PHANTOMCALLBACKSHAPE,
			BROADPHASE_SHAPEPHANTOM,
			BROADPHASE_AABBPHANTOM,
			BROADPHASE_CHAR_RIGIDBODY,
			BROADPHASE_CHAR_PROXY
		};
		bf->assign( hitTypes[(int)volTypeID], 1 );
	}
	else
	{
		HitType hitTypes[] = {
			INVALID_HIT_TYPE,
			NARROWPHASE_RIGIDBODY,
			NARROWPHASE_TRIGGERVOLUME,
			NARROWPHASE_PHANTOMCALLBACKSHAPE,
			NARROWPHASE_SHAPEPHANTOM,
			NARROWPHASE_AABBPHANTOM,
			NARROWPHASE_CHAR_RIGIDBODY,
			NARROWPHASE_CHAR_PROXY
		};
		bf->assign( hitTypes[(int)volTypeID], 1 );
	}
}
static const char variantName_0[] = "Interaction with each other, character controllers and rigid bodies";
static const char variantName_1[] = "Continuous collision detection";

static const char variantHelp_0[] = "Compares the volumes for broadphase and narrowphase detection with rigid bodies, character controllers and each other";
static const char variantHelp_1[] = "Compares the volumes for detection of fast-moving objects with continuous collision detection enabled";

static const char variantDetails_0[] = \
"A demo which shows the difference between hkpTriggerVolumes, " \
"hkpAabbPhantoms, hkpShapePhantoms and hkpPhantomCallbackShapes.\n" \
"This variant shows how to detect both broadphase and narrowphase overlaps." \
"Also, it compares the interaction of the various types of trigger volume with rigid bodies, with each other, and with both kinds of character controllers.";
static const char variantDetails_1[] = \
"A demo which shows the difference between hkpTriggerVolumes, " \
"hkpAabbPhantoms, hkpShapePhantoms and hkpPhantomCallbackShapes.\n" \
"This variant shows how each detects enter and leave events for fast moving objects " \
"with continuous collision detection enabled";


HK_DECLARE_DEMO_VARIANT( TriggerVolumeComparisonDemo, HK_DEMO_TYPE_PHYSICS_2012, variantName_0, 0, variantHelp_0, variantDetails_0);
HK_DECLARE_DEMO_VARIANT( TriggerVolumeComparisonDemo, HK_DEMO_TYPE_PHYSICS_2012, variantName_1, 1, variantHelp_1, variantDetails_1);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
