/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/UseCase/TriggerVolume/KeyframedTriggerVolume/KeyframedTriggerVolumeDemo.h>

// We will need these shapes
#include <Physics2012/Collide/Shape/Misc/PhantomCallback/hkpPhantomCallbackShape.h>
#include <Physics2012/Collide/Shape/Misc/Bv/hkpBvShape.h>

#include <Physics2012/Utilities/Collide/TriggerVolume/hkpTriggerVolume.h>

#include <Physics2012/Utilities/Dynamics/KeyFrame/hkpKeyFrameUtility.h>

// We need to display some collision based info
#include <Common/Visualize/hkDebugDisplay.h>

static const hkReal circleRadius = 5.0f;

static const hkColor::Argb bodyColors[2][3] =
{
	{ 
		hkColor::rgbFromChars( 0x70, 0x00, 0x00 ),
		hkColor::rgbFromChars( 0x00, 0x70, 0x00 ),
		hkColor::rgbFromChars( 0x00, 0x00, 0x70 ),
	},
	{
		hkColor::rgbFromChars( 0xff, 0x00, 0x00 ),
		hkColor::rgbFromChars( 0x00, 0xff, 0x00 ),
		hkColor::rgbFromChars( 0x00, 0x00, 0xff ),
	}
};

static const hkColor::Argb triggerVolumeColors[2] = 
{
	hkColor::rgbFromChars( 0xff, 0xff, 0x00, 0x70 ),
	hkColor::rgbFromChars( 0x00, 0xff, 0xff, 0x70 ),
};

// While the entry event simply colors the ball object, the exit event also applies an impulse to the ball to make
// it bounce upwards back towards the wall.
class KeyframedTriggerVolume : public hkpTriggerVolume
{
	public:
		HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DEMO );

		KeyframedTriggerVolume( hkpRigidBody* triggerBody, hkpRigidBody** bodies ) : hkpTriggerVolume( triggerBody ), m_bodies( bodies ) {}

		using hkpTriggerVolume::triggerEventCallback;
		virtual void triggerEventCallback( hkpRigidBody* body, EventType type );

	public:
		hkpRigidBody** m_bodies;
};

void KeyframedTriggerVolume::triggerEventCallback( hkpRigidBody* body, EventType type )
{
	for ( int i = 0; i < 3; ++i )
	{
		if ( m_bodies[i] == body )
		{
			if ( type & ENTERED_EVENT )
			{
				HK_SET_OBJECT_COLOR((hkUlong)body->getCollidable(), bodyColors[1][i] );
			}
			if ( type & LEFT_EVENT )
			{
				HK_SET_OBJECT_COLOR( (hkUlong)body->getCollidable(), bodyColors[0][i] );
			}
		}
	}
}


KeyframedTriggerVolumeDemo::KeyframedTriggerVolumeDemo( hkDemoEnvironment* env )
: hkDefaultPhysics2012Demo(env, DEMO_FLAGS_NO_SERIALIZE)
, m_frameCount( 0 )
{
	//
	// Setup the camera
	//
	{
		hkVector4 from( 0.0f, 0.0f, 20.0f );
		hkVector4 to( 0.0f, 0.0f, 0.0f );
		hkVector4 up( 0.0f, 1.0f, 0.0f );
		setupDefaultCameras( env, from, to, up );
	}

	//
	// Create the world
	//
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		info.setBroadPhaseWorldSize( 100.0f );
		info.m_enableDeactivation = false;
		info.m_gravity.setZero4();
		m_world = new hkpWorld( info );
		m_world->lock();

		setupGraphics();
	}

	//
	// Register the collision agents
	//
	{
		hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );
	}


	// In this demo we have three different objects; the wall, the sphere and a phantom volume. Both the wall, which is simply a box,
	// and the sphere are created in the usual manner using a hkpRigidBodyCinfo 'blueprint' and are added to the world.

	const hkReal radius = 1.0f;
	hkpShape *const shape = new hkpSphereShape( radius );

	//
	// Create three bodies - fixed, keyframed and dynamic
	//
	for ( int i = 0; i < 3; ++i )
	{
		hkpRigidBodyCinfo info;
		info.m_shape = shape;
		switch ( i )
		{
			case 0:
				//text = "Dynamic";
				info.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;
				info.m_qualityType = HK_COLLIDABLE_QUALITY_CRITICAL;
				{
					info.m_mass = 1.0f;
					hkMassProperties massProperties;
					hkpInertiaTensorComputer::computeSphereVolumeMassProperties(radius, info.m_mass, massProperties);
					info.m_inertiaTensor = massProperties.m_inertiaTensor;
					info.m_centerOfMass = massProperties.m_centerOfMass;
					info.m_mass = massProperties.m_mass;
				}
				break;

			case 1:
				//text = "Keyframed";
				info.m_motionType = hkpMotion::MOTION_KEYFRAMED;
				info.m_qualityType = HK_COLLIDABLE_QUALITY_KEYFRAMED;
				break;

			case 2:
			default:
				//text = "Fixed";
				info.m_motionType = hkpMotion::MOTION_FIXED;
				info.m_qualityType = HK_COLLIDABLE_QUALITY_FIXED;
				break;
		}
		info.m_position.set( circleRadius * hkMath::sin( HK_REAL_PI * 0.66f * i ), circleRadius * hkMath::cos( HK_REAL_PI * 0.66f * i ), 0.0f );

		// Create fixed box
		m_body[i] = new hkpRigidBody( info );
		m_world->addEntity( m_body[i] );
		HK_SET_OBJECT_COLOR((hkUlong) m_body[i]->getCollidable(), bodyColors[0][i] );
	}

	// Create two keyframed trigger volumes.
	for ( unsigned char i = 0; i < 2; ++i )
	{
		hkpRigidBodyCinfo info;
		info.m_motionType = hkpMotion::MOTION_KEYFRAMED;
		info.m_qualityType = i ? HK_COLLIDABLE_QUALITY_KEYFRAMED : HK_COLLIDABLE_QUALITY_KEYFRAMED_REPORTING;

		info.m_shape = shape;
		info.m_position.set( circleRadius * hkMath::sin( HK_REAL_PI * i ), circleRadius * hkMath::cos( HK_REAL_PI * i ), 0.0f );

		m_triggerBody[i] = new hkpRigidBody( info );
		m_world->addEntity( m_triggerBody[i] );
	       
		// Make the volume semi-transparent.
		HK_SET_OBJECT_COLOR((hkUlong) m_triggerBody[i]->getCollidable(), triggerVolumeColors[i] );
		// Convert the body into a trigger volume.
		( new KeyframedTriggerVolume( m_triggerBody[i], m_body ) )->removeReference();
	}

	shape->removeReference();
	m_world->unlock();
}

KeyframedTriggerVolumeDemo::~KeyframedTriggerVolumeDemo()
{
	for ( int i = 0; i < 2; ++i )
	{
		m_triggerBody[i]->removeReference();
	}
	for ( int i = 0; i < 3; ++i )
	{
		m_body[i]->removeReference();
	}
}

hkDemo::Result KeyframedTriggerVolumeDemo::stepDemo()
{
	m_world->lock();

	for ( int i = 0; i < 2; ++i )
	{
		hkVector4 position;
		hkReal circlePosition = ( m_frameCount * m_timestep * 0.5f ) + i;
		position.set( circleRadius * hkMath::sin( HK_REAL_PI * circlePosition ), circleRadius * hkMath::cos( HK_REAL_PI * circlePosition ), 0.0f );
		hkpKeyFrameUtility::applyHardKeyFrame( position, hkQuaternion::getIdentity(), 1 / m_timestep, m_triggerBody[i] );
	}

	m_world->unlock();

	++m_frameCount;

	return hkDefaultPhysics2012Demo::stepDemo();
}


static const char helpString[] = \
"This demo shows two keyframed trigger volumes interacting with three bodies.\n\n"
"It shows that you should use HK_COLLIDABLE_QUALITY_KEYFRAMED_REPORTING if you want to pick up "
"collisions with other keyframed or fixed bodies, but HK_COLLIDABLE_QUALITY_KEYFRAMED if you only "
"want to pick up collisions with dynamic bodies.\n\n"
"The bodies are colored as follows:\n"
"Red - a dynamic body\n"
"Green - a keyframed body\n"
"Blue - a fixed body\n\n"
"The trigger volumes are colored as follows:\n"
"Cyan - keyframed\n"
"Yellow - keyframed reporting\n";

HK_DECLARE_DEMO( KeyframedTriggerVolumeDemo, HK_DEMO_TYPE_PHYSICS_2012, "Keyframed trigger volumes and reporting types", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
