/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>

// Need to display debug lines.
#include <Common/Visualize/hkDebugDisplay.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics/Constraint/Data/BallAndSocket/hkpBallAndSocketConstraintData.h>
#include <Physics2012/Dynamics/Constraint/hkpConstraintInstance.h>
#include <Physics2012/Utilities/Collide/Filter/GroupFilter/hkpGroupFilterUtil.h>
#include <Physics2012/Utilities/Actions/AngularDashpot/hkpAngularDashpotAction.h>

#include <Demos/DemoCommon/DemoFramework/hkDefaultPhysics2012Demo.h>

class ProceduralTree;
class ProceduralTreeDemo : public hkDefaultPhysics2012Demo
{
public:
	HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);

	ProceduralTreeDemo(hkDemoEnvironment* env);
	~ProceduralTreeDemo();

	// Step the demo
	virtual Result stepDemo();

private:	
	ProceduralTree* m_tree;
};


class ProceduralTreeCinfo
{
public:
	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, ProceduralTreeCinfo);

	ProceduralTreeCinfo()
		: m_world(HK_NULL)
		, m_nbLevel(3)
		, m_nbBranchPerJoint(3)
		, m_sectionLength(1.f)
		, m_branchLengthRatio(1.f)
		, m_massDensity(50.f)
		, m_springStrength(10.f)
		, m_springDamping(0.17f)
	{
		m_pos.setZero();
		m_rot.setIdentity();
	}

	hkpWorld* m_world;
	hkVector4 m_pos;
	hkQuaternion m_rot;

	hkUint32 m_nbLevel;
	hkUint32 m_nbBranchPerJoint;
	hkReal m_sectionLength;
	hkReal m_branchLengthRatio;
	hkReal m_massDensity;

	//spring
	hkReal m_springStrength;
	hkReal m_springDamping;
};

class ProceduralTree
{
public:
	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, ProceduralTree);

	ProceduralTree(const ProceduralTreeCinfo& param);
	virtual ~ProceduralTree();

private:
	void Destroy();
	void Init(const ProceduralTreeCinfo& param);

	hkpRigidBody* CreateRigid(const hkVector4& pos, const hkQuaternion& rot, hkReal halfLength, 
		hkReal radius, hkReal massDensity, hkBool fixed, int level);
	void CreateBranches(hkpRigidBody* parent, const hkVector4& parentPos, 
		const hkQuaternion& parentRot, const ProceduralTreeCinfo& param, 
		hkUint32 nbLevel, hkUint32 nbBranchPerJoint);

	hkpWorld* m_hkWorld;

	hkpRigidBody*           m_trunk;
	hkArray<hkpRigidBody*> m_branches;

	hkArray<hkpConstraintInstance*> m_constraints;
	hkArray<hkpAction*> m_actions;
};

// Base layer ID for the tree group filter.
const int BASE_LAYER_ID = 10;

ProceduralTreeDemo::ProceduralTreeDemo(hkDemoEnvironment* env)
	: hkDefaultPhysics2012Demo(env)
{
	// Define gravity as zero for this demo.
	hkVector4 gravity( 0.0f, 0.0f, -9.8f );

	//
	// Set up the camera.
	//
	// We are using the simple demo framework so we first we set up the camera 
	// with the camera looking approximately at the origin.
	{
		hkVector4 from(0.0f, 5.0f, 20.0f);
		hkVector4 to  (0.0f, 0.0f, 0.0f);
		hkVector4 up  (0.0f, 0.0f, 1.0f);
		setupDefaultCameras( env, from, to, up );
	}
	
	//
	// Create the world.
	//
	// We create our world with broadphase extents set to +/- 100 units and specify
	// hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM which is a reasonably all purpose solver type.
	hkpGroupFilter* filter = HK_NULL;
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		info.m_gravity = gravity;	
		info.setBroadPhaseWorldSize( 100.0f );
		filter = new hkpGroupFilter;
		info.m_collisionFilter.setAndDontIncrementRefCount(filter);
		m_world = new hkpWorld( info );
		m_world->lock();
		setupGraphics();
	}

	// Register all collision agents
	hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );
    
    // Create procedural tree
    ProceduralTreeCinfo param;
    param.m_world = m_world;
    
    // override parameters here.
    //param.m_nbBranchPerJoint = 2;

	// Disable all collisions between level X and level X+1.
	for(unsigned int i = 0; i < param.m_nbLevel+1; ++i)
	{
		filter->disableCollisionsBetween(BASE_LAYER_ID+i, BASE_LAYER_ID+i+1);
	}
    m_tree = new ProceduralTree(param);


	m_world->unlock();
}

ProceduralTreeDemo::~ProceduralTreeDemo()
{
	m_world->lock();
    
    delete m_tree;
    m_tree = NULL;

	m_world->unlock();
}

hkDemo::Result ProceduralTreeDemo::stepDemo()
{
	return hkDefaultPhysics2012Demo::stepDemo();
}



ProceduralTree::ProceduralTree(const ProceduralTreeCinfo& param)
	: m_hkWorld(NULL)
	, m_trunk(NULL)
{
    Init(param);
}

ProceduralTree::~ProceduralTree()
{
    Destroy();
}

void ProceduralTree::Destroy()
{
    if (m_trunk)
    {
        if (m_trunk->isAddedToWorld())
        {
            for(hkUint32 i=0, size=m_branches.getSize(); i<size; ++i)
            {
                m_hkWorld->removeEntity(m_branches[i]);
            }
        }
        m_trunk = NULL;

        for(hkUint32 i=0, size=m_branches.getSize(); i<size; ++i)
        {
            m_branches[i]->setUserData(0);
            m_branches[i]->removeReference();
        }
        m_branches.clear();
    }

    hkArray<hkpAction*>::iterator it = m_actions.begin();
    hkArray<hkpAction*>::iterator end = m_actions.end();
    for(; it!=end; ++it)
    {
        (*it)->removeReference();
    }
    m_actions.clear();

    for(hkUint32 i=0, size=m_constraints.getSize(); i<size; ++i)
    {
        if (m_constraints[i]->getOwner())
        {
            m_hkWorld->removeConstraint(m_constraints[i]);
        }
        m_constraints[i]->setUserData(0);
        m_constraints[i]->removeReference();
    }
    m_constraints.clear();
}

void ProceduralTree::Init(const ProceduralTreeCinfo& param)
{
    m_hkWorld = param.m_world;

    hkReal halfLength = param.m_sectionLength;
    hkReal radius = param.m_sectionLength * 0.1f;
    hkQuaternion rot(param.m_rot);
    hkVector4 pos;
    pos.setRotatedDir(rot, hkVector4(0.f, 0.f, halfLength + radius));
    pos.add(param.m_pos);
    m_trunk = CreateRigid(pos, rot, halfLength, radius, param.m_massDensity, true, param.m_nbLevel+1);
    m_branches.pushBack(m_trunk);

    if(param.m_nbLevel > 0)
    {
        // Create branches
        hkVector4 attachPos;
        attachPos.setRotatedDir(rot, hkVector4(0.f, 0.f, halfLength));
        attachPos.add(pos);
        CreateBranches(m_trunk, attachPos, rot, param, param.m_nbLevel, param.m_nbBranchPerJoint);
    }

    for(hkUint32 i=0, size=m_branches.getSize(); i<size; ++i)
    {
        m_hkWorld->addEntity(m_branches[i], HK_ENTITY_ACTIVATION_DO_ACTIVATE);
    }

    for(hkUint32 i=0, size=m_constraints.getSize(); i<size; ++i)
    {
        m_hkWorld->addConstraint(m_constraints[i]);
    }

    for(hkUint32 i=0, size=m_actions.getSize(); i<size; ++i)
    {
        m_hkWorld->addAction(m_actions[i]);
    }
}

hkpRigidBody* ProceduralTree::CreateRigid(const hkVector4& pos, const hkQuaternion& rot, 
	hkReal halfLength, hkReal radius, hkReal massDensity, hkBool fixed, int level)
{
    hkpRigidBodyCinfo info;
    hkReal mass = massDensity * halfLength*2.f * radius*radius;
    hkVector4 vertexA; vertexA.set(0.f, 0.f, -halfLength);
    hkVector4 vertexB; vertexB.set(0.f, 0.f, halfLength);
    hkpCapsuleShape* capShape = new hkpCapsuleShape( vertexA, vertexB, radius );
    info.m_position = pos;
    info.m_rotation = rot;
    info.m_shape = capShape;
    info.m_maxAngularVelocity = 4.f;
	// Disable all collisions between subsystems in the same level.
    info.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(BASE_LAYER_ID+level, level+1, 1, 1);
    hkpInertiaTensorComputer::setShapeVolumeMassProperties(capShape, mass, info);
	hkReal inertiaTensorElement = 10.0f*mass;
	info.m_inertiaTensor.setDiagonalMul( 
		hkVector4(inertiaTensorElement, inertiaTensorElement, inertiaTensorElement), 
		hkMatrix3::getIdentity() );
	if(fixed)
	{
		info.m_motionType = hkpMotion::MOTION_FIXED;
		info.m_qualityType = HK_COLLIDABLE_QUALITY_FIXED;
	}
	else
	{
		info.m_motionType = hkpMotion::MOTION_THIN_BOX_INERTIA;
		info.m_qualityType = HK_COLLIDABLE_QUALITY_MOVING;
	}
	hkpRigidBody* body = new hkpRigidBody(info);
    body->setGravityFactor(0.f);
    body->setUserData(reinterpret_cast<hkUlong>(this));
    capShape->removeReference();

    return body;
}

void ProceduralTree::CreateBranches(hkpRigidBody* parent, const hkVector4& parentPos, 
	const hkQuaternion& parentRot, const ProceduralTreeCinfo& param, hkUint32 nbLevel, hkUint32 nbBranchPerJoint)
{
    const hkReal levelFraction = ((hkReal)nbLevel/(hkReal)param.m_nbLevel);
    hkReal halfLength = param.m_sectionLength * 0.5f * levelFraction * param.m_branchLengthRatio;
    hkReal radius = param.m_sectionLength * 0.1f * levelFraction;
    for(hkUint32 i=0; i<nbBranchPerJoint; ++i)
    {
        hkQuaternion rot, deltaAngle;
        rot.setFromEulerAngles(0.f, 2.f * HK_REAL_PI * ((hkReal)i/(hkReal)(nbBranchPerJoint)), HK_REAL_PI * 0.25f);
        deltaAngle = rot;
        rot.setMul(parentRot, deltaAngle);
        hkVector4 pos;
        hkVector4 extremity(0.f, 0.f, halfLength + radius);
        pos.setRotatedDir(rot, extremity);
        pos.add(parentPos);

        hkpRigidBody* branch = CreateRigid(pos, rot, halfLength, radius, param.m_massDensity, false, nbLevel);
        m_branches.pushBack(branch);

        // Setup constraint
		hkpConstraintData* constraintData = NULL;
		{
  		    hkpBallAndSocketConstraintData* bs = new hkpBallAndSocketConstraintData;
		    bs->setInWorldSpace(parent->getTransform(), branch->getTransform(), parentPos);
            constraintData = bs;
        }
        hkpConstraintInstance* instance = new hkpConstraintInstance( parent, branch, constraintData);
        constraintData->removeReference();
        m_constraints.pushBack(instance);
        instance->setUserData(reinterpret_cast<hkUlong>(this));

        // Setup action
        hkpAngularDashpotAction* ba = new hkpAngularDashpotAction(parent, branch, (hkUlong)this);
        deltaAngle.setInverse(deltaAngle);
        ba->setRotation(deltaAngle);
        ba->setStrength(param.m_springStrength * branch->getMass());
        ba->setDamping(param.m_springDamping * branch->getMass());
        
        m_actions.pushBack(ba);

        // Build sub branches
        if(nbLevel > 1)
        {
            hkVector4 attachPos;
            attachPos.setRotatedDir(rot, hkVector4(0.f, 0.f, halfLength));
            attachPos.add(pos);
            CreateBranches(branch, attachPos, rot, param, nbLevel-1, nbBranchPerJoint);
        }
    }
}


static const char helpString[] =					\
"A demo which shows a procedural tree built using "	\
"capsule as rigid bodies and linked with "	\
"constraints and hkpAngularDashpotAction.";

HK_DECLARE_DEMO(ProceduralTreeDemo, HK_DEMO_TYPE_PHYSICS_2012, "Demonstrates an procedural tree of rigid bodies", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
