/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/UseCase/PhantomEyes/PhantomEyesDemo.h>
#include <Common/Base/Math/Matrix/hkMatrix3Util.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>

#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Common/Light/hkgLightManager.h>

#include <Physics2012/Collide/Shape/HeightField/Plane/hkpPlaneShape.h>
#include <Physics2012/Collide/Shape/Misc/PhantomCallback/hkpPhantomCallbackShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/ExtendedMeshShape/hkpExtendedMeshShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>
#include <Physics2012/Collide/Shape/Misc/Bv/hkpBvShape.h>

#include <Physics2012/Dynamics/Action/hkpUnaryAction.h>
#include <Physics2012/Dynamics/Phantom/hkpAabbPhantom.h>

#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>
#include <Physics2012/Utilities/VisualDebugger/hkpPhysicsContext.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpShapeDisplayViewer.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpPhantomDisplayViewer.h>

#define MAZE_ASSET "Resources/Physics2012/PhantomEyes/Maze.hkt"
#define ROBOT_ASSET "Resources/Physics2012/PhantomEyes/Robot.hkt"

enum 
{
	GROUND_LAYER = 1,
	WALL_LAYER,
	ROBOT_LAYER,
	PHANTOM_LAYER,
};

// Placed at exits. Will reposition the robot to the centre of the maze once it reaches the exit.
class MazeExitPhantom : public hkpAabbPhantom
{
public:
	HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO);

	MazeExitPhantom( const hkAabb& aabb, hkUint32 collisionFilterInfo = 0 )
		: hkpAabbPhantom( aabb, collisionFilterInfo )
	{
		m_randomizer = new hkPseudoRandomGenerator(12345);
	}

	~MazeExitPhantom()
	{
		delete m_randomizer;
	}

	// Callback implementation, resets the rigid body
	HK_FORCE_INLINE virtual void addOverlappingCollidable( hkpCollidable* handle )
	{
		hkpRigidBody* rb = hkpGetRigidBody( handle );
		if( rb != HK_NULL )
		{ 
			hkVector4 pos; pos.set( 0.0f, 0.0f, 0.0f );
			
			hkReal angle = hkReal(m_randomizer->getRand32()%360)*HK_REAL_PI / 180.0f;
			hkQuaternion orientation(hkVector4(0.f,1.f,0.f), angle);

			rb->setPositionAndRotationAsCriticalOperation(pos, orientation);
		}
	}

	// Callback implementation 
	virtual void removeOverlappingCollidable( hkpCollidable* handle ) {}

	hkPseudoRandomGenerator* m_randomizer;
};

/// Create a table with 4 boundary boxes.
/// A small moving box gets pushed around on the table, see 
/// MyPhantomAction for details
PhantomEyesDemo::PhantomEyesDemo(hkDemoEnvironment* env):hkDefaultPhysics2012Demo(env)
{
	//
	// Setup the camera
	//
	{
		hkVector4 from(0.f, 30.f, 21.f);
		hkVector4 to(0.f, 1.2f, 0.f);
		hkVector4 up  (0.0f,  1.0f,  0.0f);
		setupDefaultCameras(env, from, to, up );

		m_followCam = true;
	}

	//Lights and shadows
	{
		hkVector4 mainLightDir(1.f, -1.f, 1.f);
		hkVector4 fillDir(-1.f, 1.f, -1.f);
		hkColor::Argb keyColor = hkColor::rgbFromFloats(1.0f, 1.0f, 1.0f);
		hkColor::Argb fillColor = hkColor::rgbFromFloats(0.95f, 0.95f, 0.95f);
		hkColor::Argb rimColor = hkColor::rgbFromFloats(0.9f, 0.9f, 0.95f);
		setThreeStageLights( env, &mainLightDir(0), &fillDir(0), keyColor, fillColor, rimColor );

		hkgAabb areaAabb;
		hkgVec3Set(areaAabb.m_min, -20.f, -5, -20.f);
		hkgVec3Set(areaAabb.m_max, 20.f, 5, 20.f);
		int numSplits = 0;
		m_env->m_window->setShadowMapSize(512); // a nice hi res single one, as we only have the one char etc
		m_env->m_window->getContext()->setShadowVsmBias(1.0e-5f, 1.0e-5f);
		setupFixedShadowFrustum( m_env, *m_env->m_displayWorld->getLightManager()->getBestShadowCaster(), areaAabb, 0, 0, numSplits, /*y up*/ 1 );

		m_env->m_options->m_enableShadows = true;
		m_env->m_sceneConverter->setShaderLibraryEnabled(true);
	}

	//
	// Create world
	//
	{
		hkpWorldCinfo info;
		info.m_gravity.set(0.0f, -9.8f, 0.0f);
		info.setBroadPhaseWorldSize(100.0f);
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM);
		info.m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_DISCRETE;
		m_world = new hkpWorld( info );
		m_world->lock();

		setupGraphics();
	}

	// Register ALL agents (though some may not be necessary)
	hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

	hkpGroupFilter* filter = new hkpGroupFilter();
	filter->disableCollisionsBetween( GROUND_LAYER, WALL_LAYER);
	filter->disableCollisionsBetween( GROUND_LAYER, PHANTOM_LAYER);
	filter->disableCollisionsBetween( WALL_LAYER, PHANTOM_LAYER);
	filter->enableCollisionsBetween( ROBOT_LAYER, WALL_LAYER);
	filter->enableCollisionsBetween( ROBOT_LAYER, GROUND_LAYER);
	m_world->setCollisionFilter( filter );
	filter->removeReference();

	{
		m_env->m_sceneConverter->clearTextureSearchOrder();
		m_env->m_sceneConverter->addTextureSearchOrder("dds"); 
		m_env->m_sceneConverter->addTextureSearchOrder("png");
		/*m_env->m_sceneConverter->addTextureSearchOrder("tga");*/

		m_env->m_sceneConverter->clearTextureSearchPaths();
		m_env->m_sceneConverter->addTextureSearchPath("Resources/Physics2012/PhantomEyes");
	}

	m_loader = new hkLoader;

	loadMaze();
	loadRobot();

	// Add phantoms to see if the robot has solved the maze
	{
		// Bottom right phantom
		{
			hkAabb bottomExit;
			bottomExit.m_min = hkVector4(-22.f,-1.f,10.f);
			bottomExit.m_max = hkVector4(22.f,2.f,20.f);
			
			MazeExitPhantom* bottomPhantom = new MazeExitPhantom(bottomExit, hkpGroupFilter::calcFilterInfo (PHANTOM_LAYER) );
			m_world->addPhantom( bottomPhantom );
			bottomPhantom->removeReference();
		}
		// Top left phantom
		{
			hkAabb topExit;
			topExit.m_min = hkVector4(-22.f,-1.f,-20.f);
			topExit.m_max = hkVector4(22.f,2.f,-10.f);

			MazeExitPhantom* topPhantom = new MazeExitPhantom(topExit, hkpGroupFilter::calcFilterInfo (PHANTOM_LAYER) );
			m_world->addPhantom( topPhantom );
			topPhantom->removeReference();
		}
	}
	
	m_world->unlock();
}

PhantomEyesDemo::~PhantomEyesDemo()
{
	if (m_world)
	{
		m_world->markForWrite();
		m_world->removeReference();
		m_world = HK_NULL;
	}

	delete m_loader;
}

void PhantomEyesDemo::setupContexts(hkArray<hkProcessContext*>& contexts)
{
	if ( m_world )
	{
		m_world->markForWrite();
		if ( (m_physicsViewersContext->findWorld(m_world) < 0) )
		{
			m_physicsViewersContext->addWorld(m_world);
		}
		m_world->unmarkForWrite();
	}
	contexts.pushBack( m_physicsViewersContext );
	
	// 
	/*{
		m_debugViewerNames.pushBack(hkpPhantomDisplayViewer::getName());
		m_debugViewerNames.pushBack(hkpShapeDisplayViewer::getName());
	}*/
}

void PhantomEyesDemo::loadMaze()
{
	hkBool enableReflection = true;

	hkStringBuf fileName(HK_GET_DEMOS_ASSET_FILENAME(MAZE_ASSET));
	hkRootLevelContainer* container = m_loader->load( fileName.cString() );
	HK_ASSERT2(0x0, container != HK_NULL, "Could not load root level object" );

	hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
	HK_ASSERT2(0x0, scene, "No Scene Data found");

	hkArray<hkpRigidBody* > environmentRigidBodies;
	// Get the physics data for the landscape
	{
		hkpPhysicsData* physicsData = static_cast<hkpPhysicsData*>( container->findObjectByType( hkpPhysicsDataClass.getName() ) );
		HK_ASSERT2(0x0, physicsData != HK_NULL, "Could not find physics data in root level object" );
		HK_ASSERT2(0x0, physicsData->getWorldCinfo() != HK_NULL, "No physics cinfo in loaded file - cannot create a hkpWorld" );

		for ( int i = 0; i < physicsData->getPhysicsSystems().getSize(); ++i )
		{
			hkpPhysicsSystem* physicsSystem = physicsData->getPhysicsSystems()[i];
			const hkArray<hkpRigidBody *>& bodies = physicsSystem->getRigidBodies();

			for (int rb = 0; rb < bodies.getSize();rb++)
			{
				hkpRigidBody* rigidBody = bodies[rb];
				hkpRigidBodyCinfo groundInfo; rigidBody->getCinfo(groundInfo);

				if(rigidBody->isFixed())
				{
					if(hkString::beginsWith(rigidBody->getName(), "floor"))
					{
						//Make a new RB with a plane shape
						hkpRigidBody* groundRB = HK_NULL;
						hkpPlaneShape* plane = HK_NULL;
						{
							//Create a plane
							{
								hkVector4 dir( 0,1,0);
								hkVector4 point( 0,0,0);
								
								hkVector4 halfExtents; 
								{
									hkAabb planeAabb; rigidBody->getCollidable()->getShape()->getAabb(rigidBody->getTransform(), 0.0f, planeAabb);
									halfExtents.setSub4(planeAabb.m_max, planeAabb.m_min);
									halfExtents.mul4(0.5f);
								}

								hkVector4 extents( halfExtents(0), 5.f, halfExtents(2));
								plane = new hkpPlaneShape(dir, point, extents);
							}

							groundInfo.m_shape = plane;
							groundInfo.m_motionType = hkpMotion::MOTION_FIXED;
							groundInfo.m_friction = 0.5f;
							groundInfo.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo( GROUND_LAYER, 0 );

							groundRB = new hkpRigidBody(groundInfo);
							groundRB->setName(rigidBody->getName());
						}

						m_world->addEntity(groundRB);
						groundRB->removeReference();
						plane->removeReference();
						environmentRigidBodies.pushBack(groundRB);

					}
					else
					{
						rigidBody->setCollisionFilterInfo(hkpGroupFilter::calcFilterInfo( WALL_LAYER, 0 ));
						m_world->addEntity(rigidBody);
						environmentRigidBodies.pushBack(rigidBody);
					}
				}
			}
		}
	}

	// Load the landscape display object and rigid bodies, turn on reflections
	{
		hkgDefaultSceneDataFilter filter;
		for(int rb=0; rb<environmentRigidBodies.getSize(); ++rb)
		{
			filter.allow(environmentRigidBodies[rb]->getName());
		}

		m_env->m_sceneConverter->convert(scene, hkgAssetConverter::CONVERT_ALL, true, &filter);

		for(int rb=0; rb<environmentRigidBodies.getSize(); ++rb)
		{
			hkpRigidBody* rigidBody = environmentRigidBodies[rb];

			int idx = m_env->m_displayWorld->findDisplayObject(rigidBody->getName());
			if(idx >= 0)
			{
				hkgDisplayObject* groundDisplayObject = m_env->m_displayWorld->getDisplayObject(idx);
				groundDisplayObject->computeAABB();

				if(hkString::beginsWith(rigidBody->getName(), "floor"))
				{
					float reflectionPlane[] = { 0,1,0, 0.0f };
					m_env->m_displayWorld->setReflections(true, reflectionPlane);
					groundDisplayObject->setStatusFlags( (groundDisplayObject->getStatusFlags() | HKG_DISPLAY_OBJECT_REFLECTOR | HKG_DISPLAY_OBJECT_SHADOWRECEIVER) & ~HKG_DISPLAY_OBJECT_SHADOWCASTER  );
				}
			}
		}
	}
}

void PhantomEyesDemo::loadRobot()
{
	hkStringBuf fileName(HK_GET_DEMOS_ASSET_FILENAME(ROBOT_ASSET));
	hkRootLevelContainer* container = m_loader->load( fileName.cString() );
	HK_ASSERT2(0x0, container != HK_NULL, "Could not load root level object" );

	hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
	HK_ASSERT2(0x0, scene, "No Scene Data found");

	hkpRigidBody* robotBodyRB = HK_NULL;
	hkpRigidBody* robotLeftEyesRB = HK_NULL;
	hkpRigidBody* robotRightEyesRB = HK_NULL;
	hkpRigidBody* robotFrontEyesRB = HK_NULL;
	// Get the physics data for the robot
	{
		hkpPhysicsData* physicsData = static_cast<hkpPhysicsData*>( container->findObjectByType( hkpPhysicsDataClass.getName() ) );
		HK_ASSERT2(0x0, physicsData != HK_NULL, "Could not find physics data in root level object" );
		HK_ASSERT2(0x0, physicsData->getWorldCinfo() != HK_NULL, "No physics cinfo in loaded file - cannot create a hkpWorld" );

		for ( int i = 0; i < physicsData->getPhysicsSystems().getSize(); ++i )
		{
			hkpPhysicsSystem* physicsSystem = physicsData->getPhysicsSystems()[i];
			const hkArray<hkpRigidBody *>& bodies = physicsSystem->getRigidBodies();
				
			for (int rb = 0; rb < bodies.getSize();rb++)
			{
				hkpRigidBody* rigidBody = bodies[rb];
				
				if(hkString::beginsWith(rigidBody->getName(), "laserbotBody"))
				{
					robotBodyRB = rigidBody;
				}
				else if(hkString::beginsWith(rigidBody->getName(), "laserbotLeftEyes"))
				{
					robotLeftEyesRB = rigidBody;
				}
				else if(hkString::beginsWith(rigidBody->getName(), "laserbotRightEyes"))
				{
					robotRightEyesRB = rigidBody;
				}
				else if(hkString::beginsWith(rigidBody->getName(), "laserbotFrontEyes"))
				{
					robotFrontEyesRB = rigidBody;
				}
			}
		}
	}

	//
	hkpListShape* listShape  = HK_NULL;
	MyPhantomCallbackShape* leftEyeShape;
	MyPhantomCallbackShape* rightEyeShape;
	MyPhantomCallbackShape* frontEyeShape;
	{
		//hkpShape* shapes[3];
		hkpShape* shapes[4];

		hkArray<const hkpShape* > triangleShapes; triangleShapes.setSize(2, HK_NULL);
		
		const hkpShape* rbShape = robotBodyRB->getCollidable()->getShape();
		shapes[0] = (hkpShape*)rbShape;
		
		const hkpShape* rbLeftEyesShape = robotLeftEyesRB->getCollidable()->getShape();
		
		// create the left eye
		{
			leftEyeShape = new MyPhantomCallbackShape();
			shapes[1] = new hkpBvShape( rbLeftEyesShape, leftEyeShape );

			leftEyeShape->removeReference();
			rbLeftEyesShape->removeReference();
		}

		const hkpShape* rbRightEyesShape = robotRightEyesRB->getCollidable()->getShape();
		{
			rightEyeShape = new MyPhantomCallbackShape();
			shapes[2] = new hkpBvShape( rbRightEyesShape, rightEyeShape );

			rightEyeShape->removeReference();
			rbRightEyesShape->removeReference();
		}

		const hkpShape* rbFrontEyesShape = robotFrontEyesRB->getCollidable()->getShape();
		{
			frontEyeShape = new MyPhantomCallbackShape();
			shapes[3] = new hkpBvShape( rbFrontEyesShape, frontEyeShape );

			frontEyeShape->removeReference();
			rbFrontEyesShape->removeReference();
		}

		listShape = new hkpListShape( shapes, 4 );
		shapes[0]->removeReference();
		shapes[1]->removeReference();
		shapes[2]->removeReference();
		shapes[3]->removeReference();
	}
	
	// 
	// Create a compound body from this list
	//
	{

		hkpRigidBodyCinfo boxInfo;
		boxInfo.m_shape = listShape;
		boxInfo.m_position.set( 0.0f, 0.0f, 0.0f );
		hkMatrix3Util::_setDiagonal( 25.f, 0.1f, 25.0f, boxInfo.m_inertiaTensor );
		//boxInfo.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;
		boxInfo.m_motionType = hkpMotion::MOTION_BOX_INERTIA;
		boxInfo.m_mass = 1.0f;

		boxInfo.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo( ROBOT_LAYER, 0 );
		m_robotRB = new hkpRigidBody(boxInfo);
		m_robotRB->setName(robotBodyRB->getName());
		
		hkVector4 angVel( 0.0f, 1.0f, 0.0f );
		m_robotRB->setAngularVelocity( angVel );

		m_world->addEntity( m_robotRB );
		m_robotRB->removeReference();

		listShape->removeReference();
	}

	//
	//	Create our special Phantom Action
	//
	{
		MyPhantomAction* cle = new MyPhantomAction(m_robotRB, leftEyeShape, rightEyeShape, frontEyeShape);
		m_world->addAction( cle );
		cle->removeReference();
	}

	// Load the landscape display object and rigid bodies, turn on reflections
	{
		hkgDefaultSceneDataFilter filter;
		filter.allow(m_robotRB->getName());
		
		m_env->m_sceneConverter->convert(scene, hkgAssetConverter::CONVERT_ALL, true, &filter);
		int idx = m_env->m_displayWorld->findDisplayObject(m_robotRB->getName());
		if(idx >= 0)
		{
			m_robotDisplayObject = m_env->m_displayWorld->getDisplayObject(idx);
		}

	}
}

MyPhantomAction::MyPhantomAction( hkpRigidBody* body, MyPhantomCallbackShape* leftEye, MyPhantomCallbackShape* rightEye, MyPhantomCallbackShape* frontEye ) : hkpUnaryAction(body)
{
	m_leftEye = leftEye;
	m_rightEye = rightEye;
	m_frontEye = frontEye;
	m_leftActive  = 0;
	m_rightActive = 0;
	m_frontActive = 0;
}




/// Just push the box forward, except when some eye sees (penetrates) something
/// If only onew does, turn in the opposite direction.
/// If both do, reverse for a while.
/// => Very basic AI.
void MyPhantomAction::applyAction( const hkStepInfo& stepInfo )
{
	hkpRigidBody* body = static_cast<hkpRigidBody*>( m_entity );

	const int phantomHitDuration = 10;

	if ( m_frontEye->m_penetratingShapesCount )
	{
		m_frontActive = phantomHitDuration;
	}

	if ( m_leftEye->m_penetratingShapesCount )
	{
		m_leftActive = phantomHitDuration;
	}

	if ( m_rightEye->m_penetratingShapesCount )
	{
		m_rightActive = phantomHitDuration;
	}

	hkVector4 forwardLocal( 0.0f, 0.0f, 6.f ); 
	hkVector4 rotateLocal( 0.0f, 0.0f, 0.0f ); //Negative Y-axis is rotate right and vice versa
	
	// Front eyes has just collided
	if(m_frontActive == phantomHitDuration)
	{
		forwardLocal.setZero4();
		
		if(m_rightActive == phantomHitDuration && m_leftActive <= 0)
		{
			rotateLocal.set(0.0f, 2.5f, 0.0f);
		}	
		else if(m_leftActive == phantomHitDuration && m_rightActive <= 0)
		{
			rotateLocal.set(0.0f, -2.5f, 0.0f);
		}
		else // everywhere is blocked so rotate around without translating
		{
			rotateLocal.set(0.0f, -2.5f, 0.0f);
		}
	}
	else
	{
		forwardLocal.set( 0.0f, 0.0f, 6.f );
		
		// Try to stick to the wall
		if(m_rightActive && m_rightActive < phantomHitDuration)
		{
			forwardLocal.setNeg3(forwardLocal);
			rotateLocal.set(0.0f, -7.5f, 0.0f);
		}
		else if(m_leftActive && m_leftActive < phantomHitDuration)
		{
			forwardLocal.setNeg3(forwardLocal);
			rotateLocal.set(0.0f, 7.5f, 0.0f);
		}
	}

	hkVector4 linVel = body->getLinearVelocity();
	hkReal linVelMag = linVel.lengthSquared3();
	if(linVelMag > 4.f)
	{
		forwardLocal.setZero4();
	}

	hkVector4 angVel = body->getAngularVelocity();
	hkReal angVelMag = angVel.lengthSquared3();

	if(angVelMag > 270.f*HK_REAL_PI / 180.0f)
	{
		rotateLocal.setZero4();
	}
	
	if ( m_leftActive ) m_leftActive--;
	if ( m_frontActive ) m_frontActive--;
	if ( m_rightActive ) m_rightActive--;
	
	hkVector4 forwardWorld; 
	forwardWorld.setRotatedDir( body->getTransform().getRotation(), forwardLocal );
	hkVector4 rotateWorld; 
	rotateWorld.setRotatedDir( body->getTransform().getRotation(), rotateLocal );
	body->applyForce( stepInfo.m_deltaTime, forwardWorld );
	body->applyTorque( stepInfo.m_deltaTime, rotateWorld );
}


MyPhantomCallbackShape:: MyPhantomCallbackShape() : hkpPhantomCallbackShape()
{
	m_penetratingShapesCount = 0;
}

void MyPhantomCallbackShape::phantomEnterEvent( const hkpCollidable* phantomColl, const hkpCollidable* otherColl, const hkpCollisionInput& env )
{
	m_penetratingShapesCount++;
	// we should really use a thread safe implementation below
	//hkInterlockedIncrement( m_penetratingShapesCount );
}

void MyPhantomCallbackShape::phantomLeaveEvent( const hkpCollidable* phantomColl, const hkpCollidable* otherColl )
{
	m_penetratingShapesCount--;
	//hkInterlockedDecrement( m_penetratingShapesCount );
}

hkDemo::Result PhantomEyesDemo::stepDemo()
{
	//
	// Step the world.
	//
	{
		hkDefaultPhysics2012Demo::stepDemo();
	}

	hkTransform robotXForm = m_robotRB->getTransform();
	m_robotDisplayObject->setTransformAligned(robotXForm);

	if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_1 ) || m_env->m_window->getKeyboard().wasKeyPressed( 'C' ))
	{
		hkgCamera* cam = m_env->m_window->getCurrentViewport()->getCamera();

		m_followCam = !m_followCam;

		if(!m_followCam)
		{
			hkVector4 desiredFrom, desiredTo, desiredUp;
			desiredFrom.set(0.f, 30.f, 21.f);
			desiredTo.set(0.f, 1.2f, 0.f);
			desiredUp.set(0.0f,  1.0f,  0.0f);

			cam->setTo((hkReal*) &desiredTo);
			cam->setFrom((hkReal*) &desiredFrom);
			cam->setUp((hkReal*) &desiredUp);

			cam->orthogonalize();
			cam->computeModelView();
			cam->computeProjection();
		}
	}

	if(m_followCam)
	{
		hkgCamera* cam = m_env->m_window->getCurrentViewport()->getCamera();

		hkVector4 desiredFrom, desiredTo, desiredUp;

		hkVector4 pos = m_robotRB->getPosition();
		desiredTo = pos;
		desiredFrom = pos;
		desiredFrom(2) += 5.f;
		desiredFrom(1) = 6.f;
		desiredUp.set(0.f,1.f,0.f,1.f);

		cam->setTo((hkReal*) &desiredTo);
		cam->setFrom((hkReal*) &desiredFrom);
		cam->setUp((hkReal*) &desiredUp);

		cam->orthogonalize();
		cam->computeModelView();
		cam->computeProjection();
	}

	return DEMO_OK;
}

//////////////////////////////////////////////


static const char helpString[] = \
"A demo which shows use of phantoms as eyes for an AI controlled character. " \
"Usually we think of phantoms and entities as distinct. In this demo " \
"the phantom triangles are part of the shape hierarchy for the bot. When " \
"the bot moves the phantom automatically moves with it.";

HK_DECLARE_DEMO(PhantomEyesDemo, HK_DEMO_TYPE_PHYSICS_2012, "Phantom Demo", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
