/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/UseCase/Creatures/CreaturesDemo.h>

#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>

#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>

#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics/Constraint/Data/Hinge/hkpHingeConstraintData.h>
#include <Physics/Constraint/Data/LimitedHinge/hkpLimitedHingeConstraintData.h>
#include <Physics2012/Dynamics/Constraint/Chain/Powered/hkpPoweredChainData.h>
#include <Physics2012/Dynamics/Constraint/Chain/hkpConstraintChainInstance.h>
#include <Physics/Constraint/Motor/Position/hkpPositionConstraintMotor.h>
#include <Physics2012/Dynamics/Phantom/hkpAabbPhantom.h>
#include <Physics2012/Dynamics/World/hkpSimulationIsland.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>

// Which creatures are we steering??
#define STEERING_SNAKE 0
#define STEERING_SPIDER 1

// snake has 3 speeds
#define SNAKE_FAST_PERIOD 0.75f
#define SNAKE_NORMAL_PERIOD 1.5f
#define SNAKE_SLOW_PERIOD 3.0f

#define NUM_SPIDERS 4
#define NUM_SNAKES 10

const hkReal GROUND_SIZE = 100.f;
const hkReal WALL_HEIGHT = 5.0f;
const hkReal MAX_CREATURE_VELOCITY = 1.5f;

const hkVector4 snakeSegmentHalfSize( 0.5f, 0.25f, 0.25f );

#define GROUND_ASSET "Resources/Physics2012/Creatures/CreatureFloor_100.hkt"

CreaturesDemo::CreaturesDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env, DEMO_FLAGS_NO_SERIALIZE), m_rand(9261), m_step(0), m_cameraFollowingSteered(true)
{
	m_loader = HK_NULL;
	
	//
	// Setup the camera
	//
	{
		hkVector4 from(42.0f, 15.0f, -3.0f);
		hkVector4 to  (-60.0f, -35.0f, 0.0f);
		hkVector4 up  (0.0f, 1.0f, 0.0f);
		setupDefaultCameras( env, from, to, up, .5f, 300.0f );
	}
	//Lights and shadows
	{
		//HK_DISPLAY_ARROW
		hkVector4 mainLightDir(1.f, -1.f, 1.f);
		setSoleDirectionLight(m_env, mainLightDir, 0xffffffff );

 		forceShadowState(false);
		m_env->m_sceneConverter->setShaderLibraryEnabled(true);
	}


	//
	// Create the world
	//
	hkpGroupFilter* filter;
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_8ITERS_MEDIUM); 
		info.setBroadPhaseWorldSize( 1000.0f );
		info.m_gravity.set(0.0f, -9.81f, 0.0f);

		m_world = new hkpWorld( info );
		m_world->lock();

		m_world->m_wantDeactivation = false;

		filter = new hkpGroupFilter();
		
		filter->disableCollisionsBetween(LAYER_GROUND, LAYER_PHANTOM);
		filter->enableCollisionsBetween(LAYER_GROUND, LAYER_SNAKE); 

		for(int i=0;i<NUM_SNAKES;i++)
		{
			filter->disableCollisionsBetween(LAYER_SNAKE + i, LAYER_SNAKE + i);
		}

		m_world->setCollisionFilter( filter );
		filter->removeReference();

		//
		// Create constraint viewer
		//
		setupGraphics();
	}

	hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );

	// floor
	{
		{
			hkVector4 size(GROUND_SIZE, 1.0f, GROUND_SIZE);
			hkVector4 position(0.0f, 0.0f, 0.0f);
			hkReal mass = 0.0f;
			hkpRigidBody* ground = GameUtils::createBox( size, mass, position );

			if(ground->hasProperty(HK_PROPERTY_DISPLAY_PTR))
			{
				ground->editProperty(HK_PROPERTY_DISPLAY_PTR, HK_NULL);
			}
			else
			{
				ground->addProperty(HK_PROPERTY_DISPLAY_PTR, HK_NULL);
			}

			m_world->addEntity(ground);
			ground->removeReference();
			ground->getCollidableRw()->setCollisionFilterInfo( hkpGroupFilter::calcFilterInfo(LAYER_GROUND) );
		}

		hkVector4 wallSize;
		hkVector4 wallPosition;

		wallSize.set(GROUND_SIZE, WALL_HEIGHT, 1.0f);
		wallPosition.set(0.0f, WALL_HEIGHT / 2.0f, GROUND_SIZE / 2.0f - 0.5f);
		createWall(wallSize, wallPosition);
		wallSize.set(GROUND_SIZE, WALL_HEIGHT, 1.0f);
		wallPosition.set(0.0f, WALL_HEIGHT / 2.0f, -(GROUND_SIZE / 2.0f - 0.5f));
		createWall(wallSize, wallPosition);
		wallSize.set(1.0f, WALL_HEIGHT, GROUND_SIZE);
		wallPosition.set(GROUND_SIZE / 2.0f - 0.5f, WALL_HEIGHT / 2.0f, 0.0f);
		createWall(wallSize, wallPosition);
		wallSize.set(1.0f, WALL_HEIGHT, GROUND_SIZE);
		wallPosition.set(-(GROUND_SIZE / 2.0f - 0.5f), WALL_HEIGHT / 2.0f, 0.0f);
		createWall(wallSize, wallPosition);

		// Invisible roof
		{
			hkVector4 size(GROUND_SIZE, 1.0f, GROUND_SIZE);
			hkVector4 position(0.0f, 5.5f, 0.0f);
			hkReal mass = 0.0f;
			hkpRigidBody* ground = GameUtils::createBox( size, mass, position );

			if(ground->hasProperty(HK_PROPERTY_DISPLAY_PTR))
			{
				ground->editProperty(HK_PROPERTY_DISPLAY_PTR, HK_NULL);
			}
			else
			{
				ground->addProperty(HK_PROPERTY_DISPLAY_PTR, HK_NULL);
			}

			m_world->addEntity(ground);
			ground->removeReference();
			ground->getCollidableRw()->setCollisionFilterInfo( hkpGroupFilter::calcFilterInfo(LAYER_GROUND) );
		}

		// Load the display object
		{
			m_env->m_sceneConverter->clearTextureSearchOrder();
			m_env->m_sceneConverter->addTextureSearchOrder("dds"); 
			m_env->m_sceneConverter->addTextureSearchOrder("png");

			m_env->m_sceneConverter->clearTextureSearchPaths();
			m_env->m_sceneConverter->addTextureSearchPath("./Resources/Physics2012/Creatures/");

			m_loader = new hkLoader;
			hkStringBuf fileName(HK_GET_DEMOS_ASSET_FILENAME(GROUND_ASSET));
			hkRootLevelContainer* container = m_loader->load( fileName.cString() );
			HK_ASSERT2(0x0, container != HK_NULL, "Could not load root level object" );

			hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
			HK_ASSERT2(0x0, scene, "No Scene Data found");

			m_env->m_sceneConverter->convert(scene, hkgAssetConverter::CONVERT_ALL, true);

			int idx = m_env->m_displayWorld->findDisplayObject("floor");
			if(idx >= 0)
			{
				hkgDisplayObject* groundDisplayObject = m_env->m_displayWorld->getDisplayObject(idx);
				groundDisplayObject->computeAABB();

				float reflectionPlane[] = { 0,1,0, 0.5f };
				m_env->m_displayWorld->setReflections(true, reflectionPlane);
				groundDisplayObject->setStatusFlags( (groundDisplayObject->getStatusFlags() | HKG_DISPLAY_OBJECT_REFLECTOR | HKG_DISPLAY_OBJECT_SHADOWRECEIVER) & ~HKG_DISPLAY_OBJECT_SHADOWCASTER  );
			}
		}
	}

	{
		// Share these so that we can use instancing
		m_snakeBox = new hkpBoxShape(snakeSegmentHalfSize);
		m_snakeSphere = new hkpSphereShape(snakeSegmentHalfSize(1) * 0.75f);
	}

	m_leggedCreatureData.setSize(NUM_SPIDERS);
	// Create the Spiders
	for(int i=0;i<NUM_SPIDERS;i++)
	{
		m_leggedCreatureData[i].m_layout.m_numLegs = ((m_rand.getRand32() % 2) + 2) * 2;
		m_leggedCreatureData[i].m_layout.m_bodyHalfExtents.set( 0.8f + (m_leggedCreatureData[i].m_layout.m_numLegs - 4) * 0.4f, 0.2f, 0.3f + (m_leggedCreatureData[i].m_layout.m_numLegs - 4) * 0.1f );
		m_leggedCreatureData[i].m_layout.m_legHalfExtents.set( 2.2f + (m_leggedCreatureData[i].m_layout.m_numLegs - 4) * 0.25f, 1.0f, 2.0f );
		m_leggedCreatureData[i].m_creatureType = SPIDER;
		m_leggedCreatureData[i].m_time = 0.0f;
		// If we are steering this spider, start it off slowly so that it looks different
		m_leggedCreatureData[i].m_velocity = (STEERING_SPIDER && (i == 0)) ? 0.1f : 1.0f;
		m_leggedCreatureData[i].m_leftFactor = 1.0f;
		m_leggedCreatureData[i].m_rightFactor = 1.0f;
		m_leggedCreatureData[i].m_layout.m_filterInfo = hkpGroupFilter::calcFilterInfo( 0, filter->getNewSystemGroup() );
		hkVector4 pos( i * 6.0f  - 10 , 2.0f, i  * 12.0f  );
		createLeggedCreature( m_world, m_leggedCreatureData[i], &pos);
	}

	// Create the Snakes
	m_snakeData.setSize(NUM_SNAKES);
	for(int i=0;i<NUM_SNAKES;i++)
	{
		hkVector4 pos( (i - NUM_SNAKES/2) * 4.0f + m_rand.getRandRange(-1.0f, 1.0f), 1.0f, (-i - 1) * 4.0f + m_rand.getRandRange(-2.5f, 2.5f) );

		// if we are steering this snake, make it bigger
		createSnake(i, &pos, STEERING_SNAKE && (i == 0));
	}
	
	m_world->unlock();
}

void CreaturesDemo::createWall(hkVector4& sideWallSize, hkVector4& sideWallPosition)
{
	hkReal sideWallMass = 0.0f;
	hkpRigidBody* wall = GameUtils::createBox( sideWallSize, sideWallMass, sideWallPosition );
	m_world->addEntity(wall);
	wall->removeReference();
	wall->getCollidableRw()->setCollisionFilterInfo( hkpGroupFilter::calcFilterInfo(LAYER_GROUND) );
	{
		hkgDisplayObject* displayObject =  m_env->m_displayHandler->findDisplayObject(hkUlong(wall->getCollidable()));//m_env->m_displayWorld->getDisplayObject(idx);
		if(displayObject)
		{
			displayObject->setStatusFlags( (displayObject->getStatusFlags()) & ~(HKG_DISPLAY_OBJECT_SHADOWRECEIVER | HKG_DISPLAY_OBJECT_SHADOWCASTER) );
		}
	}
}

void CreaturesDemo::createSnake(int index, hkVector4* createPosition, bool isBigSnake)
{
	{
		hkpRigidBody* front = HK_NULL;
		hkpRigidBody* back = HK_NULL;

		int numSegments = (isBigSnake ? 32 : 16);
		hkVector4 segmentHalfSize( 0.5f, 0.25f, 0.25f );
		hkVector4 segmentPosition;
		if(createPosition)
		{
			segmentPosition = *createPosition;
		}
		else
		{
			segmentPosition.set( numSegments * segmentHalfSize(0), 1.0f, 0.0f );
		}
		hkVector4 axis( 0.0f, 1.0f, 0.0f );

		// link together segments with powered hinges
		int iSegment;
		for( iSegment = 0; iSegment < numSegments; iSegment++ )
		{
			//
			// We create a new segment here. A body is added, and collision agents are created.
			// This function also adds constraints linking to other segments, which triggers
			// an extra verification of the collision filter (triggered via its constraint listener
			// functionality).
			//
			front = createSegment( segmentPosition, segmentHalfSize, index );

			if( back )
			{
				hkVector4 linkPosition = segmentPosition;
				//linkPosition(2) -= segmentHalfSize(2);
				linkPosition(0) += segmentHalfSize(0);
				linkSegment( front, back, linkPosition, axis, m_snakeData[index] );
			}

			//segmentPosition(2) += segmentHalfSize(2)*2.0f;
			segmentPosition(0) -= segmentHalfSize(0)*2.0f;
			back = front;

		}

		//m_segmentLength = segmentHalfSize(2)*2.0f;
		m_snakeData[index].m_segmentLength = segmentHalfSize(0)*2.0f;

		// wave front keeps track of the translation of the wave, increasing over time.
		m_snakeData[index].m_waveFront = 0.0f;
		// our steering value
		m_snakeData[index].m_steering = 0.0f;
		m_snakeData[index].m_creatureType = SNAKE;
		m_snakeData[index].m_waveLength = 4.f;
		m_snakeData[index].m_wavePeriod= 1.5f;
		m_snakeData[index].m_waveAmplitude = 1.f;
	}
}


void CreaturesDemo::doLegIk( const CreaturesDemo::calcLegMatrizesIn& in, hkVector4& pivotOut, hkRotation& mAOut, hkRotation& mBOut )
{
	hkVector4 dir; dir.setSub4( in.m_to, in.m_from );
	hkReal tdist = dir.length3();
	if ( tdist >= in.m_lenA + in.m_lenB )
	{
		hkVector4Util::buildOrthonormal( dir, in.m_up, mAOut );
		mBOut = mAOut;
		dir.normalize3();
		pivotOut.setAddMul4( in.m_from, dir, in.m_lenA );
		return;
	}
	dir.normalize3();
	hkReal la2 = in.m_lenA*in.m_lenA;
	hkReal lb2 = in.m_lenB*in.m_lenB;
	hkReal adist = 0.5f * (la2- lb2 + tdist*tdist) / tdist;
	hkReal hdist = hkMath::sqrt( la2 - adist*adist );

	hkVector4 cup;
	{
		hkVector4 h; h.setCross( dir, in.m_up );
		cup.setCross( h, dir );
		cup.normalize3();
	}

	{
		pivotOut.setAddMul4( in.m_from, dir, adist );
		pivotOut.addMul4( hdist, cup );
	}

	{
		hkVector4 d; d.setSub4( pivotOut, in.m_from );
		d.normalize3();
		hkVector4Util::buildOrthonormal( d, in.m_up, mAOut );
	}
	{
		hkVector4 d; d.setSub4( in.m_to, pivotOut );
		d.normalize3();
		hkVector4Util::buildOrthonormal( d, in.m_up, mBOut );
	}
}

static hkpRigidBody* createCapsule( hkRotation& rot, const hkVector4& aWs, const hkVector4& bWs, int filterInfo, hkReal radius )
{
	hkVector4 center; center.setInterpolate4( aWs, bWs, 0.5f );
	hkVector4 a; a.setSub4( aWs, center );
	hkVector4 b; b.setSub4( bWs, center );

	a.setRotatedInverseDir( rot, a );
	b.setRotatedInverseDir( rot, b );

	hkpRigidBodyCinfo ci;
	ci.m_shape = new hkpCapsuleShape( a, b, radius );
	ci.m_friction = 2.0f;
	ci.m_restitution = 0.0f;
	ci.m_collisionFilterInfo = filterInfo;
	ci.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;
	ci.m_position = center;
	ci.m_rotation.set( rot );

	hkpInertiaTensorComputer::setShapeVolumeMassProperties( ci.m_shape, 1.0f, ci );

	hkpRigidBody* body = new hkpRigidBody( ci );

	ci.m_shape->removeReference();

	return body;
}


void CreaturesDemo::buildLeg( hkpWorld* world, hkpRigidBody* rootBody, const CreaturesDemo::calcLegMatrizesIn& in, int filterInfo,  Leg& legOut )
{
	hkVector4 pivot;
	hkRotation mA;
	hkRotation mB;
	doLegIk( in, pivot, mA, mB );

	const hkReal radius = .1f;

	hkpRigidBody* bodyA = createCapsule( mA, in.m_from, pivot, filterInfo, radius );
	hkpRigidBody* bodyB = createCapsule( mB, pivot, in.m_to, filterInfo, radius );

	world->addEntity( bodyA )->removeReference();
	world->addEntity( bodyB )->removeReference();

	legOut.m_lenA = in.m_lenA;
	legOut.m_lenB = in.m_lenB;
	legOut.m_constraint = HK_NULL;

	hkpConstraintChainInstance* chainInstance;
	{
		//CONSTRAINT_6D
		hkpPoweredChainData* chainData = new hkpPoweredChainData();
		chainInstance = new hkpConstraintChainInstance( chainData );

		hkpPositionConstraintMotor* strongMotor = new hkpPositionConstraintMotor();
		{
			strongMotor->m_tau = 1.0f;
			strongMotor->m_maxForce = 150000.0f;
			strongMotor->m_damping = 1.0f;
			strongMotor->m_constantRecoveryVelocity = 1.0f;
			strongMotor->m_proportionalRecoveryVelocity = 1.1f;
		}
		hkpPositionConstraintMotor* weakMotor = new hkpPositionConstraintMotor();
		{
			weakMotor->m_tau = 1.0f;
			weakMotor->m_maxForce = 550.0f;
			weakMotor->m_damping = 1.0f;
			weakMotor->m_constantRecoveryVelocity = 1.0f;
			weakMotor->m_proportionalRecoveryVelocity = 1.1f;
		}

		chainInstance->addEntity( rootBody );

		{
			hkVector4 pivotinA; pivotinA.setTransformedInversePos( rootBody->getTransform(), in.m_from );
			hkVector4 pivotinB; pivotinB.setTransformedInversePos( bodyA->getTransform(),    in.m_from );

			hkQuaternion aTw; aTw.setInverse( rootBody->getRotation() );
			hkQuaternion aTb; aTb.setMul( aTw, bodyA->getRotation() );

			chainData->addConstraintInfoInBodySpace( pivotinA, pivotinB, aTb, strongMotor, strongMotor, weakMotor );
			chainInstance->addEntity( bodyA );
		}
		{
			hkVector4 pivotinA; pivotinA.setTransformedInversePos( bodyA->getTransform(), pivot );
			hkVector4 pivotinB; pivotinB.setTransformedInversePos( bodyB->getTransform(), pivot );

			hkQuaternion aTw; aTw.setInverse( bodyA->getRotation() );
			hkQuaternion aTb; aTb.setMul( aTw, bodyB->getRotation() );

			chainData->addConstraintInfoInBodySpace( pivotinA, pivotinB, aTb, strongMotor, strongMotor, strongMotor );

			chainInstance->addEntity( bodyB );
		}
		strongMotor->removeReference();
		weakMotor->removeReference();
		chainData->removeReference();
	}
	world->addConstraint( chainInstance );
	legOut.m_constraint = chainInstance;
}


void CreaturesDemo::createLeggedCreature( hkpWorld* world, LeggedCreatureData& creatureData, const hkVector4* requestCenter )
{
	hkVector4 center;

	if(requestCenter)
	{
		center = *requestCenter;
	}
	else
	{
		if(creatureData.m_layout.m_numLegs == 4)
		{
			center = hkVector4( 0, 2, 5 );
		}
		else if(creatureData.m_layout.m_numLegs == 20)
		{
			center = hkVector4( 0, 2, -5 );
		}
		else
		{
			center.setZero();
		}
	}

	// center body
	hkpRigidBody* rootBody;
	{
		hkpRigidBodyCinfo ci;
		ci.m_shape = new hkpBoxShape( creatureData.m_layout.m_bodyHalfExtents, 0.0f );
		ci.m_friction = 2.0f;
		ci.m_restitution = 0.0f;
		ci.m_collisionFilterInfo = creatureData.m_layout.m_filterInfo;
		ci.m_motionType = hkpMotion::MOTION_BOX_INERTIA;
		ci.m_position = center;
		ci.m_angularDamping = 1.0f;

		hkReal mass = 50.0f;
		hkReal inertiaFactor = 1.0f;
		hkMassProperties mp;
		hkpInertiaTensorComputer::computeBoxVolumeMassProperties( creatureData.m_layout.m_legHalfExtents, mass * inertiaFactor, mp );
		ci.m_mass = mass;
		ci.m_inertiaTensor = mp.m_inertiaTensor;

		rootBody = new hkpRigidBody( ci );

		creatureData.m_center = rootBody;
		world->addEntity( rootBody );
		rootBody->removeReference();
		ci.m_shape->removeReference();
	}

	// legs
	{
		for (int leg = 0; leg < creatureData.m_layout.m_numLegs; leg++ )
		{
			hkReal xf = leg/hkReal(creatureData.m_layout.m_numLegs-1)*2.0f - 1.0f;
			for (int z = -1; z <= 1; z+=2 )
			{
				CreaturesDemo::calcLegMatrizesIn m;
				m.m_from = center;
				m.m_from(0) += xf * creatureData.m_layout.m_bodyHalfExtents(0);
				m.m_from(2) += z  * creatureData.m_layout.m_bodyHalfExtents(2);

				m.m_to = center;
				m.m_to(0) += xf * creatureData.m_layout.m_legHalfExtents(0);
				m.m_to(1) -= creatureData.m_layout.m_legHalfExtents(1);
				m.m_to(2) += z * creatureData.m_layout.m_legHalfExtents(2);
				m.m_up.set(0,1,0);
				hkVector4 diff; diff.setSub4( m.m_to, m.m_from );
				hkReal len = diff.length3();
				m.m_lenA = 0.6f * len;
				m.m_lenB = 0.8f * len;
				buildLeg( world, rootBody, m, creatureData.m_layout.m_filterInfo, creatureData.m_legs.expandOne() );
			}
		}
	}
}

void CreaturesDemo::moveLeg( LeggedCreatureData& creatureData, CreaturesDemo::Leg& leg, hkReal z, hkReal xf, hkReal time, hkReal vel )
{
	static hkReal xControl[] = {  1.f,  0.6f, -0.6f,  -1.0f, 1.f };
	static hkReal yControl[] = {  1.f,  0.0f,  0.0f,   1.0f, 1.f };
	static hkReal tControl[] = {  0.f,  0.16f, 0.64f,  0.8f, 1.f };

	hkReal yOffset;
	hkReal xOffset;
	{
		time += xf * 0.5f + z * 0.25f;
		hkReal t = time - int(time);
		HK_ASSERT( 0, t >=-1.0 && t <=1.0f );
		if ( t < 0.0f )
		{
			t += 1.0f;
		}

		int i;
		const int num = sizeof(tControl)/sizeof(hkReal);
		for (i=0; i < num-1; i++)
		{
			if ( t < tControl[i])
			{
				break;
			}
		}
		int in = i;
		i--;
		hkReal f = (t - tControl[i])/ (tControl[in]- tControl[i]);
		xOffset = xControl[in] * f + (1-f) * xControl[i];
		yOffset = yControl[in] * f + (1-f) * yControl[i];

		xOffset *= 1.1f * vel;
		yOffset *= 0.3f;
	}

	// this is a copy of create spider
	// we calculate all the relative matrizes in the root object space
	hkVector4 center; center.setZero4();

	CreaturesDemo::calcLegMatrizesIn m;
	m.m_from = center;
	m.m_from(0) += xf * creatureData.m_layout.m_bodyHalfExtents(0);
	m.m_from(2) += z  * creatureData.m_layout.m_bodyHalfExtents(2);

	m.m_to = center;
	m.m_to(0) += xOffset + xf * creatureData.m_layout.m_legHalfExtents(0);
	m.m_to(1) += yOffset - creatureData.m_layout.m_legHalfExtents(1);
	m.m_to(2) += z * creatureData.m_layout.m_legHalfExtents(2);
	m.m_up.set(0,1,0);

	m.m_lenA = leg.m_lenA;
	m.m_lenB = leg.m_lenB;

	hkVector4 pivot;
	hkRotation mA;
	hkRotation mB;
	doLegIk( m, pivot, mA, mB );

	hkpPoweredChainData* data = static_cast<hkpPoweredChainData*>(leg.m_constraint->getData());
	{
		data->m_infos[0].m_aTc.set(mA);
	}
	{
		hkRotation invMa; invMa.setTranspose(mA);
		hkRotation aTb; aTb.setMul( invMa, mB );
		data->m_infos[1].m_aTc.set( aTb );
	}
}

void CreaturesDemo::moveLeggedCreature( LeggedCreatureData& creatureData)
{
	int legId = 0;
	for (int posId = 0; posId < creatureData.m_layout.m_numLegs; posId++ )
	{
		hkReal xf = posId/hkReal(creatureData.m_layout.m_numLegs-1)*2.0f - 1.0f;
		hkReal vel = creatureData.m_leftFactor;
		for (int z = -1; z <= 1; z+=2 )
		{
			Leg& leg = creatureData.m_legs[legId++];
			moveLeg( creatureData, leg, hkReal(z), xf, creatureData.m_time, vel);
			vel = creatureData.m_rightFactor;
		}
	}
}


CreaturesDemo::~CreaturesDemo()
{
	m_world->lock();

	for (int c=0; c < m_leggedCreatureData.getSize(); c++ )
	{
		for (int i=0; i < m_leggedCreatureData[c].m_legs.getSize(); i++ )
		{
			m_leggedCreatureData[c].m_legs[i].m_constraint->removeReference();
		}
	}

	m_world->unlock();

	if(m_loader)
	{
		delete m_loader;
	}

	m_snakeBox->removeReference();
	m_snakeSphere->removeReference();
}

hkDemo::Result CreaturesDemo::stepDemo()
{
	m_world->lock();

	handleKeyPresses();

	// Move the spider and the centipede
	for(int i=0;i<NUM_SPIDERS;i++)
	{
		moveLeggedCreature( m_leggedCreatureData[i] );

		hkQuaternion rot = m_leggedCreatureData[i].m_center->getRotation();
		hkVector4 upVec; upVec.setRotatedDir(rot, hkVector4::getConstant(HK_QUADREAL_0100));
		// Rotate (0,1,0) by the body's rotation, it should still have a positive y
		if(upVec(1) < 0.0f)
		{
			hkVector4 impulseVec(25.0f, 0.0f, 0.0f);
			m_leggedCreatureData[i].m_center->applyAngularImpulse(impulseVec);
			//HK_WARN_ALWAYS(4324, "Upside down");
		}
	}

	// drive the snake with side-to-side slithering
	for(int i=0;i<NUM_SNAKES;i++)
	{
		slither( m_timestep, m_snakeData[i] ); 
	}

	for(int i=0;i<NUM_SPIDERS;i++)
	{
		m_leggedCreatureData[i].m_time += m_timestep * m_leggedCreatureData[i].m_velocity;
	}

	if(m_cameraFollowingSteered)
	{
#if STEERING_SPIDER
		hkVector4 from(42.0f, 15.0f, -3.0f);
		hkVector4 up  (0.0f, 1.0f, 0.0f);
		setupDefaultCameras( m_env, from, m_leggedCreatureData[0].m_center->getPosition(), up, .5f, 300.0f );
#endif
	}

	m_world->unlock();

	m_step++;
	return hkDefaultPhysics2012Demo::stepDemo();
}

// create a snake body segment with a wheel on the bottom.
// the wheel provides asymetric friction, which is essential for this type of snake locomotion
hkpRigidBody* CreaturesDemo::createSegment( hkVector4& segmentPosition, hkVector4& segmentHalfSize, int index)
{
	LeglessCreatureData& snakeData = m_snakeData[index];
	hkVector4 segmentSize;
	segmentSize.setMul4( 0.5f, segmentHalfSize );
	
//	hkpRigidBody* segment = GameUtils::createBox( segmentSize, 0.5f, segmentPosition );

	hkpRigidBodyCinfo boxCinfo;
	boxCinfo.m_position = segmentPosition;
	boxCinfo.m_mass = 0.5f;
	boxCinfo.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo( LAYER_SNAKE + index );
	boxCinfo.m_shape = m_snakeBox;

	hkMassProperties massProperties;
	hkpInertiaTensorComputer::computeBoxVolumeMassProperties(snakeSegmentHalfSize, boxCinfo.m_mass, massProperties);
	boxCinfo.m_inertiaTensor = massProperties.m_inertiaTensor;
	boxCinfo.m_motionType = hkpMotion::MOTION_BOX_INERTIA;

	hkpRigidBody* segment = new hkpRigidBody(boxCinfo);

	m_world->addEntity(segment);

	{
		hkgDisplayObject* displayObject =  m_env->m_displayHandler->findDisplayObject(hkUlong(segment->getCollidable()));//m_env->m_displayWorld->getDisplayObject(idx);
		if(displayObject)
		{
			displayObject->setStatusFlags( (displayObject->getStatusFlags() | HKG_DISPLAY_OBJECT_SHADOWCASTER) & ~HKG_DISPLAY_OBJECT_SHADOWRECEIVER );
		}
	}

	segment->removeReference();

	// create the wheel rigidbody
	hkVector4 wheelPos = segmentPosition;
	wheelPos(1) -= segmentHalfSize(1) * 0.75f;
	//hkpRigidBody* wheel = GameUtils::createSphere( segmentHalfSize(1) * 0.75f, 0.5f, wheelPos );

	hkpRigidBodyCinfo sphereCinfo;
	sphereCinfo.m_position = wheelPos;
	sphereCinfo.m_mass = 0.5f;
	sphereCinfo.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo( LAYER_SNAKE + index );
	// give the wheels a bit of friction, because a snakes belly would have more friction.
	sphereCinfo.m_angularDamping = 2.0f;
	sphereCinfo.m_friction = 1.0f;
	sphereCinfo.m_shape = m_snakeSphere;

	hkpInertiaTensorComputer::computeSphereVolumeMassProperties(snakeSegmentHalfSize(1) * 0.75f, sphereCinfo.m_mass, massProperties);
	sphereCinfo.m_inertiaTensor = massProperties.m_inertiaTensor;
	sphereCinfo.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;

	hkpRigidBody* wheel = new hkpRigidBody(sphereCinfo);

	m_world->addEntity(wheel);

	{
		hkgDisplayObject* displayObject =  m_env->m_displayHandler->findDisplayObject(hkUlong(wheel->getCollidable()));//m_env->m_displayWorld->getDisplayObject(idx);
		if(displayObject)
		{
			displayObject->setStatusFlags( (displayObject->getStatusFlags() | HKG_DISPLAY_OBJECT_SHADOWCASTER) & ~HKG_DISPLAY_OBJECT_SHADOWRECEIVER );
		}
	}

	wheel->removeReference();

	// create the hinge constraint
	hkpHingeConstraintData* wheelCon = new hkpHingeConstraintData( );
	//hkVector4 axel(1.0f, 0.0f, 0.0f);
	hkVector4 axel(0.0f, 0.0f, 1.0f);
	wheelCon->setInWorldSpace( wheel->getTransform(), segment->getTransform(), wheelPos, axel );

	snakeData.m_wheels.pushBack( wheelCon );

	m_world->createAndAddConstraintInstance( wheel, segment, wheelCon )->removeReference();
	wheelCon->removeReference(); 

	return segment;
}

// link together two segments of a snake with a powered hinge and add the motor
void CreaturesDemo::linkSegment( hkpRigidBody* front, hkpRigidBody* back, hkVector4& pos, hkVector4& axis, LeglessCreatureData& snakeData )
{
	hkpLimitedHingeConstraintData* lhc = new hkpLimitedHingeConstraintData( );
	lhc->setInWorldSpace( front->getTransform(), back->getTransform(), pos, axis );

	// don't need limits
	lhc->disableLimits();

	snakeData.m_muscles.pushBack( lhc );

	hkpConstraintInstance* instance = m_world->createAndAddConstraintInstance( front,back, lhc );
	lhc->removeReference();

	// create the motor that drives the snake ( simulates the muscle ).
	// set it up as inactive to start, we control all snake power in slither()
	hkpPositionConstraintMotor* pcm = new hkpPositionConstraintMotor( 0 );
	pcm->m_maxForce = 15.0f;
	pcm->m_tau = 0.1f;

	lhc->setMotor( pcm );
	lhc->setMotorEnabled(instance->getRuntime(), true);
	pcm->removeReference();
	instance->removeReference();
}

void CreaturesDemo::handleKeyPresses()
{
	if(m_env->wasButtonPressed(HKG_PAD_BUTTON_2) || m_env->m_window->getKeyboard().wasKeyPressed('C'))
	{
		m_cameraFollowingSteered = !m_cameraFollowingSteered;
	}
#if STEERING_SPIDER
	// Manually steer spider 0
	if( m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_UP) )
	{
		m_leggedCreatureData[0].m_velocity += 2.0f * m_timestep;
	}
	if( m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_DOWN) )
	{
		m_leggedCreatureData[0].m_velocity += -2.0f * m_timestep;
	}

	m_leggedCreatureData[0].m_leftFactor += m_timestep * 2.0f;
	m_leggedCreatureData[0].m_rightFactor += m_timestep * 2.0f;

	if( m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_RIGHT) )
	{
		m_leggedCreatureData[0].m_rightFactor -= m_timestep * 4.0f;
	}
	if( m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_LEFT) )
	{
		m_leggedCreatureData[0].m_leftFactor -= m_timestep * 4.0f;
	}

	hkReal maxVel = MAX_CREATURE_VELOCITY;
	m_leggedCreatureData[0].m_velocity = hkMath::clamp( m_leggedCreatureData[0].m_velocity, -maxVel, maxVel );
	m_leggedCreatureData[0].m_leftFactor = hkMath::clamp( m_leggedCreatureData[0].m_leftFactor, hkReal(-0.5f), hkReal(1.f) );
	m_leggedCreatureData[0].m_rightFactor = hkMath::clamp( m_leggedCreatureData[0].m_rightFactor, hkReal(-0.5f), hkReal(1.f) );

#endif
#if STEERING_SNAKE

	// Steer snake 0

	m_snakeData[0].m_steering = 0.0f;
	m_snakeData[0].m_wavePeriod = SNAKE_NORMAL_PERIOD;

	// steer left and right
	if( (m_env->m_gamePad->getButtonState() & HKG_PAD_DPAD_LEFT) )
	{
		m_snakeData[0].m_steering = -5.0f;
	}

	if( (m_env->m_gamePad->getButtonState() & HKG_PAD_DPAD_RIGHT) )
	{
		m_snakeData[0].m_steering = 5.0f;
	}

	// control speed
	if( (m_env->m_gamePad->getButtonState() & HKG_PAD_DPAD_UP) )
	{
		m_snakeData[0].m_wavePeriod = SNAKE_FAST_PERIOD;
	}

	if( (m_env->m_gamePad->getButtonState() & HKG_PAD_DPAD_DOWN) )
	{
		m_snakeData[0].m_wavePeriod = SNAKE_SLOW_PERIOD;
	}
#endif

	// Do auto steering
	for(int i=(STEERING_SPIDER ? 1 : 0);i<NUM_SPIDERS;i++)
	{
		const hkReal randRange = hkReal(NUM_SPIDERS)/2.0f;
		//const hkReal randBias = hkReal(i) - hkReal(NUM_SPIDERS) / 2.0f;
		const hkReal randBias = 0.01f;
		m_leggedCreatureData[i].m_leftFactor += m_timestep * (m_rand.getRandRange(-randRange + randBias, randRange + randBias)) * 0.5f;
		m_leggedCreatureData[i].m_rightFactor -= m_timestep * (m_rand.getRandRange(-randRange, randRange)) * 0.5f;
		// Blend in velocity
		if(m_leggedCreatureData[i].m_velocity > MAX_CREATURE_VELOCITY)
		{
			m_leggedCreatureData[i].m_velocity -= m_timestep;
		}
		else
		{
			m_leggedCreatureData[i].m_velocity += m_timestep;
		}
		m_leggedCreatureData[i].m_leftFactor = hkMath::clamp( m_leggedCreatureData[i].m_leftFactor, hkReal(-0.25f), hkReal(0.5f) );
		m_leggedCreatureData[i].m_rightFactor = hkMath::clamp( m_leggedCreatureData[i].m_rightFactor, hkReal(-0.25f), hkReal(0.5f) );
	}

	for(unsigned int i=(STEERING_SNAKE ? 1 : 0);i<NUM_SNAKES;i++)
	{
		// Distribute steps, change every 2 seconds
		if((m_step % hkUint64(2.0f / m_timestep)) == (i * 10))
		{
			// 1/7 left, 1/7 right, 5/7 straight
			hkUint32 steeringRand = m_rand.getRand32() % 7;
			if(steeringRand == 0)
			{
				m_snakeData[i].m_steering = -5;
			}
			else if(steeringRand == 1)
			{
				m_snakeData[i].m_steering = 5;
			}
			else
			{
				m_snakeData[i].m_steering = 0;
			}
			m_snakeData[i].m_wavePeriod = SNAKE_NORMAL_PERIOD;
		}
	}
}

// simple snake locomotion technique.  
// http://www.snakerobots.com to see one of these in the wild 
void CreaturesDemo::slither( hkReal dt, LeglessCreatureData& snakeData )
{
	int numMuscles = snakeData.m_muscles.getSize();

	hkReal scaleStart = 1.0f;

	// start of the snake with smaller waves.  
	// I think starting the wave at the tail would work better ( while it still goes from head to tail )
	if( snakeData.m_waveFront < snakeData.m_segmentLength*4.0f )
	{
		scaleStart = snakeData.m_waveFront/(snakeData.m_segmentLength*4.0f);
	}

	int segment = (int)numMuscles-1;

	// we simply move a sin wave down the body of the snake.
	// this snake may be going backwards, but who can tell ;)
	for( ; segment >= 0; segment-- )
	{
		// map segment to phase
		hkReal phase = (snakeData.m_waveFront - (segment+1)*snakeData.m_segmentLength)/ snakeData.m_waveLength;
		phase -= hkMath::floor(phase);

		phase *= HK_REAL_PI * 2.0f;

		// map phase to curvature

		// set our motor
		hkpLimitedHingeConstraintData* lhc = static_cast<hkpLimitedHingeConstraintData*>( snakeData.m_muscles[segment] );

		// set up the muscles of the snake
		lhc->setMotorTargetAngle( hkMath::sin( phase ) * scaleStart * snakeData.m_waveAmplitude );


		// steer snake by squashing +ve or -ve side of sin curve
		if( snakeData.m_steering > 0 && lhc->getMotorTargetAngle() < 0 )
		{
			lhc->setMotorTargetAngle( lhc->getMotorTargetAngle() * 1.0f/( 1.0f + snakeData.m_steering ) );
		}

		if( snakeData.m_steering < 0 && lhc->getMotorTargetAngle() > 0 )
		{
			lhc->setMotorTargetAngle( lhc->getMotorTargetAngle() * 1.0f/( 1.0f - snakeData.m_steering ) );
		}

	}
	// wave keeps track of where the wave is in time
	snakeData.m_waveFront += dt/snakeData.m_wavePeriod * snakeData.m_waveLength;
}

////////////////////////////////////////////////////////////////////////////////


static const char helpString[] = "Snakes and spiders. Use the pad to steer the center spider\n"
"C or \x12 switches the camera";

HK_DECLARE_DEMO( CreaturesDemo, HK_DEMO_TYPE_PHYSICS_2012, "Snake and Spider", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
