/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>

#include <Graphics/Common/Window/hkgWindow.h>

#include <Demos/Physics2012/UseCase/CharacterControl/CharacterRigidBody/CharacterRbAndCollisionArtifacts/CharacterRbAndCollisionArtifactsDemo.h>
#include <Physics2012/Utilities/CharacterControl/CharacterRigidBody/hkpCharacterRigidBody.h>

#include <Physics2012/Utilities/CharacterControl/StateMachine/hkpDefaultCharacterStates.h>

// Used to generate a convex hull for convexVerticesShape
#include <Common/Internal/ConvexHull/hkGeometryUtility.h>

// Used for character controller code
#include <Physics2012/Dynamics/Phantom/hkpSimpleShapePhantom.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics2012/Collide/Query/Collector/PointCollector/hkpClosestCdPointCollector.h>

#include <Common/Internal/SimplexSolver/hkSimplexSolver.h>

// Used for graphics and I/O
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>
#include <Demos/DemoCommon/Utilities/Character/CharacterStepInput.h>

#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>

 // Used to create the MOPP 'code' object
#include <Physics2012/Collide/Shape/Compound/Collection/SimpleMesh/hkpSimpleMeshShape.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppUtility.h>

#include <Physics2012/Collide/Agent3/Machine/Nn/hkpAgentNnTrack.h>
#include <Physics2012/Dynamics/Collide/hkpSimpleConstraintContactMgr.h>


// We need to create a constraint
#include <Physics/Constraint/Data/Ragdoll/hkpRagdollConstraintData.h>

// We want to turn on constraint viewers for this demo
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpActiveContactPointViewer.h>

#include <Physics/Constraint/Motor/Position/hkpPositionConstraintMotor.h>

#include <Physics2012/Dynamics/Collide/Deprecated/hkpCollisionListener.h>
#include <Physics2012/Dynamics/Collide/ContactListener/hkpContactListener.h>
#include <Physics2012/Dynamics/Entity/hkpEntityListener.h>


#define CHARACTER_COLOR hkColor::rgbFromChars(255, 0, 0, 150)
#define CHARACTER_TRIGGER_COLOR hkColor::rgbFromChars(255, 165, 0, 150)


static const hkVector4 UP(0,0,1);

namespace {

class CharacterControllerCollisionListener : public hkpContactListener, /*public hkpCollisionListener, */public hkpEntityListener
{
public:
	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, CharacterControllerCollisionListener);

	virtual void contactPointAddedCallback(	hkpContactPointAddedEvent& event) 
	{
		if (!event.isToi() && event.m_gskCache)
		{
			const_cast<hkpGskCache*>(event.m_gskCache)->m_gskFlags |= hkpGskCache::GSK_FLAGS_ALLOW_QUICKER_CONTACT_POINT_RECREATION;
		}
	}

	virtual void entityDeletedCallback( hkpEntity* entity ) 
	{
		delete this;
	}
};

} // namespace



CharacterRbAndCollisionArtifactsDemo::CharacterRbAndCollisionArtifactsDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env)
{
	
	// Create the world
	{
		hkpWorldCinfo info;
		info.setBroadPhaseWorldSize( 500.0f );  
		info.m_gravity.set(0,0,-9.8f);
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_8ITERS_MEDIUM);
		info.m_collisionTolerance = 1.f;
		info.m_enableDeactivation = false;
		m_world = new hkpWorld( info );
		m_world->lock();

		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

		m_debugViewerNames.pushBack( hkpConstraintViewer::getName() );
		m_debugViewerNames.pushBack( hkpActiveContactPointViewer::getName());

		disableBackFaceCulling();

		setupGraphics();
	}

	// Setup collision layers 
	{
		// Replace filter
		hkpGroupFilter* groupFilter = new hkpGroupFilter();

		// We disable collisions between different layers to determine what behaviour we want
		groupFilter->disableCollisionsBetween(LAYER_PLANT, LAYER_GROUND);

		m_world->setCollisionFilter( groupFilter, true);
		groupFilter->removeReference();
	}

	
	// Add a floor
	hkpRigidBody* fixedRigidBody;
	{
		hkVector4 halfExtents(30.0f, 30.0f, 1.0f);
		hkpBoxShape* shape = new hkpBoxShape(halfExtents);
		
		hkpRigidBodyCinfo rigidBodyInfo;

		rigidBodyInfo.m_shape = shape;
		rigidBodyInfo.m_position.set(0.0f, 0.0f, 0.0f);
		rigidBodyInfo.m_angularDamping = 0.0f;
		rigidBodyInfo.m_linearDamping = 0.0f;

		rigidBodyInfo.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(LAYER_GROUND,0 );

		// If we set this to true, the body is fixed, and no mass properties need to be computed.
		rigidBodyInfo.m_motionType = hkpMotion::MOTION_FIXED;
			
		 
		// Create a rigid body (using the template above).
		fixedRigidBody = new hkpRigidBody(rigidBodyInfo);

		// Remove references since the body now "owns" the Shapes.
		shape->removeReference();

		// Finally add body so we can see it, and remove reference since the world now "owns" it.
		m_world->addEntity(fixedRigidBody);
		fixedRigidBody->removeReference();

	}

	// Create a character rigid body
	{
	
		// Construct a shape
		hkVector4 vertexA(0,0, .4f);
		hkVector4 vertexB(0,0,-.4f);		

		// Create a capsule to represent the character standing
		m_standShape = new hkpCapsuleShape(vertexA, vertexB, 1.f);

		// Construct a character rigid body
		hkpCharacterRigidBodyCinfo info;
		info.m_mass = 1000.0f;
		info.m_shape = m_standShape;
		info.m_friction = 0.2f;
		info.m_vdbColor = hkColor::rgbFromChars( 240, 0, 0, 150 );  // transparent red
		info.m_maxSlope = HK_REAL_PI / 3.0f;
		info.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo( LAYER_CHARACTER,0 );

		info.m_maxForce = 100000.0f;
		info.m_up = UP;
		info.m_position.set( 0.497625f, -0.108334f, 2.45f );

		m_characterRigidBody = new hkpCharacterRigidBody( info );
		m_world->addEntity( m_characterRigidBody->getRigidBody() );
	}

	// Attach collision listener to improve quality of collisions with the character controller.
	if (env->m_variantId == 1)
	{
		CharacterControllerCollisionListener* listener = new CharacterControllerCollisionListener();
		m_characterRigidBody->getRigidBody()->addContactListener(listener);
		m_characterRigidBody->getRigidBody()->addEntityListener(listener);
	}
	
	// create a "plant" underneath
	// one rigid body constrained to the ground via a ragdoll constraint with a spring damper motor
	{
		// construct another capsule
		hkVector4 vertexA(0,0,  0.1f);
		hkVector4 vertexB(0,0, -0.1f);
		hkpShape* plantShape = new hkpCapsuleShape(vertexA, vertexB, 0.1f);

		// construct the rigid body
		hkpRigidBodyCinfo info;
		info.m_mass = 1.f;
		info.m_shape = plantShape;
		info.m_friction = 0.f;
		info.m_position.set(-0.328245f, 0.00235754f, 1.35769f );
		{
			hkRotation rotationP;
			rotationP.setCols(hkVector4(0.720247f, -0.00436423f, 0.693704f), hkVector4(0.00485492f, 0.999987f, 0.00125044f), hkVector4(-0.693701f, 0.00246725f, 0.720259f));
			info.m_rotation.set(rotationP);
		}
		info.m_linearDamping = 0.f;
		info.m_angularDamping = 0.01f;
		info.m_qualityType = HK_COLLIDABLE_QUALITY_BULLET;
		info.m_solverDeactivation = hkpRigidBodyCinfo::SOLVER_DEACTIVATION_LOW;
		info.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(LAYER_PLANT,0 );

		hkMassProperties props;
		hkpInertiaTensorComputer::computeShapeVolumeMassProperties(plantShape, info.m_mass, props);
		info.m_inertiaTensor = props.m_inertiaTensor;

		hkpRigidBody* plantRigidBody = new hkpRigidBody(info);
		plantShape->removeReference();

		m_world->addEntity(plantRigidBody);

		// add constraint
		//
		// CREATE POWERED RAGDOLL CONSTRAINT
		// 
		{
			hkVector4 pivot(0.0f, 0.0f, 1.0f);
			hkReal planeMin =  HK_REAL_PI * -0.4f;
			hkReal planeMax =  HK_REAL_PI * 0.4f;
			hkReal twistMin =  HK_REAL_PI * -0.1f;
			hkReal twistMax =  HK_REAL_PI *  0.4f;
			hkReal coneMin  =  HK_REAL_PI * 0.4f;

			hkpRagdollConstraintData* rdc = new hkpRagdollConstraintData();

			rdc->setConeAngularLimit(coneMin);
			rdc->setPlaneMinAngularLimit(planeMin);
			rdc->setPlaneMaxAngularLimit(planeMax);
			rdc->setTwistMinAngularLimit(twistMin);
			rdc->setTwistMaxAngularLimit(twistMax);

			hkVector4 twistAxis(0.0f, 0.0f, 1.0f);
			hkVector4 planeAxis(0.0f, 1.0f, 0.0f);
			rdc->setInWorldSpace(plantRigidBody->getTransform(), fixedRigidBody->getTransform(), pivot, twistAxis, planeAxis);


			//
			//	Create and add the constraint
			//
			hkpConstraintInstance* constraint = new hkpConstraintInstance(plantRigidBody, fixedRigidBody, rdc );
			constraint->setPriority(hkpConstraintInstance::PRIORITY_TOI);
			m_world->addConstraint(constraint);

			hkpPositionConstraintMotor* motor = new hkpPositionConstraintMotor( 0 );
			motor->m_tau = 0.6f;
			motor->m_maxForce = 100.0f;

			rdc->setTwistMotor( motor ); 
			rdc->setConeMotor( motor ); 
			rdc->setPlaneMotor( motor ); 
			rdc->setMotorsEnabled(constraint->getRuntime(), true);

			motor->removeReference();

			hkRotation bRa;
			bRa.setTranspose( fixedRigidBody->getTransform().getRotation() );
			bRa.mul( plantRigidBody->getTransform().getRotation() );

			rdc->setTargetRelativeOrientationOfBodies( bRa );

			rdc->removeReference();
			constraint->removeReference();
		}

		plantRigidBody->removeReference();
	}

	// Create the Character state machine and context
	{
		hkpCharacterState* state;
		hkpCharacterStateManager* manager = new hkpCharacterStateManager();

		hkpCharacterStateOnGround* gstate = new hkpCharacterStateOnGround();
		manager->registerState( gstate,	HK_CHARACTER_ON_GROUND);
		gstate->removeReference();

		state = new hkpCharacterStateInAir();
		manager->registerState( state,	HK_CHARACTER_IN_AIR);
		state->removeReference();

		state = new hkpCharacterStateJumping();
		manager->registerState( state,	HK_CHARACTER_JUMPING);
		state->removeReference();

		state = new hkpCharacterStateClimbing();
		manager->registerState( state,	HK_CHARACTER_CLIMBING);
		state->removeReference();

		m_characterContext = new hkpCharacterContext(manager, HK_CHARACTER_ON_GROUND);
		m_characterContext->setCharacterType(hkpCharacterContext::HK_CHARACTER_RIGIDBODY);
		manager->removeReference();
	}


	// Current camera angle about up
	m_currentAngle = 0.4f;
	// Snap mouse back to middle
	m_env->m_window->setMousePosition(m_env->m_window->getWidth() >> 1, getWindowHeight() >> 1);

	m_world->unlock();
}

CharacterRbAndCollisionArtifactsDemo::~CharacterRbAndCollisionArtifactsDemo()
{
	m_world->lock();

	m_characterRigidBody->removeReference();

	m_standShape->removeReference();
	
	m_world->unlock();

	delete m_characterContext;
}

hkDemo::Result CharacterRbAndCollisionArtifactsDemo::stepDemo()
{
	hkpCharacterInput input;

	{
		m_world->lock();

		//	Get user input data
		hkQuaternion orient;
		
		hkReal posX = 0.f;
		hkReal posY = 0.f;
		{
			hkReal deltaAngle = 0.f;
			CharacterUserInput::getUserInputForCharacter(m_env, deltaAngle, posX, posY);
			m_currentAngle += deltaAngle;
			orient.setAxisAngle(UP, m_currentAngle);
		}

		hkpCharacterOutput output;
		{
			input.m_inputLR = posX / 100.f;
			input.m_inputUD = posY / 100.f;

			input.m_wantJump = m_env->m_window->getMouse().wasButtonPressed(HKG_MOUSE_LEFT_BUTTON)
				|| m_env->m_gamePad->wasButtonPressed(HKG_PAD_BUTTON_1);
			input.m_atLadder = false;

			input.m_up = UP;
			input.m_forward.set(1,0,0);
			input.m_forward.setRotatedDir( orient, input.m_forward );

			hkStepInfo stepInfo;
			stepInfo.m_deltaTime = m_timestep;
			stepInfo.m_invDeltaTime = 1.0f/m_timestep;
			
			input.m_stepInfo = stepInfo;
			input.m_characterGravity.set(0,0,-16);
			input.m_velocity = m_characterRigidBody->getLinearVelocity();
			input.m_position = m_characterRigidBody->getPosition();

			m_characterRigidBody->checkSupport(stepInfo,input.m_surfaceInfo);
		}

		// Apply the character state machine
		{
			HK_TIMER_BEGIN( "update character state", HK_NULL );

			m_characterContext->update(input, output);

			HK_TIMER_END();
		}

		// Display state
		{
			hkpCharacterStateType state = m_characterContext->getState();
			const char * stateStr;

			switch (state)
			{
			case HK_CHARACTER_ON_GROUND:
				stateStr = "On Ground";	break;
			case HK_CHARACTER_JUMPING:
				stateStr = "Jumping"; break;
			case HK_CHARACTER_IN_AIR:
				stateStr = "In Air"; break;
			case HK_CHARACTER_CLIMBING:
				stateStr = "Climbing"; break;
			default:
				stateStr = "Other";	break;
			}
			char buffer[255];
			const int& h = getWindowHeight();
			hkString::snprintf(buffer, 255, "State : %s", stateStr);
			m_env->m_textDisplay->outputText(buffer, 20, h-40, 0xffffffff);
		}
		
		//Apply the player character controller
		{
			HK_TIMER_BEGIN( "simulate character", HK_NULL );

			// Feed output velocity from state machine into character rigid body
			m_characterRigidBody->setLinearVelocity(output.m_velocity, m_timestep);

			HK_TIMER_END();
		}

		m_world->unlock();
	}

	// Step world
	{
		hkDefaultPhysics2012Demo::stepDemo();
	}
	
	{
		m_world->lock();

		// Camera Handling
		{
			const hkReal offset = -0.6f;

			hkVector4 from, to;
			to = m_characterRigidBody->getRigidBody()->getPosition();
			to.addMul4(offset, UP);

			const hkReal height = .7f;

			hkVector4 dir;
			dir.setMul4( height, UP );
			dir.addMul4( -2.f, input.m_forward);

			from.setAdd4(to, dir);
			setupDefaultCameras(m_env, from, to, UP, 1.0f);
		}

		m_world->unlock();
	}

	return hkDemo::DEMO_OK;
}




static const char helpString[] = \
" This demonstrates rare collision detection artifacts when bodies are pushed into each other, and when gsk agents fails to create points to avoid jitter.\n";

//HK_DECLARE_DEMO(CharacterRbAndCollisionArtifactsDemo, HK_DEMO_TYPE_PHYSICS_2012, "Character rigid body and collision artifacts.", helpString);
HK_DECLARE_DEMO_VARIANT(CharacterRbAndCollisionArtifactsDemo, HK_DEMO_TYPE_PHYSICS_2012, "with standard quality", 0, "Character rigid body and collision artifacts.", helpString);
HK_DECLARE_DEMO_VARIANT(CharacterRbAndCollisionArtifactsDemo, HK_DEMO_TYPE_PHYSICS_2012, "with extra contact regeneration policy", 1, "Character rigid body and collision artifacts.", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
