/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Common/Base/Math/Matrix/hkMatrix3Util.h>
#include <Demos/Physics2012/ShowCase/VehicleStyles/VehicleStylesBase.h>
#include <Demos/Physics2012/ShowCase/VehicleStyles/CruiserSetup.h>
#include <Demos/Physics2012/ShowCase/VehicleStyles/SportsCarSetup.h>
#include <Demos/Physics2012/ShowCase/VehicleStyles/BigRigSetup.h>
#include <Demos/Physics2012/ShowCase/VehicleStyles/BigRigTrailerSetup.h>
#include <Demos/Physics2012/ShowCase/VehicleStyles/BikeSetup.h>
#include <Demos/Physics2012/ShowCase/VehicleStyles/GoKartSetup.h>

#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>
#include <Demos/DemoCommon/Utilities/GameUtils/Landscape/FlatLand.h>
#include <Demos/DemoCommon/Utilities/GameUtils/TweakerUtils.h>
#include <Demos/DemoCommon/Utilities/VehicleApi/VehicleApiUtils.h>
#include <Demos/DemoCommon/Utilities/VehicleDisplay/VehicleDisplayUtils.h>

#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Thread/Pool/hkCpuThreadPool.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/DisplayObject/hkgInstancedDisplayObject.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>
#include <Graphics/Common/Material/hkgMaterial.h>

#include <Physics2012/Collide/Agent/hkpProcessCollisionInput.h>
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastOutput.h>
#include <Physics2012/Collide/Shape/HeightField/Plane/hkpPlaneShape.h>
#include <Physics2012/Collide/Query/Collector/PointCollector/hkpFixedBufferCdPointCollector.h>
#include <Physics2012/Collide/Query/Multithreaded/CollisionQuery/hkpCollisionQueryJobQueueUtils.h>
#include <Physics/Constraint/Data/Ragdoll/hkpRagdollConstraintData.h>
#include <Physics2012/Dynamics/Phantom/hkpAabbPhantom.h>
#include <Physics2012/Dynamics/World/hkpPhysicsSystem.h>
#include <Physics2012/Utilities/Actions/Reorient/hkpReorientAction.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>
#include <Physics2012/Utilities/VisualDebugger/hkpPhysicsContext.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpPhantomDisplayViewer.h>
#include <Physics2012/Vehicle/hkpVehicleInstance.h>
#include <Physics2012/Vehicle/Manager/RayCastBatchingManager/hkpVehicleRayCastBatchingManager.h>
#include <Physics2012/Vehicle/Manager/MultithreadedVehicle/hkpMultithreadedVehicleManager.h>
#include <Physics2012/Vehicle/Manager/MultithreadedVehicle/hkpVehicleJobQueueUtils.h>

#include <Common/Visualize/Process/hkDebugDisplayProcess.h>

// One per vehicle type.
struct VehicleStylesAssetInfo
{
	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,VehicleStylesAssetInfo);

	const char* m_modelName;
	const char* m_assetName;
};

VehicleStylesAssetInfo s_vehicleModelInfo[] = {
	{"Cruiser", HK_ASSET_NAME("Resources/Physics2012/Showcase/VehicleStyles/Vehicle/cruiser/cruiser.hkt")},
	{"SportsCar", HK_ASSET_NAME("Resources/Physics2012/Showcase/VehicleStyles/Vehicle/SportsCar/SportsCar.hkt")},
	{"GoKart", HK_ASSET_NAME("Resources/Physics2012/Showcase/VehicleStyles/Vehicle/GoKart/GoKart.hkt")},
	{"BigRig", HK_ASSET_NAME("Resources/Physics2012/Showcase/VehicleStyles/Vehicle/Truck/BigRig.hkt")},
	{"BigRigTrailer", HK_ASSET_NAME("Resources/Physics2012/Showcase/VehicleStyles/Vehicle/Truck/BigRig.hkt")},
	// Bike not currently in use.
	{HK_NULL, HK_NULL},
	{"Bike", HK_ASSET_NAME("Resources/Physics2012/Showcase/VehicleStyles/Vehicle/Bike/Bike.hkt")}
};

#define PLANE_LANDSCAPE_ASSET HK_ASSET_NAME("Resources/Physics2012/Showcase/VehicleStyles/Vehicle/PlaneLandscape.hkt")

VehicleStylesAssetInfo s_vehicleLandscapeInfo[] = {
	{"PlaneLandscape", PLANE_LANDSCAPE_ASSET}
};

// To be placed under the ground. Will reposition the car or any props that fall off the ground
class ScenePhantom : public hkpAabbPhantom
{
public:
	HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO);

	ScenePhantom( const hkAabb& aabb, hkUint32 collisionFilterInfo = 0 )
		: hkpAabbPhantom( aabb, collisionFilterInfo )
	{}

	// Callback implementation, resets the rigid body
	HK_FORCE_INLINE virtual void addOverlappingCollidable( hkpCollidable* handle )
	{
		hkpRigidBody* rb = hkpGetRigidBody( handle );
		if( rb != HK_NULL )
		{
			//Special case where we ignore the trailer as it violates the constraint and causes problems
			{
				hkUint32 filterInfo = rb->getCollidable()->getCollisionFilterInfo();
				hkUint32 trailerFilterInfo = hkpGroupFilter::calcFilterInfo(VehicleStylesDemoBase::TRUCK_LAYER, hkpGroupFilter::getSystemGroupFromFilterInfo(filterInfo),
																				VehicleStylesDemoBase::TRAILER_LAYER, VehicleStylesDemoBase::CHASSIS_LAYER);
				if(filterInfo == trailerFilterInfo)
				{
					return;
				}
			}

			hkVector4 currentPos = rb->getPosition();
			hkVector4 newPosition = currentPos;
			hkVector4 oppositeSide(-1.0f, 1.0f, -1.0f);
			newPosition.mul4(oppositeSide);
			newPosition(0) = hkMath::clamp(newPosition(0), hkReal(-108.f), hkReal(108.f));
			newPosition(1) = hkMath::clamp(newPosition(1), hkReal(0.0f), newPosition(1));
			newPosition(2) = hkMath::clamp(newPosition(2), hkReal(-108.f), hkReal(108.f));

			rb->setPositionAndRotationAsCriticalOperation(newPosition, rb->getRotation());

			const int numConstraints = rb->getNumConstraints();
			for(int c=0; c<numConstraints; ++c)
			{
				const hkpConstraintInstance* constraint = rb->getConstraint(c);
				if( constraint->getData()->getType() == hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL)
				{
					hkpRigidBody* trailerRB = constraint->getRigidBodyB();
					hkVector4 currentTrailerPos = trailerRB->getPosition();
					hkVector4 newTrailerPos = currentTrailerPos;
					hkVector4 translate; translate.setSub4(newPosition, currentPos);
					newTrailerPos.add4(translate);
					newTrailerPos(1) = currentTrailerPos(1);
					newTrailerPos(1) = hkMath::clamp(hkReal(newTrailerPos(1)), hkReal(0.f), hkReal(1.f));

					trailerRB->setPositionAndRotationAsCriticalOperation(newTrailerPos, trailerRB->getRotation());
				}
			}
		}
	}

	// Callback implementation
	virtual void removeOverlappingCollidable( hkpCollidable* handle ) {}
};

const float VehicleStylesDemoBase::GROUND_SIZE = 250.f;
const float VehicleStylesDemoBase::WORLD_SIZE = VehicleStylesDemoBase::GROUND_SIZE*2.0f;

VehicleStylesDemoBase::VehicleStylesDemoBase( hkDemoEnvironment* env, VehicleStylesBaseOptions options, hkBool fireCollisionCallbacks, hkBool createWorld )
:	hkDefaultPhysics2012Demo(env),
	m_numVehiclesPerType(options.m_numVehiclesPerType),
	m_tweaking(false),
	m_vehicleManager(HK_NULL),
	m_controlledVehicleID(0),
	m_demoOptions(options)
{

	// We use the thread buttons for car flip etc
	m_allowChangingNumThreads = false;

	// Initially "controller" is at (0,0), ie. neither pointing left/right nor up/down.
	m_inputXPosition = 0.0f;
	m_inputYPosition = 0.0f;

	//
	// Setup the camera. Actually overwritten by step function, and when we first add the vehicle.
	//
	{
		hkVector4 from(0.0f, 0.0f, 10.0f);
		hkVector4 to(0.0f, 0.0f, 0.0f);
		hkVector4 up(0.0f, 1.0f, 0.0f);
		setupDefaultCameras( m_env, from, to, up );
	}

	m_env->m_sceneConverter->setShaderLibraryEnabled(true);
	//Lights and shadows
	{
		hkVector4 mainLightDir(1.f, -1.f, 1.f);
		hkVector4 fillDir(-1.f, -1.f, -1.f);
		hkColor::Argb keyColor = hkColor::rgbFromFloats(1.0f, 1.0f, 1.0f);
		hkColor::Argb fillColor = hkColor::rgbFromFloats(0.95f, 0.95f, 0.95f);
		hkColor::Argb rimColor = hkColor::rgbFromFloats(0.9f, 0.9f, 0.95f);
		setThreeStageLights( env, &mainLightDir(0), &fillDir(0), keyColor, fillColor, rimColor );

		if(m_demoOptions.m_enableVehicleShadows || m_demoOptions.m_enablePropShadows)
		{
			hkgAabb areaAabb;
			hkgVec3Set(areaAabb.m_min, -GROUND_SIZE*0.5f, -5.0f, -GROUND_SIZE*0.5f);
			hkgVec3Set(areaAabb.m_max, GROUND_SIZE*0.5f, 5.0f, GROUND_SIZE*0.5f);
			int numSplits = 0;
			m_env->m_window->setShadowMapSize(512);
			setupFixedShadowFrustum( m_env, *m_env->m_displayWorld->getLightManager()->getBestShadowCaster(), areaAabb, 3, 5, numSplits, /*y up*/ 1 );

			m_env->m_window->getContext()->setShadowVsmBias(1.0e-5f, 1.0e-5f);
			m_env->m_window->getShadowUtil().setNearDistanceClamp( 5, 10 ); // fix first split over car

			forceShadowState(true);
		}
		else
		{
			forceShadowState(false);
		}
	}

	// Should probably be disabled as cruiser model not display correctly, however the door interiors are not modelled
	//setGraphicsState( HKG_ENABLED_CULLFACE, false );

	// Setup up tweaking variables
	{
		m_tweaking = false;
		m_tweakName = "/VehicleSettings";
	}

	// Create the world.
	{
		hkpWorldCinfo info;
		info.m_collisionTolerance = 0.1f;
		info.m_gravity.set(0.0f, -19.8f, 0.0f);
		info.setBroadPhaseWorldSize(WORLD_SIZE) ;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM);
		info.m_fireCollisionCallbacks = fireCollisionCallbacks;
		m_world = new hkpWorld( info );
	}

	m_world->lock();

	{
		// Register all agents.
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
		// Graphics.
		setupGraphics();
	}

	//
	// Create a filter, so that the wheels of the car do not collide with the chassis.
	//
	{
		hkpGroupFilter* filter = new hkpGroupFilter();
		filter->disableCollisionsBetween( CHASSIS_LAYER, WHEEL_LAYER );
		filter->disableCollisionsBetween( TRUCK_LAYER, WHEEL_LAYER );
		filter->disableCollisionsBetween( PHANTOM_LAYER, GROUND_LAYER );
		filter->disableCollisionsBetween( PHANTOM_LAYER, TRAILER_LAYER );
		filter->disableCollisionsBetween( PHANTOM_LAYER, PHANTOM_LAYER );

		m_world->setCollisionFilter( filter );
		filter->removeReference();
	}

	// Setup the vehicle manager
	{
		switch (options.m_vehicleManagerType)
		{
		case VehicleStylesBaseOptions::NO_VEHICLE_MANAGER:
			m_vehicleManager = HK_NULL;
			break;
		case VehicleStylesBaseOptions::WHEEL_BATCHED_RAY_CAST:
		case VehicleStylesBaseOptions::SINGLETHREADED_RAY_CAST:
			m_vehicleManager = new hkpVehicleRayCastBatchingManager;
			break;
		case VehicleStylesBaseOptions::MULTITHREADED_RAY_CAST:
			m_vehicleManager = new hkpMultithreadedVehicleManager();
			break;
		}

		m_numJobs = m_threadPool->getNumThreads() + 1;
		if(m_vehicleManager)
		{
			//
			// Setup for multithreading.
			//
			hkpCollisionQueryJobQueueUtils::registerWithJobQueue( m_jobQueue );
			hkpVehicleJobQueueUtils::registerWithJobQueue( m_jobQueue );

			// register the default addCdPoint() function; you are free to register your own implementation here though
			hkpFixedBufferCdPointCollector::registerDefaultAddCdPointFunction();

			// Special case for this demo variant: we do not allow the # of active SPUs to drop to zero as this can cause a deadlock.
			if ( options.m_vehicleManagerType == VehicleStylesBaseOptions::WHEEL_BATCHED_RAY_CAST)
			{
				m_allowZeroActiveSpus = false;
			}
		}
	}

	m_loader = new hkLoader();

	_loadLandscape();

	// Setup vehicles and their display objects
	{
		// Texture search paths are handled on a per vehicle basis to reduce the search time
		{
			m_env->m_sceneConverter->clearTextureSearchOrder();
			m_env->m_sceneConverter->addTextureSearchOrder("dds");
			/*m_env->m_sceneConverter->addTextureSearchOrder("png");
			m_env->m_sceneConverter->addTextureSearchOrder("tga");*/
		}

		m_vehicleTypeDisplayData.setSize(VehicleStylesBaseOptions::NUM_VEHICLE_TYPES, HK_NULL);
		// Only loading one model
		if(options.m_vehicleType < VehicleStylesBaseOptions::NUM_VEHICLE_TYPES)
		{
			VehicleStylesAssetInfo& modelInfo = s_vehicleModelInfo[options.m_vehicleType];

			_loadVehicle(modelInfo.m_modelName, HK_GET_DEMOS_ASSET_FILENAME(modelInfo.m_assetName), options.m_vehicleType, options.m_numVehiclesPerType);

			//Special case for the big rig, need to load the trailer as well
			if(options.m_vehicleType == VehicleStylesBaseOptions::BIG_RIG)
			{
				VehicleStylesAssetInfo& trailerInfo = s_vehicleModelInfo[VehicleStylesBaseOptions::BIG_RIG_TRAILER];

				_loadVehicle(trailerInfo.m_modelName, HK_GET_DEMOS_ASSET_FILENAME(trailerInfo.m_assetName), VehicleStylesBaseOptions::BIG_RIG_TRAILER, m_numVehiclesPerType);
				_attachTrailers(false);
			}
		}
		else
		{
			for(int t=0; t<VehicleStylesBaseOptions::NUM_VEHICLE_TYPES;++t)
			{
				if(t == VehicleStylesBaseOptions::BIG_RIG_TRAILER)
				{
					VehicleStylesAssetInfo& trailerInfo = s_vehicleModelInfo[VehicleStylesBaseOptions::BIG_RIG_TRAILER];

					_loadVehicle(trailerInfo.m_modelName, HK_GET_DEMOS_ASSET_FILENAME(trailerInfo.m_assetName), t, m_numVehiclesPerType);
					_attachTrailers(true);
				}
				else
				{
					VehicleStylesAssetInfo& modelInfo = s_vehicleModelInfo[t];

					_loadVehicle(modelInfo.m_modelName, HK_GET_DEMOS_ASSET_FILENAME(modelInfo.m_assetName), t, options.m_numVehiclesPerType);
				}
			}
		}
	}

	m_cameraMode = VehicleStylesDemoBase::THIRD_PERSON_BEHIND;
	_setupCamera(m_vehiclesData[m_controlledVehicleID]->m_vehicleType);

	// Setup skybox, background color, and fog
	{
		setupSkyBox(m_env, "Resources/Physics2012/Showcase/VehicleStyles/Vehicle/Skybox/darksky");

		float cc[] = { 0.0f, 0.0f, 0.0f };
		m_env->m_window->setClearColor(cc);

		HKG_FOG_MODE fogMode = HKG_FOG_EXP;
		hkReal density = 0.02f;
		setGraphicsState(HKG_ENABLED_FOG, true);
		m_env->m_window->getContext()->setLinearFogParameters(300.f, 550.f);
		m_env->m_window->getContext()->setExpFogParameters(fogMode, float(density));

	}

	bindKeyPressed('T', "Tweaker, use arrow keys", KeyPressCallback::Method(&VehicleStylesDemoBase::_dummyCallback, this));
	bindKeyPressed('C' /*HKG_PAD_BUTTON_R2*/, "Change Camera", KeyPressCallback::Method(&VehicleStylesDemoBase::_dummyCallback, this));
	if(m_demoOptions.m_switchControlledVehicle)
	{
		bindKeyPressed('X' /*HKG_PAD_BUTTON_L2*/, "Change Vehicle", KeyPressCallback::Method(&VehicleStylesDemoBase::_dummyCallback, this));
	}
	bindKeyPressed('R' /*HKG_PAD_BUTTON_R1*/, "Reorient flipped vehicle", KeyPressCallback::Method(&VehicleStylesDemoBase::_dummyCallback, this));

	m_env->m_mousePickingEnabled = false;
	env->m_window->setWantDrawMousePointer(false);

	m_world->unlock();
}

void VehicleStylesDemoBase::_attachTrailers(bool isMultiVehicleDemo)
{
	m_world->lock();

	// Attach BigRig with their Trailers
	int startVehicleId = isMultiVehicleDemo ? (m_numVehiclesPerType * VehicleStylesBaseOptions::BIG_RIG) : 0;
	int endVehicleId = startVehicleId + m_numVehiclesPerType;
	int startTrailerId = endVehicleId;
	int numVehiclesWithTrailer = m_numVehiclesPerType;

	HK_ASSERT(0x526edf5d, m_world->getCollisionFilter() && m_world->getCollisionFilter()->m_type == hkpCollisionFilter::HK_FILTER_GROUP);

	for (int vIdx=0; vIdx<numVehiclesWithTrailer; ++vIdx)
	{
		hkpVehicleInstance* tractorVehicle = m_vehiclesData[startVehicleId + vIdx]->m_vehicleInstance;
		hkpVehicleInstance* trailerVehicle = m_vehiclesData[startTrailerId + vIdx]->m_vehicleInstance;

		hkpRigidBody* tractorChassis = tractorVehicle->getChassis();
		hkpRigidBody* trailerChassis = trailerVehicle->getChassis();
		//
		// Use a ragdoll constraint to connect the tractor to the trailer
		//
		{
			hkReal planeMin =  HK_REAL_PI * -0.45f;
			hkReal planeMax =  HK_REAL_PI * 0.45f;
			hkReal twistMin =  HK_REAL_PI * -0.005f;
			hkReal twistMax =  HK_REAL_PI *  0.005f;
			hkReal coneMin  =  HK_REAL_PI * -0.55f;
			hkReal coneMax = HK_REAL_PI * 0.55f;

			hkpRagdollConstraintData* rdc = new hkpRagdollConstraintData();
			rdc->setPlaneMinAngularLimit(planeMin);
			rdc->setPlaneMaxAngularLimit(planeMax);
			rdc->setTwistMinAngularLimit(twistMin);
			rdc->setTwistMaxAngularLimit(twistMax);

			hkVector4 twistAxisA(1.0f, 0.0f, 0.0f);
			hkVector4 planeAxisA(0.0f, 1.0f, 0.0f);
			hkVector4 twistAxisB(1.0f, 0.0f, 0.0f);
			hkVector4 planeAxisB(0.0f, 1.0f, 0.0f);
			hkVector4 point1(-1.20f, 1.11f, 0.0f);
			hkVector4 point2( 4.92f, 1.11f, 0.0f);

			rdc->setInBodySpace(point1, point2, planeAxisA, planeAxisB, twistAxisA, twistAxisB);
			rdc->setAsymmetricConeAngle(coneMin, coneMax);
			m_world->createAndAddConstraintInstance(tractorChassis, trailerChassis, rdc)->removeReference();
			rdc->removeReference();
		}
	}
	m_world->unlock();
}

VehicleStylesDemoBase::InstancedDisplayData* VehicleStylesDemoBase::_setupInstancedGeometry(hkxScene* scene, hkArray<hkStringBuf* >& meshNames, int numObjects, hkBool enableShadows )
{
	m_env->m_window->getContext()->lock();


	InstancedDisplayData* displayData = new InstancedDisplayData();
	displayData->m_numObjects = numObjects;

	for(int i=0; i<meshNames.getSize(); i++)
	{
		hkgMaterial* shaderMat = HK_NULL;
		hkgGeometry* displayGeom = HK_NULL;

		hkgDisplayObject* m_sourceObject = m_env->m_sceneConverter->convertMesh(scene, meshNames[i]->cString(), false, HKG_DISPLAY_OBJECT_DEEP_COPY, hkgAssetConverter::CONVERT_ALL, false );

		if(m_sourceObject == HK_NULL)
			continue;

		shaderMat = m_sourceObject->getGeometry(0)->getMaterialFaceSet(0)->getMaterial();
		displayGeom = m_sourceObject->getGeometry(0);

		hkgInstancedDisplayObject* displayObj = hkgInstancedDisplayObject::create( m_env->m_window->getContext() );
		{
			const hkgShaderEffectCollection* shaderCollection = shaderMat->getShaderCollection();
			if (!shaderCollection)
			{
				shaderCollection = m_env->m_window->getContext()->getDefaultShaderEffectCollection();
			}
			bool runningWithInstancedShaders = (m_env->m_window->getContext()->getRenderCaps().m_vertexShaderMajorVersion >= 2) && shaderCollection;

			if (runningWithInstancedShaders)
			{
				runningWithInstancedShaders = false;
				for (int j=0; j < shaderCollection->getNumShaderEffects(); ++j)
				{
					if (shaderCollection->getShaderEffect(j)->getStyleOR() & HKG_SHADER_RENDER_INSTANCING_USING_TCOORDS)
					{
						displayObj->setRenderMode( HKG_INSTANCED_DISPLAY_TCOORD123 );
						runningWithInstancedShaders = true;
						break;
					}
				}
			}

			if (!runningWithInstancedShaders)
			{
				// CPU, but still better than individual objects
				displayObj->setRenderMode( HKG_INSTANCED_DISPLAY_CPU );
				HK_REPORT("No hardware support for instancing, fallback to CPU.");
			}
		}


		displayObj->addGeometry( displayGeom );

		if(enableShadows)
		{
			displayObj->setStatusFlags( displayObj->getStatusFlags() | HKG_DISPLAY_OBJECT_SHADOW);
		}
		else
		{
			displayObj->setStatusFlags( displayObj->getStatusFlags() & ~HKG_DISPLAY_OBJECT_SHADOW);
		}

		if( hkString::strStr(meshNames[i]->cString(), "Wheel") == HK_NULL)
		{
			displayData->m_bodyDisplayObjects.pushBack(displayObj);
		}
		else
		{
			displayData->m_wheelDisplayObjects.pushBack(displayObj);
		}

		displayObj->setMaxNumObjects( displayData->m_numObjects, true );
		displayObj->setNumObjects(displayData->m_numObjects);

		m_env->m_displayWorld->addDisplayObject(displayObj);
		displayObj->removeReference();

		m_env->m_displayWorld->removeDisplayObject( m_sourceObject );

	}


	m_env->m_window->getContext()->unlock();
	return displayData;
}

//void VehicleStylesDemo::_loadVehicle(int variantID, int numVehicles)
void VehicleStylesDemoBase::_loadVehicle(const char* vehicleModelName, const char* assetFileName, int vehicleType, int numVehicles)
{
	m_env->m_sceneConverter->clearTextureSearchPaths();
	m_env->m_sceneConverter->addTextureSearchPath("./Resources/Physics2012/Showcase/VehicleStyles/Vehicle");
	m_env->m_sceneConverter->addTextureSearchPath("./Resources/Physics2012/Showcase/VehicleStyles/Vehicle/Skybox");

	switch (vehicleType)
	{
		case VehicleStylesBaseOptions::CRUISER:
			m_env->m_sceneConverter->addTextureSearchPath("./Resources/Physics2012/Showcase/VehicleStyles/Vehicle/cruiser");
			break;
		case VehicleStylesBaseOptions::SPORTS_CAR:
			m_env->m_sceneConverter->addTextureSearchPath("./Resources/Physics2012/Showcase/VehicleStyles/Vehicle/SportsCar");
			break;
		case VehicleStylesBaseOptions::BIG_RIG:
		case VehicleStylesBaseOptions::BIG_RIG_TRAILER:
			m_env->m_sceneConverter->addTextureSearchPath("./Resources/Physics2012/Showcase/VehicleStyles/Vehicle/Truck/Textures");
			break;
		case VehicleStylesBaseOptions::GOKART:
			m_env->m_sceneConverter->addTextureSearchPath("./Resources/Physics2012/Showcase/VehicleStyles/Vehicle/GoKart/Textures");
			break;
		case VehicleStylesBaseOptions::MOTORBIKE:
			m_env->m_sceneConverter->addTextureSearchPath("./Resources/Physics2012/Showcase/VehicleStyles/Vehicle/bike");
			break;
	}

	// Note this is ordered, chassis should always be the first element
	const char* importVehicleParts[] = {
		"Chassis",
		"Interior",
		"Cab",
		"Windows",
		"WheelFL",
		"WheelFR",
		"WheelBL",
		"WheelBR",
		"WheelML",
		"WheelMR",
	};

	const int numVehicleParts = sizeof(importVehicleParts) / sizeof(char*);

	hkArray<hkStringBuf* > specificVehicleParts;
	for(int i=0; i<numVehicleParts; ++i)
	{
		hkStringBuf* newVehiclePart = new hkStringBuf(vehicleModelName);
		newVehiclePart->append(importVehicleParts[i]);
		specificVehicleParts.pushBack(newVehiclePart);
	}

	// Create the chassis body.
	hkpRigidBody* chassisRigidBody = HK_NULL;
	{
		hkStringBuf fileName(assetFileName);hkAssetManagementUtil::getFilePath(fileName);
		hkRootLevelContainer* container = m_loader->load( fileName.cString() );
		HK_ASSERT2(0x0, container != HK_NULL, "Could not load root level object" );

		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
		HK_ASSERT2(0x0, scene, "No Scene Data found");

		m_vehicleTypeDisplayData[vehicleType] = _setupInstancedGeometry(scene, specificVehicleParts, numVehicles, m_demoOptions.m_enableVehicleShadows);

		// Load the chassis rigid body for the vehicle
		{
			hkpPhysicsData* physicsData = static_cast<hkpPhysicsData*>( container->findObjectByType( hkpPhysicsDataClass.getName() ) );
			HK_ASSERT2(0x0, physicsData != HK_NULL, "Could not find physics data in root level object" );
			HK_ASSERT2(0x0, physicsData->getWorldCinfo() != HK_NULL, "No physics cinfo in loaded file - cannot create a hkpWorld" );

			for ( int i = 0; i < physicsData->getPhysicsSystems().getSize(); ++i )
			{
				hkpPhysicsSystem* physicsSystem = physicsData->getPhysicsSystems()[i];
				const hkArray<hkpRigidBody *>& bodies = physicsSystem->getRigidBodies();

				for (int rb = 0; rb < bodies.getSize();rb++)
				{
					hkpRigidBody* rigidBody = bodies[rb];
					hkpRigidBodyCinfo rbInfo;
					rigidBody->getCinfo(rbInfo);

					if(hkString::beginsWith(rigidBody->getName(), specificVehicleParts[0]->cString()))
					{
						chassisRigidBody = rigidBody;
						break;
					}
				}
			}

		}

		HK_ASSERT2(0x0, chassisRigidBody != HK_NULL, "No rigid body for the chassis was loaded.");
		const hkpShape* chassisShape = chassisRigidBody->getCollidable()->getShape();

		hkpGroupFilter* groupFilter = static_cast<hkpGroupFilter*>(const_cast<hkpCollisionFilter*>(m_world->getCollisionFilter()));

		for(int vehicleDisplayID=0; vehicleDisplayID<numVehicles; ++vehicleDisplayID)
		{
			int vehicleID = m_vehiclesData.getSize();
			// Create vehicle's chassis rigid body based on the one loaded from the asset.
			{
				hkpRigidBodyCinfo chassisInfo;
				chassisRigidBody->getCinfo(chassisInfo);
				chassisInfo.m_shape = chassisShape;
				chassisInfo.m_friction = 0.8f;
				chassisInfo.m_motionType = hkpMotion::MOTION_BOX_INERTIA;

				//// Inertia tensor will be set by VehicleSetup?
				int xIdx = int(vehicleID/5.f);
				int zIdx = vehicleID%(5);
				//// Position chassis on the ground.
				chassisInfo.m_position.set(-110.0f + xIdx*20.f, 0.f, -30.0f + zIdx*20.f);
				//chassisInfo.m_position.setZero4();
				hkMatrix3Util::_setDiagonal(1.0f, 1.0f, 1.0f, chassisInfo.m_inertiaTensor);
				//chassisInfo.m_centerOfMass.set( 0.0f, 0.143f, 0.0f);

				chassisRigidBody = new hkpRigidBody(chassisInfo);
			}

			// This hkpAction flips the car upright if it turns over.
			if(vehicleID%m_numVehiclesPerType == 0)
			{
				hkVector4 rotationAxis(1.0f, 0.0f, 0.0f);
				hkVector4 upAxis(0.0f, 1.0f, 0.0f);
				hkpRigidBodyCinfo chassisInfo;
				chassisRigidBody->getCinfo(chassisInfo);
				hkReal strength = chassisInfo.m_mass*0.1f;
				hkpReorientAction* reorientAction = new hkpReorientAction(chassisRigidBody, rotationAxis, upAxis, strength);
				m_reorientActions.pushBack(reorientAction);
			}

			hkpVehicleInstance* vehicleInstance = HK_NULL;

			switch (vehicleType)
			{
				case VehicleStylesBaseOptions::CRUISER:
					{
						CruiserSetup setup;
						vehicleInstance = new hkpVehicleInstance( chassisRigidBody );
						setup.buildVehicle( m_world, *vehicleInstance );
						vehicleInstance->getChassis()->setCollisionFilterInfo(hkpGroupFilter::calcFilterInfo(CHASSIS_LAYER, 0));
					}
					break;
				case VehicleStylesBaseOptions::SPORTS_CAR:
					{
						SportsCarSetup setup;
						vehicleInstance = new hkpVehicleInstance( chassisRigidBody );
						setup.buildVehicle( m_world, *vehicleInstance );
						vehicleInstance->getChassis()->setCollisionFilterInfo(hkpGroupFilter::calcFilterInfo(CHASSIS_LAYER, 0));
					}
					break;
				case VehicleStylesBaseOptions::BIG_RIG:
					{
						BigRigSetup setup;
						vehicleInstance = new hkpVehicleInstance( chassisRigidBody );
						setup.buildVehicle( m_world, *vehicleInstance );
						int vehicleGroup = groupFilter->getNewSystemGroup();
						vehicleInstance->getChassis()->setCollisionFilterInfo(hkpGroupFilter::calcFilterInfo(TRUCK_LAYER, vehicleGroup, CHASSIS_LAYER, TRAILER_LAYER));
					}
					break;
				case VehicleStylesBaseOptions::BIG_RIG_TRAILER:
					{
						hkpRigidBody* truckBody = m_vehiclesData[vehicleID - numVehicles]->m_vehicleInstance->getChassis();
						hkTransform truckXForm = truckBody->getTransform();
						hkVector4 trailerPos = truckXForm.getTranslation();
						trailerPos(0) -= 5.9295f;
						truckXForm.setTranslation(trailerPos);

						chassisRigidBody->setTransform(truckXForm);
						BigRigTrailerSetup setup;
						vehicleInstance = new hkpVehicleInstance( chassisRigidBody );
						setup.buildVehicle( m_world, *vehicleInstance );
						int vehicleGroup = groupFilter->getSystemGroupFromFilterInfo(truckBody->getCollisionFilterInfo());
						vehicleInstance->getChassis()->setCollisionFilterInfo(hkpGroupFilter::calcFilterInfo(TRUCK_LAYER, vehicleGroup, TRAILER_LAYER, CHASSIS_LAYER));
					}
					break;
				case VehicleStylesBaseOptions::GOKART:
					{
						GoKartSetup setup;
						vehicleInstance = new hkpVehicleInstance( chassisRigidBody );
						setup.buildVehicle( m_world, *vehicleInstance );
						vehicleInstance->getChassis()->setCollisionFilterInfo(hkpGroupFilter::calcFilterInfo(CHASSIS_LAYER, 0));
					}
					break;
				case VehicleStylesBaseOptions::MOTORBIKE:
					{
						BikeSetup setup;
						vehicleInstance = new hkpVehicleInstance( chassisRigidBody );
						setup.buildVehicle( m_world, *vehicleInstance );
						vehicleInstance->getChassis()->setCollisionFilterInfo(hkpGroupFilter::calcFilterInfo(CHASSIS_LAYER, 0));
					}
					break;
				default:
					HK_ASSERT2(0x0, false, "Vehicle Type is not supported" );
					break;
			}

			vehicleInstance->addToWorld( m_world );
			chassisRigidBody->removeReference();
			vehicleInstance->m_wheelCollide->setCollisionFilterInfo( hkpGroupFilter::calcFilterInfo( WHEEL_LAYER, 0 ) );

			if(m_vehicleManager != HK_NULL)
			{
				m_vehicleManager->addVehicle( vehicleInstance );
			}
			else
			{
				m_world->addAction( vehicleInstance );
			}

			VehicleInstanceData* vehicleInstanceData = new VehicleInstanceData;
			vehicleInstanceData->m_vehicleInstance = vehicleInstance;
			vehicleInstanceData->m_instancedDisplayID = vehicleDisplayID;
			vehicleInstanceData->m_vehicleType = vehicleType;
			vehicleInstanceData->m_simulated = true;
			m_vehiclesData.pushBack(vehicleInstanceData);
		}
	}

	for(int i=0; i<specificVehicleParts.getSize(); ++i)
	{
		delete specificVehicleParts[i];
	}
	specificVehicleParts.clear();
}

void VehicleStylesDemoBase::_setupCamera(int cameraType)
{
	// Initialise the camera.
	switch (cameraType)
	{
	case VehicleStylesBaseOptions::BIG_RIG:
	case VehicleStylesBaseOptions::BIG_RIG_TRAILER:
		_setupBigRigCamera();
		break;
	case VehicleStylesBaseOptions::GOKART:
		_setupGoKartCamera();
		break;
	case VehicleStylesBaseOptions::SPORTS_CAR:
		_setupSportsCarCamera();
		break;
	case VehicleStylesBaseOptions::CRUISER:
	case VehicleStylesBaseOptions::MOTORBIKE:
	default:
		_setupCruiserCamera();
		break;
	}
}

void VehicleStylesDemoBase::_setupBigRigCamera()
{
	hkp1dAngularFollowCamCinfo cinfo;
	{
		cinfo.m_yawSignCorrection = 1.0f;
		cinfo.m_upDirWS.set(0.0f, 1.0f, 0.0f);
		cinfo.m_rigidBodyForwardDir.set(1.0f, 0.0f, 0.0f);
		cinfo.m_set[0].m_velocity = 10.0f;
		cinfo.m_set[1].m_velocity = 50.0f;
		cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 1.0f;
		cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 1.0f;
		cinfo.m_set[0].m_angularRelaxation = 3.5f;
		cinfo.m_set[1].m_angularRelaxation = 4.5f;

		if(m_cameraMode == THIRD_PERSON_BEHIND)
		{
			cinfo.m_set[0].m_positionUS.set( -20.0f, 6.5f, 0.0f);
			cinfo.m_set[1].m_positionUS.set( -20.0f, 6.5f, 0.0f);
			cinfo.m_set[0].m_lookAtUS.set ( 8.0f, 4.f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 8.0f, 4.f, 0.0f );
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
		}
		else if(m_cameraMode == THIRD_PERSON_FRONT)
		{
			cinfo.m_set[0].m_positionUS.set ( 12.0f, 1.5f, 0.0f );
			cinfo.m_set[1].m_positionUS.set ( 12.0f, 1.5f, 0.0f );
			cinfo.m_set[0].m_lookAtUS.set( -20.0f, 1.0f, 0.0f);
			cinfo.m_set[1].m_lookAtUS.set( -20.0f, 1.0f, 0.0f);
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
		}
		else if(m_cameraMode == THIRD_PERSON_SIDE)
		{
			cinfo.m_set[0].m_positionUS.set( 0.7f, 1.25f, 12.f);
			cinfo.m_set[1].m_positionUS.set( 0.7f, 1.25f, 12.f);
			cinfo.m_set[0].m_lookAtUS.set ( 0.7f, 1.1f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 0.7f, 1.1f, 0.0f );
			cinfo.m_set[0].m_fov = 50.0f;
			cinfo.m_set[1].m_fov = 75.0f;
		}
		else if(m_cameraMode == TOP_DOWN)
		{
			cinfo.m_set[0].m_positionUS.set( -15.f, 15.f, 0.0f);
			cinfo.m_set[1].m_positionUS.set( -15.f, 15.f, 0.0f);
			cinfo.m_set[0].m_lookAtUS.set ( 0.0f, 5.0f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 0.0f, 5.0f, 0.0f );
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
			cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 2.f;
			cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 2.f;
		}
	}
	m_camera.reinitialize( cinfo );
}

void VehicleStylesDemoBase::_setupCruiserCamera()
{
	// Initialise the camera.
	hkp1dAngularFollowCamCinfo cinfo;
	{
		cinfo.m_yawSignCorrection = 1.0f;
		cinfo.m_upDirWS.set(0.0f, 1.0f, 0.0f);
		cinfo.m_rigidBodyForwardDir.set(1.0f, 0.0f, 0.0f);
		cinfo.m_set[0].m_velocity = 10.0f;
		cinfo.m_set[1].m_velocity = 50.0f;
		cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 1.0f;
		cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 1.0f;
		cinfo.m_set[0].m_angularRelaxation = 3.5f;
		cinfo.m_set[1].m_angularRelaxation = 14.5f;

		if(m_cameraMode == THIRD_PERSON_BEHIND)
		{
			cinfo.m_set[0].m_positionUS.set( -5.0f, 2.5f, 0.0f);
			cinfo.m_set[1].m_positionUS.set( -5.0f, 2.5f, 0.0f);
			cinfo.m_set[0].m_lookAtUS.set ( 0.0f, 0.0f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 0.0f, 0.0f, 0.0f );
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
			cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 2.0f;
			cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 2.0f;
		}
		else if(m_cameraMode == THIRD_PERSON_FRONT)
		{
			cinfo.m_set[0].m_positionUS.set ( 8.0f, 1.5f, 0.0f );
			cinfo.m_set[1].m_positionUS.set ( 8.0f, 1.5f, 0.0f );
			cinfo.m_set[0].m_lookAtUS.set( -5.0f, 0.f, 0.0f);
			cinfo.m_set[1].m_lookAtUS.set( -5.0f, 0.f, 0.0f);

			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
			cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 2.0f;
			cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 2.0f;
		}
		else if(m_cameraMode == THIRD_PERSON_SIDE)
		{
			cinfo.m_set[0].m_positionUS.set( 0.7f, 1.25f, 8.f);
			cinfo.m_set[1].m_positionUS.set( 0.7f, 1.25f, 8.f);
			cinfo.m_set[0].m_lookAtUS.set ( 0.7f, 1.1f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 0.7f, 1.1f, 0.0f );
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
		}
		else if(m_cameraMode == TOP_DOWN)
		{
			cinfo.m_set[0].m_positionUS.set( -15.f, 15.f, 0.0f);
			cinfo.m_set[1].m_positionUS.set( -15.f, 15.f, 0.0f);
			cinfo.m_set[0].m_lookAtUS.set ( 0.0f, 5.0f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 0.0f, 5.0f, 0.0f );
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
			cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 2.f;
			cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 2.f;
		}
	}
	m_camera.reinitialize( cinfo );
}

void VehicleStylesDemoBase::_setupSportsCarCamera()
{
	// Initialise the camera.
	hkp1dAngularFollowCamCinfo cinfo;
	{
		cinfo.m_yawSignCorrection = 1.0f;
		cinfo.m_upDirWS.set(0.0f, 1.0f, 0.0f);
		cinfo.m_rigidBodyForwardDir.set(1.0f, 0.0f, 0.0f);
		cinfo.m_set[0].m_velocity = 10.0f;
		cinfo.m_set[1].m_velocity = 50.0f;
		cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 1.0f;
		cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 1.0f;
		cinfo.m_set[0].m_angularRelaxation = 3.5f;
		cinfo.m_set[1].m_angularRelaxation = 14.5f;

		if(m_cameraMode == THIRD_PERSON_BEHIND)
		{
			cinfo.m_set[0].m_positionUS.set( -5.0f, 1.5f, 0.0f);
			cinfo.m_set[1].m_positionUS.set( -6.0f, 1.5f, 0.0f);
			cinfo.m_set[0].m_lookAtUS.set ( 0.0f, 0.0f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 0.0f, 0.0f, 0.0f );
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
			cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 2.0f;
			cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 2.0f;
		}
		else if(m_cameraMode == THIRD_PERSON_FRONT)
		{
			cinfo.m_set[0].m_positionUS.set ( 8.0f, 1.5f, 0.0f );
			cinfo.m_set[1].m_positionUS.set ( 8.0f, 1.5f, 0.0f );
			cinfo.m_set[0].m_lookAtUS.set( -5.0f, 0.f, 0.0f);
			cinfo.m_set[1].m_lookAtUS.set( -6.0f, 0.f, 0.0f);
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
			cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 2.0f;
			cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 2.0f;
		}
		else if(m_cameraMode == THIRD_PERSON_SIDE)
		{
			cinfo.m_set[0].m_positionUS.set( 0.7f, 1.25f, 8.f);
			cinfo.m_set[1].m_positionUS.set( 0.7f, 1.25f, 8.f);
			cinfo.m_set[0].m_lookAtUS.set ( 0.7f, 1.1f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 0.7f, 1.1f, 0.0f );
			cinfo.m_set[0].m_fov = 50.0f;
			cinfo.m_set[1].m_fov = 75.0f;
		}
		else if(m_cameraMode == TOP_DOWN)
		{
			cinfo.m_set[0].m_positionUS.set( -15.f, 15.f, 0.0f);
			cinfo.m_set[1].m_positionUS.set( -15.f, 15.f, 0.0f);
			cinfo.m_set[0].m_lookAtUS.set ( 0.0f, 5.0f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 0.0f, 5.0f, 0.0f );
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
			cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 2.f;
			cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 2.f;
		}
	}
	m_camera.reinitialize( cinfo );
}

void VehicleStylesDemoBase::_setupGoKartCamera()
{
	// Initialise the camera.
	hkp1dAngularFollowCamCinfo cinfo;
	{
		cinfo.m_yawSignCorrection = 1.0f;
		cinfo.m_upDirWS.set(0.0f, 1.0f, 0.0f);
		cinfo.m_rigidBodyForwardDir.set(1.0f, 0.0f, 0.0f);
		cinfo.m_set[0].m_velocity = 10.0f;
		cinfo.m_set[1].m_velocity = 30.0f;
		cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 1.0f;
		cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 1.0f;
		cinfo.m_set[0].m_angularRelaxation = 3.5f;
		cinfo.m_set[1].m_angularRelaxation = 14.5f;

		if(m_cameraMode == THIRD_PERSON_BEHIND)
		{
			cinfo.m_set[0].m_positionUS.set( -2.75f, 1.0f, 0.0f);
			cinfo.m_set[1].m_positionUS.set( -2.75f, 1.0f, 0.0f);
			cinfo.m_set[0].m_lookAtUS.set ( 0.0f, 0.5f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 0.0f, 0.5f, 0.0f );
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
		}
		else if(m_cameraMode == THIRD_PERSON_FRONT)
		{
			cinfo.m_set[0].m_positionUS.set ( 5.0f, 1.0f, 0.0f );
			cinfo.m_set[1].m_positionUS.set ( 5.0f, 1.0f, 0.0f );
			cinfo.m_set[0].m_lookAtUS.set( -2.75f, 0.5f, 0.0f);
			cinfo.m_set[1].m_lookAtUS.set( -2.75f, 0.5f, 0.0f);
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
		}
		else if(m_cameraMode == THIRD_PERSON_SIDE)
		{
			cinfo.m_set[0].m_positionUS.set( 0.7f, 1.0f, 3.f);
			cinfo.m_set[1].m_positionUS.set( 0.7f, 1.0f, 3.f);
			cinfo.m_set[0].m_lookAtUS.set ( 0.7f, 1.1f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 0.7f, 1.1f, 0.0f );
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
		}
		else if(m_cameraMode == TOP_DOWN)
		{
			cinfo.m_set[0].m_positionUS.set( -15.f, 15.f, 0.0f);
			cinfo.m_set[1].m_positionUS.set( -15.f, 15.f, 0.0f);
			cinfo.m_set[0].m_lookAtUS.set ( 0.0f, 5.0f, 0.0f );
			cinfo.m_set[1].m_lookAtUS.set ( 0.0f, 5.0f, 0.0f );
			cinfo.m_set[0].m_fov = 60.0f;
			cinfo.m_set[1].m_fov = 60.0f;
			cinfo.m_set[0].m_speedInfluenceOnCameraDirection = 2.f;
			cinfo.m_set[1].m_speedInfluenceOnCameraDirection = 2.f;
		}
	}
	m_camera.reinitialize( cinfo );
}

void VehicleStylesDemoBase::_loadLandscape()
{
	m_env->m_sceneConverter->clearTextureSearchOrder();
	m_env->m_sceneConverter->addTextureSearchOrder("dds");
	/*m_env->m_sceneConverter->addTextureSearchOrder("png");
	m_env->m_sceneConverter->addTextureSearchOrder("tga");*/

	m_env->m_sceneConverter->clearTextureSearchPaths();
	m_env->m_sceneConverter->addTextureSearchPath("./Resources/Physics2012/Showcase/VehicleStyles/Vehicle");
	m_env->m_sceneConverter->addTextureSearchPath("./Resources/Physics2012/Showcase/VehicleStyles/Skybox");

	VehicleStylesAssetInfo& landscapeInfo = s_vehicleLandscapeInfo[m_demoOptions.m_landscapeType];
	const char* landscapeAsset = landscapeInfo.m_assetName;

	hkBool enableReflection = false;

	if (m_demoOptions.m_enableReflection)
	{
		enableReflection = true;
	}

	hkStringBuf fileName(HK_GET_DEMOS_ASSET_FILENAME(landscapeAsset)); hkAssetManagementUtil::getFilePath(fileName);
	hkRootLevelContainer* container = m_loader->load( fileName.cString() );
	HK_ASSERT2(0x0, container != HK_NULL, "Could not load root level object" );

	hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
	HK_ASSERT2(0x0, scene, "No Scene Data found");

	hkArray<hkpRigidBody* > groundRigidBodiesToAdd;
	// Get the physics data for the landscape
	{
		hkpPhysicsData* physicsData = static_cast<hkpPhysicsData*>( container->findObjectByType( hkpPhysicsDataClass.getName() ) );
		HK_ASSERT2(0x0, physicsData != HK_NULL, "Could not find physics data in root level object" );
		HK_ASSERT2(0x0, physicsData->getWorldCinfo() != HK_NULL, "No physics cinfo in loaded file - cannot create a hkpWorld" );

		for ( int i = 0; i < physicsData->getPhysicsSystems().getSize(); ++i )
		{
			hkpPhysicsSystem* physicsSystem = physicsData->getPhysicsSystems()[i];
			const hkArray<hkpRigidBody *>& bodies = physicsSystem->getRigidBodies();

			for (int rb = 0; rb < bodies.getSize();rb++)
			{
				hkpRigidBody* rigidBody = bodies[rb];

				if(rigidBody->isFixed())
				{
					const hkpShape* originalShape = static_cast<const hkpShape*> (rigidBody->getCollidable()->getShape());

					if(originalShape->getType() == hkcdShapeType::BOX)
					{
						//Make a new RB with a plane shape
						hkpRigidBody* groundRB = HK_NULL;
						hkpPlaneShape* plane = HK_NULL;
						{
							hkpRigidBodyCinfo groundInfo; rigidBody->getCinfo(groundInfo);
							{
								hkVector4 dir( 0,1,0);
								hkVector4 point( 0,0,0);
								hkVector4 extents( GROUND_SIZE*0.5f, GROUND_SIZE*0.5f, GROUND_SIZE*0.5f);
								plane = new hkpPlaneShape(dir, point, extents);
							}

							groundInfo.m_shape = plane;
							groundInfo.m_motionType = hkpMotion::MOTION_FIXED;
							groundInfo.m_friction = 0.5f;
							groundInfo.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo( GROUND_LAYER, 0 );

							groundRB = new hkpRigidBody(groundInfo);
							groundRB->setName(rigidBody->getName());
						}

						m_world->addEntity(groundRB);
						groundRB->removeReference();
						plane->removeReference();
						groundRigidBodiesToAdd.pushBack(groundRB);

					}
					else
					{
						m_world->addEntity(rigidBody);
						groundRigidBodiesToAdd.pushBack(rigidBody);
					}
					break;
				}
			}
		}
	}

	// Load the landscape display object and rigid bodies, turn on reflections
	{
		hkgDefaultSceneDataFilter filter;
		for(int rb=0; rb<groundRigidBodiesToAdd.getSize(); ++rb)
		{
			hkpRigidBody* groundRB = groundRigidBodiesToAdd[rb];
			filter.allow(groundRB->getName());
		}

		m_env->m_sceneConverter->convert(scene, hkgAssetConverter::CONVERT_ALL, true, &filter);



		for(int rb=0; rb<groundRigidBodiesToAdd.getSize(); ++rb)
		{
			hkpRigidBody* groundRB = groundRigidBodiesToAdd[rb];

			int idx = m_env->m_displayWorld->findDisplayObject(groundRB->getName());
			if(idx >= 0)
			{
				hkgDisplayObject* groundDisplayObject = m_env->m_displayWorld->getDisplayObject(idx);
				groundDisplayObject->computeAABB();

				if(enableReflection)
				{
					float reflectionPlane[] = { 0,1,0, 0.0f };
					m_env->m_displayWorld->setReflections(true, reflectionPlane);
					groundDisplayObject->setStatusFlags( (groundDisplayObject->getStatusFlags() | HKG_DISPLAY_OBJECT_REFLECTOR ) );
				}
				else
				{
					m_env->m_displayWorld->setReflections(false, HK_NULL);
				}

				// Altering state so should remove\add back display object?
				if(m_demoOptions.m_enableVehicleShadows || m_demoOptions.m_enablePropShadows)
				{
					groundDisplayObject->setStatusFlags( (groundDisplayObject->getStatusFlags() | HKG_DISPLAY_OBJECT_SHADOWRECEIVER) & ~HKG_DISPLAY_OBJECT_SHADOWCASTER  );
				}
				else
				{
					groundDisplayObject->setStatusFlags( (groundDisplayObject->getStatusFlags()) & ~(HKG_DISPLAY_OBJECT_SHADOW) );
				}

				m_env->m_displayWorld->removeDisplayObject( groundDisplayObject );
				m_env->m_displayWorld->addDisplayObject( groundDisplayObject );
				groundDisplayObject->removeReference(); // passed back by the removeDobj call


			}
		}
		// Create phantoms to keep track of the rigid bodies that fall off the ground
		{
			// North Phantom
			{
				hkAabb northAABB;
				northAABB.m_min = hkVector4(GROUND_SIZE*0.5f -1.f, -WORLD_SIZE*0.5f, -WORLD_SIZE*0.5f, 1.0f);
				northAABB.m_max = hkVector4(WORLD_SIZE*0.5f, WORLD_SIZE*0.5f, WORLD_SIZE*0.5f, 1.0f);

				ScenePhantom* northPhantom = new ScenePhantom(northAABB, hkpGroupFilter::calcFilterInfo (PHANTOM_LAYER) );
				m_world->addPhantom( northPhantom );
				northPhantom->removeReference();
			}

			// South Phantom
			{
				hkAabb southAABB;
				southAABB.m_min = hkVector4(-WORLD_SIZE*0.5f, -WORLD_SIZE*0.5f, -WORLD_SIZE*0.5f, 1.0f);
				southAABB.m_max = hkVector4(-GROUND_SIZE*0.5f + 1.f, WORLD_SIZE*0.5f, WORLD_SIZE*0.5f, 1.0f);

				ScenePhantom* southPhantom = new ScenePhantom(southAABB, hkpGroupFilter::calcFilterInfo (PHANTOM_LAYER) );
				m_world->addPhantom( southPhantom );
				southPhantom->removeReference();
			}

			// West Phantom
			{
				hkAabb westAABB;
				westAABB.m_min = hkVector4(-WORLD_SIZE*0.5f, -WORLD_SIZE*0.5f, -WORLD_SIZE*0.5f, 1.0f);
				westAABB.m_max = hkVector4(WORLD_SIZE*0.5f, WORLD_SIZE*0.5f, -GROUND_SIZE*0.5f + 1.f, 1.0f);

				ScenePhantom* westPhantom = new ScenePhantom(westAABB, hkpGroupFilter::calcFilterInfo (PHANTOM_LAYER) );
				m_world->addPhantom( westPhantom );
				westPhantom->removeReference();
			}

			// East Phantom
			{
				hkAabb eastAABB;
				eastAABB.m_min = hkVector4(-WORLD_SIZE*0.5f, -WORLD_SIZE*0.5f, GROUND_SIZE*0.5f - 1.f, 1.0f);
				eastAABB.m_max = hkVector4(WORLD_SIZE*0.5f, WORLD_SIZE*0.5f, WORLD_SIZE*0.5f, 1.0f);

				ScenePhantom* eastPhantom = new ScenePhantom(eastAABB, hkpGroupFilter::calcFilterInfo (PHANTOM_LAYER) );
				m_world->addPhantom( eastPhantom );
				eastPhantom->removeReference();
			}
		}
	}
}

VehicleStylesDemoBase::~VehicleStylesDemoBase( )
{
	m_world->lock();

	startGraphicsCodeSection();

	shutdownVDB();

	for(int v=0; v<m_vehiclesData.getSize(); ++v)
	{
		if(v == m_controlledVehicleID || !m_demoOptions.m_switchControlledVehicle)
		{
			m_vehiclesData[v]->m_vehicleInstance->removeFromWorld();
		}

		m_vehiclesData[v]->m_vehicleInstance->removeReference();
		delete m_vehiclesData[v];
	}

	if(m_vehicleManager != HK_NULL)
	{
		delete m_vehicleManager;
	}

	for(int r=0; r<m_reorientActions.getSize(); ++r)
	{
		m_reorientActions[r]->removeReference();
	}

	if (m_env->m_displayWorld)
	{
		for(int v=0; v<m_vehicleTypeDisplayData.getSize(); ++v)
		{
			if(m_vehicleTypeDisplayData[v] == HK_NULL)
				continue;

			for(int d=0; d<m_vehicleTypeDisplayData[v]->m_bodyDisplayObjects.getSize(); d++)
			{
				m_env->m_displayWorld->removeDisplayObject( m_vehicleTypeDisplayData[v]->m_bodyDisplayObjects[d] );
				m_vehicleTypeDisplayData[v]->m_bodyDisplayObjects[d]->removeReference();
			}
			for(int d=0; d<m_vehicleTypeDisplayData[v]->m_wheelDisplayObjects.getSize(); d++)
			{
				m_env->m_displayWorld->removeDisplayObject( m_vehicleTypeDisplayData[v]->m_wheelDisplayObjects[d] );
				m_vehicleTypeDisplayData[v]->m_wheelDisplayObjects[d]->removeReference();
			}

			delete m_vehicleTypeDisplayData[v];
		}
		m_vehicleTypeDisplayData.clear();
	}

	m_world->unlock();

	if(m_world != HK_NULL)
	{
		m_world->markForWrite();
		m_world->removeReference();
		m_world = HK_NULL;
	}
	if(m_loader != HK_NULL)
		delete m_loader;

	endGraphicsCodeSection();

	// Reset FOV
	m_env->m_window->getCurrentViewport()->getCamera()->setFOV(45.f);//hkgCamera::DEFAULT_FOV);
}

void VehicleStylesDemoBase::setupContexts(hkArray<hkProcessContext*>& contexts)
{
	if ( m_world )
	{
		m_world->markForWrite();
		if ( (m_physicsViewersContext->findWorld(m_world) < 0) )
		{
			m_physicsViewersContext->addWorld(m_world);
		}
		m_world->unmarkForWrite();
	}
	contexts.pushBack( m_physicsViewersContext );
	//m_debugViewerNames.pushBack(hkpPhantomDisplayViewer::getName());
	m_debugViewerNames.pushBack(hkDebugDisplayProcess::getName());
}

int VehicleStylesDemoBase::_dummyCallback()
{
	return 0;
}

void VehicleStylesDemoBase::_addVehicle(VehicleInstanceData* vehicleInstanceData, const hkTransform& initialTransform)
{
	m_world->lock();

	hkpRigidBody* chassisRB = vehicleInstanceData->m_vehicleInstance->getChassis();
	chassisRB->setLinearVelocity(hkVector4::getZero());
	chassisRB->setAngularVelocity(hkVector4::getZero());
	chassisRB->setTransform(initialTransform);

	vehicleInstanceData->m_simulated = true;
	vehicleInstanceData->m_vehicleInstance->addToWorld(m_world);
	m_world->addAction(vehicleInstanceData->m_vehicleInstance);

	m_world->unlock();
}

void VehicleStylesDemoBase::_removeVehicle(VehicleInstanceData* vehicleInstanceData)
{
	m_world->lock();
	vehicleInstanceData->m_vehicleInstance->removeFromWorld();
	m_world->unlock();

	vehicleInstanceData->m_simulated = false;

	// Set the num instanced display objects to zero for this vehicle type
	{
		int vehicleType = vehicleInstanceData->m_vehicleType;
		InstancedDisplayData* displayData = m_vehicleTypeDisplayData[vehicleType];
		for(int b=0; b<displayData->m_bodyDisplayObjects.getSize(); b++)
		{
			displayData->m_bodyDisplayObjects[b]->setNumObjects(0);
		}
		for(int w=0; w<displayData->m_wheelDisplayObjects.getSize(); w++)
		{
			displayData->m_wheelDisplayObjects[w]->setNumObjects(0);
		}
	}
}

void VehicleStylesDemoBase::_handleBaseKeyPresses()
{

	// Change camera
	if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_R2 ) || m_env->m_window->getKeyboard().wasKeyPressed( 'C' ))
	{
		// Cycle through the camera modes
		m_cameraMode = int((m_cameraMode + 1)%m_demoOptions.m_numCameras);
		if(m_cameraMode == NAVIGATE_CAMERA)
		{
			hkVector4 camFrom(-80.863f, 165.353f, 72.922f);
			hkVector4 camTo(-76.462f, 155.220f, 68.726f);
			hkVector4 camUp(0.f, 1.f, 0.f);
			setupDefaultCameras( m_env, camFrom, camTo, camUp );
		}
		if(m_cameraMode != PAN_CAMERA)
		{
			_setupCamera(m_vehiclesData[m_controlledVehicleID]->m_vehicleType);
		}
	}

	// Change which vehicle type is controlled
	if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_L2 ) || m_env->m_window->getKeyboard().wasKeyPressed( 'X' ))
	{
		if( (m_demoOptions.m_vehicleType == VehicleStylesBaseOptions::NUM_VEHICLE_TYPES) &&
			(m_demoOptions.m_switchControlledVehicle ) )
		{
			int prevControlledVehicleID = m_controlledVehicleID;
			const int totalNumVehicles = m_numVehiclesPerType*VehicleStylesBaseOptions::NUM_VEHICLE_TYPES;
			m_controlledVehicleID = (m_controlledVehicleID + m_numVehiclesPerType)%(totalNumVehicles);

			// Can't controll the trailer
			m_controlledVehicleID = (m_vehiclesData[m_controlledVehicleID]->m_vehicleType == VehicleStylesBaseOptions::BIG_RIG_TRAILER)?
				(m_controlledVehicleID + m_numVehiclesPerType)%totalNumVehicles : m_controlledVehicleID;

			_removeVehicle(m_vehiclesData[prevControlledVehicleID]);
			if(m_vehiclesData[prevControlledVehicleID]->m_vehicleType == VehicleStylesBaseOptions::BIG_RIG )
			{
				int trailerID = (prevControlledVehicleID + m_numVehiclesPerType)%totalNumVehicles;
				_removeVehicle(m_vehiclesData[trailerID]);
			}

			hkTransform initialTransform; initialTransform.setIdentity();
			initialTransform.setTranslation(hkVector4(-110.0f, 0.0f, 0.0f));
			initialTransform.setRotation(hkQuaternion::getIdentity());
			_addVehicle(m_vehiclesData[m_controlledVehicleID], initialTransform);

			if(m_vehiclesData[m_controlledVehicleID]->m_vehicleType == VehicleStylesBaseOptions::BIG_RIG )
			{
				hkTransform initialTrailerTransform; initialTrailerTransform.setIdentity();
				initialTrailerTransform.setTranslation(hkVector4(-110.0f - 5.9295f, 0.0f, 0.0f));
				initialTrailerTransform.setRotation(hkQuaternion::getIdentity());

				int trailerID = (m_controlledVehicleID + m_numVehiclesPerType)%totalNumVehicles;
				_addVehicle(m_vehiclesData[trailerID], initialTrailerTransform);
				_attachTrailers(true);
			}

			// Setup camera based on the vehicle we are controlling
			if(m_cameraMode != PAN_CAMERA)
			{
				_setupCamera(m_vehiclesData[m_controlledVehicleID]->m_vehicleType);
			}

		}
	}

	//
	// Reorient the vehicle if it has flipped
	//
	if( m_reorientActions.getSize() > 0 && m_world )
	{
		m_world->lock();
		VehicleApiUtils::reorient( (m_env->m_gamePad->isButtonPressed( HKG_PAD_BUTTON_R1 ) || m_env->m_window->getKeyboard().wasKeyPressed( 'R' )), m_reorientActions[m_controlledVehicleID], m_world );
		m_world->unlock();
	}

	if ( m_env->m_window->getKeyboard().wasKeyPressed( 'T' ) )
	{
		// Flip the tweak state
		m_tweaking = !m_tweaking;
	}

	// Tweaking
	if( m_tweaking )
	{
		bool strengthChanged = false;
		bool hardPointChanged = false;

		hkReal prevHardPoint = m_settings.m_suspensionHardPoint;

		extern const hkClass VehicleSettingsClass;

		const hkClass& klass = VehicleSettingsClass;
		void* tweakee = static_cast< void * >( &m_settings );

		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_DOWN ) )
		{
			m_tweakName = TweakerUtils::getNextSiblingPath( m_tweakName, &m_settings, klass );
		}

		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_UP ) )
		{
			m_tweakName = TweakerUtils::getPrevSiblingPath( m_tweakName, &m_settings, klass );
		}


		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_RIGHT ) )
		{
			hkBool neg = m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT );

			if(m_tweakName.endsWith("suspensionStrength"))
			{
				strengthChanged = true;
				float delta = 5.0f;
				TweakerUtils::tweakData( m_tweakName, tweakee, klass, !neg? delta : -delta, 1e-5f, TweakerUtils::ADDITIVE);

				m_settings.m_suspensionStrength = m_settings.m_suspensionStrength < 1.f ? 1.f : m_settings.m_suspensionStrength;
				m_settings.m_suspensionStrength = m_settings.m_suspensionStrength > 1000.f ? 1000.f : m_settings.m_suspensionStrength;
			}
			else if(m_tweakName.endsWith("suspensionHardPoint"))
			{
				hardPointChanged = true;

				float delta = 0.1f;
				TweakerUtils::tweakData( m_tweakName, tweakee, klass, !neg? delta : -delta, 1e-5f, TweakerUtils::ADDITIVE);

				m_settings.m_suspensionHardPoint = m_settings.m_suspensionHardPoint < -5.f ? -5.f : m_settings.m_suspensionHardPoint;
				m_settings.m_suspensionHardPoint = m_settings.m_suspensionHardPoint > 2.f ? 2.f : m_settings.m_suspensionHardPoint;
			}
		}

		TweakerUtils::DisplayOptions displayOptions(HK_NULL, 0, 0xFFFF8040);
		TweakerUtils::displayData( m_tweakName, tweakee, klass, *m_env->m_textDisplay, 10.0f, 120.0f, &displayOptions);

		if(strengthChanged || hardPointChanged)
		{
			hkpVehicleInstance* vehicleInstance = m_vehiclesData[m_controlledVehicleID]->m_vehicleInstance;
			hkpVehicleDefaultSuspension* suspension = static_cast< hkpVehicleDefaultSuspension*>((*vehicleInstance).m_suspension);
			if(strengthChanged)
			{
				suspension->m_wheelSpringParams[0].m_strength = m_settings.m_suspensionStrength;
				suspension->m_wheelSpringParams[1].m_strength = m_settings.m_suspensionStrength;
				suspension->m_wheelSpringParams[2].m_strength = m_settings.m_suspensionStrength;
				suspension->m_wheelSpringParams[3].m_strength = m_settings.m_suspensionStrength;

				if(vehicleInstance->m_wheelsInfo.getSize() == 6)
				{
					suspension->m_wheelSpringParams[4].m_strength = m_settings.m_suspensionStrength;
					suspension->m_wheelSpringParams[5].m_strength = m_settings.m_suspensionStrength;
				}
			}
			if(hardPointChanged)
			{
				hkVector4 hardPointPos;
				hardPointPos = suspension->m_wheelParams[0].m_hardpointChassisSpace;
				hardPointPos(1) += m_settings.m_suspensionHardPoint - prevHardPoint;
				suspension->m_wheelParams[0].m_hardpointChassisSpace.set ( hardPointPos(0), hardPointPos(1), hardPointPos(2));

				hardPointPos = suspension->m_wheelParams[1].m_hardpointChassisSpace;
				hardPointPos(1) += m_settings.m_suspensionHardPoint - prevHardPoint;
				suspension->m_wheelParams[1].m_hardpointChassisSpace.set ( hardPointPos(0), hardPointPos(1), hardPointPos(2));

				hardPointPos = suspension->m_wheelParams[2].m_hardpointChassisSpace;
				hardPointPos(1) += m_settings.m_suspensionHardPoint - prevHardPoint;
				suspension->m_wheelParams[2].m_hardpointChassisSpace.set ( hardPointPos(0), hardPointPos(1), hardPointPos(2));

				hardPointPos = suspension->m_wheelParams[3].m_hardpointChassisSpace;
				hardPointPos(1) += m_settings.m_suspensionHardPoint - prevHardPoint;
				suspension->m_wheelParams[3].m_hardpointChassisSpace.set ( hardPointPos(0), hardPointPos(1), hardPointPos(2));

				if((*vehicleInstance).m_wheelsInfo.getSize() == 6)
				{
					hardPointPos = suspension->m_wheelParams[4].m_hardpointChassisSpace;
					hardPointPos(1) += m_settings.m_suspensionHardPoint - prevHardPoint;
					suspension->m_wheelParams[4].m_hardpointChassisSpace.set ( hardPointPos(0), hardPointPos(1), hardPointPos(2));

					hardPointPos = suspension->m_wheelParams[5].m_hardpointChassisSpace;
					hardPointPos(1) += m_settings.m_suspensionHardPoint - prevHardPoint;
					suspension->m_wheelParams[5].m_hardpointChassisSpace.set ( hardPointPos(0), hardPointPos(1), hardPointPos(2));
				}
			}

		}
	}
}

hkDemo::Result VehicleStylesDemoBase::stepDemo()
{
	_handleBaseKeyPresses();

	{
		m_world->markForWrite();

		// Keep the vehicle activated.
		hkpVehicleInstance* vehicleInstace = m_vehiclesData[m_controlledVehicleID]->m_vehicleInstance;
		vehicleInstace->getChassis()->activate();

		// Don't steer if we are tweaking (using the same keys to handle both)
		if(!m_tweaking && m_cameraMode != VehicleStylesDemoBase::NAVIGATE_CAMERA)
		{
			// Steer the vehicle from user input.
			VehicleApiUtils::steer( m_env->m_gamePad, m_inputXPosition, m_inputYPosition, *vehicleInstace, m_timestep);
		}

		m_world->unmarkForWrite();
	}

	if(m_demoOptions.m_switchControlledVehicle)
	{
		for(int vId=0; vId<m_vehiclesData.getSize(); ++vId)
		{
			if(vId == m_controlledVehicleID)
				continue;

			if( (m_vehiclesData[m_controlledVehicleID]->m_vehicleType == VehicleStylesBaseOptions::BIG_RIG)
					&& (vId == m_controlledVehicleID + m_numVehiclesPerType) )
				continue;

			hkpVehicleInstance* vehicleInstance = m_vehiclesData[vId]->m_vehicleInstance;
			hkpVehicleDriverInputAnalogStatus* deviceStatus = (hkpVehicleDriverInputAnalogStatus*)vehicleInstance->m_deviceStatus;
			deviceStatus->m_positionY = 0;
			deviceStatus->m_positionX = 0;

			// Otherwise causes a problem with the truck
			if(m_vehiclesData[vId]->m_vehicleType != VehicleStylesBaseOptions::BIG_RIG_TRAILER)
				deviceStatus->m_handbrakeButtonPressed = 1;
		}
	}

	// Vehicle manager
	if(m_vehicleManager != HK_NULL)
	{
		VehicleStylesBaseOptions::VehicleManagerType vmType = m_demoOptions.m_vehicleManagerType;

		// Step info must be updated s.th. the vehicles current centreOfMass can be determined correctly
		hkStepInfo updatedStepInfo = m_world->m_dynamicsStepInfo.m_stepInfo;
		updatedStepInfo.m_startTime = updatedStepInfo.m_endTime;
		updatedStepInfo.m_endTime = updatedStepInfo.m_startTime + updatedStepInfo.m_deltaTime;

		if(vmType == VehicleStylesBaseOptions::SINGLETHREADED_RAY_CAST)
		{
			// Singlethreaded
			m_world->markForWrite();
			m_vehicleManager->stepVehicles( updatedStepInfo );
			m_world->unmarkForWrite();
		}
		else if(vmType == VehicleStylesBaseOptions::WHEEL_BATCHED_RAY_CAST)
		{
			hkpVehicleCastBatchingManager* batchingManager = static_cast<hkpVehicleCastBatchingManager*>( m_vehicleManager );

			hkLocalArray< hkpVehicleInstance* > activeVehicles( batchingManager->m_registeredVehicles.getSize() );
			batchingManager->getActiveVehicles( activeVehicles );

			// Batch wheel-cast jobs
			m_world->markForWrite();
			batchingManager->stepVehiclesSynchronously( m_world, updatedStepInfo, m_threadPool, m_jobQueue, m_numJobs, activeVehicles );
			m_world->unmarkForWrite();
		}
		else if(vmType == VehicleStylesBaseOptions::MULTITHREADED_RAY_CAST)
		{
			hkpMultithreadedVehicleManager* multithreadedManager = static_cast<hkpMultithreadedVehicleManager*>( m_vehicleManager );

			// Multithreaded vehicle manager
			multithreadedManager->stepVehiclesSynchronously( m_world, updatedStepInfo, m_threadPool, m_jobQueue, m_numJobs );
		}
	}


	//
	// Step the world.
	//
	{
		hkDefaultPhysics2012Demo::stepDemo();
	}

	// Update the display objects transform based on the vehicle's parts
	if(m_demoOptions.m_vehicleType != VehicleStylesBaseOptions::NUM_VEHICLE_TYPES)
	{
		hkVector4    pos;
		hkQuaternion rot;

		for(int vehicleID=0; vehicleID<m_vehiclesData.getSize(); ++vehicleID)
		{
			int displayObjectId = m_vehiclesData[vehicleID]->m_instancedDisplayID;
			int vehicleType = m_vehiclesData[vehicleID]->m_vehicleType;

			hkpVehicleInstance* vehicleInstance = m_vehiclesData[vehicleID]->m_vehicleInstance;
			hkpRigidBody* chassisRB = vehicleInstance->getChassis();
			hkpVehicleSuspension* suspension = vehicleInstance->m_suspension;
			hkTransform chassisTransform = chassisRB->getTransform();

			const int numWheelDisplayObjs = m_vehicleTypeDisplayData[vehicleType]->m_wheelDisplayObjects.getSize();
			for(int wheelIdx=0; wheelIdx<numWheelDisplayObjs; ++wheelIdx)
			{
				hkgInstancedDisplayObject* wheelMesh = m_vehicleTypeDisplayData[vehicleType]->m_wheelDisplayObjects[wheelIdx];
				vehicleInstance->calcCurrentPositionAndRotation(chassisRB, suspension, wheelIdx, pos, rot );
				rot.normalize();
				hkTransform trans(rot, pos);

				wheelMesh->setTransformAligned(trans, displayObjectId);
			}

			const int numChassisDisplayObjs = m_vehicleTypeDisplayData[vehicleType]->m_bodyDisplayObjects.getSize();
			for(int chassisIdx=0; chassisIdx<numChassisDisplayObjs; ++chassisIdx)
			{
				hkgInstancedDisplayObject* chassisMesh = m_vehicleTypeDisplayData[vehicleType]->m_bodyDisplayObjects[chassisIdx];
				chassisMesh->setTransformAligned(chassisTransform, displayObjectId);

			}

			//VehicleDisplayUtils::updateTyremarks( m_timestep, m_vehicleInstances[v] );
		}
	}
	else // Only render vehicles that are visible to the camera
	{
		hkVector4    pos;
		hkQuaternion rot;

		hkArray<int > numVisibleObjectsPerVehicle; numVisibleObjectsPerVehicle.setSize(m_vehicleTypeDisplayData.getSize(), 0);

		hkgCamera* cam = m_env->m_window->getViewport(0)->getCamera();

		for(int vehicleID=0; vehicleID<m_vehiclesData.getSize(); ++vehicleID)
		{
			int vehicleType = m_vehiclesData[vehicleID]->m_vehicleType;

			if(m_vehicleTypeDisplayData[vehicleType] == HK_NULL || !m_vehiclesData[vehicleID]->m_simulated)
				continue;

			hkpVehicleInstance* vehicleInstance = m_vehiclesData[vehicleID]->m_vehicleInstance;
			hkpRigidBody* chassisRB = vehicleInstance->getChassis();
			hkpVehicleSuspension* suspension = vehicleInstance->m_suspension;
			hkTransform chassisTransform = chassisRB->getTransform();
            HK_ALIGN_REAL(float chassisPos[4]); chassisTransform.getTranslation().store<4>(&chassisPos[0]);

			float sphereRadius = 2.0f;
			if(vehicleType == VehicleStylesBaseOptions::BIG_RIG)
			{
				sphereRadius = 3.0f;
			}
			else if(vehicleType == VehicleStylesBaseOptions::BIG_RIG_TRAILER)
			{
				sphereRadius = 5.0f;
			}

			if(cam->sphereVisible(chassisPos, sphereRadius))
			{
				const int numWheelDisplayObjs = m_vehicleTypeDisplayData[vehicleType]->m_wheelDisplayObjects.getSize();
				for(int wheelIdx=0; wheelIdx<numWheelDisplayObjs; ++wheelIdx)
				{
					hkgInstancedDisplayObject* wheelMesh = m_vehicleTypeDisplayData[vehicleType]->m_wheelDisplayObjects[wheelIdx];
					vehicleInstance->calcCurrentPositionAndRotation(chassisRB, suspension, wheelIdx, pos, rot );
					hkTransform trans(rot, pos);

					wheelMesh->setTransformAligned( trans, numVisibleObjectsPerVehicle[vehicleType] );
				}

				const int numChassisDisplayObjs = m_vehicleTypeDisplayData[vehicleType]->m_bodyDisplayObjects.getSize();
				for(int chassisIdx=0; chassisIdx<numChassisDisplayObjs; ++chassisIdx)
				{
					hkgInstancedDisplayObject* chassisMesh = m_vehicleTypeDisplayData[vehicleType]->m_bodyDisplayObjects[chassisIdx];
					chassisMesh->setTransformAligned(chassisTransform, numVisibleObjectsPerVehicle[vehicleType]);
				}

				numVisibleObjectsPerVehicle[vehicleType] += 1;
			}

			//VehicleDisplayUtils::updateTyremarks( m_timestep, m_vehicleInstances[v] );
		}

		// Set the number of instanced display objects based on the number of them that are visible
		for(int vehicleType=0; vehicleType<m_vehicleTypeDisplayData.getSize(); vehicleType++)
		{
			if(m_vehicleTypeDisplayData[vehicleType] == HK_NULL)
				continue;

			const int numWheelDisplayObjs = m_vehicleTypeDisplayData[vehicleType]->m_wheelDisplayObjects.getSize();
			for(int wheelIdx=0; wheelIdx<numWheelDisplayObjs; ++wheelIdx)
			{
				if(m_vehicleTypeDisplayData[vehicleType]->m_wheelDisplayObjects[wheelIdx]->getNumObjects() != numVisibleObjectsPerVehicle[vehicleType])
				{
					m_vehicleTypeDisplayData[vehicleType]->m_wheelDisplayObjects[wheelIdx]->setNumObjects(numVisibleObjectsPerVehicle[vehicleType]);
				}
			}

			const int numChassisDisplayObjs = m_vehicleTypeDisplayData[vehicleType]->m_bodyDisplayObjects.getSize();
			for(int chassisIdx=0; chassisIdx<numChassisDisplayObjs; ++chassisIdx)
			{
				if(m_vehicleTypeDisplayData[vehicleType]->m_bodyDisplayObjects[chassisIdx]->getNumObjects() != numVisibleObjectsPerVehicle[vehicleType])
				{
					m_vehicleTypeDisplayData[vehicleType]->m_bodyDisplayObjects[chassisIdx]->setNumObjects(numVisibleObjectsPerVehicle[vehicleType]);
				}
			}
		}
	}

	// Update the camera
	if(m_cameraMode == PAN_CAMERA)
	{
		m_env->m_window->getViewport(0)->setNavigationMode(HKG_CAMERA_NAV_DISABLED);
		_panCamera();
	}
	else if(m_cameraMode != NAVIGATE_CAMERA)
	{
		m_world->markForWrite();
		hkpVehicleInstance* vehicleInstace = m_vehiclesData[m_controlledVehicleID]->m_vehicleInstance;
		VehicleApiUtils::updateCamera( m_env, *vehicleInstace->getChassis(), m_timestep, m_camera);

		m_world->unmarkForWrite();
	}
	else
	{
		m_env->m_window->getViewport(0)->setNavigationMode(HKG_CAMERA_NAV_FLY);
	}

	// Only display the shadow of the first car
	if(m_demoOptions.m_enableVehicleShadows)
	{
		m_world->markForWrite();

		hkgAabb vehicleAabb;
		{
			hkAabb chassisAabb;
			{
				hkpVehicleInstance* vehicleInstance = m_vehiclesData[m_controlledVehicleID]->m_vehicleInstance;
				hkpRigidBody* chassisRB = vehicleInstance->getChassis();
				hkTransform chassisTransform = chassisRB->getTransform();

				chassisRB->getCollidable()->getShape()->getAabb( chassisTransform, 0.0f, chassisAabb);
			}

#ifdef HK_REAL_IS_DOUBLE
			{
				HK_ALIGN32(float) tmp[4];
				chassisAabb.m_min.store<4>(tmp);	vehicleAabb.addPoint(tmp);
				chassisAabb.m_max.store<4>(tmp);	vehicleAabb.addPoint(tmp);
			}
#else
			vehicleAabb.addPoint( (float*)&chassisAabb.m_min );
			vehicleAabb.addPoint( (float*)&chassisAabb.m_max );
#endif

			// Check if vehicle has a trailer and include it's aabb
			{
				hkBool hasTrailer = (m_vehiclesData[m_controlledVehicleID]->m_vehicleType == VehicleStylesBaseOptions::BIG_RIG)? true : false;
				int vehicleTrailerID = (m_controlledVehicleID + m_numVehiclesPerType)%m_vehiclesData.getSize();
				if(hasTrailer && m_vehiclesData[vehicleTrailerID]->m_vehicleType == VehicleStylesBaseOptions::BIG_RIG_TRAILER)
				{
					hkAabb trailerAabb;
					{
						hkpVehicleInstance* trailerInstance = m_vehiclesData[vehicleTrailerID]->m_vehicleInstance;
						hkpRigidBody* trailerRB = trailerInstance->getChassis();
						hkTransform trailerTransform = trailerRB->getTransform();

						trailerRB->getCollidable()->getShape()->getAabb( trailerTransform, 0.0f, trailerAabb);
					}

#ifdef HK_REAL_IS_DOUBLE
					{
						HK_ALIGN32(float) tmp[4];
						trailerAabb.m_min.store<4>(tmp);	vehicleAabb.addPoint(tmp);
						trailerAabb.m_max.store<4>(tmp);	vehicleAabb.addPoint(tmp);
					}
#else
					vehicleAabb.addPoint( (float*)&trailerAabb.m_min );
					vehicleAabb.addPoint( (float*)&trailerAabb.m_max );
#endif
				}
			}
		}


		m_world->unmarkForWrite();

		int numSplits = 0; //m_vehiclesData.getSize () > 1 ? 1 : 0;
		setupFixedShadowFrustum( m_env, *m_env->m_displayWorld->getLightManager()->getBestShadowCaster(), vehicleAabb, 2, 2, numSplits, /*y up*/ 1 );
	}

	if(m_vehiclesData[m_controlledVehicleID]->m_vehicleType != VehicleStylesBaseOptions::BIG_RIG)
	{
		hkpVehicleInstance* vehicleInstance = m_vehiclesData[m_controlledVehicleID]->m_vehicleInstance;
		updateTyremarks( m_timestep, vehicleInstance );
	}

	return DEMO_OK;
}

void VehicleStylesDemoBase::updateTyremarks( hkReal timestep, hkpVehicleInstance* vehicle )
{
	vehicle->m_tyreMarks->updateTyremarksInfo( timestep, vehicle );

	for (int wheel = 0; wheel < vehicle->m_data->m_numWheels; wheel++ )
	{
		const hkpTyremarksWheel &tyremarks_wheel = *vehicle->m_tyreMarks->m_tyremarksWheel[wheel];
		const int num_points = tyremarks_wheel.m_tyremarkPoints.getSize();

		const hkVector4* startl = &tyremarks_wheel.getTyremarkPoint(0).m_pointLeft;
		const hkVector4* startr = &tyremarks_wheel.getTyremarkPoint(0).m_pointRight;

		for (int p_it=1; p_it < num_points; p_it++)
		{
			const hkpTyremarkPoint* tyremark_point = &tyremarks_wheel.getTyremarkPoint(p_it);

			const hkVector4* endl = &tyremark_point->m_pointLeft;
			const hkVector4* endr = &tyremark_point->m_pointRight;

			hkReal alpha = (*startl)(3);
			if ( alpha >= 1.0f )
			{
				// Note: alpha display does not work with the havok graphics engine but
				// the next lines should be ok. To trim the alpha values you can use
				// m_minSkidmarkEnergy and m_maxSkidmarkEnergy.
				const long color = int( alpha ) <<24;
				HK_DISPLAY_TRIANGLE( *startl, *startr, *endr, color );
				HK_DISPLAY_TRIANGLE( *startl, *endl, *endr, color );
				HK_DISPLAY_TRIANGLE( *endr, *startr, *startl, color );
				HK_DISPLAY_TRIANGLE( *endr, *endl, *startl, color );

			}

			startl = endl;
			startr = endr;
		}
	}
}

void VehicleStylesDemoBase::_panCamera()
{
	// NB: This code is based on ClothAssetTesterDemo spin camera code
	m_world->markForWrite();
	m_env->m_window->getContext()->lock();

	// Get the current camera info
	hkgCamera* cam = m_env->m_window->getCurrentViewport()->getCamera();
	hkVector4 currentTo, currentFrom, currentUp, currentDir, dist;
	{
		float f[4];
		cam->getTo(f);
		currentTo.load4(f);
		cam->getFrom(f);
		currentFrom.load4(f);
		cam->getUp(f);
		currentUp.load4(f);
		cam->getDir(f);
		currentDir.load4(f);
	}

	// Get the first vehicle's aabb
	hkpVehicleInstance* vehicleInstace = m_vehiclesData[m_controlledVehicleID]->m_vehicleInstance;
	hkpRigidBody* chassisRB = vehicleInstace->getChassis();

	hkTransform chassisTrans = chassisRB->getTransform();
	hkAabb unionAabb; chassisRB->getCollidable()->getShape()->getAabb(chassisTrans, 0.0f, unionAabb);

	const hkSimdReal camDrive = 0.5f;
	hkReal camSpinRate = 0.05f;


	hkVector4 O;
	hkReal size = 2.f;
	unionAabb.m_min.setAdd4(hkVector4(-size,-size, -size), chassisTrans.getTranslation());
	unionAabb.m_max.setAdd4(hkVector4(size,size, size), chassisTrans.getTranslation());
	unionAabb.m_min(1) = 0.5f;
	unionAabb.m_max(1) = 1.5f;

	hkReal rangeFactor = 2.5f;
	if(m_vehiclesData[m_controlledVehicleID]->m_vehicleType == VehicleStylesBaseOptions::BIG_RIG)
	{
		unionAabb.m_max(1) = 2.0f;
		rangeFactor = 5.2f;
	}
	else if(m_vehiclesData[m_controlledVehicleID]->m_vehicleType == VehicleStylesBaseOptions::GOKART)
	{
		unionAabb.m_max(1) = 1.0f;
		rangeFactor = 1.5f;
	}

	O.setAdd4(unionAabb.m_min, unionAabb.m_max);
	O.mul4(0.5f);


	hkVector4 C = unionAabb.m_max;//chassisTrans.getTranslation();
	hkVector4 OC; OC.setSub4(C, O);
	hkReal camDistance = rangeFactor * (hkReal)OC.length3();

	hkVector4 F; F.setAddMul4(O, OC, rangeFactor);

	hkVector4 YVec(0,1,0);
	OC.normalize3IfNotZero();
	hkReal cosAzimuth = OC.dot3(YVec);
	const hkReal AZIMUTH_TOL = 5.0e-2f;
	if (cosAzimuth<AZIMUTH_TOL) cosAzimuth = AZIMUTH_TOL;
	if (cosAzimuth>1.0f-AZIMUTH_TOL) cosAzimuth = 1.0f-AZIMUTH_TOL;

	hkReal sinAzimuth = hkMath::sqrt( 1.0f - cosAzimuth * cosAzimuth );


	hkReal phi = 2.0f * HK_REAL_PI * camSpinRate * m_env->m_frameTimer.getElapsedSeconds();
	hkReal X = camDistance * sinAzimuth * cos(phi);
	hkReal Y = camDistance * cosAzimuth;
	hkReal Z = camDistance * sinAzimuth * sin(phi);
	hkVector4 desiredFrom(X, Y, Z);
	desiredFrom.add4(O);

	hkVector4 desiredTo = O;
	hkVector4 desiredUp (0,1,0);

	hkVector4 diffTo, diffFrom, diffUp;
	diffTo.setSub4(desiredTo, currentTo);
	diffFrom.setSub4(desiredFrom, currentFrom);
	diffUp.setSub4(desiredUp, currentUp);

	desiredTo.setAddMul4(currentTo, diffTo, camDrive);
	desiredFrom.setAddMul4(currentFrom, diffFrom, camDrive);
	desiredUp.setAddMul4(currentUp, diffUp, camDrive);

	cam->setTo((hkReal*) &desiredTo);
	cam->setFrom((hkReal*) &desiredFrom);
	cam->setUp((hkReal*) &desiredUp);


	cam->setFOV(60.f);
	cam->computeModelView();
	cam->computeProjection();
	cam->orthogonalize();

	m_env->m_window->getContext()->unlock();
	m_world->unmarkForWrite();
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
