/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Vehicle/Wheel/WheelVehicleDemo.h>
#include <Demos/Physics2012/Api/Vehicle/Wheel/UnicycleSetup.h>
#include <Demos/Physics2012/Api/Vehicle/Wheel/FrontSetup.h>
#include <Demos/Physics2012/Api/Vehicle/Wheel/BackSetup.h>

#include <Demos/DemoCommon/Utilities/VehicleApi/VehicleSetup.h>
#include <Demos/DemoCommon/Utilities/VehicleApi/VehicleApiUtils.h>
#include <Demos/DemoCommon/Utilities/GameUtils/Landscape/FlatLand.h>
#include <Demos/DemoCommon/Utilities/GameUtils/Landscape/RoundTrack.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>

#include <Physics/Constraint/Data/Fixed/hkpFixedConstraintData.h>
#include <Physics2012/Collide/Shape/HeightField/SampledHeightField/hkpSampledHeightFieldShape.h>
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastOutput.h>
#include <Physics2012/Collide/Shape/Convex/Cylinder/hkpCylinderShape.h>
#include <Physics2012/Dynamics/World/hkpPhysicsSystem.h>
#include <Physics2012/Vehicle/Camera/hkp1dAngularFollowCam.h>
#include <Physics2012/Vehicle/hkpVehicleInstance.h>
#include <Physics2012/Vehicle/Brake/hkpVehicleBrake.h>
#include <Physics2012/Vehicle/DriverInput/hkpVehicleDriverInput.h>
#include <Physics2012/Vehicle/Transmission/hkpVehicleTransmission.h>
#include <Physics2012/Vehicle/Wheel/hkpRaycastWheel.h>
#include <Physics2012/Vehicle/Wheel/hkpLinearCastWheel.h>

#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Visualize/Shape/hkDisplayGeometry.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/Material/hkgMaterial.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>

extern const hkClass WheelVehicleDemoTweakerOptionsClass;
WheelVehicleDemoTweakerOptions WheelVehicleDemo::s_tweakerOptions;

//
// Demo variants setup
//

struct WheelVehicleDemoVariant
{
	const char* m_name;
	const char* m_details;
	enum
	{
		CAR,
		UNICYCLE,
		ARTICULATED
	} m_vehicleType;
};

static const WheelVehicleDemoVariant WheelVehicleDemoVariants[] =
{
	{ "Car", "Drive a car", WheelVehicleDemoVariant::CAR },
	{ "Unicycle", "Drive a unicycle", WheelVehicleDemoVariant::UNICYCLE },
	{ "Articulated", "Drive an articulated vehicle", WheelVehicleDemoVariant::ARTICULATED }
};

WheelVehicleDemo::WheelVehicleDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env)
{
	m_mouseTweakerSettings.m_tweakee = static_cast< void * >( &s_tweakerOptions );
	m_mouseTweakerSettings.m_klass = &WheelVehicleDemoTweakerOptionsClass;
	m_mouseTweakerSettings.m_demoEnvironment = m_env;

	m_track = HK_NULL;
	m_bootstrapIterations = 150;

	setupWorld();
	setupVehicles();

	VehicleApiUtils::createCamera( m_camera );
}

void WheelVehicleDemo::setupVehicles()
{
	m_world->lock();

	int vehicleType = WheelVehicleDemoVariants[m_variantId].m_vehicleType;
	switch ( vehicleType )
	{
		case WheelVehicleDemoVariant::CAR:
		case WheelVehicleDemoVariant::UNICYCLE:
			createCar();
			break;

		case WheelVehicleDemoVariant::ARTICULATED:
			createArticulated();
			break;
	}

	//
	// Final vehicle setup common to all variants
	//

	for (int vehicleId = 0; vehicleId < m_vehicles.getSize(); vehicleId++)
	{
		m_vehicles[vehicleId].m_lastRPM = 0.0f;
		static_cast<hkpVehicleSteeringAckerman*>(m_vehicles[vehicleId].m_vehicle->m_steering)->m_maxSpeedFullSteeringAngle = HK_REAL_MAX;

		// Set the body's inertia
		hkpRigidBody* chassis = m_vehicles[vehicleId].m_vehicle->getChassis();
		if (vehicleType == WheelVehicleDemoVariant::UNICYCLE)
		{
			// Give infinite inertia to the unicycle
			hkMatrix3 invInertia;
			invInertia.setZero();
			chassis->setInertiaInvLocal(invInertia);
		}
		else
		{
			// Calculate box inertia
			hkAabb chassisAabb; chassis->getCollidable()->getShape()->getAabb(hkTransform::getIdentity(), 0.0f, chassisAabb);
			hkVector4 chassisHalfExtents; chassisAabb.getHalfExtents(chassisHalfExtents);
			hkMassProperties chassisMassProperties;
			hkInertiaTensorComputer::computeBoxVolumeMassProperties(chassisHalfExtents, chassis->getMass(), chassisMassProperties);
			chassis->setInertiaLocal(chassisMassProperties.m_inertiaTensor);
		}

		HK_SET_OBJECT_COLOR((hkUlong)m_vehicles[vehicleId].m_vehicle->getChassis()->getCollidable(), hkColor::semiTransparent(hkColor::LIGHTGREY));
	}

	createDisplayWheels();

	m_world->unlock();
}

void WheelVehicleDemo::createArticulated()
{
	FrontSetup frontSetup;
	BackSetup backSetup;
	VehicleSetup* setups[] = { &frontSetup, &backSetup };

	hkVector4 halfExtents[] = { hkVector4(0.75f, 0.75f, 1.25f), hkVector4(1.5f, 0.75f, 1.25f) };
	hkVector4 positions[] = { hkVector4(0.0f, 0.0f, 0.0f), hkVector4(-2.5f, 0.0f, 0.0f) };
	hkpRigidBody* bodies[2];

	for (int i = 0; i < 2; i++)
	{
		// Add to position
		positions[i].add(hkVector4(-3.f, 0.5f, 0.f));

		hkpRigidBodyCinfo chassisInfo;
		chassisInfo.m_motionType = hkpMotion::MOTION_BOX_INERTIA;
		chassisInfo.m_mass = 750.0f;
		chassisInfo.m_collisionFilterInfo = m_chassisLayer;
		chassisInfo.m_enableDeactivation = false;
		chassisInfo.m_position = positions[i];
		chassisInfo.m_shape = new hkpBoxShape(halfExtents[i]);
		
		// Lower the center of mass
		chassisInfo.m_centerOfMass = hkVector4(0.0f, -halfExtents[i](1) / 2.0f, 0.0f);

		bodies[i] = new hkpRigidBody(chassisInfo);
		chassisInfo.m_shape->removeReference();

		createVehicle( *setups[i], bodies[i]);
		bodies[i]->removeReference();
	}

	//Pivot in the center of the space between front and back
	hkVector4 pivot = hkVector4(((positions[0](0) - halfExtents[0](0)) + (positions[1](0) + halfExtents[1](0))) / 2.0f, positions[1](1), positions[0](2));

	// Simple articulated vehicle, not really articulated :) lock the bodies together. 
	hkpFixedConstraintData* fixedData = new hkpFixedConstraintData();
	hkTransform pivotTransform(hkQuaternion::getIdentity(), pivot);
	fixedData->setInWorldSpace(bodies[0]->getTransform(), bodies[1]->getTransform(), pivotTransform);

	hkpConstraintInstance* instance = new hkpConstraintInstance(bodies[0], bodies[1], fixedData);
	fixedData->removeReference();

	m_world->addConstraint(instance);
	instance->removeReference();
}

void WheelVehicleDemo::createCar()
{
	VehicleSetup carSetup;
	UnicycleSetup unicycleSetup;
	VehicleSetup* setup = HK_NULL;

	int vehicleType = WheelVehicleDemoVariants[m_variantId].m_vehicleType;

	hkpRigidBodyCinfo chassisInfo;
	chassisInfo.m_mass = 750.0f;	
	chassisInfo.m_motionType = hkpMotion::MOTION_BOX_INERTIA;
	chassisInfo.m_gravityFactor = 1.5f;
	chassisInfo.m_collisionFilterInfo = m_chassisLayer;
	chassisInfo.m_enableDeactivation = false;
	chassisInfo.m_position.set(-10.0f, -4.f, 3.0f);

	// Set up parameters that vary between vehicle types
	switch (vehicleType)
	{
	case WheelVehicleDemoVariant::CAR:
		chassisInfo.m_shape = VehicleApiUtils::createCarChassisShape();
		chassisInfo.m_centerOfMass.set( 0.0f, -0.1f, 0.0f);	// Lower the center of mass
		setup = &carSetup;
		break;
	case WheelVehicleDemoVariant::UNICYCLE:
		chassisInfo.m_shape = new hkpBoxShape(hkVector4(0.5f, 0.5f, 0.5f));
		setup = &unicycleSetup;
		break;
	default:
		break;
	}

	hkpRigidBody* chassisRigidBody = new hkpRigidBody(chassisInfo);
	chassisInfo.m_shape->removeReference();

	createVehicle( *setup, chassisRigidBody);
	chassisRigidBody->removeReference();
}

WheelVehicleDemo::~WheelVehicleDemo( )
{
	m_world->lock();

	for (int vehicleId = 0; vehicleId < m_vehicles.getSize(); vehicleId++ )
	{
		// Check size of each element as not all vehicles have same number of wheels.
		for (int wheelNum = 0; wheelNum < m_displayWheelId[vehicleId].getSize(); wheelNum++)
		{
			m_env->m_displayHandler->removeGeometry(m_displayWheelId[vehicleId][wheelNum], m_tag, 0);
			m_wheels[vehicleId][wheelNum]->removeReference();
		}

		m_vehicles[vehicleId].m_vehicle->removeFromWorld();
		m_vehicles[vehicleId].m_vehicle->removeReference();
	}

	// Release the underlying geometry data.
	// Note: normally this data is owned by the graphics system (not in our case though).
	if (m_track)
	{
		m_track->removeReference();
	}

	m_world->unlock();
}

void WheelVehicleDemo::makeFakeInput()
{
	m_env->m_gamePad->forceButtonPressed(HKG_PAD_DPAD_UP);
	m_env->m_gamePad->forceButtonPressed(HKG_PAD_DPAD_RIGHT);
}

///[stepGame]
/// This is called every simulation timestep. We need to
/// - Steer the first vehicle based on user input.
/// - Step the simulation.
/// - Sync each vehicle's display wheels.
/// - Update the camera that follows the first vehicle.
/// - Draw skidmarks for the first vehicle if it is skidding.
/// - Update the RPM meter and speedometer.
///<
hkDemo::Result WheelVehicleDemo::stepDemo()
{
	DemoMouseTweaker::tweak(m_mouseTweakerSettings);
	
	m_world->lock();

	hkStepInfo stepInfo;
	stepInfo.set(m_world->getCurrentPsiTime(), m_world->getCurrentPsiTime() + m_timestep);

	// Steer the vehicle from user input.
	steer();

	for (int i = 0; i < m_vehicles.getSize(); i++)
	{
		hkArray<hkpWheel*>& wheels = m_wheels[i];
		hkpVehicleInstance* vehicle = m_vehicles[i].m_vehicle;

		//
		// Update the wheels and pass collision results to the other components
		//

		for (int j = 0; j < wheels.getSize(); j++)
		{
			hkpWheel* wheel = wheels[j];
			wheel->update( stepInfo );

			hkpVehicleInstance::WheelInfo& wheelInfo = m_vehicles[i].m_vehicle->m_wheelsInfo[j];

			hkVector4 contactPosition = wheel->getContactPosition();
			hkVector4 contactNormal = wheel->getContactNormal();

			wheelInfo.m_hardPointWs = wheel->getHardPoint();
			wheelInfo.m_contactPoint.set(contactPosition, contactNormal, 0.0f);
			wheelInfo.m_noSlipIdealSpinVelocity = wheel->m_noSlipIdealSpinVelocity;
			wheelInfo.m_forwardSlipVelocity = wheel->m_forwardSlipVelocity;
			wheelInfo.m_skidEnergyDensity = wheel->m_skidEnergyDensity;
			wheelInfo.m_sideForce = wheel->m_sideForce;
			wheelInfo.m_spinVelocity = wheel->getSpinVelocity();
			wheelInfo.m_spinAngle = wheel->m_spinAngle;
		}

		//
		// Update other components to compute steering and torque
		//

		hkpVehicleBrake::WheelBreakingOutput wheelBreakingInfo;
		hkpVehicleDriverInput::FilteredDriverInputOutput filteredDriverInputInfo;
		hkpVehicleAerodynamics::AerodynamicsDragOutput aerodynamicsDragInfo;
		hkpVehicleTransmission::TransmissionOutput transmissionInfo;
		transmissionInfo.m_wheelsTransmittedTorque = hkAllocateStack<hkReal>( wheels.getSize() );

		vehicle->updateDriverInput( m_timestep, filteredDriverInputInfo );
		vehicle->updateSteering( m_timestep, filteredDriverInputInfo );
		vehicle->updateTransmission( m_timestep, transmissionInfo );
		vehicle->updateEngine( m_timestep, filteredDriverInputInfo, transmissionInfo );
		vehicle->updateBrake( m_timestep, filteredDriverInputInfo, wheelBreakingInfo );
		vehicle->updateAerodynamics( m_timestep, aerodynamicsDragInfo );

		//
		// Update wheels with steering and torque computed by the other components
		//

		for (int j = 0; j < wheels.getSize(); j++)
		{
			hkpWheel* wheel = wheels[j];
			wheel->m_torque = wheelBreakingInfo.m_brakingTorque[j] + transmissionInfo.m_wheelsTransmittedTorque[j];
			wheel->m_steeringAngle = vehicle->m_wheelsSteeringAngle[j];
			wheel->m_axle->m_isFixed = vehicle->m_isFixed[j];

			wheel->preStep( stepInfo );
		}

		hkDeallocateStack(transmissionInfo.m_wheelsTransmittedTorque, wheels.getSize());
	}

	m_world->unlock();

	//
	// Step the world.
	//
	{
		hkDefaultPhysics2012Demo::stepDemo();
	}

	m_world->markForWrite();

	for (int vehicleId = 0; vehicleId < m_vehicles.getSize(); vehicleId++ )
	{
		//
		// Sync wheels. This is necessary only because they do not "exist" as bodies in simulation,
		// and so are not automatically updated by the current display. We must explicity tell the
		// display that some other "display obects" (objects which are drawn but are not physical)
		// have moved.
		//
		hkArray<hkpWheel*>& wheels = m_wheels[vehicleId];
		for (int i = 0; i < wheels.getSize(); i++)
		{
			hkpWheel* wheel = wheels[i];

			// Post-step update each wheel
			wheel->postStep( stepInfo );

			hkpVehicleInstance::WheelInfo& wheelInfo = m_vehicles[vehicleId].m_vehicle->m_wheelsInfo[i];
			wheelInfo.m_skidEnergyDensity = wheel->m_skidEnergyDensity;

			// Sync the display wheel
			hkVector4    pos;
			hkQuaternion rot;

			//
			// XXX Check if this is the same value as the m_hardPointWS in wheelsInfo
			//
			wheels[i]->calcCurrentPositionAndRotation( pos, rot );
			hkTransform trans(rot, pos);
			m_env->m_displayHandler->updateGeometry(trans, m_displayWheelId[vehicleId][i], m_tag);
		}
	}

	// Update the "follow" camera.
	VehicleDisplayUtils::VehicleDataAndDisplayInfo& playerVehicle = m_vehicles[0];
	VehicleApiUtils::updateCamera( m_env, *playerVehicle.m_vehicle->getChassis(), m_timestep, m_camera, 0.01f);
	VehicleDisplayUtils::updateTyremarks( m_timestep, playerVehicle.m_vehicle );
	VehicleDisplayUtils::updateInfo( m_env, m_vehicles[0] );

	m_world->unmarkForWrite();

	return hkDemo::DEMO_OK;
}

void WheelVehicleDemo::steer()
{
	//
	// Activate chassis if any of the driving/reorientation keys have been pressed
	//
	if(m_env->m_gamePad->isButtonPressed(HKG_PAD_BUTTON_0)
		|| m_env->m_gamePad->isButtonPressed(HKG_PAD_BUTTON_2)
		|| m_env->m_gamePad->isButtonPressed(HKG_PAD_BUTTON_3)
		|| m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_UP)
		|| m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_DOWN)
		|| m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_RIGHT)
		|| m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_LEFT)
		|| m_env->m_gamePad->isButtonPressed(HKG_PAD_BUTTON_R2)
		)
	{
		m_vehicles[0].m_vehicle->getChassis()->activate();
	}

	// Steer the vehicle.

	VehicleApiUtils::steer( m_env->m_gamePad,
							m_inputXPosition,
							m_inputYPosition,
						   *m_vehicles[0].m_vehicle,
							m_timestep);
}


class CarSampledHeightFieldShape: public hkpSampledHeightFieldShape
{
	public:
		CarSampledHeightFieldShape( const hkpSampledHeightFieldBaseCinfo& ci )
			: hkpSampledHeightFieldShape(ci)
		{
		}

		HK_FORCE_INLINE hkReal getHeightAtImpl( int x, int z ) const
		{
			HK_ASSERT(0x31f89011,  x>=0 && x < m_xRes );
			HK_ASSERT(0x4c32b1bc,  z>=0 && z < m_zRes );
			if (0)
			{
				hkReal sx = hkMath::sin( x * 0.8f );
				hkReal sy = hkMath::sin( z * 0.99f );

				return sx + sy;
			}
			return (x == m_xRes/2) && (z == m_zRes/2);
		}

		/// this should return true if the two triangles share the edge p00-p11
		///      otherwise it should return false if the triangles shere the edge p01-p10
		HK_FORCE_INLINE hkBool getTriangleFlipImpl() const
		{	
			return false;
		}

		virtual void collideSpheres( const CollideSpheresInput& input, SphereCollisionOutput* outputArray) const
		{
			hkSampledHeightFieldShape_collideSpheres(*this, input, outputArray);
		}
};


void WheelVehicleDemo::buildLandscape()
{
	// Setup some materials to use
	hkgMaterial* floorMaterial;
	{
		hkgDisplayContext* ctx = m_env->m_window->getContext();

		floorMaterial = hkgMaterial::create();
		floorMaterial->setDiffuseColor(1.0f, 1.0f, 1.0f, 1.0f);
		floorMaterial->setSpecularColor(1.0f, 0.7f, 0.7f);
		floorMaterial->setSpecularPower(10);
		hkgTexture* texture = hkgTexture::create(ctx);			
		if ( !texture->loadFromFile("./Resources/Animation/Showcase/Level/bricks1.png" ) )
		{
			HK_REPORT( "Failed to load texture from file.\n" );
			texture->loadBuiltIn( HKG_TEXTURE_PERLINNOISE );
		}
		texture->setAutoMipMaps(true);
		texture->realize();
		floorMaterial->addTexture(texture);
		texture->release();

		m_env->m_displayHandler->setTextureMapScale(0.5f);
		m_env->m_displayHandler->setTextureMapLocalMode(false);
	}

	if (s_tweakerOptions.m_landscape)
	{
		//
		// Create a MOPP landscape
		//
		{
			hkpRigidBodyCinfo groundInfo;


			//
			//	Set the if condition to 0 if you want to test the heightfield
			//
			if ( 1 )
			{
				FlatLand* fl = new FlatLand();
				m_track = fl;
				groundInfo.m_shape = fl->createMoppShapeForSpu();
				groundInfo.m_position.set(5.0f, -2.0f, 5.0f);
			}
			else
			{
				hkpSampledHeightFieldBaseCinfo ci;
				ci.m_xRes = 64;
				ci.m_zRes = 64;
				ci.m_scale.set( 4.0f, 1.5f, 4.0f );
				m_track = HK_NULL;
				groundInfo.m_shape = new CarSampledHeightFieldShape( ci );
				groundInfo.m_position.set(-0.5f * ci.m_xRes * ci.m_scale(0), -2.0f, -0.5f * ci.m_zRes * ci.m_scale(2));
			}

			{
				groundInfo.m_motionType = hkpMotion::MOTION_FIXED;
				groundInfo.m_friction = 0.5f;
				hkpRigidBody* groundbody = new hkpRigidBody(groundInfo);
				m_world->addEntity(groundbody);
				groundbody->removeReference();

				hkgDisplayObject* dispObj = m_env->m_displayHandler->findDisplayObject( (hkUlong)groundbody->getCollidable() );
				if (dispObj)
				{
					dispObj->setMaterialOnAll(floorMaterial);
				}
			}

			groundInfo.m_shape->removeReference();
		}

		if (1)
		{
			hkVector4 halfExtents(0.5f, 0.5f, 0.5f);
			hkVector4 startPos(-240.0f, -7.0f, 0.0f);
			hkVector4 diffPos (30.0f, 0.0f, 0.0f);
			createDodgeBoxes(5, halfExtents, startPos, diffPos);
		}

		if (1)
		{
			int gridSize = 1 + int(hkMath::sqrt( hkReal(m_env->m_cpuMhz/100) ));
			createRagdollGrid( m_world, gridSize, gridSize, 4.0f, 4.0f );
		}

		if (1)
		{
			// Make a low friction ice body
			hkpRigidBodyCinfo icyBoxInfo;
			icyBoxInfo.m_position.set(5.0f, -2.0f, 5.0f);
			icyBoxInfo.m_motionType = hkpMotion::MOTION_FIXED;
			icyBoxInfo.m_friction = 0.01f;	
			hkVector4 halfExtentsBox(30.0f, 2.0f, 30.0f);	
			icyBoxInfo.m_shape = new hkpBoxShape(halfExtentsBox, 0 );
			icyBoxInfo.m_position.set(-120.0f, -9.90f, 130.0f);

			hkpRigidBody* rb = new hkpRigidBody(icyBoxInfo);
			m_world->addEntity(rb);
			rb->removeReference();

			// Color so we can see it
			HK_SET_OBJECT_COLOR((hkUlong)rb->getCollidable(), hkColor::rgbFromChars(200, 200, 255));	

			icyBoxInfo.m_shape->removeReference();
		}
	}
	else
	{
		// Build a simple, flat environment
		hkpRigidBodyCinfo floorInfo;
		floorInfo.m_shape = new hkpBoxShape(hkVector4(400.f, 1.f, 400.f));
		floorInfo.m_motionType = hkpMotion::MOTION_FIXED;
		floorInfo.m_position.set(-5.f, -7.f, 0.0f);
		for (int i = 0; i < 1 + (s_tweakerOptions.m_hill ? 1 : 0); i++)
		{
			if (i)
			{
				floorInfo.m_rotation.setAxisAngle(hkVector4(0, 0, 1), 0.2f);
			}
			hkpRigidBody* body = new hkpRigidBody(floorInfo);
			m_world->addEntity(body);

			hkgDisplayObject* dispObj = m_env->m_displayHandler->findDisplayObject( (hkUlong)body->getCollidable() );
			if (dispObj)
			{
				dispObj->setMaterialOnAll(floorMaterial);
			}

			body->removeReference();
		}
		floorInfo.m_shape->removeReference();
	}

	floorMaterial->removeReference();
}

void WheelVehicleDemo::createDodgeBoxes(hkUint16 num, hkVector4& halfExtents, hkVector4& startPos, hkVector4& diffPos)
{
	for(int i = 0; i < num; i++)
	{
		/////////////////// SHAPE CONSTRUCTION ////////////////
		hkpBoxShape* shape = new hkpBoxShape(halfExtents);
		///////////////////////////////////////////////////////


		hkVector4 gapp;
		gapp.setMul4((hkReal)i, diffPos);
		
		// To illustrate using the shape, create a rigid body by first defining a template.
		hkpRigidBodyCinfo rigidBodyInfo;

		rigidBodyInfo.m_shape = shape;
		rigidBodyInfo.m_position.setAdd4(startPos, gapp);
		rigidBodyInfo.m_angularDamping = 0.0f;
		rigidBodyInfo.m_linearDamping = 0.0f;

		rigidBodyInfo.m_motionType = hkpMotion::MOTION_FIXED;

		// If we set this to true, the body is fixed, and no mass properties need to be computed.
		//rigidBodyInfo.m_motionType = hkpMotion::MOTION_BOX_INERTIA;

		// If we need to compute mass properties, we'll do this using the hkpInertiaTensorComputer.
		if (rigidBodyInfo.m_motionType != hkpMotion::MOTION_FIXED)
		{
			hkReal mass = 10.0f;
			hkMassProperties massProperties;
			hkpInertiaTensorComputer::computeBoxVolumeMassProperties(halfExtents, mass, massProperties);

			rigidBodyInfo.m_inertiaTensor = massProperties.m_inertiaTensor;
			rigidBodyInfo.m_centerOfMass = massProperties.m_centerOfMass;
			rigidBodyInfo.m_mass = massProperties.m_mass;			
		}	
			
		 
		// Create a rigid body (using the template above).
		hkpRigidBody* rigidBody = new hkpRigidBody(rigidBodyInfo);

		// Remove reference since the body now "owns" the Shape.
		shape->removeReference();

		// Finally add body so we can see it, and remove reference since the world now "owns" it.
		m_world->addEntity(rigidBody);
		rigidBody->removeReference();
	}

}

//
// Create a grid of ragdolls
//
void WheelVehicleDemo::createRagdollGrid( hkpWorld* world, int x_size, int y_size, hkReal xStep, hkReal yStep)
{
	int systemGroup = 2;
	hkReal ragdollHeight = 2.50f;
	
	for( int x = 0; x < x_size; x++ )
	{
		for( int y = 0; y < y_size; y++ )
		{
			hkVector4 position; 
			// do a raycast to place the ragdoll
			{
				hkpWorldRayCastInput ray;
				ray.m_from.set( x * xStep,  10, y * yStep );
				ray.m_to.  set( x * xStep, -10, y * yStep );
				hkpWorldRayCastOutput result;
				world->castRay( ray, result );
				position.setInterpolate4( ray.m_from, ray.m_to, result.m_hitFraction );
				position(1) += ragdollHeight* 0.5f;
			}

			hkQuaternion	rotation; rotation.setIdentity();

			rotation.setAxisAngle( hkTransform::getIdentity().getColumn(0), HK_REAL_PI  * -0.5f );
			
			hkpPhysicsSystem* ragdoll = GameUtils::createRagdoll( ragdollHeight, position, rotation, systemGroup, GameUtils::RPT_CAPSULE );
			{
				for ( int i = 0; i < ragdoll->getRigidBodies().getSize(); i++)
				{
					hkpRigidBody* rb = ragdoll->getRigidBodies()[i];
					rb->setCollisionFilterInfo( hkpGroupFilter::calcFilterInfo( m_collisionLayer, systemGroup ) );
				}
			}
			systemGroup++;
	
			world->addPhysicsSystem(ragdoll);
			ragdoll->getRigidBodies()[0]->deactivate();
			ragdoll->removeReference();
		}
	}
}

void WheelVehicleDemo::setupWorld()
{
	m_tag = 0;

	// Initially "controller" is at (0,0), ie. neither pointing left/right nor up/down.
	m_inputXPosition = 0.0f;
	m_inputYPosition = 0.0f;

	//
	// Setup the camera. Actually overwritten by step function, and when we first add the vehicle.
	//
	{
		hkVector4 from(0.0f, 0.0f, 10.0f);
		hkVector4 to(0.0f, 0.0f, 0.0f);
		hkVector4 up(0.0f, 1.0f, 0.0f);
		setupDefaultCameras( m_env, from, to, up );
	}

	//
	// Create the world.
	//
	{
		hkpWorldCinfo info;
		info.m_collisionTolerance = 0.1f;
		info.m_gravity.set(0.0f, -9.8f, 0.0f);
		info.setBroadPhaseWorldSize(2050.0f) ;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM);
		info.m_solverMicrosteps = 1;
		m_world = new hkpWorld( info );
		m_world->lock();

		// Register all agents.
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
		// Graphics.
		setupGraphics();
	}

	//
	// Create a filter, so that the raycasts of car does not collide with the ragdolls
	// and linear casts don't collide with the chassis
	//
	{
		hkpGroupFilter* filter = new hkpGroupFilter();
		filter->disableCollisionsBetween(m_collisionLayer, m_castLayer);
		filter->disableCollisionsBetween(m_chassisLayer, m_castLayer);
		filter->disableCollisionsBetween(m_chassisLayer, m_chassisLayer);

		m_world->setCollisionFilter( filter );
		filter->removeReference();
	}

	buildLandscape();

	m_world->unlock();
}

void WheelVehicleDemo::createVehicle( VehicleSetup& vehicleSetup, hkpRigidBody* chassis)
{
	HK_SET_OBJECT_COLOR(hkUlong(chassis->getCollidable()), hkColor::semiTransparent(hkColor::LIGHTGREY));

	hkpVehicleInstance* vehicle;
	//
	// Create the basic vehicle.
	//
	{
		vehicle = new hkpVehicleInstance( chassis );
		vehicleSetup.buildVehicle( m_world, *vehicle );
	}

	//
	// Add the vehicle's entities and phantoms to the world
	// Don't add the action--we're only using some of the vehicle's components,
	// the hkpWheels will actually drive the chassis
	//
	{
		vehicle->addToWorld( m_world );
	}

	//
	// Create the wheels
	//

	const hkReal wheelMass = 400.0f;
	hkReal wheelRadius = vehicle->m_data->m_wheelParams[0].m_radius;	// Assume all wheels have the same radius

	int numAxles = vehicle->m_data->m_numWheels;
	if (s_tweakerOptions.m_axles && vehicle->m_data->m_numWheels > 1)	// Can't use shared axles for the unicycle, since there's only one
	{
		numAxles /= 2;
	}
	int wheelsPerAxle = vehicle->m_data->m_numWheels / numAxles;

	m_axles.expandOne();
	m_axles.back().reserveExactly(numAxles);
	for (int i = 0; i < numAxles; i++)
	{
		const hkReal axleMass = wheelMass * wheelsPerAxle;
		hkReal invInertia = 2.0f / (axleMass * wheelRadius * wheelRadius);
		m_axles.back().expandOne().init(wheelsPerAxle, invInertia);
	}

	m_wheels.expandOne();
	for (int i = 0; i < vehicle->m_data->m_numWheels; i++)
	{
		const hkpVehicleSuspension::SuspensionWheelParameters& suspensionParams = vehicle->m_suspension->m_wheelParams[i];
		const hkpVehicleData::WheelComponentParams& wheelParams = vehicle->m_data->m_wheelParams[i];

		hkpWheelFrictionConstraintAtom::Axle* axle = s_tweakerOptions.m_axles ? &m_axles.back()[i / 2] : &m_axles.back()[i];

		hkpWheel* wheel;
		if (s_tweakerOptions.m_linearCast)
		{
			hkpCylinderShape* wheelShape = new hkpCylinderShape( hkVector4( -wheelParams.m_width / 2.0f, 0.0f, 0.0f ), hkVector4( wheelParams.m_width / 2.0f, 0.0f, 0.0f ), wheelParams.m_radius );
			wheel = new hkpLinearCastWheel( wheelShape );
			wheelShape->removeReference();
		}
		else
		{
			wheel = new hkpRaycastWheel();
		}
		wheel->init(chassis, axle, wheelParams.m_radius, wheelParams.m_width, vehicle->m_data->m_chassisOrientation.getColumn<2>(), suspensionParams.m_hardpointChassisSpace, suspensionParams.m_directionChassisSpace, suspensionParams.m_length);
		m_wheels.back().pushBack(wheel);

		wheel->m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(m_castLayer);

		const hkpVehicleDefaultSuspension::WheelSpringSuspensionParameters& wheelSpringParams = static_cast<hkpVehicleDefaultSuspension*>(vehicle->m_suspension)->m_wheelSpringParams[i];
		wheel->m_suspensionStrength = wheelSpringParams.m_strength;
		wheel->m_suspensionDampingCompression = wheelSpringParams.m_dampingCompression;
		wheel->m_suspensionDampingRelaxation = wheelSpringParams.m_dampingRelaxation;

		wheel->m_normalClippingAngleCos = vehicle->m_data->m_normalClippingAngleCos;
	}

	m_vehicles.expandOne();
	m_vehicles.back().m_vehicle = vehicle;
}

void WheelVehicleDemo::createDisplayWheels(hkReal radius, hkReal thickness)
{
	//
	// Create the wheel display geometries. Note: we use a Havok hkDisplayGeometry here, to
	// make binding to the Havok demo framework and Visual Debugger graphics engine easier.
	// However you can use whatever you want!
	//

	// The id for the wheel displayGeometry is an odd number. This prevents it conflicting
	// with hkpCollidable pointers which are used as ids for rigid bodies by convention in 
	// the Visual Debugger.
	int displayId = 1;

	m_displayWheelId.setSize( m_vehicles.getSize() );

	for (int vehicleId = 0; vehicleId < m_vehicles.getSize(); vehicleId++)
	{
		m_displayWheelId[vehicleId].setSize( m_vehicles[vehicleId].m_vehicle->m_wheelsInfo.getSize() );

		// Create a display wheel for each wheel on the vehicle.
		for(int i = 0; i < m_displayWheelId[vehicleId].getSize(); i++)
		{
			VehicleApiUtils::createDisplayWheels(m_env, m_tag, displayId, radius, thickness);

			m_displayWheelId[vehicleId][i] = displayId;
			m_env->m_displayHandler->setGeometryColor(hkColor::semiTransparent(hkColor::YELLOW), m_env->m_displayHandler->findDisplayObject(displayId));

			displayId += 2;
		}
	}
}


static const char helpString[] = "Controls:\n" \
"\x14\x15 - accelerate / brake " \
"\x16\x17 - steer\n" \
"\x11 - handbrake " \
"\x13 - brake / reverse " \
"\x1D - reorient\n"
"Turn right from start position to find 'icy patch'.";

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( WheelVehicleDemo, HK_DEMO_TYPE_PHYSICS_2012, WheelVehicleDemoVariant, WheelVehicleDemoVariants, "Drive a car using the hkpWheel");

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
