/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Dynamics/RigidBodies/InitialPenetration/InitialPenetrationDemo.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>

#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppBvTreeShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/SimpleMesh/hkpSimpleMeshShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/StorageExtendedMesh/hkpStorageExtendedMeshShape.h>

#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppUtility.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppCompilerInput.h>

#include <Physics2012/Utilities/Actions/EasePenetration/hkpEasePenetrationAction.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>
#include <Physics2012/Utilities/Collide/ShapeUtils/ShapeScaling/hkpShapeScalingUtility.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpShapeDisplayViewer.h>

#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>

#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>


enum VariantType
{
	DOING_NOTHING,
	FADEIN_ALLOWED_PENETRATION_DEPTH,
	FADEIN_PENETRATION_DISTANCE_TOO,
};

enum BodyQualityType
{
	QUALITY_CRITICAL,
	QUALITY_DEBRIS
};

enum GroundType
{
	GROUND_MESH,
	GROUND_CONVEX
};

enum DynamicObjectType
{
	DYNAMIC_BOXES,
	DYNAMIC_MIXED,
	DYNAMIC_COMPLEX
};

struct BenchmarkSuiteVariant
{
	const char*	 m_name;
	VariantType  m_type;
	BodyQualityType	m_bodyQualityType;
	GroundType m_groundType;
	DynamicObjectType m_dynamicObjectType;
	const char* m_details;
	int m_objectStackSize;
};

#define COMPLEX_OBJECT_MASS 20.0f

static const BenchmarkSuiteVariant g_variants[] =
{
	{ "No correction, massive TOIs, very high CPU usage", DOING_NOTHING, QUALITY_CRITICAL, GROUND_MESH, DYNAMIC_BOXES,  "This variant shows CPU usage when a batch of bodies is added in penetrating state, and no correction is done. Note very high number of TOIs generated.", 5 },
	{ "Scaling allowed penetration depth", FADEIN_ALLOWED_PENETRATION_DEPTH, QUALITY_CRITICAL, GROUND_MESH, DYNAMIC_BOXES, "This is the same scene as the one above. Each body has an EasePenetrationAction attached which scales it's allowed penetration depth over the period of 1 second.", 5 },
	{ "Scaling allowed penetration depth and contact depth", FADEIN_PENETRATION_DISTANCE_TOO, QUALITY_CRITICAL, GROUND_MESH, DYNAMIC_BOXES, "This is the same scene as the one above. Additionally contact depth is also scaled, which results with less jitter and softer penetration recovery.", 5 },
	{ "Scaling allowed penetration depth and contact depth, complex objects", FADEIN_PENETRATION_DISTANCE_TOO, QUALITY_CRITICAL, GROUND_CONVEX, DYNAMIC_COMPLEX, "This is the same scene as the one above. Additionally contact depth is also scaled, which results with less jitter and softer penetration recovery.", 5 },
	{ "No correction, debris bodies, no TOIs", DOING_NOTHING, QUALITY_DEBRIS, GROUND_MESH, DYNAMIC_BOXES, "This is the same scene as the one above, without any EasePenetrationActions, with debris bodies, and no TOIs are created.", 5 },
	{ "No correction, debris bodies, no TOIs, flat ground", DOING_NOTHING, QUALITY_DEBRIS, GROUND_CONVEX, DYNAMIC_MIXED, "This is the scene with simple, flat ground body, without any EasePenetrationActions, with debris bodies, and no TOIs are created.", 5 },
};

static hkpShape* createMoppShape(hkpShapeCollection* meshShape)
{
	hkpMoppCompilerInput moppInput;
	moppInput.m_enableChunkSubdivision = true;

	hkpMoppCode* code = hkpMoppUtility::buildCode( meshShape, moppInput );
	hkpShape* moppShape = new hkpMoppBvTreeShape(meshShape, code);

	code->removeReference();

	return moppShape;
}

static void CreatePhysicsTerrain( hkpWorld* world, GroundType groundType, const int side, float deltaHeight = 0.0f )
{
	const hkReal shapeRadius = 0.05f;

	hkpRigidBodyCinfo ci;

	if (groundType == GROUND_MESH)
	{
		const float triangleEdgeLen = 1.0f;
		hkpSimpleMeshShape* meshShape = new hkpSimpleMeshShape( shapeRadius );

		{
			meshShape->m_vertices.setSize( side * side );
			for(int x = 0; x < side; x++)
			{
				for (int y = 0; y < side; y++ )
				{
					float height = 0.0f;
					if ( (x&1) && (y&1) )
					{
						height = -deltaHeight;
					}

					hkVector4 vertex (
						triangleEdgeLen * (x * 1.0f - (side-1) * 0.5f),
						triangleEdgeLen * height,
						triangleEdgeLen * (y * 1.0f - (side-1) * 0.5f));
					meshShape->m_vertices[x*side + y] = vertex ;
				}
			}
		}

		{
			meshShape->m_triangles.setSize( (side-1) * (side-1) * 2);
			int corner = 0;
			int curTri = 0;
			for(int i = 0; i < side - 1; i++)
			{
				for (int j = 0; j < side - 1; j++ )
				{
					meshShape->m_triangles[curTri].m_a = corner+1;
					meshShape->m_triangles[curTri].m_b = corner+side;
					meshShape->m_triangles[curTri].m_c = corner;
					curTri++;

					meshShape->m_triangles[curTri].m_a = corner+side+1;
					meshShape->m_triangles[curTri].m_b = corner+side;
					meshShape->m_triangles[curTri].m_c = corner+1;
					curTri++;
					corner++;
				}
				corner++;
			}
		}

		hkpStorageExtendedMeshShape* extendedMesh = new hkpStorageExtendedMeshShape();

		hkpExtendedMeshShape::TrianglesSubpart part;
		part.m_numTriangleShapes = meshShape->m_triangles.getSize();
		part.m_numVertices = meshShape->m_vertices.getSize();
		part.m_vertexBase = &meshShape->m_vertices.begin()[0](0);
		part.m_stridingType = hkpExtendedMeshShape::INDICES_INT32;
		part.m_vertexStriding = sizeof(hkVector4);
		part.m_indexBase = meshShape->m_triangles.begin();
		part.m_indexStriding = sizeof(hkpSimpleMeshShape::Triangle);
		extendedMesh->addTrianglesSubpart( part );

		// No longer need the simple mesh
		meshShape->removeReference();

		ci.m_shape = createMoppShape( extendedMesh );
		ci.m_position.set( 0.0f, 0.0f, 0.0f );

		extendedMesh->removeReference();
	}
	else
	{
		ci.m_shape = new hkpBoxShape(hkVector4(0.5f * side, 0.5f * deltaHeight, 0.5f * side), shapeRadius);
		ci.m_position.set( 0.0f, -0.5f * deltaHeight, 0.0f );
	}

	ci.m_restitution = 0.5f;
	ci.m_friction = 0.3f;
	ci.m_motionType = hkpMotion::MOTION_FIXED;

	world->addEntity( new hkpRigidBody( ci ) )->removeReference();
	ci.m_shape->removeReference();
}

void InitialPenetrationDemo::LoadComplexObjects()
{
	hkStringBuf fileName(HK_GET_DEMOS_ASSET_FILENAME("Resources/Physics2012/Compoundbodies/compoundbodies_scaled.hkt"));
	hkAssetManagementUtil::getFilePath(fileName);
	m_debrisPackfileData = hkSerializeUtil::load(fileName);

	HK_ASSERT( 0x215d080c, m_debrisPackfileData );

	hkRootLevelContainer* container = m_debrisPackfileData->getContents<hkRootLevelContainer>();
	HK_ASSERT2(0xa6451543, container != HK_NULL, "Could not load root level object" );

	hkpPhysicsData* physicsData = static_cast<hkpPhysicsData*>( container->findObjectByType( hkpPhysicsDataClass.getName() ) );
	HK_ASSERT2(0xa6451544, physicsData != HK_NULL, "Could not find physics data in root level object" );

	m_complexRBs = physicsData->getPhysicsSystems()[0]->getRigidBodies();
	m_complexShapes.setSize(m_complexRBs.getSize());

	for(int i=0;i<m_complexRBs.getSize();i++)
  	{
		hkpCollidable* collidable = const_cast<hkpCollidable*>(m_complexRBs[i]->getCollidable());
		hkpShape* shape = const_cast<hkpShape*>(collidable->getShape());
		m_complexShapes[i] = shape;
  	}
}

InitialPenetrationDemo::~InitialPenetrationDemo()
{
	// destroy the world first, as bits of it are contained in the packfile
	destroyWorld();
	if(m_debrisPackfileData)
	{
		m_debrisPackfileData->removeReference();
	}
	hkError::getInstance().setEnabled(0x20096AAE, true); // Critical operations generated during simulation's step / determinism.
}

InitialPenetrationDemo::InitialPenetrationDemo(hkDemoEnvironment* env)
	:	hkDefaultPhysics2012Demo(env), m_debrisPackfileData(NULL), m_step(0)
{
	const BenchmarkSuiteVariant& variant =  g_variants[m_variantId];
	
	// EasePenetrationAction will cause this when removing itself
	hkError::getInstance().setEnabled(0x20096AAE, false); // Critical operations generated during simulation's step / determinism.
	
	//
	// Setup the camera
	//
	{
		hkVector4 from(55.0f, 50.0f, 55.0f);
		hkVector4 to  ( 0.0f,  0.0f,   0.0f);
		hkVector4 up  ( 0.0f,  1.0f,   0.0f);
		setupDefaultCameras(env, from, to, up, 0.1f, 20000.0f);
	}

	//
	// Create the world
	//
	{
		hkpWorldCinfo worldInfo;
		m_world = new hkpWorld(worldInfo);
		m_world->lock();

		// Register ALL agents (though some may not be necessary)
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

		setupGraphics();
	}

// TODO: Do we need this??
	//hkpShapeDisplayViewer* shapeViewer = getLocalViewer<hkpShapeDisplayViewer>();
	//if (shapeViewer)
//	{
//		shapeViewer->setInstancingEnabled( false );
//	}

	//
	// Setup the camera
	//
	{
		hkVector4 from(12.0f, 3.0f, 12.0f);
		hkVector4 to  (-3.0f, -5.0f, -6.0f);
		hkVector4 up  (0.0f, 1.0f, 0.0f);
		setupDefaultCameras( env, from, to, up );
	}


	{
		int side = 32;
		CreatePhysicsTerrain(m_world, variant.m_groundType, side, 1.0f);

		if(variant.m_dynamicObjectType == DYNAMIC_COMPLEX)
		{
			LoadComplexObjects();
		}

		hkPseudoRandomGenerator gen(234);
		const int numBodies = variant.m_dynamicObjectType != DYNAMIC_MIXED ? 50 : 150;
		for (int i = 0; i < numBodies; i++)
		{
			hkVector4 size(0.5f, 0.5f, 0.5f);
			hkVector4 pos;
			gen.getRandomVector11(pos);
			pos.mul4(hkVector4(5.0f, 0.1f, 5.0f));
			pos(1) += 0.1f;
			hkpRigidBody* body = HK_NULL;
			hkpRigidBodyCinfo createRB;
			hkMassProperties massProps;

			switch(variant.m_dynamicObjectType)
			{
				case DYNAMIC_BOXES: 
					body = GameUtils::createBox(size, 1.0f, pos);
					break;
				case DYNAMIC_MIXED: 
					pos(1) = (pos(1)-0.1f)*5.0f-0.3f;
					body = i%2 ? GameUtils::createBox(size, 1.0f, pos) : GameUtils::createSphere(0.75f * size(0), 1.0f, pos);
					break;
				case DYNAMIC_COMPLEX:
					createRB.m_shape = m_complexShapes[i % m_complexRBs.getSize()];
					createRB.m_position = pos;
					gen.getRandomRotation(createRB.m_rotation);
					createRB.m_mass = COMPLEX_OBJECT_MASS;

					hkpInertiaTensorComputer::computeShapeVolumeMassProperties(createRB.m_shape, COMPLEX_OBJECT_MASS, massProps);
					createRB.setMassProperties(massProps);
					body = new hkpRigidBody(createRB);

					break;
				default:
					HK_ASSERT2(0xad341321, false, "Unhandled case.");
					break;
			}

			if(variant.m_dynamicObjectType != DYNAMIC_COMPLEX)
			{
				hkQuaternion rotation;
				gen.getRandomRotation(rotation);
				body->setRotation(rotation);
			}
			 
			switch(variant.m_bodyQualityType)
			{
				case QUALITY_CRITICAL: body->setQualityType(HK_COLLIDABLE_QUALITY_CRITICAL); break;
				case QUALITY_DEBRIS: body->setQualityType(HK_COLLIDABLE_QUALITY_DEBRIS); break;
				default: HK_ASSERT2(0xad341324, false, "Unhandled case.");
			}
			
			m_world->addEntity(body);
			body->removeReference();

			switch(variant.m_type)
			{
			case DOING_NOTHING:
				{
					// nothing				
				}
				break;
			case FADEIN_ALLOWED_PENETRATION_DEPTH:
				{
					// Attach hkpEasePenetrationAction, disable contact penetration depth correction
					hkpEasePenetrationAction* action = new hkpEasePenetrationAction(body, 1.0f);
					action->m_reducePenetrationDistance = false;
					m_world->addAction(action);
					action->removeReference();
				}
				break;
			case FADEIN_PENETRATION_DISTANCE_TOO:
				{
					// Attach hkpEasePenetrationAction, use the default option of correcting contact point penetration depth.
					hkpAction* action = new hkpEasePenetrationAction(body, 1.0f);
					m_world->addAction(action);
					action->removeReference();
				}
				break;
			}
		}

		if(variant.m_objectStackSize > 0)
		{
			const float boxWidth = 0.5f;
			const float scalePosition = 0.8f; // scale positions by this. < 1.0f will cause penetration
			hkVector4 halfExtents(boxWidth / 2.0f, boxWidth / 2.0f, boxWidth / 2.0f);
			hkpBoxShape* cube = new hkpBoxShape(halfExtents, 0.0f );	// Note: We use HALF-extents for boxes

			hkVector4 basePosition(10.0f, 0.5f, 0.0f);

			for(int stackLevelSize=variant.m_objectStackSize; stackLevelSize > 0 ; stackLevelSize--)
			{
				for(int i=0;i<stackLevelSize;i++)
				{
					for(int j=0;j<stackLevelSize;j++)
					{
						hkVector4 offset((hkReal(i) - hkReal(stackLevelSize) / 2.0f) * boxWidth * scalePosition, (variant.m_objectStackSize - stackLevelSize) * boxWidth * scalePosition, (hkReal(j) - hkReal(stackLevelSize) / 2.0f) * boxWidth * scalePosition);
						// Put in a random jitter in the positions so that it doesn't look too regular
						hkVector4 randomOffset; gen.getRandomVector11(randomOffset); randomOffset.mul4(boxWidth * 0.1f);
						hkVector4 position; position.setAdd4(basePosition, offset); position.setAdd4(position, randomOffset);

						hkpRigidBodyCinfo boxInfo;
						boxInfo.m_mass = 1.0f;
						hkMassProperties massProperties;
						hkpInertiaTensorComputer::computeBoxVolumeMassProperties(halfExtents, 1.0f, massProperties);
						boxInfo.m_inertiaTensor = massProperties.m_inertiaTensor;
						boxInfo.m_motionType = hkpMotion::MOTION_BOX_INERTIA;

						gen.getRandomRotation(boxInfo.m_rotation);
						boxInfo.m_shape = cube;
						boxInfo.m_position = position;

						hkpRigidBody* body = new hkpRigidBody(boxInfo);

						m_world->addEntity(body);
						body->removeReference();
					}
				}
			}

			cube->removeReference();
		}
	}

	//
	//	Some values
	//

	m_world->unlock();
}

hkDemo::Result InitialPenetrationDemo::stepDemo()
{
	if (m_step == 0)
	{
		m_step++;
		return DEMO_PAUSED;
	}

	return hkDefaultPhysics2012Demo::stepDemo();
}




static const char helpString[] = "When adding bodies to the world, it's not uncommon that they're placed so that they penetrate the landscape. "\
"This demo shows how this results in enourmous CPU hits, and how this can be counteracted with the use of hkpEasePenetrationAction.";

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( InitialPenetrationDemo, HK_DEMO_TYPE_PHYSICS_2012, BenchmarkSuiteVariant, g_variants, helpString );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
