/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>

#include <Graphics/Common/Window/hkgWindow.h>

#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabbUtil.h>

#include <Physics2012/Internal/Dynamics/Constraints/hkpConstraintProjector.h>
#include <Physics2012/Dynamics/World/hkpPhysicsSystem.h>
#include <Physics2012/Dynamics/World/Util/hkpWorldCallbackUtil.h>
#include <Physics/Constraint/Data/BallAndSocket/hkpBallAndSocketConstraintData.h>
#include <Physics2012/Dynamics/Constraint/Util/hkpConstraintStabilizationUtil.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>


#include <Graphics/Common/Input/Pad/hkgPad.h>

#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>

#include <Demos/Physics2012/Api/Dynamics/RigidBodies/ConstraintProjection/ConstraintProjectionDemo.h>


struct ConstraintProjectionVariant
{
	const char*	 m_name;
	bool m_stabilizationEnabled;
	bool m_inertiaTweaking;
	const char* m_details;
};

static const char variantHelpString[] = "Press 'P' to toggle projection. Press keys 1 to 4 to apply impulses to the character.";

// suggestions for variants: motor tau + dumping,
// with(out) collision
static const ConstraintProjectionVariant g_variants[] =
{
	{ "Constraint stabilization OFF", false, false, variantHelpString },
	{ "Constraint stabilization ON" , true, false, variantHelpString },
	{ "Constraint stabilization ON, Body inertias stabilized" , true, true, variantHelpString },
};


/// Simple example of constraints projection usage
class ConstraintProjectorListener : public hkpConstraintListener
{
	public:

		HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO,ConstraintProjectorListener);

		/// constructor
		ConstraintProjectorListener(hkpWorld* world)
		{
			m_world = world;
			m_world->lock();
			m_world->addConstraintListener(this);
			m_world->unlock();
		}

		/// destructor
		virtual ~ConstraintProjectorListener()
		{
			m_world->lock();
			m_world->removeConstraintListener(this);
			m_world->unlock();
		}

		/// implementation of hkpConstraintListener
		void	constraintViolatedCallback( hkpConstraintInstance* constraint )
		{
			m_criticalSection.enter();
			m_constraintsToProject.pushBack(constraint);
			constraint->addReference();
			m_criticalSection.leave();
		}
			
		/// project and clear violated constraints.
		void	projectAndClear()
		{
			m_criticalSection.enter();
			if(m_constraintsToProject.getSize())
			{
				hkpConstraintProjector	cp(hkpConstraintProjector::HK_REPAIR_VIOLATIONS);
				cp.project(m_constraintsToProject);
			}
			clear();
			m_criticalSection.leave();		
		}

		/// clear violated constraints.
		void	clear()
		{
			m_criticalSection.enter();
			hkReferencedObject::removeReferences( m_constraintsToProject.begin(), m_constraintsToProject.getSize() );
			m_constraintsToProject.clear();
			m_criticalSection.leave();
		}

	public:

		hkpWorld*												m_world;
		hkCriticalSection										m_criticalSection;
		hkArray<hkpConstraintInstance*>							m_constraintsToProject;
};

ConstraintProjectionDemo::ConstraintProjectionDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env)
,	m_rndGen(180673)
,	m_doProjection(true)
{
	//
	//	Build a world
	//
	{
		hkpWorldCinfo worldinfo;
		worldinfo.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM);
		worldinfo.m_gravity.set( 0, 0, -9.81f );		
		worldinfo.m_enableDeactivation = false;
		worldinfo.setBroadPhaseWorldSize( 200.0f );
		worldinfo.m_maxConstraintViolation = 0.05f;	// Set maximum constraint violation to 5cm.
		m_world = new hkpWorld( worldinfo );
		m_world->lock();

		hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );
	}

	//
	// Create a collision filter, to filter collisions between ragdoll limbs
	//
	{
		hkpGroupFilter* filter = new hkpGroupFilter();
		m_world->setCollisionFilter( filter );
		filter->removeReference();
	}

	//
	// Create and attach our listener
	//
	{
		m_violatedConstraintListener = new ConstraintProjectorListener(m_world);
	}

	//
	// Visualize constraints
	//
	{
		m_debugViewerNames.pushBack( hkpConstraintViewer::getName() );
		hkpConstraintViewer::m_scale = 0.1f;
	}

	//
	// Misc
	//

	m_shootGunDelay	=	0;
	
	//
	// Create rag dolls
	//
	{
		m_numRagdolls = 0;
		//for(int i=0,ni=1;i<ni;i++)
		{
			static const hkReal	spacing  = 2;
			static const hkReal	height   = 2;
			hkQuaternion		rotation = hkQuaternion::getIdentity();
			hkVector4			position = hkVector4::getZero();
			position(0)	=	0;//(hkReal)(-(ni/2)*spacing+i*spacing);
			position(2)	=	5;


			GameUtils::RagdollCinfo info;
			info.m_height = height;
			info.m_position = position;
			info.m_rotation = rotation;
			info.m_boneShapeType = GameUtils::RPT_CAPSULE;
			info.m_ragdollInterBoneCollisions = GameUtils::DISABLE_ONLY_ADJOINING_BONE_COLLISIONS;
			hkpPhysicsSystem*	ragdoll   = GameUtils::createRagdoll( info );

			//hkpPhysicsSystem*	ragdoll   = GameUtils::createRagdoll( height, position,	rotation, 0, GameUtils::RPT_CAPSULE );
			m_numRagdolls++;

			m_world->addPhysicsSystem(ragdoll);
			m_ragdolls.pushBack( ragdoll );

			// Attach i th limb to world	
			{
				const hkArray<hkpRigidBody*>& rigidbodies = ragdoll->getRigidBodies();
				hkpRigidBody*					body	=	rigidbodies[10];
				hkpBallAndSocketConstraintData*	cd		=	new hkpBallAndSocketConstraintData();
				hkVector4						pivot	=	body->getPosition();
				cd->setInWorldSpace(body->getTransform(), hkTransform::getIdentity(), pivot);
				m_ragdollAttachmentPoints.pushBack(pivot);
				hkpConstraintInstance*			ci		=	new hkpConstraintInstance(body,HK_NULL,cd);
				m_world->addConstraint(ci);
				ci->removeReference();
				cd->removeReference();
			}
		}
	}

	//hkVector4 extents;
	//h

	//extents.set( 0.5f * head, 1.0f * head, 0.5f * head, info.m_height );
	//position.set( 0.75f * extents(0), 0, extents(2) );
	//rotation.setRows( hkVector4( 0, 0, -1 ), hkVector4( 0, -1, 0 ), hkVector4( -1, 0, 0 ) );
	//offset.getTranslation().set( 0.5f * extents(2), 0.25f * extents(1), 0 );
	//offset.getRotation() = rotation;
	//mass = 1.0f * scale;

	//hkVector4 p; p.setRotatedDir( offset.getRotation(), offset.getTranslation() ); p.add4( position );
	//rb = createRagdollPart( extents, mass, p, info.m_boneShapeType, (0 == filename.getLength()) ? HK_NULL : filename );

	//ragdoll->addRigidBody( rb );
	//rb->removeReference(); // we don't want the ref anymore, leave with the system.


	//hkpRigidBody* rb = GameUtils::createBox( 5, 1000, hkVector4::getZero() );
	//m_world->addEntity( rb );
	//rb->removeReference();


	// Keep reference to a list of rigid bodies to allow 'similar' test & deterministic body selection for applied impulses.
	m_rigidBodyCache = m_world->getWorldAsOneSystem();

	//
	//	Create a ground box
	//
	if (0)
	{
		hkVector4 size( 25, 25, 1 );

		hkpRigidBody* rb = GameUtils::createBox( size, 0, hkVector4::getZero() );
		m_world->addEntity( rb );
		rb->removeReference();
	}

	//
	// Setup the camera
	//
	{
		hkVector4 from(   0.0f, -10.0f, 5.0f );
		hkVector4 to  (   0.0f,   0.0f, 5.0f );
		hkVector4 up  (   0.0f,   0.0f, 1.0f );
		setupDefaultCameras( env, from, to, up );

		hkVector4 ldir( 0, 0.5f, -1.0f );
		setSoleDirectionLight( m_env, ldir, 0xffffffff );

		setupGraphics();
	}

	// Enable stabilized solver and pre-stabilize inertias
	if (g_variants[m_env->m_variantId].m_stabilizationEnabled)
	{
		hkpConstraintStabilizationUtil::setConstraintsSolvingMethod(m_world, hkpConstraintAtom::METHOD_STABILIZED);
	}
	else
	{
		hkpConstraintStabilizationUtil::setConstraintsSolvingMethod(m_world, hkpConstraintAtom::METHOD_OLD);
	}

	if (g_variants[m_env->m_variantId].m_inertiaTweaking)
	{
		hkpConstraintStabilizationUtil::stabilizePhysicsWorldInertias(m_world);
	}
 
	m_world->unlock();

	m_framesSinceImpact = 0;
}

ConstraintProjectionDemo::~ConstraintProjectionDemo()
{
	m_rigidBodyCache->removeReference();

	m_world->lock();

	delete m_violatedConstraintListener;
	
	for( int i = 0; i < m_ragdolls.getSize(); i++ )
	{
		m_ragdolls[i]->removeReference();
	}	

	m_world->unlock();
}

hkDemo::Result ConstraintProjectionDemo::stepDemo()
{
	if(	//m_env->m_gamePad->wasButtonPressed(HKG_PAD_BUTTON_2)||
		m_env->m_window->getKeyboard().wasKeyPressed('P'))
	{
		m_doProjection=!m_doProjection;
	}

	// Clear display of old impulses
	if (++m_framesSinceImpact > 20)
	{
		m_lastImpactPositions.clear();
		m_lastImpactDirections.clear();
	}

	// Apply an impulse on keypress
	{
		hkVector4 impulse;
		do {m_rndGen.getRandomVector11(impulse);} 
		while (impulse.length3() < HK_REAL_EPSILON);
		impulse.normalize3();

		     if (m_env->m_window->getKeyboard().wasKeyPressed('1')) { impulse.mul4( 500); }
		else if (m_env->m_window->getKeyboard().wasKeyPressed('2')) { impulse.mul4(2000); }
		else if (m_env->m_window->getKeyboard().wasKeyPressed('3')) { impulse.mul4(4000); }
		else if (m_env->m_window->getKeyboard().wasKeyPressed('4')) { impulse.mul4(8000); }
		else														{ impulse.mul4(0); }

		if (impulse.length3() > HK_REAL_EPSILON)
		{
			m_world->lock();

			m_lastImpactPositions.clear();
			m_lastImpactDirections.clear();

			hkVector4 impulseNormalized = impulse; impulseNormalized.normalize3();

			const hkArray<hkpRigidBody*>&	rigidBodies(m_rigidBodyCache->getRigidBodies());
			int numBodiesInFirstRagdoll = rigidBodies.getSize() / m_numRagdolls;
			int bodyIdx = m_rndGen.getRand32()%numBodiesInFirstRagdoll;
			for (int i = bodyIdx; i < rigidBodies.getSize(); i += numBodiesInFirstRagdoll)
			{
				hkpRigidBody* rb = rigidBodies[i]; // body to push

				rb->setMaxLinearVelocity(14143.0f);
				rb->applyLinearImpulse(impulse);

				m_lastImpactPositions.pushBack(rb->getPosition());
				m_lastImpactDirections.pushBack(impulseNormalized);
			}

			m_world->unlock();

			m_framesSinceImpact = 0;
		}
	}


	/* Apply a strong impulse every half second	*/ 
	if (false)
	{
		m_world->lock();
		const int			maxDelay=30;
		if((++m_shootGunDelay)>maxDelay)
		{
			const hkArray<hkpRigidBody*>&	rigidBodies(m_rigidBodyCache->getRigidBodies());
			const int						randomBody(m_rndGen.getRand32()%rigidBodies.getSize());
			hkpRigidBody*					bodyToPush(rigidBodies[randomBody]);
			hkVector4						impulse;
			m_rndGen.getRandomVector11(impulse);
			impulse.mul4(10000);
			// Prevent the resulting velocity exceeding the maximum velocity
			bodyToPush->setMaxLinearVelocity(14143.0f);
			bodyToPush->applyLinearImpulse(impulse);
			m_shootGunDelay					=	0;
			m_lastImpactPositions.clear();
			m_lastImpactPositions.pushBack(bodyToPush->getPosition());
			impulse.normalize3();
			m_lastImpactDirections.clear();
			m_lastImpactDirections.pushBack(impulse);
		}
		m_world->unlock();
	}

	/* Step										*/ 

	hkDemo::Result	result=hkDefaultPhysics2012Demo::stepDemo();
	
	/* Project constraints if required			*/ 
	m_world->lock();
	const int	numViolations	=	m_violatedConstraintListener->m_constraintsToProject.getSize();
	if(m_doProjection && numViolations)
	{		
		m_violatedConstraintListener->projectAndClear();
		hkpWorldCallbackUtil::firePostSimulationCallback(m_world);
	}
	else
	{
		m_violatedConstraintListener->clear();
	}
	m_world->unlock();	
	
	/* Display info text						*/ 
	hkStringBuf	statusText;
	if(m_doProjection)
		statusText.printf("Projection [ON] , %d constraints projected",numViolations);
	else
		statusText.printf("Projection [OFF]");
	m_env->m_textDisplay->outputText(statusText.cString(),8,(int)getWindowHeight()-24);

	/* Display last impulses						*/ 
	HK_ASSERT2(0xad843211, m_lastImpactPositions.getSize() == m_lastImpactDirections.getSize(), "Num positions & directions don't match.");
	const int numImpacts = m_lastImpactPositions.getSize();

	// Draw last applied impluses
	for (int i = 0; i < numImpacts; i++)
	{
		hkAabb	aabb;
		hkAabbUtil::calcAabb(m_lastImpactPositions[i],0.1f,aabb);
		HK_DISPLAY_ARROW(m_lastImpactPositions[i],m_lastImpactDirections[i],hkColor::WHITE);
		HK_DISPLAY_BOUNDING_BOX(aabb,hkColor::YELLOW);
		HK_DISPLAY_3D_TEXT("X",m_lastImpactPositions[i],hkColor::RED);
	}

	// Draw aabbs for ragdoll attachemnts
	for (int i = 0; i < m_ragdollAttachmentPoints.getSize(); i++)
	{
		hkAabb	aabb;
		hkAabbUtil::calcAabb(m_ragdollAttachmentPoints[i],0.1f,aabb);
		HK_DISPLAY_BOUNDING_BOX(aabb,hkColor::BLUE);
	}

	return result;
}




static const char helpString[] = "Using constraint projections to fix large errors in articulated bodies";

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( ConstraintProjectionDemo, HK_DEMO_TYPE_PHYSICS_2012, ConstraintProjectionVariant, g_variants, helpString );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
