/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>

#include <Graphics/Common/Window/hkgWindow.h>

#include <Demos/Physics2012/Api/Dynamics/Phantoms/PhantomBatchMove/PhantomBatchMoveDemo.h>

#include <Common/Visualize/hkDebugDisplay.h>

#include <Physics2012/Utilities/Dynamics/PhantomBatchMove/hkpPhantomBatchMoveUtil.h>

#include <Physics2012/Utilities/CharacterControl/CharacterProxy/hkpCharacterProxy.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics2012/Dynamics/Phantom/hkpSimpleShapePhantom.h>
#include <Physics2012/Dynamics/Phantom/hkpAabbPhantom.h>

#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>


PhantomBatchMoveDemo::PhantomBatchMoveDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env), m_time(0.0f)
{
	//
	// Setup the camera
	//
	{
		hkVector4 from(  0.0f, 20.0f, -80.0f);
		hkVector4 to  (  0.0f,  0.0f,   0.0f);
		hkVector4 up  (  0.0f,  1.0f,   0.0f);
		setupDefaultCameras( env, from, to, up );	
		
	}

	//
	// Create the world
	//
	{
		hkpWorldCinfo info;
		info.setBroadPhaseWorldSize( 350.0f ); 
		info.m_collisionTolerance = 0.1f;
		m_world = new hkpWorld( info );		
	}

	setupGraphics();

	m_world->lock();
	{

		hkPseudoRandomGenerator random(1337);

		//
		//	Create the character proxies
		//
		{
			hkpPropertyValue shapeType;

			m_characterProxy.setSize(NUM_SHAPE_PHANTOMS);
			for( int i = 0; i < NUM_SHAPE_PHANTOMS; i++ )
			{		
				//
				// Create a random shape to represent the character
				//
				hkpConvexShape* shape = HK_NULL;
				{
					hkReal scale = random.getRandReal01() * 0.25f + 0.75f;

					shapeType.setInt( random.getRand32() % 3 );

					switch (shapeType.getInt())
					{
						case 0: // Capsule
						{
							hkVector4 top(0, scale * 0.4f, 0);
							hkVector4 bottom(0, -scale * 0.4f, 0);					
							shape = new hkpCapsuleShape(top, bottom, scale * 0.6f);
							break;
						}
						case 1: // Box
						{
							hkVector4 size(scale * 0.5f, scale , scale * 0.5f);
							shape = new hkpBoxShape(size);
							break;
						}						
						default: // Sphere
						{
							shape = new hkpSphereShape(scale);
							break;
						}
					}
				}
				
				//
				// Construct shape phantoms for the characters (could also be a hkpCachingShapePhantom)
				//

				hkpShapePhantom* phantom = new hkpSimpleShapePhantom( shape, hkTransform::getIdentity() );
				shape->removeReference();

				//
				// Add the phantom to the world
				//

				m_world->addPhantom(phantom);
				phantom->removeReference();
				HK_SET_OBJECT_COLOR( (hkUlong)phantom->getCollidable(), 0xafffffff & hkColor::getRandomColor() );

				//
				// Construct a character proxy
				//

				hkpCharacterProxyCinfo cpci;
				random.getRandomVector11( cpci.m_position );
				cpci.m_position.mul4(32);
				cpci.m_position(1) = 10;
				cpci.m_up.setNeg4( m_world->getGravity() );
				cpci.m_up.normalize3();

				cpci.m_shapePhantom = phantom;
				m_characterProxy[i] = new hkpCharacterProxy( cpci );
			}			
		}

		//
		// Create some AABB phantoms too
		//
		m_aabbPhantoms.setSize(NUM_AABB_PHANTOMS);
		for (int i = 0; i < NUM_AABB_PHANTOMS; i++)
		{
			hkVector4 position;
			random.getRandomVector11(position);
			position.mul4(32);
			position(1) = 10;

			hkVector4 a,b;
			random.getRandomVector11(a);
			random.getRandomVector11(b);
			a.mul4(3.0f);
			b.mul4(3.0f);

			hkAabb aabb;
			aabb.m_max.setMax4(a,b);
			aabb.m_min.setMin4(a,b);

			aabb.m_max.add4(position);
			aabb.m_min.add4(position);

			m_aabbPhantoms[i] = new hkpAabbPhantom(aabb);

			m_world->addPhantom(m_aabbPhantoms[i]);
		}

		m_stopwatch.reset();
	}
	m_world->unlock();
}

PhantomBatchMoveDemo::~PhantomBatchMoveDemo()
{
	m_world->lock();

	for (int i=0; i < m_characterProxy.getSize(); i++)
	{
		m_characterProxy[i]->removeReference();
	}

	for (int i=0; i < m_aabbPhantoms.getSize(); i++)
	{
		m_aabbPhantoms[i]->removeReference();
	}

	m_world->unlock();
}


hkDemo::Result PhantomBatchMoveDemo::stepDemo()
{

	m_time += m_timestep;

	m_world->lock();
	{
		hkLocalArray<hkVector4> newPositions( TOTAL_PHANTOMS );
		newPositions.setSize( TOTAL_PHANTOMS );

		hkLocalArray<hkpPhantom*> phantoms( TOTAL_PHANTOMS );
		phantoms.setSize( TOTAL_PHANTOMS );

		//
		// Update shape phantoms first
		//
		for( int i = 0; i < NUM_SHAPE_PHANTOMS; i++ )
		{
			hkVector4 newVel; newVel.set(hkMath::sin(m_time), 0.0f, hkMath::cos(m_time));
			hkReal mult = (hkReal)((i % 7 + 1) * (2 * (i % 2) - 1));
			newVel.mul4(mult);
			hkVector4 newPos; newPos.setAddMul4(m_characterProxy[i]->getPosition(), newVel, m_timestep);
			newPositions[i] = newPos;
			phantoms[i] = m_characterProxy[i]->getShapePhantom();
		}
			
		//
		// Then the AABB phantoms
		//
		for( int i = NUM_SHAPE_PHANTOMS; i < TOTAL_PHANTOMS ; i++ )
		{
			hkVector4 newVel; newVel.set(hkMath::sin(m_time), 0.0f, hkMath::cos(m_time));
			hkReal mult = (hkReal)((i % 7 + 1) * (2 * (i % 2) - 1));
			newVel.mul4(mult);
			hkpAabbPhantom* aabbPhantom = m_aabbPhantoms[i - NUM_SHAPE_PHANTOMS];
			hkAabb aabb = aabbPhantom->getAabb();
			hkVector4 oldPos; oldPos.setInterpolate4(aabb.m_min, aabb.m_max, .5f);
			hkVector4 newPos; newPos.setAddMul4(oldPos, newVel, m_timestep);
			newPositions[i] = newPos;
			phantoms[i] = aabbPhantom;
		}

		
		if(m_variantId == 0)
		{
			//
			// Set the position sequentially
			//

			HK_TIMER_BEGIN("setPositionIndividual", HK_NULL);
			m_stopwatch.start();
			for( int i = 0; i < NUM_SHAPE_PHANTOMS; i++ )
			{
				m_characterProxy[i]->getShapePhantom()->setPosition(newPositions[i]);
			}

			for( int aabbIdx = 0; aabbIdx < NUM_AABB_PHANTOMS; aabbIdx++ )
			{
				hkAabb oldAabb = m_aabbPhantoms[aabbIdx]->getAabb();
				hkVector4 midpoint; midpoint.setInterpolate4(oldAabb.m_min, oldAabb.m_max, hkSimdReal::getConstant<HK_QUADREAL_INV_2>());
				hkVector4 offset; offset.setSub4(midpoint, newPositions[aabbIdx + NUM_AABB_PHANTOMS]);
				hkAabb newAabb = oldAabb;
				newAabb.m_max.add4(offset);
				newAabb.m_min.add4(offset);
				HK_ASSERT(0x0, newAabb.isValid());
				m_aabbPhantoms[aabbIdx]->setAabb(newAabb);
			}
			m_stopwatch.stop();
			HK_TIMER_END();
		}
		else
		{

			//
			// Batch set the position
			//

			HK_TIMER_BEGIN("setPositionBatch", HK_NULL);
			m_stopwatch.start();
			hkpPhantomBatchMoveUtil::setPositionBatch( phantoms , newPositions );
			m_stopwatch.stop();
			HK_TIMER_END();
		}

		char msg[2048];
		
		hkUint64 ticks = m_stopwatch.getElapsedTicks();
		hkDouble64 tpf = hkDouble64(ticks/m_stopwatch.getNumTimings());
		hkDouble64 tpms = hkDouble64(m_stopwatch.getTicksPerSecond()/1000000);
		hkDouble64 mspf = tpf / tpms;
		hkString::sprintf(msg, "Total movement time: %f", mspf);
		const int h = getWindowHeight();
		m_env->m_textDisplay->outputText( msg, 20, h-100, 0xffffffff);
	}
	m_world->unlock();

	// Step the world
	hkDefaultPhysics2012Demo::stepDemo();

	return hkDemo::DEMO_OK;
}


static const char helpString[] = \
"These demos compare the performance difference between moving phantoms sequentially or all together.\n\n"\
"There are 100 hkpCharacterProxies which have hkpSimpleShape phantoms and 100 hkAAbbPhantoms.\n"\
"If moving sequentially, the broadphase will be updated for each phantom per simulation step.\n"\
"If batch moving, the broadphase will be updated once using a batch of AABBs representing the given phantoms.\n";

HK_DECLARE_DEMO_VARIANT(PhantomBatchMoveDemo, HK_DEMO_TYPE_PHYSICS_2012, "Move Sequentially", 0, "Update broadphase one at a time", helpString );
HK_DECLARE_DEMO_VARIANT(PhantomBatchMoveDemo, HK_DEMO_TYPE_PHYSICS_2012, "Batch Move" , 1, "Update broadphase all at once", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
