/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */
#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Dynamics/MemoryIssues/ResizeSolverBuffer/ResizeSolverBufferDemo.h>

#include <Physics2012/Dynamics/World/hkpWorld.h>
#include <Common/Base/Memory/Allocator/Solver/hkSolverAllocator.h>
#include <Common/Visualize/hkDebugDisplay.h> 

// anonymous namespace for helper functions setting up physics world
namespace 
{
	void createGroundPlane( hkpWorld* world )
	{
		hkVector4 baseSize( 135.0f, 0.5f, 135.0f);
		hkpConvexShape* shape = new hkpBoxShape( baseSize , 0 );
		hkpRigidBodyCinfo ci;
		{
			ci.m_shape = shape;
			ci.m_restitution = 0.5f;
			ci.m_friction = 0.3f;
			ci.m_position.set( 0.0f, -0.5f, 0.0f );
			ci.m_motionType = hkpMotion::MOTION_FIXED;
		}
		world->addEntity( new hkpRigidBody( ci ) )->removeReference();
		shape->removeReference();
	}

	void createWall( hkpWorld* world, hkpShape* shape, hkVector4Parameter position, hkQuaternionParameter rotation )
	{
		hkpRigidBodyCinfo ci;
		{
			ci.m_shape = shape;
			ci.m_restitution = 0.5f;
			ci.m_friction = 0.3f;
			ci.m_position = position;
			ci.m_rotation = rotation;
			ci.m_motionType = hkpMotion::MOTION_FIXED;
		}
		hkpRigidBody* rb = new hkpRigidBody( ci );
		world->addEntity( rb )->removeReference();
		HK_SET_OBJECT_COLOR((hkUlong)rb->getCollidable(), hkColor::rgbFromChars( 192, 192, 192, 125 ) );
	}

	void createWalls( hkpWorld* world, float size, hkVector4Parameter positionOffset )
	{
		hkReal height = size;
		hkReal width = size;
		hkReal thickness = size*0.1f;
		hkVector4 halfWallSize( width*0.5f, height*0.5f, thickness*0.5f ); 
		hkpConvexShape* shape = new hkpBoxShape( halfWallSize );

		{
			hkVector4 pos( 0.0f, height*0.5f, (width+thickness)*0.5f ); pos.add(positionOffset);
			hkQuaternion rot( 0,0,0,1 );
			createWall( world, shape, pos, rot );
		}

		{
			hkVector4 pos( (width+thickness)*0.5f, height*0.5f, 0.0f ); pos.add(positionOffset);
			hkQuaternion rot( hkVector4(0,1,0), HK_REAL_PI*0.5f );
			createWall( world, shape, pos, rot );
		}

		{
			hkVector4 pos( 0.0f, height*0.5f, -(width+thickness)*0.5f ); pos.add(positionOffset);
			hkQuaternion rot( 0,0,0,1 );
			createWall( world, shape, pos, rot );
		}

		{
			hkVector4 pos( -(width+thickness)*0.5f, height*0.5f, 0.0f ); pos.add(positionOffset);
			hkQuaternion rot( hkVector4(0,1,0), HK_REAL_PI*0.5f );
			createWall( world, shape, pos, rot );
		}

		shape->removeReference();

	}


	void createBoxArray( hkpWorld* world, int size, float yPos = 0.0f )
	{
		const hkReal cubeSize  = 1.0f;					// This is the size of the cube side of the box

		const hkReal convexRadius = cubeSize * 0.01f;
		const hkReal gapSize = cubeSize * 0.05f;

		hkReal extendedBoxDim = cubeSize + gapSize;

		hkVector4 startPos( 0.0f , yPos, 0.0f );
		{
			hkVector4 boxRadii(cubeSize *.5f, cubeSize *.5f, cubeSize *.5f);

			hkpShape* boxShape = new hkpBoxShape( boxRadii , convexRadius );

			for(int i=0; i<size; i++)
			{
				// This constructs a row, from left to right
				hkVector4 start(-(size*extendedBoxDim*0.5f), 0,  (i*extendedBoxDim)-(size*extendedBoxDim*0.5f));
				for(int j=0; j< size; j++)
				{
					hkVector4 boxPos(start);
					hkVector4 shift(j * extendedBoxDim, 0.0f, 0.0f);
					boxPos.setAdd4(boxPos, shift);
					boxPos.setAdd4(boxPos, startPos);

					hkpRigidBodyCinfo boxInfo;
					{
						boxInfo.m_mass = 100.0f;
						// calculate the correct inertia
						hkReal d = boxInfo.m_mass * cubeSize * cubeSize / 6.0f;
						boxInfo.m_inertiaTensor.setIdentity();
						boxInfo.m_inertiaTensor.mul(d);

						boxInfo.m_shape = boxShape;
						boxInfo.m_motionType = hkpMotion::MOTION_DYNAMIC;
						boxInfo.m_position = boxPos;
						boxInfo.m_solverDeactivation = hkpRigidBodyCinfo::SOLVER_DEACTIVATION_MAX;
					}

					hkpRigidBody* boxRigidBody = new hkpRigidBody(boxInfo);

					world->addEntity( boxRigidBody );
					boxRigidBody->removeReference();	// Remove reference, since we no longer want to remember this
				}
			}
			boxShape->removeReference();
		}
	}
}


ResizeSolverBufferDemo::ResizeSolverBufferDemo(hkDemoEnvironment* env)
	:	hkDefaultPhysics2012Demo(env), m_originalBuffer(HK_NULL), m_originalBufferSize(0), m_myBuffer(HK_NULL), m_myBufferSize(0)
{
	// Replace the solver buffer
	{
		hkSolverAllocator* solverAllocator = static_cast<hkSolverAllocator*>(&hkMemoryRouter::getInstance().solver());

		// keep track of the original buffer allocated by the base system
		m_originalBuffer = solverAllocator->m_bufferStart;
		m_originalBufferSize = (int)(solverAllocator->m_bufferEnd - solverAllocator->m_bufferStart);

		// create our own buffer - start with a small size
		m_myBufferSize = 1000;
		m_myBuffer = hkMemHeapBlockAlloc<char>( m_myBufferSize );
		solverAllocator->setBuffer( m_myBuffer, m_myBufferSize );
	}

	// Setup the camera
	{
		hkVector4 from(50.0f, 50.0f, 50.0f);
		hkVector4 to  (0.0f, 0.0f, 0.0f);
		hkVector4 up  (0.0f, 1.0f, 0.0f);
		setupDefaultCameras( env, from, to, up );
	}

	// Create the world
	{
		hkpWorldCinfo worldInfo;
		{
			worldInfo.m_gravity.set(-0.0f, -9.81f, -0.0f);
			worldInfo.setBroadPhaseWorldSize(500.0f);
			worldInfo.setupSolverInfo( hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM );
			worldInfo.m_enableDeactivation = false;
		}
		m_world = new hkpWorld(worldInfo);
		m_world->lock();

		// Register ALL agents (though some may not be necessary)
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

		setupGraphics();
	}

	// Setup the physics scene
	{
		createGroundPlane( m_world );
		createWalls( m_world, 12.0f, hkVector4(0.0f,0.5f,0.0f) );
		for (int q = 0; q <= 10; q++)
		{
			createBoxArray(m_world, 10, 1.0f + q * 10.0f);
		}
	}

	m_world->unlock();
}

ResizeSolverBufferDemo::~ResizeSolverBufferDemo()
{
	// Restore old buffer from backup
	if (m_originalBuffer != HK_NULL)
	{
		// This is safe because m_originalBuffer is set after we have determined the allocator is a hkSolverAllocator.
		hkSolverAllocator* solverAllocator = static_cast<hkSolverAllocator*>(&hkMemoryRouter::getInstance().solver());
		solverAllocator->setBuffer(m_originalBuffer, m_originalBufferSize);
	}

	// Deallocate my buffer
	hkMemHeapBlockFree( m_myBuffer, m_myBufferSize );
}


hkDefaultPhysics2012Demo::Result ResizeSolverBufferDemo::stepDemo()
{
	m_world->lock();
	{
		// Get memory figures:
		//   MemUsageInfo::m_maxRuntimeBlockSize is size of memory needed by the largest island, hence minimum amount needed by solver buffer
		//   MemUsageInfo::m_sumRuntimeBlockSize is total memory needed by solver buffer for all simulation islands
		m_world->calcRequiredSolverBufferSize( m_memUsageInfo );

		// Check if we need to resize the solver buffer
		if( m_memUsageInfo.m_maxRuntimeBlockSize > m_myBufferSize )
		{
			int newBufferSize = m_memUsageInfo.m_maxRuntimeBlockSize; // allocate the minimum necessary - there may be some performance hit
			//int newBufferSize = m_memUsageInfo.m_sumRuntimeBlockSize; // alternatively, we could be a bit more generous

			hkprintf("Solver Buffer: freeing %d bytes, allocating %d bytes\n", m_myBufferSize, newBufferSize);

			// free my current buffer
			hkMemHeapBlockFree( m_myBuffer, m_myBufferSize );

			// allocate a new buffer with enough memory to accommodate the solver
			m_myBufferSize = newBufferSize;
			m_myBuffer = hkMemHeapBlockAlloc<char>( m_myBufferSize );

			// set the solver buffer to my new buffer
			hkSolverAllocator* solverAllocator = static_cast<hkSolverAllocator*>(&hkMemoryRouter::getInstance().solver());
			solverAllocator->setBuffer( m_myBuffer, m_myBufferSize );
		}
	}
	m_world->unlock();

	return hkDefaultPhysics2012Demo::stepDemo();
}


static const char helpString[] = "Resize the solver buffer at runtime as the required size grows.";

HK_DECLARE_DEMO( ResizeSolverBufferDemo, HK_DEMO_TYPE_PHYSICS_2012 | HK_DEMO_TYPE_DONT_BOOTSTRAP, "How to maintain a variable-sized solver buffer.", helpString );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
