/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#ifndef HK_BUOYANCY_ACTION_H
#define HK_BUOYANCY_ACTION_H

#include <Physics2012/Dynamics/Action/hkpUnaryAction.h>

class hkpRigidBody;
class hkStepInfo;

class BuoyancyAction: public hkpUnaryAction
{
public:
	HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
	
	/// Environment : base interface store and query informations about the environment.
	struct Environment
	{
		HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,Environment);
						Environment()	:	m_waterDensity(1000)
										,	m_drag(600)
										,	m_enableBuoyancy(true)
										,	m_enableDrag(true)		{}
		virtual			~Environment() {}

		/// Get the plane defining the interface between air and water, assume that negative distances to this plane are under water.
		virtual	void	getWaterPlaneInWorldSpace(const hkVector4& bodyCOM, hkVector4& planeOut) const=0;

		hkReal			m_waterDensity;		///< Water density (about 1000 kg/m^3 for seawater).
		hkReal			m_drag;				///< Aerodynamic drag coefficient.
		hkBool			m_enableBuoyancy;	///< True to compute and apply buoyancy force.
		hkBool			m_enableDrag;		///< True to compute and apply drag forces.
	};

	/// Surface triangle used to for forces integration.
	struct SurfaceTriangle 
	{
		hkVector4	m_plane;			///< Surface plane (pointing outward).
		int			m_vertices[3];		///< Surface vertex indices.
	};

public:

	/// Constructor
										BuoyancyAction(const Environment* environment,hkpRigidBody* body);
		
	/// Apply the action
	void								applyAction(const hkStepInfo& stepInfo );
	
	/// Clone action
	hkpAction*							clone(const hkArray<hkpEntity*>&, const hkArray<hkpPhantom*>&) const { HK_ERROR(0xF894A7BF, "Clonning not implemented"); return HK_NULL; }

	/// Compute the surface area of the triangle abc
	static HK_FORCE_INLINE hkSimdReal	triangleTwiceArea(const hkVector4& a, const hkVector4& b, const hkVector4& c);

	/// Transform a plane in local space
	static HK_FORCE_INLINE void			planeToLocal(const hkTransform& transform, const hkVector4& plane, hkVector4& planeOut);

	/// Transform a plane in world space
	static HK_FORCE_INLINE void			planeToWorld(const hkTransform& transform, const hkVector4& plane, hkVector4& planeOut);

public:
	const Environment*			m_env;			///< Environment back-pointer.
	hkReal						m_bodyDensity;	///< Density of the rigid body (automaticaly computed from mass and volume).
	hkArray<hkVector4>			m_vertices;		///< Vertices of the surface.
	hkArray<SurfaceTriangle>	m_triangles;	///< Triangles of the surface.
};

//
// Inline's
//

//
HK_FORCE_INLINE hkSimdReal	BuoyancyAction::triangleTwiceArea(const hkVector4& a, const hkVector4& b, const hkVector4& c)
{
	hkVector4	ba,ca;
	ba.setSub(b,a);
	ca.setSub(c,a);
	hkVector4	cross;
	cross.setCross(ba,ca);
	return cross.length<3>();
}

//
HK_FORCE_INLINE void		BuoyancyAction::planeToLocal(const hkTransform& transform, const hkVector4& plane, hkVector4& planeOut)
{
	planeOut._setRotatedInverseDir(transform.getRotation(),plane);
	planeOut.setW( plane.getW() + plane.dot<3>(transform.getTranslation()) );
}

//
HK_FORCE_INLINE void		BuoyancyAction::planeToWorld(const hkTransform& transform, const hkVector4& plane, hkVector4& planeOut)
{
	planeOut._setRotatedDir(transform.getRotation(),plane);
	planeOut.setW( plane.getW() - planeOut.dot<3>(transform.getTranslation()) );
}

#endif // HK_BUOYANCY_ACTION_H

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
