/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Dynamics/Actions/BuoyancyAction/BuoyancyAction.h>

#include <Common/Internal/ConvexHull/hkGeometryUtility.h>
#include <Common/GeometryUtilities/Misc/hkGeometryUtils.h>
#include <Common/Visualize/Shape/hkDisplayGeometry.h>

#include <Physics2012/Utilities/VisualDebugger/Viewer/hkpShapeDisplayBuilder.h>
#include <Physics2012/Utilities/Dynamics/ImpulseAccumulator/hkpImpulseAccumulator.h>
#include <Physics2012/Utilities/Dynamics/Inertia/hkpInertiaTensorComputer.h>

BuoyancyAction::BuoyancyAction(const Environment* environment,hkpRigidBody* body)
	:	hkpUnaryAction(body)	
	,	m_env(environment)
{
	// Extract density from shape.
	hkMassProperties	massProperties;
	hkpInertiaTensorComputer::computeShapeVolumeMassProperties(body->getCollidable()->getShape(),1,massProperties);
	m_bodyDensity	=	body->getMass()/massProperties.m_volume;

	// Create surface geometry from shape.
	hkpShapeDisplayBuilder::hkpShapeDisplayBuilderEnvironment	env;
	hkpShapeDisplayBuilder										sdb(env);
	hkArray<hkDisplayGeometry*>									displayGeometries;
	sdb.buildDisplayGeometries(body->getCollidable()->getShape(),displayGeometries);
	for(int i=0;i<displayGeometries.getSize();++i)
	{
		// Extract and weld each geometry.
		displayGeometries[i]->buildGeometry();
		hkGeometry*	geom=displayGeometries[i]->getGeometry();
		const hkTransform& transform = displayGeometries[i]->getTransform();
		hkGeometryUtils::weldVertices(*geom);

		// Add vertices.
		const int	indexOffset=m_vertices.getSize();
		m_vertices.append(geom->m_vertices.begin(), geom->m_vertices.getSize());
		
		// Add each triangle to the probing surfaces.
		m_triangles.reserveExactly(m_triangles.getSize()+geom->m_triangles.getSize()+1);
		for(int j=0;j<geom->m_triangles.getSize();++j)
		{
			SurfaceTriangle&	st	=	m_triangles.expandOne();
 			const hkVector4&	a	=	geom->m_vertices[geom->m_triangles[j].m_a];
 			const hkVector4&	b	=	geom->m_vertices[geom->m_triangles[j].m_b];
 			const hkVector4&	c	=	geom->m_vertices[geom->m_triangles[j].m_c];
			st.m_vertices[0]	=	geom->m_triangles[j].m_a+indexOffset;
			st.m_vertices[1]	=	geom->m_triangles[j].m_b+indexOffset;
			st.m_vertices[2]	=	geom->m_triangles[j].m_c+indexOffset;
			hkVector4			ba;	ba.setSub4(b,a);
			hkVector4			ca; ca.setSub4(c,a);
			st.m_plane.setCross(ba,ca);
			if(HK_SUCCESS == st.m_plane.normalize3IfNotZero())
			{
				const hkReal dist = -st.m_plane.dot3(a);
				// This is algebraically equivalent to transforming the vertices first.
				st.m_plane.setRotatedDir( transform.getRotation(), st.m_plane );
				st.m_plane(3)	=	dist;
			}
			else
			{
				m_triangles.popBack();
			}
		}
		delete displayGeometries[i];
	}
}

//
void BuoyancyAction::applyAction( const hkStepInfo& stepInfo )
{
	hkpRigidBody*			body				=	getRigidBody();
	const hkTransform&		transform			=	body->getTransform();
	
	// Declare and initalize an impulse accumulator.
	hkpImpulseAccumulator	impulseAcc(body,stepInfo.m_deltaTime);

	// SIMD constants.
	const hkSimdReal		bodyDensityCst		=	m_bodyDensity;
	const hkSimdReal		waterDensityCst		=	m_env->m_waterDensity;

	const hkSimdReal		epsilonCst			=	hkSimdReal_Eps;
	const hkSimdReal		oneThirdCst			=	hkSimdReal_Inv3;
	const hkSimdReal		oneFourthCst		=	hkSimdReal_Inv4;
	const hkSimdReal		oneSixthCst			=	hkSimdReal_Inv6;

	// Allow the environment to choose a water plane on a per body center of mass basis.
	hkVector4		worldWaterPlane;
	m_env->getWaterPlaneInWorldSpace(body->getCenterOfMassInWorld(), worldWaterPlane);
	
	// We work mostly in local space, so lets transform the water plane.		
	hkVector4		localWaterPlane;
	planeToLocal(transform, worldWaterPlane, localWaterPlane);

	// Pick an origin that allow us to skip tetrahedrons made of clipped vertices.
	hkVector4		origin;
	origin.setAddMul(body->getCenterOfMassLocal(), localWaterPlane, -localWaterPlane.dot4xyz1(body->getCenterOfMassLocal()));

	// Store signed distance to clipping plane in vertices W component.	
	{
		#define COMPUTE_D2P_IN_W(_index_)	stream[_index_].setW(localWaterPlane.dot4xyz1(stream[_index_]))
		const unsigned		numVertices = m_vertices.getSize();
		hkVector4*			stream = m_vertices.begin();
		unsigned			index = 3;
		for(;index<numVertices;index+=4)		{ COMPUTE_D2P_IN_W(index-3); COMPUTE_D2P_IN_W(index-2); COMPUTE_D2P_IN_W(index-1); COMPUTE_D2P_IN_W(index-0); }
		for(index-=3;index<numVertices;++index) { COMPUTE_D2P_IN_W(index); }
		#undef COMPUTE_D2P_IN_W
	}
	
	// Clip every surface triangle against the water plane and apply forces.
	hkSimdReal		volume; volume.setZero();
	hkVector4		com; com.setZero();
	for(int i=0;i<m_triangles.getSize();++i)
	{
		const SurfaceTriangle&	tri(m_triangles[i]);
		const hkVector4*		vtx[]={&m_vertices[tri.m_vertices[0]], &m_vertices[tri.m_vertices[1]], &m_vertices[tri.m_vertices[2]]};
		const hkSimdReal		distanceToPlanes[3]={vtx[0]->getW(),vtx[1]->getW(),vtx[2]->getW()};
		const hkBool32			isClipped = distanceToPlanes[0].isLessZero() |
											distanceToPlanes[1].isLessZero() |
											distanceToPlanes[2].isLessZero() ;
		if(isClipped)
		{
			// Generate under-water vertices by clipping edge against the water plane.
			hkVector4	clipped[4];
			int			numVertices=0;
			for(int j=2,k=0;k<3;j=k++)
			{
				const hkSimdReal	dj	=	distanceToPlanes[j];
				const hkSimdReal	dk	=	distanceToPlanes[k];
				if((dj*dk+epsilonCst).isLessZero())
				{
					clipped[numVertices].setInterpolate(*vtx[j],*vtx[k],dj/(dj-dk));
					clipped[numVertices++].sub(origin);
				}
				if(dk.isLessZero()) clipped[numVertices++].setSub(*vtx[k],origin);
			}
			
			// Compute sub-merged volume and apply drag force.
			for(int j=2;j<numVertices;++j)
			{
				const hkVector4&	a=clipped[0];
				const hkVector4&	b=clipped[j-1];
				const hkVector4&	c=clipped[j];
				hkVector4			ctr; ctr.setCross(a,b);
				hkSimdReal			vol6=ctr.dot<3>(c);
				volume	=	volume + vol6;
				ctr.setAdd(a,b);
				ctr.add(c);
				com.addMul(vol6,ctr);

				// Apply drag force
				if(m_env->m_enableDrag)
				{
					ctr.mul(oneThirdCst);
					ctr.add(origin);
					ctr._setTransformedPos(transform,ctr);
					
					hkVector4	worldPlane;
					planeToWorld(transform,tri.m_plane,worldPlane);
					
					hkVector4	worldVelocity;
					body->getPointVelocity(ctr,worldVelocity);
					
					hkSimdReal	velocityWrtWaterPlaneNormal = -worldVelocity.dot<3>(worldPlane);
					if(velocityWrtWaterPlaneNormal.isLessZero())
					{
						hkSimdReal	cosAngle		=	velocityWrtWaterPlaneNormal * worldVelocity.lengthInverse<3>();
						hkSimdReal	velocitySquared	=	worldVelocity.lengthSquared<3>();
						hkSimdReal	drag			=	m_env->m_drag;
						hkSimdReal	halfArea		=	triangleTwiceArea(a,b,c)*oneFourthCst;
						hkVector4	force;
						force.setMul(drag * halfArea * velocitySquared * cosAngle, worldPlane);
						impulseAcc.addForce(ctr,force);
					}
				}
			}
		}
	}	
	
	// Apply buoyancy force
	if(m_env->m_enableBuoyancy && volume.isGreaterZero())
	{			
		// Finalize COM and volume
		com.mul(oneFourthCst/volume);
		com.add(origin);
		com._setTransformedPos(transform,com);
		
		// Calculate and apply buoyancy force
		hkVector4	force;
		hkSimdReal	acceleration	=	body->getWorld()->getGravity().length<3>();
		hkSimdReal	pressureGrad	=	waterDensityCst / bodyDensityCst;
		force.setMul( volume * oneSixthCst * bodyDensityCst * acceleration * pressureGrad, worldWaterPlane);
		impulseAcc.addForce(com,force);
	}
	
	// Flush accumulated forces
	impulseAcc.flushImpulses();
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
