/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>

#include <Graphics/Common/Window/hkgWindow.h>

#include <Demos/Physics2012/Api/Dynamics/Actions/BuoyancyAction/Buoyancy.h>

#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Base/Types/Geometry/hkStridedVertices.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>

#include <Common/Visualize/hkDebugDisplay.h>

#include <Physics2012/Collide/Shape/Convex/Cylinder/hkpCylinderShape.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexVertices/hkpConvexVerticesShape.h>

// Implement our own environment
struct	MyEnvironment : public BuoyancyAction::Environment
{
	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,MyEnvironment);

	// Constructor
	MyEnvironment(BuoyancyDemo* demo) : m_demo(demo) {}

	// The demo implement a simple wave function to animate the water surface.
	void		getWaterPlaneInWorldSpace(const hkVector4& bodyCOM, hkVector4& planeOut) const
	{
		m_demo->evaluateWaveFunctionDerivative(bodyCOM, planeOut);
	}

	BuoyancyDemo*	m_demo;
};


static hkVector4 calculateBodyPosition(int index)
{
	const hkReal xPos[4] = {-4.0f, 4.0f, 4.0f, -4.0f};
	const hkReal yPos[4] = {-4.0f, -4.0f, 4.0f, 4.0f};
	return hkVector4(xPos[index%4], (1+index/4)*9.0f, yPos[index%4]);
}

BuoyancyDemo::BuoyancyDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env)
{	
	// Setup the camera.
	hkVector4 from(-1.540f, 12.895f, 61.078f);
	hkVector4 to(-1.549f, 11.998f, 57.053f);
	hkVector4 up(0.0f, 4.0f, 0.0f);
	setupDefaultCameras(env, from, to, up);
	
	// Create the world
	hkpWorldCinfo info;
	info.m_gravity.set(0.0f, -10.0f, 0.0f);	
	info.setBroadPhaseWorldSize( 1000.0f );
	info.m_simulationType  = info.SIMULATION_TYPE_CONTINUOUS;
	m_world = new hkpWorld(info);
	m_world->lock();

	m_timeAccumulator	=	0;

	// Set environment
	m_environment					=	new MyEnvironment(this);
	m_environment->m_enableDrag		=	true;
	m_environment->m_enableBuoyancy	=	true;
	m_environment->m_waterDensity	=	1000;
	m_environment->m_drag			=	600;

	hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );
	
	setupGraphics();
	forceShadowState(false);

	hkpRigidBodyCinfo rigidBodyInfo;
	rigidBodyInfo.m_angularDamping		=	0.1f;
	rigidBodyInfo.m_linearDamping		=	0;
	rigidBodyInfo.m_restitution			=	0;
	rigidBodyInfo.m_motionType			=	hkpMotion::MOTION_BOX_INERTIA;
	rigidBodyInfo.m_enableDeactivation	=	false;	

	//
	// Add random rigid bodies and attach actions
	//
	hkPseudoRandomGenerator	rnd(180674);
	const int				numConvexVerticesShapes	=	32;
	const int				numConvexCylinderShapes	=	4;	
	const int				numConvexCapsuleShapes	=	4;
	const hkSimdReal		shapesDensity			=	hkSimdReal::fromFloat(600.0f);	// Oak like.
	const hkReal			convexRadius			=	0.01f;
	int						shapeIndex				=	0;	
	
	// Add random convex vertices shapes
	
	for(int iShape=0;iShape<numConvexVerticesShapes;++iShape)
	{		
		const int	numVertices = 16;
		hkVector4	scale(2,1,3,0);
		rnd.getRandomVector01(scale);
		scale.mul4(scale);
		scale.mul4(4);
		scale.add4(hkVector4(0.5f,0.5f,0.5f));
	
		hkArray<hkVector4>	vertices;
		for(int i=0;i<numVertices;++i)
		{
			rnd.getRandomVector11(vertices.expandOne());
			vertices.back().mul4(scale);
		}

		hkpConvexVerticesShape::BuildConfig	config;
		config.m_convexRadius		=	convexRadius;
		hkpShape* shape = new hkpConvexVerticesShape(vertices,config);

		rigidBodyInfo.m_position = calculateBodyPosition(++shapeIndex);
		rnd.getRandomRotation(rigidBodyInfo.m_rotation);
		rigidBodyInfo.m_shape		=	shape;
		
		hkMassProperties	massProperties;
		hkpInertiaTensorComputer::computeShapeVolumeMassProperties(shape,1,massProperties);
		massProperties.scaleToDensity(shapesDensity);
		rigidBodyInfo.setMassProperties(massProperties);
				
		hkpRigidBody* rigidBody = new hkpRigidBody(rigidBodyInfo);

		shape->removeReference();

		m_world->addEntity(rigidBody);
		rigidBody->removeReference();

		hkpAction* action=new BuoyancyAction(m_environment,rigidBody);
		m_world->addAction(action);
		action->removeReference();
	}

	// Add random cylinder shapes
	for(int iShape=0;iShape<numConvexCylinderShapes;++iShape)
	{		
		hkVector4	endPoints[2];
		rnd.getRandomVector11(endPoints[0]);
		rnd.getRandomVector11(endPoints[1]);
		if(iShape&1)
		{
			endPoints[0].mul4(4);
			endPoints[1].mul4(4);
		}
		else
		{
			endPoints[0].mul4(0.5f);
			endPoints[1].mul4(0.5f);
		}
		hkpShape* shape = new hkpCylinderShape(endPoints[0],endPoints[1],rnd.getRandRange(0.25,2),convexRadius);

		rigidBodyInfo.m_position = calculateBodyPosition(++shapeIndex);
		rigidBodyInfo.m_shape		=	shape;		

		hkMassProperties	massProperties;
		hkpInertiaTensorComputer::computeShapeVolumeMassProperties(shape,1,massProperties);
		massProperties.scaleToDensity(shapesDensity);
		rigidBodyInfo.setMassProperties(massProperties);

		hkpRigidBody* rigidBody = new hkpRigidBody(rigidBodyInfo);

		shape->removeReference();

		m_world->addEntity(rigidBody);
		rigidBody->removeReference();

		hkpAction* action=new BuoyancyAction(m_environment,rigidBody);
		m_world->addAction(action);
		action->removeReference();
	}

	// Add random capsule shapes
	for(int iShape=0;iShape<numConvexCapsuleShapes;++iShape)
	{		
		hkVector4	endPoints[2];
		rnd.getRandomVector11(endPoints[0]);endPoints[0].mul4(5);
		rnd.getRandomVector11(endPoints[1]);endPoints[1].mul4(5);
		hkpShape* shape = new hkpCapsuleShape(endPoints[0],endPoints[1],rnd.getRandRange(0.25,2)+convexRadius);

		rigidBodyInfo.m_position = calculateBodyPosition(++shapeIndex);
		rigidBodyInfo.m_shape		=	shape;
		
		hkMassProperties	massProperties;
		hkpInertiaTensorComputer::computeShapeVolumeMassProperties(shape,1,massProperties);
		massProperties.scaleToDensity(shapesDensity);
		rigidBodyInfo.setMassProperties(massProperties);

		hkpRigidBody* rigidBody = new hkpRigidBody(rigidBodyInfo);

		shape->removeReference();

		m_world->addEntity(rigidBody);
		rigidBody->removeReference();

		
		hkpAction* action=new BuoyancyAction(m_environment,rigidBody);
		m_world->addAction(action);
		action->removeReference();
	}
	
	m_world->unlock();
}

//
				BuoyancyDemo::~BuoyancyDemo()
{
	delete m_environment;
}

//
hkReal			BuoyancyDemo::evaluateWaveFunction(const hkVector4& xz) const
{
	const hkReal	x = xz(0) + m_timeAccumulator*8;
	const hkReal	y = xz(2) + m_timeAccumulator*4;
	const hkReal	s0 = 10;
	const hkReal	s1 = -20;
	const hkReal	sx = hkMath::sin(x/s0);
	const hkReal	cx = hkMath::cos(y/s1);
	const hkReal	sh = hkMath::cos(x/s1);
	return sx*cx+sh;
}

//
void			BuoyancyDemo::evaluateWaveFunctionDerivative(const hkVector4& xz, hkVector4& planeOut) const
{
	hkReal			h = 0.01f;
	hkVector4		x = xz; x(0) = x(0)+h;
	hkVector4		z = xz; z(2) = z(2)+h;
	const hkReal	o = evaluateWaveFunction(xz);
	const hkReal	dx = (evaluateWaveFunction(x)-o)/h;
	const hkReal	dz = (evaluateWaveFunction(z)-o)/h;
	x.set(1,dx,0);
	z.set(0,dz,1);
	planeOut.setCross(z,x);
	planeOut.normalize3();
	x=xz; x(1)=o;
	planeOut(3) = -planeOut.dot3(x);
}

static inline hkReal _getFresnelReflectance(hkVector4Parameter incident, hkVector4Parameter normal, float eta)
{
    // Schlick approximation
    hkReal f0 = (1.f-eta) / (1.f+eta);
    f0 *= f0;
    hkReal t = 1.f - hkMath::fabs( incident.dot3(normal) );
    hkReal tt = t * t;
    hkReal ttttt = tt * tt * t;
    return f0 + ttttt * (1.f-f0);
}

//
void			BuoyancyDemo::recurseDrawWater(const hkVector4& a,const hkVector4& b,const hkVector4& c, int depth) const
{
	if(depth>0)
	{
		hkVector4	m; m.setInterpolate4(a,b,0.5f);
		m(1) = evaluateWaveFunction(m);
		recurseDrawWater(b,c,m,depth-1);
		recurseDrawWater(c,a,m,depth-1);
	}
	else
	{
		hkVector4 ab; ab.setSub4(b, a);
		hkVector4 ac; ac.setSub4(c, a);
		hkVector4 normal; normal.setCross(ab, ac);
		if (normal.normalize3IfNotZero() == HK_SUCCESS)
		{
			hkVector4 centroid;
			centroid.setMul4(1/3.f, a);
			centroid.addMul4(1/3.f, b);
			centroid.addMul4(1/3.f, c);

			hkVector4 incident;
			m_env->m_window->getCurrentViewport()->getCamera()->getFrom(incident);
			incident.sub4(centroid);
			incident.normalize3();

			hkReal fr = _getFresnelReflectance( incident, normal, 1.f/1.34f );
			hkVector4 col; col.setInterpolate4( hkVector4(0.f,0.2f,0.3f), hkVector4(0.69f,0.84f,1.f), fr );
			HK_DISPLAY_TRIANGLE( a, b, c, hkColor::rgbFromFloats(col(0), col(1), col(2), 0.8f) );
		}
	}
}

//
void			BuoyancyDemo::drawWater(const hkVector4& a,const hkVector4& b,const hkVector4& c, int depth) const
{
	hkVector4	v[]={a,b,c};
	v[0](1) = evaluateWaveFunction(v[0]);
	v[1](1) = evaluateWaveFunction(v[1]);
	v[2](1) = evaluateWaveFunction(v[2]);
	recurseDrawWater(v[0],v[1],v[2],depth);
}

//
hkDemo::Result	BuoyancyDemo::stepDemo()
{
	// Handle keys.
	if(m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_NUMPAD1))	m_environment->m_enableBuoyancy=!m_environment->m_enableBuoyancy;
	if(m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_NUMPAD2))	m_environment->m_enableDrag=!m_environment->m_enableDrag;

	if(m_env->m_window->getKeyboard().getKeyState(HKG_VKEY_NUMPAD4))	m_environment->m_waterDensity-=10.f;
	if(m_env->m_window->getKeyboard().getKeyState(HKG_VKEY_NUMPAD5))	m_environment->m_waterDensity+=10.f;

	m_environment->m_waterDensity	=	hkMath::clamp(m_environment->m_waterDensity,0,10000);

	if(m_env->m_window->getKeyboard().getKeyState(HKG_VKEY_NUMPAD7))	m_environment->m_drag-=1.f;
	if(m_env->m_window->getKeyboard().getKeyState(HKG_VKEY_NUMPAD8))	m_environment->m_drag+=1.f;

	m_environment->m_drag	=	hkMath::clamp(m_environment->m_drag,0,1000);
	
	// Output infos.
	hkStringBuf	str;
	str.printf("Num pad : [1] Buoyancy(%s)   [4,5] Water density(%f)      [2(7,8)] Drag(%s,%f)",m_environment->m_enableBuoyancy?"ON":"OFF",m_environment->m_waterDensity,m_environment->m_enableDrag?"ON":"OFF",m_environment->m_drag);
	m_env->m_textDisplay->outputText(str.cString(),10,(int)getWindowHeight()-24);

	// Animate water plane.
	m_timeAccumulator	+=	m_timestep;
	
	// Draw water plane.
	int				resolution = 10;
	
	drawWater(hkVector4(-100,0,-100,0),hkVector4(+100,0,+100,0),hkVector4(+100,0,-100,0),resolution);
	drawWater(hkVector4(+100,0,+100,0),hkVector4(-100,0,-100,0),hkVector4(-100,0,+100,0),resolution);

	return hkDefaultPhysics2012Demo::stepDemo();
}



static const char helpString[] = \
"Buoyancy and drag forces on convex shapes";

HK_DECLARE_DEMO(BuoyancyDemo, HK_DEMO_TYPE_TEST, "Buoyancy and drag", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
